/**
 * FreeRDP: A Remote Desktop Protocol Implementation
 * Input Interface API
 *
 * Copyright 2011 Marc-Andre Moreau <marcandre.moreau@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef FREERDP_INPUT_H
#define FREERDP_INPUT_H

#include <freerdp/api.h>
#include <freerdp/types.h>
#include <freerdp/scancode.h>

#include <winpr/crt.h>
#include <winpr/collections.h>

/* keyboard Flags */
#define KBD_FLAGS_EXTENDED 0x0100
#define KBD_FLAGS_EXTENDED1 0x0200
#define KBD_FLAGS_DOWN \
	0x4000 /**< Presence of this flag indicates the key was already down previously */
#define KBD_FLAGS_RELEASE \
	0x8000 /**< Presence of this flag indicates a key was released. Absence a key press */

/* Pointer Flags */
#define PTR_FLAGS_HWHEEL 0x0400
#define PTR_FLAGS_WHEEL 0x0200
#define PTR_FLAGS_WHEEL_NEGATIVE 0x0100
#define PTR_FLAGS_MOVE 0x0800
#define PTR_FLAGS_DOWN 0x8000
#define PTR_FLAGS_BUTTON1 0x1000 /* left */
#define PTR_FLAGS_BUTTON2 0x2000 /* right */
#define PTR_FLAGS_BUTTON3 0x4000 /* middle */
#define WheelRotationMask 0x01FF

/* Extended Pointer Flags */
#define PTR_XFLAGS_DOWN 0x8000
#define PTR_XFLAGS_BUTTON1 0x0001
#define PTR_XFLAGS_BUTTON2 0x0002

/* Keyboard Toggle Flags */
enum KBD_SYNC_FLAGS
{
	KBD_SYNC_SCROLL_LOCK = 0x00000001,
	KBD_SYNC_NUM_LOCK = 0x00000002,
	KBD_SYNC_CAPS_LOCK = 0x00000004,
	KBD_SYNC_KANA_LOCK = 0x00000008
};

#define RDP_CLIENT_INPUT_PDU_HEADER_LENGTH 4

#ifdef __cplusplus
extern "C"
{
#endif

	typedef struct rdp_input rdpInput;

	/* defined inside libfreerdp-core */
	typedef struct rdp_input_proxy rdpInputProxy;

	/* Input Interface */

	typedef BOOL (*pSynchronizeEvent)(rdpInput* input, UINT32 flags);
	typedef BOOL (*pKeyboardEvent)(rdpInput* input, UINT16 flags, UINT8 code);
	typedef BOOL (*pUnicodeKeyboardEvent)(rdpInput* input, UINT16 flags, UINT16 code);
	typedef BOOL (*pMouseEvent)(rdpInput* input, UINT16 flags, UINT16 x, UINT16 y);
	typedef BOOL (*pRelMouseEvent)(rdpInput* input, UINT16 flags, INT16 xDelta, INT16 yDelta);
	typedef BOOL (*pQoEEvent)(rdpInput* input, UINT32 timestampMS);
	typedef BOOL (*pExtendedMouseEvent)(rdpInput* input, UINT16 flags, UINT16 x, UINT16 y);
	typedef BOOL (*pFocusInEvent)(rdpInput* input, UINT16 toggleStates);
	typedef BOOL (*pKeyboardPauseEvent)(rdpInput* input);

	struct rdp_input
	{
		rdpContext* context;     /* 0 */
		void* param1;            /* 1 */
		UINT32 paddingA[16 - 2]; /* 2 */

		WINPR_ATTR_NODISCARD pSynchronizeEvent SynchronizeEvent;         /* 16 */
		WINPR_ATTR_NODISCARD pKeyboardEvent KeyboardEvent;               /* 17 */
		WINPR_ATTR_NODISCARD pUnicodeKeyboardEvent UnicodeKeyboardEvent; /* 18 */
		WINPR_ATTR_NODISCARD pMouseEvent MouseEvent;                     /* 19 */
		WINPR_ATTR_NODISCARD pExtendedMouseEvent ExtendedMouseEvent;     /* 20 */
		WINPR_ATTR_NODISCARD pFocusInEvent FocusInEvent;                 /*21 */
		WINPR_ATTR_NODISCARD pKeyboardPauseEvent KeyboardPauseEvent;     /* 22 */
		WINPR_ATTR_NODISCARD pRelMouseEvent RelMouseEvent;               /* 23 */
		WINPR_ATTR_NODISCARD pQoEEvent QoEEvent;                         /* 24 */

		UINT32 paddingB[32 - 25]; /* 25 */
	};

	FREERDP_API BOOL freerdp_input_send_synchronize_event(rdpInput* input, UINT32 flags);

	FREERDP_API BOOL freerdp_input_send_keyboard_event(rdpInput* input, UINT16 flags, UINT8 code);

	FREERDP_API BOOL freerdp_input_send_keyboard_event_ex(rdpInput* input, BOOL down, BOOL repeat,
	                                                      UINT32 rdp_scancode);

	FREERDP_API BOOL freerdp_input_send_keyboard_pause_event(rdpInput* input);

	FREERDP_API BOOL freerdp_input_send_unicode_keyboard_event(rdpInput* input, UINT16 flags,
	                                                           UINT16 code);

	FREERDP_API BOOL freerdp_input_send_mouse_event(rdpInput* input, UINT16 flags, UINT16 x,
	                                                UINT16 y);

	FREERDP_API BOOL freerdp_input_send_rel_mouse_event(rdpInput* input, UINT16 flags, INT16 xDelta,
	                                                    INT16 yDelta);

	FREERDP_API BOOL freerdp_input_send_qoe_timestamp(rdpInput* input, UINT32 timestampMS);

	FREERDP_API BOOL freerdp_input_send_extended_mouse_event(rdpInput* input, UINT16 flags,
	                                                         UINT16 x, UINT16 y);

	FREERDP_API BOOL freerdp_input_send_focus_in_event(rdpInput* input, UINT16 toggleStates);

	/** @brief stringify \b enum KBD_SYNC_FLAGS
	 *
	 *  @param flags \b enum KBD_SYNC_FLAGS flags
	 *  @param buffer A buffer to store the resulting string
	 *  @param len The length of the buffer in bytes
	 *
	 *  @return A pointer to \ref buffer or \b NULL in case of failure
	 *  @since version 3.19.0
	 */
	WINPR_ATTR_NODISCARD
	FREERDP_API const char* freerdp_input_keyboard_flags_string(uint32_t flags, char* buffer,
	                                                            size_t len);

#ifdef __cplusplus
}
#endif

#endif /* FREERDP_INPUT_H */
