/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.artifacts.ivyservice.resolveengine.strict;

import com.google.common.collect.ImmutableSet;
import org.gradle.api.artifacts.ModuleIdentifier;
import org.jspecify.annotations.NullMarked;

import java.util.Set;

@NullMarked
public class StrictVersionConstraints {

    public static final StrictVersionConstraints EMPTY = new StrictVersionConstraints(ImmutableSet.of()) {
        @Override
        public StrictVersionConstraints union(StrictVersionConstraints other) {
            return other;
        }

        @Override
        public StrictVersionConstraints intersect(StrictVersionConstraints other) {
            return EMPTY;
        }

        @Override
        public StrictVersionConstraints minus(StrictVersionConstraints other) {
            return EMPTY;
        }

        @Override
        public boolean isEmpty() {
            return true;
        }

        @Override
        public boolean contains(ModuleIdentifier module) {
            return false;
        }

        @Override
        public String toString() {
            return "no modules";
        }
    };

    private final ImmutableSet<ModuleIdentifier> modules;

    private StrictVersionConstraints(ImmutableSet<ModuleIdentifier> modules) {
        this.modules = modules;
    }

    public static StrictVersionConstraints of(ImmutableSet<ModuleIdentifier> modules) {
        if (modules.isEmpty()) {
            return EMPTY;
        }
        return new StrictVersionConstraints(modules);
    }

    public ImmutableSet<ModuleIdentifier> getModules() {
        return modules;
    }

    public boolean isEmpty() {
        return this == EMPTY;
    }

    public boolean contains(ModuleIdentifier module) {
        return modules.contains(module);
    }

    public StrictVersionConstraints union(StrictVersionConstraints other) {
        if (other == EMPTY) {
            return this;
        }
        if (this == other) {
            // this happens quite a lot!
            return this;
        }
        if (this.modules.equals(other.modules)) {
            return this;
        }
        ImmutableSet.Builder<ModuleIdentifier> builder = ImmutableSet.builderWithExpectedSize(modules.size() + other.modules.size());
        builder.addAll(modules);
        builder.addAll(other.modules);
        return of(builder.build());
    }

    public StrictVersionConstraints intersect(StrictVersionConstraints other) {
        if (other.modules == modules) {
            return this;
        }
        if (other == EMPTY) {
            return EMPTY;
        }

        Set<ModuleIdentifier> smaller = (modules.size() < other.modules.size()) ? modules : other.modules;
        Set<ModuleIdentifier> larger = (smaller == modules) ? other.modules : modules;
        ImmutableSet.Builder<ModuleIdentifier> builder = ImmutableSet.builderWithExpectedSize(smaller.size());

        // Iterating over the smaller set to minimize the number of contains() checks
        for (ModuleIdentifier module : smaller) {
            if (larger.contains(module)) {
                builder.add(module);
            }
        }

        return of(builder.build());
    }

    @Override
    public String toString() {
        return "modules=" + modules;
    }

    public StrictVersionConstraints minus(StrictVersionConstraints other) {
        if (other == EMPTY) {
            return this;
        }

        if (this == other || this == EMPTY) {
            return EMPTY;
        }

        ImmutableSet.Builder<ModuleIdentifier> builder = ImmutableSet.builderWithExpectedSize(modules.size());
        for (ModuleIdentifier module : modules) {
            if (!other.modules.contains(module)) {
                builder.add(module);
            }
        }
        return of(builder.build());
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        StrictVersionConstraints that = (StrictVersionConstraints) o;
        return modules.equals(that.modules);
    }

    @Override
    public int hashCode() {
        return modules.hashCode();
    }

}
