/*  Part of SWI-Prolog

    Author:        Jan Wielemaker
    E-mail:        J.Wielemaker@vu.nl
    WWW:           www.swi-prolog.org
    Copyright (c)  2009, University of Amsterdam
                         VU University Amsterdam
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in
       the documentation and/or other materials provided with the
       distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
    FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
    COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
    LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
    CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
    ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
    POSSIBILITY OF SUCH DAMAGE.
*/

:- module(test_record,
	  [ test_record/0
	  ]).
:- use_module(library(plunit)).

/** <module> Test set for library(record)


@tbd	A lot more tests.  In particular test and precise the relation
	between records and modules.
*/

test_record :-
	run_tests([ record
		  ]).

:- begin_tests(record).
:- use_module(library(record)).

:- record foo(x:integer).
:- record bar(x:foo).
:- record r(x:list(integer)).
:- record zot(x).

test(record_type, true) :-
	make_foo([x(10)], Foo),
	make_bar([x(Foo)], Bar),
	is_bar(Bar).
test(record_type, foo(1) == Foo) :-
	default_foo(Foo),
	foo_x(Foo, 1),
	is_foo(Foo).
test(record_type, fail) :-
	default_foo(Foo),
	is_foo(Foo).
test(list_type, true) :-
	make_r([x([1,2,3])], _).
test(list_type, error(type_error(_,_))) :-
	make_r([x([1,2,a])], _).
test(record_no_type, zot(_) =@= Zot) :-
	make_zot([], Zot),
	is_zot(Zot).

% Example from the documentation

:- record point(x:integer=0, y:integer=0).

test(point, R == point(1,2)) :-
    make_point([y(2), x(1)], R),
    is_point(R),
    point_x(R, X), X==1,
    point_y(R, Y), Y==2.
test(point, R == point(0,2)) :-
    make_point([y(2)], R),
    is_point(R),
    point_x(R, X), X==0,
    point_y(R, Y), Y==2.
test(point, fail) :-
    is_point(_R).
test(point, true) :-
    is_point(point(1,2)).
test(point, fail) :-
    is_point(point(_,2)).
test(point, error(instantiation_error)) :-
    make_point([x(1),y(_)], _R).
test(point, error(type_error(integer,foo))) :-
    make_point([x(1),y(foo)], _R).
test(point, fail) :-
    make_point([x(1),z(3)], _R).

test(point, P-R == point(1,10)-[z(123)]) :-
    make_point([y(0), x(1), y(10), z(123)], P, R).
test(point, P-R == point(1,10)-[z(123)]) :-
    make_point([z(123), y(0), y(10), x(1)], P, R).

test(point, M:T == plunit_record:point(x:integer=0,y:integer=0)) :-
    current_record(point, M:T).
test(point, set(M:T == [plunit_record:point(x:integer=0,y:integer=0)])) :-
    current_record(N, M:T),
    N = point.

% No need for testing most of the remainder of the predicates bbecause
% they're used in implementing the predicates tested above.

:- end_tests(record).
