/*
   Bacula(R) - The Network Backup Solution

   Copyright (C) 2000-2025 Kern Sibbald

   The original author of Bacula is Kern Sibbald, with contributions
   from many others, a complete list can be found in the file AUTHORS.

   You may use this file and others of this release according to the
   license defined in the LICENSE file, which includes the Affero General
   Public License, v3.0 ("AGPLv3") and some additional permissions and
   terms pursuant to its AGPLv3 Section 7.

   This notice must be preserved when any source code is
   conveyed and/or propagated.

   Bacula(R) is a registered trademark of Kern Sibbald.
*/
/*
 *  Bacula File Daemon Status routines
 *
 *    Kern Sibbald, August MMI
 *
 */

#include "bacula.h"
#include "filed.h"

#define USE_LIST_TERMINATED_JOBS
#define USE_LIST_RESOURCE_LIMITS
#include "lib/status.h"

extern bool GetWindowsVersionString(char *buf, int maxsiz);


/* Forward referenced functions */
static void  list_running_jobs(STATUS_PKT *sp);
static void  list_status_header(STATUS_PKT *sp);
static void  list_collectors_status(STATUS_PKT *sp, char *collname);
static void  show_config(STATUS_PKT *sp);
static void  api_collectors_status(STATUS_PKT *sp, char *collname);

/* Static variables */
static char qstatus1[] = ".status %127s\n";
static char qstatus2[] = ".status %127s api=%d api_opts=%127s";

static char OKqstatus[]   = "2000 OK .status\n";
static char DotStatusJob[] = "JobId=%d JobStatus=%c JobErrors=%d\n";

#if defined(HAVE_WIN32)
static int privs = 0;
#endif
#ifdef WIN32_VSS
#include "vss.h"
#define VSS " VSS"
#else
#define VSS ""
#endif

/*
 * General status generator
 */
void output_status(STATUS_PKT *sp)
{
   list_status_header(sp);
   list_running_jobs(sp);
   list_terminated_jobs(sp);    /* defined in lib/status.h */
}

/* Get have_lzo, have_zstd, ... */
#include "compress.h"

/* from filed_conf.c */
extern s_ct ciphertypes[];
extern s_ct digesttypes[];

static void api_list_status_header(STATUS_PKT *sp)
{
   char *p;
   char buf[300];
   const char *cipher=NULL, *digest=NULL;
   alist tlist(10, not_owned_by_alist);
   OutputWriter wt(sp->api_opts);
   *buf = 0;

#if defined(HAVE_WIN32)
   if (!GetWindowsVersionString(buf, sizeof(buf))) {
      *buf = 0;
   }
#endif
   /* Display the keyword instead of the index */
   if (me->pki_encrypt || me->pki_sign) {
      for (int i = 0; ciphertypes[i].type_name ; i++) {
         if (me->pki_cipher == ciphertypes[i].type_value) {
            cipher = ciphertypes[i].type_name;
            break;
         }
      }
      for (int i = 0; digesttypes[i].type_name ; i++) {
         if (me->pki_digest == digesttypes[i].type_value) {
            digest = digesttypes[i].type_name;
            break;
         }
      }
   }
   wt.start_group("header");
   wt.get_output(
      OT_START_OBJ,
      OT_STRING, "name",        my_name,
      OT_STRING, "version",     VERSION " (" BDATE ")",
      OT_STRING, "uname",       HOST_OS " " DISTNAME " " DISTVER,
      OT_UTIME,  "started",     daemon_start_time,
      OT_INT64,  "pid",         (int64_t)getpid(),
      OT_INT,    "jobs_run",    num_jobs_run,
      OT_INT,    "jobs_running",job_count(),
      OT_STRING, "winver",      buf,
      OT_INT64,  "debug",       debug_level,
      OT_INT,    "trace",       get_trace(),
      OT_ALIST_STR, "tags",     debug_get_tags_list(&tlist, debug_level_tags),
      OT_INT64,  "bwlimit",     me->max_bandwidth_per_job,
      OT_PLUGINS, "plugins",    b_plugin_list,
      OT_INT,     "pkiencryption", (int)me->pki_encrypt,
      OT_INT,     "pkisignature", (int)me->pki_sign,
      OT_STRING,  "pkicipher",  NPRTB(cipher),
      OT_STRING,  "pkidigest",  NPRTB(digest),
      OT_INT32,   "fips",       crypto_get_fips(),
      OT_STRING,  "crypto",     crypto_get_version(),
      OT_BOOL,    "gpfs",       GPFSLIB::enabled(),
      OT_END_OBJ,
      OT_END);
   p = wt.end_group();
   sendit(p, strlen(p), sp);
}

static void  list_status_header(STATUS_PKT *sp)
{
   POOL_MEM msg(PM_MESSAGE);
   char b1[32], b2[32], b3[32], b4[32], b5[35];
   int64_t memused = heap_used();
   int len;
   char dt[MAX_TIME_LENGTH];

   if (sp->api) {
      api_list_status_header(sp);
      return;
   }

   LockRes();
   CLIENT *client = (CLIENT *)GetNextRes(R_CLIENT, NULL);
   UnlockRes();

   len = Mmsg(msg, _("%s %sVersion: %s (%s) %s %s %s %s\n"),
              my_name, BDEMO, VERSION, BDATE, VSS, HOST_OS,
              DISTNAME, DISTVER);
   sendit(msg.c_str(), len, sp);
   bstrftime_nc(dt, sizeof(dt), daemon_start_time);
   len = Mmsg(msg, _("Daemon started %s. Jobs: run=%d running=%d max=%ld.\n"),
        dt, num_jobs_run, job_count(), client->MaxConcurrentJobs);
   sendit(msg.c_str(), len, sp);
#if defined(HAVE_WIN32)
   char buf[300];
   if (GetWindowsVersionString(buf, sizeof(buf))) {
      len = Mmsg(msg, "%s\n", buf);
      sendit(msg.c_str(), len, sp);
   }
   memused = get_memory_info(buf, sizeof(buf));
   if (debug_level > 0) {
      if (!privs) {
         privs = enable_backup_privileges(NULL, 1);
      }
      len = Mmsg(msg, "Priv 0x%x\n", privs);
      sendit(msg.c_str(), len, sp);

      /* Display detailed information that we got from get_memory_info() */
      len = Mmsg(msg, "Memory: %s\n", buf);
      sendit(msg.c_str(), len, sp);

      len = Mmsg(msg, "APIs=%sOPT,%sATP,%sLPV,%sCFA,%sCFW,\n",
                 p_OpenProcessToken?"":"!",
                 p_AdjustTokenPrivileges?"":"!",
                 p_LookupPrivilegeValue?"":"!",
                 p_CreateFileA?"":"!",
                 p_CreateFileW?"":"!");
      sendit(msg.c_str(), len, sp);
      len = Mmsg(msg, " %sWUL,%sWMKD,%sGFAA,%sGFAW,%sGFAEA,%sGFAEW,%sSFAA,%sSFAW,%sBR,%sBW,%sSPSP,\n",
                 p_wunlink?"":"!",
                 p_wmkdir?"":"!",
                 p_GetFileAttributesA?"":"!",
                 p_GetFileAttributesW?"":"!",
                 p_GetFileAttributesExA?"":"!",
                 p_GetFileAttributesExW?"":"!",
                 p_SetFileAttributesA?"":"!",
                 p_SetFileAttributesW?"":"!",
                 p_BackupRead?"":"!",
                 p_BackupWrite?"":"!",
                 p_SetProcessShutdownParameters?"":"!");
      sendit(msg.c_str(), len, sp);
      len = Mmsg(msg, " %sWC2MB,%sMB2WC,%sFFFA,%sFFFW,%sFNFA,%sFNFW,%sSCDA,%sSCDW,\n",
                 p_WideCharToMultiByte?"":"!",
                 p_MultiByteToWideChar?"":"!",
                 p_FindFirstFileA?"":"!",
                 p_FindFirstFileW?"":"!",
                 p_FindNextFileA?"":"!",
                 p_FindNextFileW?"":"!",
                 p_SetCurrentDirectoryA?"":"!",
                 p_SetCurrentDirectoryW?"":"!");
      sendit(msg.c_str(), len, sp);
      len = Mmsg(msg, " %sGCDA,%sGCDW,%sGVPNW,%sGVNFVMPW,%sLZO,%sEFS,%sZSTD\n",
                 p_GetCurrentDirectoryA?"":"!",
                 p_GetCurrentDirectoryW?"":"!",
                 p_GetVolumePathNameW?"":"!",
                 p_GetVolumeNameForVolumeMountPointW?"":"!",
                 have_lzo?"":"!",
                 (BEEF>0)?"":"!",
                 have_zstd?"":"!");
      sendit(msg.c_str(), len, sp);
   }
#endif

   if (debug_level > 0) {
      int64_t nofile_l = 1000;
      int64_t memlock_l=0;
      list_resource_limits(sp, nofile_l, memlock_l);
   }

   len = Mmsg(msg, _(" Heap: heap=%s smbytes=%s max_bytes=%s bufs=%s max_bufs=%s\n"),
         edit_uint64_with_commas(memused, b1),
         edit_uint64_with_commas(sm_bytes, b2),
         edit_uint64_with_commas(sm_max_bytes, b3),
         edit_uint64_with_commas(sm_buffers, b4),
         edit_uint64_with_commas(sm_max_buffers, b5));
   sendit(msg.c_str(), len, sp);
   len = Mmsg(msg, _(" Sizes: boffset_t=%d size_t=%d debug=%s trace=%d "
                     "mode=%d,%d bwlimit=%skB/s\n"),
              sizeof(boffset_t), sizeof(size_t),
              edit_uint64(debug_level, b2), get_trace(), (int)DEVELOPER_MODE, (int)BEEF,
              edit_uint64_with_commas(me->max_bandwidth_per_job/1024, b1)
      );
   sendit(msg.c_str(), len, sp);

   len = Mmsg(msg, " Crypto: fips=%s crypto=%s\n", crypto_get_fips_enabled(), crypto_get_version());
   sendit(msg.c_str(), len, sp);

   if (chk_dbglvl(1)) {
      len = Mmsg(msg, " APIs: %sGPFS\n", GPFSLIB::enabled()?"":"!");
      sendit(msg.c_str(), len, sp);
   }

   if (b_plugin_list && b_plugin_list->size() > 0) {
      Plugin *plugin;
      int len, maxlen=80;
      pm_strcpy(msg, " Plugin: ");
      foreach_alist(plugin, b_plugin_list) {
         len = pm_strcat(msg, plugin->name);
         if (plugin->pinfo) {
            pInfo *info = (pInfo *)plugin->pinfo;
            pm_strcat(msg, "(");
            pm_strcat(msg, NPRT(info->plugin_version));
            len = pm_strcat(msg, ")");
         }
         if (len > maxlen) {
            maxlen = maxlen * 2; /* Let's display an other 80c line */
            pm_strcat(msg, "\n   ");
         } else {
            pm_strcat(msg, " ");
         }
      }
      len = pm_strcat(msg, "\n");
      sendit(msg.c_str(), len, sp);
   }
}

/*
 * List running jobs in for humans.
 */
static void  list_running_jobs_plain(STATUS_PKT *sp)
{
   int total_sec, inst_sec;
   uint64_t total_bps, inst_bps;
   POOL_MEM msg(PM_MESSAGE), fname(PM_FNAME);
   BREGEXP *re;
   char b1[50], b2[50], b3[50], b4[50], b5[50], b6[50];
   int len;
   bool found = false;
   JCR *njcr;
   time_t now = time(NULL);
   char dt[MAX_TIME_LENGTH];

   re = new_bregexp("!password=[^ ]+!password=***!i");
   Dmsg0(1000, "Begin status jcr loop.\n");
   len = Mmsg(msg, _("\nRunning Jobs:\n"));
   sendit(msg.c_str(), len, sp);
   foreach_jcr(njcr) {
      const char *vss = "";
#ifdef WIN32_VSS
      if (njcr->pVSSClient && njcr->pVSSClient->IsInitialized()) {
         vss = "VSS ";
      }
#endif
      bstrftime_nc(dt, sizeof(dt), njcr->start_time);
      if (!njcr->director){
         /* skip any internal system jobs */
         continue;
      }
      if (njcr->JobId == 0) {
         len = Mmsg(msg, _("Director connected %sat: %s\n"),
                    (njcr->dir_bsock && njcr->dir_bsock->tls)?_("using TLS "):"",
                    dt);
      } else {
         len = Mmsg(msg, _("JobId %d Job %s is running.\n"),
                    njcr->JobId, njcr->Job);
         sendit(msg.c_str(), len, sp);
         len = Mmsg(msg, _("    %s%s %s Job started: %s\n"),
                    vss, job_level_to_str(njcr->getJobLevel()),
                    job_type_to_str(njcr->getJobType()), dt);
      }
      sendit(msg.c_str(), len, sp);
      if (njcr->JobId == 0) {
         continue;
      }
      if (njcr->last_time == 0) {
         njcr->last_time = njcr->start_time;
      }
      total_sec = now - njcr->start_time;
      inst_sec = now - njcr->last_time;
      if (total_sec <= 0) {
         total_sec = 1;
      }
      if (inst_sec <= 0) {
         inst_sec = 1;
      }
      /* Instanteous bps not smoothed */
      inst_bps = (njcr->JobBytes - njcr->LastJobBytes) / inst_sec;
      if (njcr->LastRate <= 0) {
         njcr->LastRate = inst_bps;
      }
      /* Smooth the instantaneous bps a bit */
      inst_bps = (2 * njcr->LastRate + inst_bps) / 3;
      /* total bps (AveBytes/sec) since start of job */
      total_bps = njcr->JobBytes / total_sec;
      len = Mmsg(msg,  _("    Files=%s Bytes=%s AveBytes/sec=%s LastBytes/sec=%s Errors=%d\n"
                         "    Bwlimit=%s ReadBytes=%s\n"),
           edit_uint64_with_commas(njcr->JobFiles, b1),
           edit_uint64_with_commas(njcr->JobBytes, b2),
           edit_uint64_with_commas(total_bps, b3),
           edit_uint64_with_commas(inst_bps, b4),
           njcr->JobErrors, edit_uint64_with_commas(njcr->max_bandwidth, b5),
           edit_uint64_with_commas(njcr->ReadBytes, b6));
      sendit(msg.c_str(), len, sp);

      if (njcr->is_JobType(JT_RESTORE)) {
         if (njcr->ExpectedFiles > 0) {
            len = Mmsg(msg, _("    Files: Restored=%s Expected=%s Completed=%d%%\n"),
                       edit_uint64_with_commas(njcr->num_files_examined, b1),
                       edit_uint64_with_commas(njcr->ExpectedFiles, b2),
                       (100*njcr->num_files_examined)/njcr->ExpectedFiles);

         } else {
            len = Mmsg(msg, _("    Files: Restored=%s\n"),
                       edit_uint64_with_commas(njcr->num_files_examined, b1));
         }
      } else {
         len = Mmsg(msg, _("    Files: Examined=%s Backed up=%s\n"),
            edit_uint64_with_commas(njcr->num_files_examined, b1),
            edit_uint64_with_commas(njcr->JobFiles, b2));
      }
      /* Update only every 10 seconds */
      if (now - njcr->last_time > 10) {
         njcr->LastRate = inst_bps;
         njcr->LastJobBytes = njcr->JobBytes;
         njcr->last_time = now;
      }
      sendit(msg.c_str(), len, sp);
      if (njcr->JobFiles > 0) {
         njcr->lock();
         pm_strcpy(fname, njcr->last_fname);
         njcr->unlock();

         /* We strip the password from the filename if we find it */
         len = Mmsg(msg, _("    Processing file: %s\n"), re->replace(fname.c_str()));
         sendit(msg.c_str(), len, sp);
      }

      found = true;
      if (njcr->store_bsock) {
         len = Mmsg(msg, "    SDReadSeqNo=%" lld " fd=%d SDtls=%d\n",
                    njcr->store_bsock->read_seqno, njcr->store_bsock->m_fd,
                    (njcr->store_bsock->tls)?1:0);
         sendit(msg.c_str(), len, sp);
      } else {
         len = Mmsg(msg, _("    SDSocket closed.\n"));
         sendit(msg.c_str(), len, sp);
      }
   }
   endeach_jcr(njcr);

   if (!found) {
      len = Mmsg(msg, _("No Jobs running.\n"));
      sendit(msg.c_str(), len, sp);
   }
   sendit(_("====\n"), 5, sp);
   free_bregexp(re);
}

/*
 * List running jobs for Bat or Bweb in a format
 *  simpler to parse. Be careful when changing this
 *  subroutine.
 */
static void  list_running_jobs_api(STATUS_PKT *sp)
{
   OutputWriter ow(sp->api_opts);
   int sec, bps, brps, val;
   bool add_sep=false;
   char *p;
   JCR *njcr;

   /* API v1, edit with comma, space before the name, sometime ' ' as separator */
   ow.start_group("running");
   p = ow.get_output(OT_START_ARRAY, OT_END);
   sendit(p, strlen(p), sp);

   foreach_jcr(njcr) {
      int vss = 0;
#ifdef WIN32_VSS
      if (njcr->pVSSClient && njcr->pVSSClient->IsInitialized()) {
         vss = 1;
      }
#endif
      if (add_sep) {
         p = ow.get_output(OT_CLEAR, OT_SEP, OT_START_OBJ, OT_END);

      } else {
         p = ow.get_output(OT_CLEAR, OT_START_OBJ, OT_END);
         add_sep = true;
      }

      if (njcr->JobId == 0) {
         val = (njcr->dir_bsock && njcr->dir_bsock->tls)?1:0;
         ow.get_output(OT_UTIME, "DirectorConnected", (utime_t)njcr->start_time,
                       OT_INT, "DirTLS", val,
                       OT_END_OBJ,
                       OT_END);
      } else {
         ow.get_output(OT_INT32,   "JobId", njcr->JobId,
                       OT_STRING,  "Job",   njcr->Job,
                       OT_INT,     "VSS",   vss,
                       OT_JOBLEVEL,"Level", njcr->getJobLevel(),
                       OT_JOBTYPE, "Type",  njcr->getJobType(),
                       OT_JOBSTATUS, "Status", njcr->getJobStatus(),
                       OT_UTIME,   "StartTime", (utime_t)njcr->start_time,
                       OT_END);

      }

      sendit(p, strlen(p), sp);
      if (njcr->JobId == 0) {
         continue;
      }

      sec = time(NULL) - njcr->start_time;
      if (sec <= 0) {
         sec = 1;
      }
      bps = (int)(njcr->JobBytes / sec);
      brps = (int)(njcr->ReadBytes / sec);
      ow.get_output(OT_CLEAR,
                    OT_SEP,     // We have started to display info
                    OT_INT32,   "JobFiles",  njcr->JobFiles,
                    OT_SIZE,    "JobBytes",  njcr->JobBytes,
                    OT_INT,     "Bytes/sec", bps,
                    OT_INT32,   "Errors",    njcr->JobErrors,
                    OT_INT64,   "Bwlimit",   njcr->max_bandwidth,
                    OT_SIZE,    "ReadBytes", njcr->ReadBytes,
                    OT_INT,     "ReadBytes/sec", brps,
                    OT_END);

      ow.get_output(OT_INT32,  "Files Examined",  njcr->num_files_examined, OT_END);

      if (njcr->is_JobType(JT_RESTORE) && njcr->ExpectedFiles > 0) {
         ow.get_output(OT_INT32,  "Expected Files",  njcr->ExpectedFiles,
                       OT_INT32,  "Percent Complete", 100*(njcr->num_files_examined/njcr->ExpectedFiles),
                       OT_END);
      }

      sendit(p, strlen(p), sp);
      ow.get_output(OT_CLEAR, OT_SEP, OT_END);

      if (njcr->JobFiles > 0) {
         njcr->lock();
         ow.get_output(OT_STRING,  "Processing file", njcr->last_fname, OT_END);
         njcr->unlock();
      }

      if (njcr->store_bsock) {
         val = (njcr->store_bsock->tls)?1:0;
         ow.get_output(OT_INT64, "SDReadSeqNo", (int64_t)njcr->store_bsock->read_seqno,
                       OT_INT,   "fd",          njcr->store_bsock->m_fd,
                       OT_INT,   "SDtls",       val,
                       OT_END);
      } else {
         ow.get_output(OT_STRING, "SDSocket", "closed", OT_END);
      }
      val = (njcr->dir_bsock && njcr->dir_bsock->tls) ? 1 : 0;
      ow.get_output(OT_INT, "DirTLS", val,
                    OT_END);

      ow.get_output(OT_END_OBJ, OT_END);
      sendit(p, strlen(p), sp);
   }
   endeach_jcr(njcr);

   ow.get_output(OT_CLEAR, OT_END_ARRAY, OT_END);
   p = ow.end_group();
   sendit(p, strlen(p), sp);
}

static void  list_running_jobs(STATUS_PKT *sp)
{
   if (sp->api) {
      list_running_jobs_api(sp);
   } else {
      list_running_jobs_plain(sp);
   }
}

/*
 * Status command from Director
 */
int status_cmd(JCR *jcr)
{
   BSOCK *user = jcr->dir_bsock;
   STATUS_PKT sp;

   user->fsend("\n");
   sp.bs = user;
   sp.api = false;                         /* no API output */
   output_status(&sp);

   user->signal(BNET_EOD);
   return 1;
}

/*
 * .status command from Director
 */
int qstatus_cmd(JCR *jcr)
{
   BSOCK *dir = jcr->dir_bsock;
   POOLMEM *cmd;
   JCR *njcr;
   char *collname;
   s_last_job* job;
   STATUS_PKT sp;

   sp.bs = dir;
   cmd = get_memory(dir->msglen+1);

   if (sscanf(dir->msg, qstatus2, cmd, &sp.api, sp.api_opts) != 3) {
      if (sscanf(dir->msg, qstatus1, cmd) != 1) {
         pm_strcpy(&jcr->errmsg, dir->msg);
         Jmsg1(jcr, M_FATAL, 0, _("Bad .status command: %s\n"), jcr->errmsg);
         dir->fsend(_("2900 Bad .status command, missing argument.\n"));
         dir->signal(BNET_EOD);
         free_memory(cmd);
         return 0;
      }
   }
   unbash_spaces(cmd);

   collname = strchr(cmd, '=');
   if (collname){
      *collname = 0;
      collname++;
   }
   if (strcasecmp(cmd, "current") == 0) {
      dir->fsend(OKqstatus, cmd);
      foreach_jcr(njcr) {
         if (njcr->JobId != 0) {
            dir->fsend(DotStatusJob, njcr->JobId, njcr->JobStatus, njcr->JobErrors);
         }
      }
      endeach_jcr(njcr);
   } else if (strcasecmp(cmd, "last") == 0) {
      dir->fsend(OKqstatus, cmd);
      if ((last_jobs) && (last_jobs->size() > 0)) {
         job = (s_last_job*)last_jobs->last();
         dir->fsend(DotStatusJob, job->JobId, job->JobStatus, job->Errors);
      }
   } else if (strcasecmp(cmd, "header") == 0) {
       sp.api = true;
       list_status_header(&sp);
   } else if (strcasecmp(cmd, "running") == 0) {
       sp.api = true;
       list_running_jobs(&sp);
   } else if (strcasecmp(cmd, "terminated") == 0) {
       sp.api = MAX(sp.api, 1);
       list_terminated_jobs(&sp); /* defined in lib/status.h */
   } else if (strcasecmp(cmd, "statistics") == 0) {
       sp.api = MAX(sp.api, 1);
       list_collectors_status(&sp, collname);
   } else if (strcasecmp(cmd, "resources") == 0) {
       sp.api = MAX(sp.api, 1);
       show_config(&sp);
   } else {
      pm_strcpy(&jcr->errmsg, dir->msg);
      Jmsg1(jcr, M_FATAL, 0, _("Bad .status command: %s\n"), jcr->errmsg);
      dir->fsend(_("2900 Bad .status command, wrong argument.\n"));
      dir->signal(BNET_EOD);
      free_memory(cmd);
      return 0;
   }

   dir->signal(BNET_EOD);
   free_memory(cmd);
   return 1;
}

static void show_config(STATUS_PKT *sp)
{
   LockRes();
   CLIENT *client = (CLIENT *)GetNextRes(R_CLIENT, NULL);
   UnlockRes();

   dump_resource(R_CLIENT, (RES *)client, sendit, sp);
}

static void list_collectors_status(STATUS_PKT *sp, char *collname)
{
   URES *res;
   int len;
   POOL_MEM buf(PM_MESSAGE);

   Dmsg2(200, "enter list_collectors_status() api=%i coll=%s\n", sp->api, NPRTB(collname));
   if (sp->api > 1) {
      api_collectors_status(sp, collname);
      return;
   }

   LockRes();
   foreach_res(res, R_COLLECTOR) {
      if (collname && !bstrcmp(collname, res->res_collector.hdr.name)){
         continue;
      }
      Dmsg1(500, "processing: %s\n", res->res_collector.hdr.name);
      len = render_collector_status(res->res_collector, buf);
      sendit(buf.c_str(), len, sp);
   };
   UnlockRes();
   if (!collname){
      len = render_updcollector_status(buf);
      sendit(buf.c_str(), len, sp);
   }
   Dmsg0(200, "leave list_collectors_status()\n");
}

static void api_collectors_status(STATUS_PKT *sp, char *collname)
{
   URES *res;
   OutputWriter ow(sp->api_opts);
   POOLMEM *buf;

   Dmsg1(200, "enter api_collectors_status() %s\n", NPRTB(collname));
   ow.start_group("collector");
   ow.get_output(OT_START_OBJ, OT_END);
   ow.start_list("collector_backends");
   LockRes();
   foreach_res(res, R_COLLECTOR) {
      if (collname && !bstrcmp(collname, res->res_collector.hdr.name)){
         continue;
      }
      Dmsg1(500, "processing: %s\n", res->res_collector.hdr.name);
      api_render_collector_status(res->res_collector, ow);
   };
   UnlockRes();
   ow.end_list();
   if (!collname){
      ow.get_output(OT_SEP, OT_LABEL, "collector_update", OT_END);
      api_render_updcollector_status(ow);
   }
   ow.get_output(OT_END_OBJ, OT_END);
   buf = ow.end_group();
   sendit(buf, strlen(buf), sp);
   Dmsg0(200, "leave api_collectors_status()\n");
};
