/**************************************************************************/
/*  locales.h                                                             */
/**************************************************************************/
/*                         This file is part of:                          */
/*                             GODOT ENGINE                               */
/*                        https://godotengine.org                         */
/**************************************************************************/
/* Copyright (c) 2014-present Godot Engine contributors (see AUTHORS.md). */
/* Copyright (c) 2007-2014 Juan Linietsky, Ariel Manzur.                  */
/*                                                                        */
/* Permission is hereby granted, free of charge, to any person obtaining  */
/* a copy of this software and associated documentation files (the        */
/* "Software"), to deal in the Software without restriction, including    */
/* without limitation the rights to use, copy, modify, merge, publish,    */
/* distribute, sublicense, and/or sell copies of the Software, and to     */
/* permit persons to whom the Software is furnished to do so, subject to  */
/* the following conditions:                                              */
/*                                                                        */
/* The above copyright notice and this permission notice shall be         */
/* included in all copies or substantial portions of the Software.        */
/*                                                                        */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,        */
/* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF     */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. */
/* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY   */
/* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,   */
/* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE      */
/* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                 */
/**************************************************************************/

#pragma once

// Windows has some weird locale identifiers which do not honor the ISO 639-1
// standardized nomenclature. Whenever those don't conflict with existing ISO
// identifiers, we override them.
//
// Reference:
// - https://msdn.microsoft.com/en-us/library/windows/desktop/ms693062(v=vs.85).aspx

static const char *locale_renames[][2] = {
	{ "in", "id" }, //  Indonesian
	{ "iw", "he" }, //  Hebrew
	{ "no", "nb" }, //  Norwegian Bokmål
	{ "C", "en" }, // Locale is not set, fallback to English.
	{ nullptr, nullptr }
};

// Additional script information to preferred scripts.
// Language code, script code, default country, supported countries.
// Reference:
// - https://lh.2xlibre.net/locales/
// - https://www.localeplanet.com/icu/index.html
// - https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-lcid/70feba9f-294e-491e-b6eb-56532684c37f

static const char *locale_scripts[][4] = {
	{ "az", "Latn", "", "AZ" },
	{ "az", "Arab", "", "IR" },
	{ "bs", "Latn", "", "BA" },
	{ "ff", "Latn", "", "BF,CM,GH,GM,GN,GW,LR,MR,NE,NG,SL,SN" },
	{ "pa", "Arab", "PK", "PK" },
	{ "pa", "Guru", "IN", "IN" },
	{ "sd", "Arab", "PK", "PK" },
	{ "sd", "Deva", "IN", "IN" },
	{ "shi", "Tfng", "", "MA" },
	{ "sr", "Cyrl", "", "BA,RS,XK" },
	{ "sr", "Latn", "", "ME" },
	{ "uz", "Latn", "", "UZ" },
	{ "uz", "Arab", "AF", "AF" },
	{ "vai", "Vaii", "", "LR" },
	{ "yue", "Hans", "CN", "CN" },
	{ "yue", "Hant", "HK", "HK" },
	{ "zh", "Hans", "CN", "CN,SG" },
	{ "zh", "Hant", "TW", "HK,MO,TW" },
	{ nullptr, nullptr, nullptr, nullptr }
};

// Additional mapping for outdated, temporary or exceptionally reserved country codes.
// Reference:
// - https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2
// - https://www.iso.org/obp/ui/#search/code/

static const char *country_renames[][2] = {
	{ "BU", "MM" }, // Burma, name changed to Myanmar.
	{ "KV", "XK" }, // Kosovo (temporary FIPS code to European Commission code), no official ISO code assigned.
	{ "TP", "TL" }, // East Timor, name changed to Timor-Leste.
	{ "UK", "GB" }, // United Kingdom, exceptionally reserved code.
	{ nullptr, nullptr }
};

// Country code, country name.
// Reference:
// - https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2
// - https://www.iso.org/obp/ui/#search/code/

static const char *country_names[][2] = {
	{ "AC", "Ascension Island" }, // Exceptionally reserved.
	{ "AD", "Andorra" },
	{ "AE", "United Arab Emirates" },
	{ "AF", "Afghanistan" },
	{ "AG", "Antigua and Barbuda" },
	{ "AI", "Anguilla" },
	{ "AL", "Albania" },
	{ "AM", "Armenia" },
	{ "AN", "Netherlands Antilles" }, // Transitionally reserved, divided into BQ, CW and SX.
	{ "AO", "Angola" },
	{ "AQ", "Antarctica" },
	{ "AR", "Argentina" },
	{ "AS", "American Samoa" },
	{ "AT", "Austria" },
	{ "AU", "Australia" },
	{ "AW", "Aruba" },
	{ "AX", "Åland Islands" },
	{ "AZ", "Azerbaijan" },
	{ "BA", "Bosnia and Herzegovina" },
	{ "BB", "Barbados" },
	{ "BD", "Bangladesh" },
	{ "BE", "Belgium" },
	{ "BF", "Burkina Faso" },
	{ "BG", "Bulgaria" },
	{ "BH", "Bahrain" },
	{ "BI", "Burundi" },
	{ "BJ", "Benin" },
	{ "BL", "St. Barthélemy" },
	{ "BM", "Bermuda" },
	{ "BN", "Brunei" },
	{ "BO", "Bolivia" },
	{ "BQ", "Caribbean Netherlands" },
	{ "BR", "Brazil" },
	{ "BS", "Bahamas" },
	{ "BT", "Bhutan" },
	{ "BV", "Bouvet Island" },
	{ "BW", "Botswana" },
	{ "BY", "Belarus" },
	{ "BZ", "Belize" },
	{ "CA", "Canada" },
	{ "CC", "Cocos (Keeling) Islands" },
	{ "CD", "Congo - Kinshasa" },
	{ "CF", "Central African Republic" },
	{ "CG", "Congo - Brazzaville" },
	{ "CH", "Switzerland" },
	{ "CI", "Côte d'Ivoire" },
	{ "CK", "Cook Islands" },
	{ "CL", "Chile" },
	{ "CM", "Cameroon" },
	{ "CN", "China" },
	{ "CO", "Colombia" },
	{ "CP", "Clipperton Island" }, // Exceptionally reserved.
	{ "CR", "Costa Rica" },
	{ "CQ", "Island of Sark" }, // Exceptionally reserved.
	{ "CU", "Cuba" },
	{ "CV", "Cabo Verde" },
	{ "CW", "Curaçao" },
	{ "CX", "Christmas Island" },
	{ "CY", "Cyprus" },
	{ "CZ", "Czechia" },
	{ "DE", "Germany" },
	{ "DG", "Diego Garcia" }, // Exceptionally reserved.
	{ "DJ", "Djibouti" },
	{ "DK", "Denmark" },
	{ "DM", "Dominica" },
	{ "DO", "Dominican Republic" },
	{ "DZ", "Algeria" },
	{ "EA", "Ceuta and Melilla" }, // Exceptionally reserved.
	{ "EC", "Ecuador" },
	{ "EE", "Estonia" },
	{ "EG", "Egypt" },
	{ "EH", "Western Sahara" },
	{ "ER", "Eritrea" },
	{ "ES", "Spain" },
	{ "ET", "Ethiopia" },
	{ "EU", "European Union" }, // Exceptionally reserved.
	{ "EZ", "Eurozone" }, // Exceptionally reserved.
	{ "FI", "Finland" },
	{ "FJ", "Fiji" },
	{ "FK", "Falkland Islands" },
	{ "FM", "Micronesia" },
	{ "FO", "Faroe Islands" },
	{ "FR", "France" },
	{ "FX", "France, Metropolitan" }, // Exceptionally reserved.
	{ "GA", "Gabon" },
	{ "GB", "United Kingdom" },
	{ "GD", "Grenada" },
	{ "GE", "Georgia" },
	{ "GF", "French Guiana" },
	{ "GG", "Guernsey" },
	{ "GH", "Ghana" },
	{ "GI", "Gibraltar" },
	{ "GL", "Greenland" },
	{ "GM", "Gambia" },
	{ "GN", "Guinea" },
	{ "GP", "Guadeloupe" },
	{ "GQ", "Equatorial Guinea" },
	{ "GR", "Greece" },
	{ "GS", "South Georgia and South Sandwich Islands" },
	{ "GT", "Guatemala" },
	{ "GU", "Guam" },
	{ "GW", "Guinea-Bissau" },
	{ "GY", "Guyana" },
	{ "HK", "Hong Kong" },
	{ "HM", "Heard Island and McDonald Islands" },
	{ "HN", "Honduras" },
	{ "HR", "Croatia" },
	{ "HT", "Haiti" },
	{ "HU", "Hungary" },
	{ "IC", "Canary Islands" }, // Exceptionally reserved.
	{ "ID", "Indonesia" },
	{ "IE", "Ireland" },
	{ "IL", "Israel" },
	{ "IM", "Isle of Man" },
	{ "IN", "India" },
	{ "IO", "British Indian Ocean Territory" },
	{ "IQ", "Iraq" },
	{ "IR", "Iran" },
	{ "IS", "Iceland" },
	{ "IT", "Italy" },
	{ "JE", "Jersey" },
	{ "JM", "Jamaica" },
	{ "JO", "Jordan" },
	{ "JP", "Japan" },
	{ "KE", "Kenya" },
	{ "KG", "Kyrgyzstan" },
	{ "KH", "Cambodia" },
	{ "KI", "Kiribati" },
	{ "KM", "Comoros" },
	{ "KN", "St. Kitts and Nevis" },
	{ "KP", "North Korea" },
	{ "KR", "South Korea" },
	{ "KW", "Kuwait" },
	{ "KY", "Cayman Islands" },
	{ "KZ", "Kazakhstan" },
	{ "LA", "Laos" },
	{ "LB", "Lebanon" },
	{ "LC", "St. Lucia" },
	{ "LI", "Liechtenstein" },
	{ "LK", "Sri Lanka" },
	{ "LR", "Liberia" },
	{ "LS", "Lesotho" },
	{ "LT", "Lithuania" },
	{ "LU", "Luxembourg" },
	{ "LV", "Latvia" },
	{ "LY", "Libya" },
	{ "MA", "Morocco" },
	{ "MC", "Monaco" },
	{ "MD", "Moldova" },
	{ "ME", "Montenegro" },
	{ "MF", "St. Martin" },
	{ "MG", "Madagascar" },
	{ "MH", "Marshall Islands" },
	{ "MK", "North Macedonia" },
	{ "ML", "Mali" },
	{ "MM", "Myanmar" },
	{ "MN", "Mongolia" },
	{ "MO", "Macao" },
	{ "MP", "Northern Mariana Islands" },
	{ "MQ", "Martinique" },
	{ "MR", "Mauritania" },
	{ "MS", "Montserrat" },
	{ "MT", "Malta" },
	{ "MU", "Mauritius" },
	{ "MV", "Maldives" },
	{ "MW", "Malawi" },
	{ "MX", "Mexico" },
	{ "MY", "Malaysia" },
	{ "MZ", "Mozambique" },
	{ "NA", "Namibia" },
	{ "NC", "New Caledonia" },
	{ "NE", "Niger" },
	{ "NF", "Norfolk Island" },
	{ "NG", "Nigeria" },
	{ "NI", "Nicaragua" },
	{ "NL", "Netherlands" },
	{ "NO", "Norway" },
	{ "NP", "Nepal" },
	{ "NR", "Nauru" },
	{ "NU", "Niue" },
	{ "NZ", "New Zealand" },
	{ "OM", "Oman" },
	{ "PA", "Panama" },
	{ "PE", "Peru" },
	{ "PF", "French Polynesia" },
	{ "PG", "Papua New Guinea" },
	{ "PH", "Philippines" },
	{ "PK", "Pakistan" },
	{ "PL", "Poland" },
	{ "PM", "St. Pierre and Miquelon" },
	{ "PN", "Pitcairn Islands" },
	{ "PR", "Puerto Rico" },
	{ "PS", "Palestine" },
	{ "PT", "Portugal" },
	{ "PW", "Palau" },
	{ "PY", "Paraguay" },
	{ "QA", "Qatar" },
	{ "RE", "Réunion" },
	{ "RO", "Romania" },
	{ "RS", "Serbia" },
	{ "RU", "Russia" },
	{ "RW", "Rwanda" },
	{ "SA", "Saudi Arabia" },
	{ "SB", "Solomon Islands" },
	{ "SC", "Seychelles" },
	{ "SD", "Sudan" },
	{ "SE", "Sweden" },
	{ "SG", "Singapore" },
	{ "SH", "St. Helena, Ascension and Tristan da Cunha" },
	{ "SI", "Slovenia" },
	{ "SJ", "Svalbard and Jan Mayen" },
	{ "SK", "Slovakia" },
	{ "SL", "Sierra Leone" },
	{ "SM", "San Marino" },
	{ "SN", "Senegal" },
	{ "SO", "Somalia" },
	{ "SR", "Suriname" },
	{ "SS", "South Sudan" },
	{ "ST", "Sao Tome and Principe" },
	{ "SV", "El Salvador" },
	{ "SX", "Sint Maarten" },
	{ "SY", "Syria" },
	{ "SZ", "Eswatini" },
	{ "TA", "Tristan da Cunha" }, // Exceptionally reserved.
	{ "TC", "Turks and Caicos Islands" },
	{ "TD", "Chad" },
	{ "TF", "French Southern Territories" },
	{ "TG", "Togo" },
	{ "TH", "Thailand" },
	{ "TJ", "Tajikistan" },
	{ "TK", "Tokelau" },
	{ "TL", "Timor-Leste" },
	{ "TM", "Turkmenistan" },
	{ "TN", "Tunisia" },
	{ "TO", "Tonga" },
	{ "TR", "Turkey" },
	{ "TT", "Trinidad and Tobago" },
	{ "TV", "Tuvalu" },
	{ "TW", "Taiwan" },
	{ "TZ", "Tanzania" },
	{ "UA", "Ukraine" },
	{ "UG", "Uganda" },
	{ "UM", "U.S. Outlying Islands" },
	{ "US", "United States of America" },
	{ "UY", "Uruguay" },
	{ "UZ", "Uzbekistan" },
	{ "VA", "Holy See" },
	{ "VC", "St. Vincent and the Grenadines" },
	{ "VE", "Venezuela" },
	{ "VG", "British Virgin Islands" },
	{ "VI", "U.S. Virgin Islands" },
	{ "VN", "Viet Nam" },
	{ "VU", "Vanuatu" },
	{ "WF", "Wallis and Futuna" },
	{ "WS", "Samoa" },
	{ "XK", "Kosovo" }, // Temporary code, no official ISO code assigned.
	{ "YE", "Yemen" },
	{ "YT", "Mayotte" },
	{ "ZA", "South Africa" },
	{ "ZM", "Zambia" },
	{ "ZW", "Zimbabwe" },
	{ nullptr, nullptr }
};

// Languages code, language name.
// Reference:
// - https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes
// - https://www.localeplanet.com/icu/index.html
// - https://lh.2xlibre.net/locales/

static const char *language_list[][2] = {
	{ "aa", "Afar" },
	{ "ab", "Abkhazian" },
	{ "ace", "Achinese" },
	{ "ach", "Acoli" },
	{ "ada", "Adangme" },
	{ "ady", "Adyghe" },
	{ "ae", "Avestan" },
	{ "aeb", "Tunisian Arabic" },
	{ "af", "Afrikaans" },
	{ "afh", "Afrihili" },
	{ "agq", "Aghem" },
	{ "ain", "Ainu" },
	{ "agr", "Aguaruna" },
	{ "ak", "Akan" },
	{ "akk", "Akkadian" },
	{ "akz", "Alabama" },
	{ "ale", "Aleut" },
	{ "aln", "Gheg Albanian" },
	{ "alt", "Southern Altai" },
	{ "am", "Amharic" },
	{ "an", "Aragonese" },
	{ "ang", "Old English" },
	{ "anp", "Angika" },
	{ "ar", "Arabic" },
	{ "arc", "Aramaic" },
	{ "arn", "Mapudungun" },
	{ "aro", "Araona" },
	{ "arp", "Arapaho" },
	{ "arq", "Algerian Arabic" },
	{ "ars", "Najdi Arabic" },
	{ "arw", "Arawak" },
	{ "ary", "Moroccan Arabic" },
	{ "arz", "Egyptian Arabic" },
	{ "as", "Assamese" },
	{ "asa", "Asu" },
	{ "ase", "American Sign Language" },
	{ "ast", "Asturian" },
	{ "av", "Avaric" },
	{ "avk", "Kotava" },
	{ "awa", "Awadhi" },
	{ "ayc", "Southern Aymara" },
	{ "ay", "Aymara" },
	{ "az", "Azerbaijani" },
	{ "ba", "Bashkir" },
	{ "bal", "Baluchi" },
	{ "ban", "Balinese" },
	{ "bar", "Bavarian" },
	{ "bas", "Bassa" },
	{ "bax", "Bamun" },
	{ "bbc", "Batak Toba" },
	{ "bbj", "Ghomala" },
	{ "be", "Belarusian" },
	{ "bej", "Beja" },
	{ "bem", "Bemba" },
	{ "ber", "Berber" },
	{ "bew", "Betawi" },
	{ "bez", "Bena" },
	{ "bfd", "Bafut" },
	{ "bfq", "Badaga" },
	{ "bg", "Bulgarian" },
	{ "bhb", "Bhili" },
	{ "bgn", "Western Balochi" },
	{ "bho", "Bhojpuri" },
	{ "bi", "Bislama" },
	{ "bik", "Bikol" },
	{ "bin", "Bini" },
	{ "bjn", "Banjar" },
	{ "bkm", "Kom" },
	{ "bla", "Siksika" },
	{ "bm", "Bambara" },
	{ "bn", "Bengali" },
	{ "bo", "Tibetan" },
	{ "bpy", "Bishnupriya" },
	{ "bqi", "Bakhtiari" },
	{ "br", "Breton" },
	{ "brh", "Brahui" },
	{ "brx", "Bodo" },
	{ "bs", "Bosnian" },
	{ "bss", "Akoose" },
	{ "bua", "Buriat" },
	{ "bug", "Buginese" },
	{ "bum", "Bulu" },
	{ "byn", "Bilin" },
	{ "byv", "Medumba" },
	{ "ca", "Catalan" },
	{ "cad", "Caddo" },
	{ "car", "Carib" },
	{ "cay", "Cayuga" },
	{ "cch", "Atsam" },
	{ "ccp", "Chakma" },
	{ "ce", "Chechen" },
	{ "ceb", "Cebuano" },
	{ "cgg", "Chiga" },
	{ "ch", "Chamorro" },
	{ "chb", "Chibcha" },
	{ "chg", "Chagatai" },
	{ "chk", "Chuukese" },
	{ "chm", "Mari" },
	{ "chn", "Chinook Jargon" },
	{ "cho", "Choctaw" },
	{ "chp", "Chipewyan" },
	{ "chr", "Cherokee" },
	{ "chy", "Cheyenne" },
	{ "cic", "Chickasaw" },
	{ "ckb", "Central Kurdish" },
	{ "cmn", "Mandarin Chinese" },
	{ "co", "Corsican" },
	{ "cop", "Coptic" },
	{ "cps", "Capiznon" },
	{ "cr", "Cree" },
	{ "crh", "Crimean Tatar" },
	{ "crs", "Seselwa Creole French" },
	{ "cs", "Czech" },
	{ "csb", "Kashubian" },
	{ "cu", "Church Slavic" },
	{ "cv", "Chuvash" },
	{ "cy", "Welsh" },
	{ "da", "Danish" },
	{ "dak", "Dakota" },
	{ "dar", "Dargwa" },
	{ "dav", "Taita" },
	{ "de", "German" },
	{ "del", "Delaware" },
	{ "den", "Slave" },
	{ "dgr", "Dogrib" },
	{ "din", "Dinka" },
	{ "dje", "Zarma" },
	{ "doi", "Dogri" },
	{ "dsb", "Lower Sorbian" },
	{ "dtp", "Central Dusun" },
	{ "dua", "Duala" },
	{ "dum", "Middle Dutch" },
	{ "dv", "Dhivehi" },
	{ "dyo", "Jola-Fonyi" },
	{ "dyu", "Dyula" },
	{ "dz", "Dzongkha" },
	{ "dzg", "Dazaga" },
	{ "ebu", "Embu" },
	{ "ee", "Ewe" },
	{ "efi", "Efik" },
	{ "egl", "Emilian" },
	{ "egy", "Ancient Egyptian" },
	{ "eka", "Ekajuk" },
	{ "el", "Greek" },
	{ "elx", "Elamite" },
	{ "en", "English" },
	{ "enm", "Middle English" },
	{ "eo", "Esperanto" },
	{ "es", "Spanish" },
	{ "esu", "Central Yupik" },
	{ "et", "Estonian" },
	{ "eu", "Basque" },
	{ "ewo", "Ewondo" },
	{ "ext", "Extremaduran" },
	{ "fa", "Persian" },
	{ "fan", "Fang" },
	{ "fat", "Fanti" },
	{ "ff", "Fulah" },
	{ "fi", "Finnish" },
	{ "fil", "Filipino" },
	{ "fit", "Tornedalen Finnish" },
	{ "fj", "Fijian" },
	{ "fo", "Faroese" },
	{ "fon", "Fon" },
	{ "fr", "French" },
	{ "frc", "Cajun French" },
	{ "frm", "Middle French" },
	{ "fro", "Old French" },
	{ "frp", "Arpitan" },
	{ "frr", "Northern Frisian" },
	{ "frs", "Eastern Frisian" },
	{ "fur", "Friulian" },
	{ "fy", "Western Frisian" },
	{ "ga", "Irish" },
	{ "gaa", "Ga" },
	{ "gag", "Gagauz" },
	{ "gan", "Gan Chinese" },
	{ "gay", "Gayo" },
	{ "gba", "Gbaya" },
	{ "gbz", "Zoroastrian Dari" },
	{ "gd", "Scottish Gaelic" },
	{ "gez", "Geez" },
	{ "gil", "Gilbertese" },
	{ "gl", "Galician" },
	{ "glk", "Gilaki" },
	{ "gmh", "Middle High German" },
	{ "gn", "Guarani" },
	{ "goh", "Old High German" },
	{ "gom", "Goan Konkani" },
	{ "gon", "Gondi" },
	{ "gor", "Gorontalo" },
	{ "got", "Gothic" },
	{ "grb", "Grebo" },
	{ "grc", "Ancient Greek" },
	{ "gsw", "Swiss German" },
	{ "gu", "Gujarati" },
	{ "guc", "Wayuu" },
	{ "gur", "Frafra" },
	{ "guz", "Gusii" },
	{ "gv", "Manx" },
	{ "gwi", "Gwichʼin" },
	{ "ha", "Hausa" },
	{ "hai", "Haida" },
	{ "hak", "Hakka Chinese" },
	{ "haw", "Hawaiian" },
	{ "he", "Hebrew" },
	{ "hi", "Hindi" },
	{ "hif", "Fiji Hindi" },
	{ "hil", "Hiligaynon" },
	{ "hit", "Hittite" },
	{ "hmn", "Hmong" },
	{ "ho", "Hiri Motu" },
	{ "hne", "Chhattisgarhi" },
	{ "hr", "Croatian" },
	{ "hsb", "Upper Sorbian" },
	{ "hsn", "Xiang Chinese" },
	{ "ht", "Haitian" },
	{ "hu", "Hungarian" },
	{ "hup", "Hupa" },
	{ "hus", "Huastec" },
	{ "hy", "Armenian" },
	{ "hz", "Herero" },
	{ "ia", "Interlingua" },
	{ "iba", "Iban" },
	{ "ibb", "Ibibio" },
	{ "id", "Indonesian" },
	{ "ie", "Interlingue" },
	{ "ig", "Igbo" },
	{ "ii", "Sichuan Yi" },
	{ "ik", "Inupiaq" },
	{ "ilo", "Iloko" },
	{ "inh", "Ingush" },
	{ "io", "Ido" },
	{ "is", "Icelandic" },
	{ "it", "Italian" },
	{ "iu", "Inuktitut" },
	{ "izh", "Ingrian" },
	{ "ja", "Japanese" },
	{ "jam", "Jamaican Creole English" },
	{ "jbo", "Lojban" },
	{ "jgo", "Ngomba" },
	{ "jmc", "Machame" },
	{ "jpr", "Judeo-Persian" },
	{ "jrb", "Judeo-Arabic" },
	{ "jut", "Jutish" },
	{ "jv", "Javanese" },
	{ "ka", "Georgian" },
	{ "kaa", "Kara-Kalpak" },
	{ "kab", "Kabyle" },
	{ "kac", "Kachin" },
	{ "kaj", "Jju" },
	{ "kam", "Kamba" },
	{ "kaw", "Kawi" },
	{ "kbd", "Kabardian" },
	{ "kbl", "Kanembu" },
	{ "kcg", "Tyap" },
	{ "kde", "Makonde" },
	{ "kea", "Kabuverdianu" },
	{ "ken", "Kenyang" },
	{ "kfo", "Koro" },
	{ "kg", "Kongo" },
	{ "kgp", "Kaingang" },
	{ "kha", "Khasi" },
	{ "kho", "Khotanese" },
	{ "khq", "Koyra Chiini" },
	{ "khw", "Khowar" },
	{ "ki", "Kikuyu" },
	{ "kiu", "Kirmanjki" },
	{ "kj", "Kuanyama" },
	{ "kk", "Kazakh" },
	{ "kkj", "Kako" },
	{ "kl", "Kalaallisut" },
	{ "kln", "Kalenjin" },
	{ "km", "Central Khmer" },
	{ "kmb", "Kimbundu" },
	{ "kn", "Kannada" },
	{ "ko", "Korean" },
	{ "koi", "Komi-Permyak" },
	{ "kok", "Konkani" },
	{ "kos", "Kosraean" },
	{ "kpe", "Kpelle" },
	{ "kr", "Kanuri" },
	{ "krc", "Karachay-Balkar" },
	{ "kri", "Krio" },
	{ "krj", "Kinaray-a" },
	{ "krl", "Karelian" },
	{ "kru", "Kurukh" },
	{ "ks", "Kashmiri" },
	{ "ksb", "Shambala" },
	{ "ksf", "Bafia" },
	{ "ksh", "Colognian" },
	{ "ku", "Kurdish" },
	{ "kum", "Kumyk" },
	{ "kut", "Kutenai" },
	{ "kv", "Komi" },
	{ "kw", "Cornish" },
	{ "ky", "Kirghiz" },
	{ "la", "Latin" },
	{ "lad", "Ladino" },
	{ "lag", "Langi" },
	{ "lah", "Lahnda" },
	{ "lam", "Lamba" },
	{ "lb", "Luxembourgish" },
	{ "lez", "Lezghian" },
	{ "lfn", "Lingua Franca Nova" },
	{ "lg", "Ganda" },
	{ "li", "Limburgan" },
	{ "lij", "Ligurian" },
	{ "liv", "Livonian" },
	{ "lkt", "Lakota" },
	{ "lmo", "Lombard" },
	{ "ln", "Lingala" },
	{ "lo", "Lao" },
	{ "lol", "Mongo" },
	{ "lou", "Louisiana Creole" },
	{ "loz", "Lozi" },
	{ "lrc", "Northern Luri" },
	{ "lt", "Lithuanian" },
	{ "ltg", "Latgalian" },
	{ "lu", "Luba-Katanga" },
	{ "lua", "Luba-Lulua" },
	{ "lui", "Luiseno" },
	{ "lun", "Lunda" },
	{ "luo", "Luo" },
	{ "lus", "Mizo" },
	{ "luy", "Luyia" },
	{ "lv", "Latvian" },
	{ "lzh", "Literary Chinese" },
	{ "lzz", "Laz" },
	{ "mad", "Madurese" },
	{ "maf", "Mafa" },
	{ "mag", "Magahi" },
	{ "mai", "Maithili" },
	{ "mak", "Makasar" },
	{ "man", "Mandingo" },
	{ "mas", "Masai" },
	{ "mde", "Maba" },
	{ "mdf", "Moksha" },
	{ "mdr", "Mandar" },
	{ "men", "Mende" },
	{ "mer", "Meru" },
	{ "mfe", "Morisyen" },
	{ "mg", "Malagasy" },
	{ "mga", "Middle Irish" },
	{ "mgh", "Makhuwa-Meetto" },
	{ "mgo", "Metaʼ" },
	{ "mh", "Marshallese" },
	{ "mhr", "Eastern Mari" },
	{ "mi", "Māori" },
	{ "mic", "Mi'kmaq" },
	{ "min", "Minangkabau" },
	{ "miq", "Mískito" },
	{ "mjw", "Karbi" },
	{ "mk", "Macedonian" },
	{ "ml", "Malayalam" },
	{ "mn", "Mongolian" },
	{ "mnc", "Manchu" },
	{ "mni", "Manipuri" },
	{ "mnw", "Mon" },
	{ "mos", "Mossi" },
	{ "moh", "Mohawk" },
	{ "mr", "Marathi" },
	{ "mrj", "Western Mari" },
	{ "ms", "Malay" },
	{ "mt", "Maltese" },
	{ "mua", "Mundang" },
	{ "mus", "Muscogee" },
	{ "mwl", "Mirandese" },
	{ "mwr", "Marwari" },
	{ "mwv", "Mentawai" },
	{ "my", "Burmese" },
	{ "mye", "Myene" },
	{ "myv", "Erzya" },
	{ "mzn", "Mazanderani" },
	{ "na", "Nauru" },
	{ "nah", "Nahuatl" },
	{ "nan", "Min Nan Chinese" },
	{ "nap", "Neapolitan" },
	{ "naq", "Nama" },
	{ "nan", "Min Nan Chinese" },
	{ "nb", "Norwegian Bokmål" },
	{ "nd", "North Ndebele" },
	{ "nds", "Low German" },
	{ "ne", "Nepali" },
	{ "new", "Newari" },
	{ "nhn", "Central Nahuatl" },
	{ "ng", "Ndonga" },
	{ "nia", "Nias" },
	{ "niu", "Niuean" },
	{ "njo", "Ao Naga" },
	{ "nl", "Dutch" },
	{ "nmg", "Kwasio" },
	{ "nn", "Norwegian Nynorsk" },
	{ "nnh", "Ngiemboon" },
	{ "nog", "Nogai" },
	{ "non", "Old Norse" },
	{ "nov", "Novial" },
	{ "nqo", "N'ko" },
	{ "nr", "South Ndebele" },
	{ "nso", "Pedi" },
	{ "nus", "Nuer" },
	{ "nv", "Navajo" },
	{ "nwc", "Classical Newari" },
	{ "ny", "Nyanja" },
	{ "nym", "Nyamwezi" },
	{ "nyn", "Nyankole" },
	{ "nyo", "Nyoro" },
	{ "nzi", "Nzima" },
	{ "oc", "Occitan" },
	{ "oj", "Ojibwa" },
	{ "om", "Oromo" },
	{ "or", "Odia" },
	{ "os", "Ossetic" },
	{ "osa", "Osage" },
	{ "ota", "Ottoman Turkish" },
	{ "pa", "Panjabi" },
	{ "pag", "Pangasinan" },
	{ "pal", "Pahlavi" },
	{ "pam", "Pampanga" },
	{ "pap", "Papiamento" },
	{ "pau", "Palauan" },
	{ "pcd", "Picard" },
	{ "pcm", "Nigerian Pidgin" },
	{ "pdc", "Pennsylvania German" },
	{ "pdt", "Plautdietsch" },
	{ "peo", "Old Persian" },
	{ "pfl", "Palatine German" },
	{ "phn", "Phoenician" },
	{ "pi", "Pali" },
	{ "pl", "Polish" },
	{ "pms", "Piedmontese" },
	{ "pnt", "Pontic" },
	{ "pon", "Pohnpeian" },
	{ "pr", "Pirate" },
	{ "prg", "Prussian" },
	{ "pro", "Old Provençal" },
	{ "prs", "Dari" },
	{ "ps", "Pushto" },
	{ "pt", "Portuguese" },
	{ "qu", "Quechua" },
	{ "quc", "K'iche" },
	{ "qug", "Chimborazo Highland Quichua" },
	{ "quy", "Ayacucho Quechua" },
	{ "quz", "Cusco Quechua" },
	{ "raj", "Rajasthani" },
	{ "rap", "Rapanui" },
	{ "rar", "Rarotongan" },
	{ "rgn", "Romagnol" },
	{ "rif", "Riffian" },
	{ "rm", "Romansh" },
	{ "rn", "Rundi" },
	{ "ro", "Romanian" },
	{ "rof", "Rombo" },
	{ "rom", "Romany" },
	{ "rtm", "Rotuman" },
	{ "ru", "Russian" },
	{ "rue", "Rusyn" },
	{ "rug", "Roviana" },
	{ "rup", "Aromanian" },
	{ "rw", "Kinyarwanda" },
	{ "rwk", "Rwa" },
	{ "sa", "Sanskrit" },
	{ "sad", "Sandawe" },
	{ "sah", "Sakha" },
	{ "sam", "Samaritan Aramaic" },
	{ "saq", "Samburu" },
	{ "sas", "Sasak" },
	{ "sat", "Santali" },
	{ "saz", "Saurashtra" },
	{ "sba", "Ngambay" },
	{ "sbp", "Sangu" },
	{ "sc", "Sardinian" },
	{ "scn", "Sicilian" },
	{ "sco", "Scots" },
	{ "sd", "Sindhi" },
	{ "sdc", "Sassarese Sardinian" },
	{ "sdh", "Southern Kurdish" },
	{ "se", "Northern Sami" },
	{ "see", "Seneca" },
	{ "seh", "Sena" },
	{ "sei", "Seri" },
	{ "sel", "Selkup" },
	{ "ses", "Koyraboro Senni" },
	{ "sg", "Sango" },
	{ "sga", "Old Irish" },
	{ "sgs", "Samogitian" },
	{ "sh", "Serbo-Croatian" },
	{ "shi", "Tachelhit" },
	{ "shn", "Shan" },
	{ "shs", "Shuswap" },
	{ "shu", "Chadian Arabic" },
	{ "si", "Sinhala" },
	{ "sid", "Sidamo" },
	{ "sk", "Slovak" },
	{ "sl", "Slovenian" },
	{ "sli", "Lower Silesian" },
	{ "sly", "Selayar" },
	{ "sm", "Samoan" },
	{ "sma", "Southern Sami" },
	{ "smj", "Lule Sami" },
	{ "smn", "Inari Sami" },
	{ "sms", "Skolt Sami" },
	{ "sn", "Shona" },
	{ "snk", "Soninke" },
	{ "so", "Somali" },
	{ "sog", "Sogdien" },
	{ "son", "Songhai" },
	{ "sq", "Albanian" },
	{ "sr", "Serbian" },
	{ "srn", "Sranan Tongo" },
	{ "srr", "Serer" },
	{ "ss", "Swati" },
	{ "ssy", "Saho" },
	{ "st", "Southern Sotho" },
	{ "stq", "Saterland Frisian" },
	{ "su", "Sundanese" },
	{ "suk", "Sukuma" },
	{ "sus", "Susu" },
	{ "sux", "Sumerian" },
	{ "sv", "Swedish" },
	{ "sw", "Swahili" },
	{ "swb", "Comorian" },
	{ "swc", "Congo Swahili" },
	{ "syc", "Classical Syriac" },
	{ "syr", "Syriac" },
	{ "szl", "Silesian" },
	{ "ta", "Tamil" },
	{ "tcy", "Tulu" },
	{ "te", "Telugu" },
	{ "tem", "Timne" },
	{ "teo", "Teso" },
	{ "ter", "Tereno" },
	{ "tet", "Tetum" },
	{ "tg", "Tajik" },
	{ "th", "Thai" },
	{ "the", "Chitwania Tharu" },
	{ "ti", "Tigrinya" },
	{ "tig", "Tigre" },
	{ "tiv", "Tiv" },
	{ "tk", "Turkmen" },
	{ "tkl", "Tokelau" },
	{ "tkr", "Tsakhur" },
	{ "tl", "Tagalog" },
	{ "tlh", "Klingon" },
	{ "tli", "Tlingit" },
	{ "tly", "Talysh" },
	{ "tmh", "Tamashek" },
	{ "tn", "Tswana" },
	{ "to", "Tongan" },
	{ "tog", "Nyasa Tonga" },
	{ "tpi", "Tok Pisin" },
	{ "tr", "Turkish" },
	{ "tru", "Turoyo" },
	{ "trv", "Taroko" },
	{ "ts", "Tsonga" },
	{ "tsd", "Tsakonian" },
	{ "tsi", "Tsimshian" },
	{ "tt", "Tatar" },
	{ "ttt", "Muslim Tat" },
	{ "tum", "Tumbuka" },
	{ "tvl", "Tuvalu" },
	{ "tw", "Twi" },
	{ "twq", "Tasawaq" },
	{ "ty", "Tahitian" },
	{ "tyv", "Tuvinian" },
	{ "tzm", "Central Atlas Tamazight" },
	{ "udm", "Udmurt" },
	{ "ug", "Uyghur" },
	{ "uga", "Ugaritic" },
	{ "uk", "Ukrainian" },
	{ "umb", "Umbundu" },
	{ "unm", "Unami" },
	{ "ur", "Urdu" },
	{ "uz", "Uzbek" },
	{ "vai", "Vai" },
	{ "ve", "Venda" },
	{ "vec", "Venetian" },
	{ "vep", "Veps" },
	{ "vi", "Vietnamese" },
	{ "vls", "West Flemish" },
	{ "vmf", "Main-Franconian" },
	{ "vo", "Volapük" },
	{ "vot", "Votic" },
	{ "vro", "Võro" },
	{ "vun", "Vunjo" },
	{ "wa", "Walloon" },
	{ "wae", "Walser" },
	{ "wal", "Wolaytta" },
	{ "war", "Waray" },
	{ "was", "Washo" },
	{ "wbp", "Warlpiri" },
	{ "wo", "Wolof" },
	{ "wuu", "Wu Chinese" },
	{ "xal", "Kalmyk" },
	{ "xh", "Xhosa" },
	{ "xmf", "Mingrelian" },
	{ "xog", "Soga" },
	{ "yao", "Yao" },
	{ "yap", "Yapese" },
	{ "yav", "Yangben" },
	{ "ybb", "Yemba" },
	{ "yi", "Yiddish" },
	{ "yo", "Yoruba" },
	{ "yrl", "Nheengatu" },
	{ "yue", "Yue Chinese" },
	{ "yuw", "Papua New Guinea" },
	{ "za", "Zhuang" },
	{ "zap", "Zapotec" },
	{ "zbl", "Blissymbols" },
	{ "zea", "Zeelandic" },
	{ "zen", "Zenaga" },
	{ "zgh", "Standard Moroccan Tamazight" },
	{ "zh", "Chinese" },
	{ "zu", "Zulu" },
	{ "zun", "Zuni" },
	{ "zza", "Zaza" },
	{ nullptr, nullptr },
};

// Scripts used by languages.
// Reference:
// - https://www.unicode.org/cldr/charts/48/supplemental/languages_and_scripts.html

static const char *language_script_list[][2] = {
	{ "aa", "Latn" },
	{ "ab", "Cyrl" },
	{ "ace", "Latn" },
	{ "ach", "Latn" },
	{ "ada", "Latn" },
	{ "ady", "Cyrl" },
	{ "ae", "Avst" },
	{ "aeb", "Arab" },
	{ "af", "Latn" },
	{ "afh", "Latn" },
	{ "agq", "Latn" },
	{ "ain", "Kana Latn" },
	{ "agr", "Latn" },
	{ "ak", "Latn" },
	{ "akk", "Xsux" },
	{ "akz", "Latn" },
	{ "ale", "Latn" },
	{ "aln", "Latn" },
	{ "alt", "Cyrl" },
	{ "am", "Ethi" },
	{ "an", "Latn" },
	{ "ang", "Latn" },
	{ "anp", "Deva" },
	{ "ar", "Arab Syrc" },
	{ "arc", "Armi Nbat Palm" },
	{ "arn", "Latn" },
	{ "aro", "Latn" },
	{ "arp", "Latn" },
	{ "arq", "Arab" },
	{ "ars", "Arab" },
	{ "arw", "Latn" },
	{ "ary", "Arab" },
	{ "arz", "Arab" },
	{ "as", "Beng" },
	{ "asa", "Latn" },
	{ "ast", "Latn" },
	{ "av", "Cyrl" },
	{ "avk", "Latn" },
	{ "awa", "Deva" },
	{ "ayc", "Latn" },
	{ "ay", "Latn" },
	{ "az", "Latn Arab Cyrl" },
	{ "ba", "Cyrl" },
	{ "bal", "Arab Latn" },
	{ "ban", "Latn Bali" },
	{ "bar", "Latn" },
	{ "bas", "Latn" },
	{ "bax", "Bamu" },
	{ "bbc", "Latn Batk" },
	{ "bbj", "Latn" },
	{ "be", "Cyrl" },
	{ "bej", "Arab" },
	{ "bem", "Latn" },
	{ "ber", "Latn Arab Tfng" },
	{ "bew", "Latn" },
	{ "bez", "Latn" },
	{ "bfd", "Latn" },
	{ "bfq", "Taml" },
	{ "bg", "Cyrl" },
	{ "bhb", "Deva" },
	{ "bgn", "Arab" },
	{ "bho", "Deva" },
	{ "bi", "Latn" },
	{ "bik", "Latn" },
	{ "bin", "Latn" },
	{ "bjn", "Latn" },
	{ "bkm", "Latn" },
	{ "bla", "Latn" },
	{ "bm", "Latn Nkoo" },
	{ "bn", "Beng" },
	{ "bo", "Tibt" },
	{ "bpy", "Beng" },
	{ "bqi", "Arab" },
	{ "br", "Latn" },
	{ "brh", "Arab Latn" },
	{ "brx", "Deva" },
	{ "bs", "Latn Cyrl" },
	{ "bss", "Latn" },
	{ "bua", "Cyrl" },
	{ "bug", "Latn Bug" },
	{ "bum", "Latn" },
	{ "byn", "Ethi" },
	{ "byv", "Latn" },
	{ "ca", "Latn" },
	{ "cad", "Latn" },
	{ "car", "Latn" },
	{ "cay", "Latn" },
	{ "cch", "Latn" },
	{ "ccp", "Cakm Beng" },
	{ "ce", "Cyrl" },
	{ "ceb", "Latn" },
	{ "cgg", "Latn" },
	{ "ch", "Latn" },
	{ "chb", "Latn" },
	{ "chg", "Arab" },
	{ "chk", "Latn" },
	{ "chm", "Cyrl" },
	{ "chn", "Latn" },
	{ "cho", "Latn" },
	{ "chp", "Latn Cans" },
	{ "chr", "Cher" },
	{ "chy", "Latn" },
	{ "cic", "Latn" },
	{ "ckb", "Arab" },
	{ "cmn", "Hant Hans Hani" },
	{ "co", "Latn" },
	{ "cop", "Copt Arab Grek" },
	{ "cps", "Latn" },
	{ "cr", "Cans Latn" },
	{ "crh", "Cyrl" },
	{ "crs", "Latn" },
	{ "cs", "Latn" },
	{ "csb", "Latn" },
	{ "cu", "Cyrl" },
	{ "cv", "Cyrl" },
	{ "cy", "Latn" },
	{ "da", "Latn" },
	{ "dak", "Latn" },
	{ "dar", "Cyrl" },
	{ "dav", "Latn" },
	{ "de", "Latn Runr" },
	{ "del", "Latn" },
	{ "den", "Latn Cans" },
	{ "dgr", "Latn" },
	{ "din", "Latn" },
	{ "dje", "Latn" },
	{ "doi", "Deva Arab Takr" },
	{ "dsb", "Latn" },
	{ "dtp", "Latn" },
	{ "dua", "Latn" },
	{ "dum", "Latn" },
	{ "dv", "Thaa" },
	{ "dyo", "Latn Arab" },
	{ "dyu", "Latn" },
	{ "dz", "Tibt" },
	{ "dzg", "Latn" },
	{ "ebu", "Latn" },
	{ "ee", "Latn" },
	{ "efi", "Latn" },
	{ "egl", "Latn" },
	{ "egy", "Egyp" },
	{ "eka", "Latn" },
	{ "el", "Grek" },
	{ "elx", "Xsux" },
	{ "en", "Latn Dsrt Shaw" },
	{ "enm", "Latn" },
	{ "eo", "Latn" },
	{ "es", "Latn" },
	{ "esu", "Latn" },
	{ "et", "Latn" },
	{ "eu", "Latn" },
	{ "ewo", "Latn" },
	{ "ext", "Latn" },
	{ "fa", "Arab" },
	{ "fan", "Latn" },
	{ "fat", "Latn" },
	{ "ff", "Latn Adlm" },
	{ "fi", "Latn" },
	{ "fil", "Latn Tglg" },
	{ "fit", "Latn" },
	{ "fj", "Latn" },
	{ "fo", "Latn" },
	{ "fon", "Latn" },
	{ "fr", "Latn Dupl" },
	{ "frc", "Latn" },
	{ "frm", "Latn" },
	{ "fro", "Latn" },
	{ "frp", "Latn" },
	{ "frr", "Latn" },
	{ "frs", "Latn" },
	{ "fur", "Latn" },
	{ "fy", "Latn" },
	{ "ga", "Latn" },
	{ "gaa", "Latn" },
	{ "gag", "Latn Cyrl" },
	{ "gan", "Hans Hani" },
	{ "gay", "Latn" },
	{ "gba", "Latn" },
	{ "gbz", "Arab" },
	{ "gd", "Latn" },
	{ "gez", "Ethi" },
	{ "gil", "Latn" },
	{ "gl", "Latn" },
	{ "glk", "Arab" },
	{ "gmh", "Latn" },
	{ "gn", "Latn" },
	{ "goh", "Latn" },
	{ "gom", "Deav Latn Knda Mlym Arab" },
	{ "gon", "Deva Telu" },
	{ "gor", "Latn" },
	{ "got", "Goth" },
	{ "grb", "Latn" },
	{ "grc", "Grek" },
	{ "gsw", "Latn" },
	{ "gu", "Gujr" },
	{ "guc", "Latn" },
	{ "gur", "Latn" },
	{ "guz", "Latn" },
	{ "gv", "Latn" },
	{ "gwi", "Latn" },
	{ "ha", "Latn Arab" },
	{ "hai", "Latn" },
	{ "hak", "Hans Hant Hani" },
	{ "haw", "Latn" },
	{ "he", "Hebr" },
	{ "hi", "Deva Latn Mahj" },
	{ "hif", "Deva Latn" },
	{ "hil", "Latn" },
	{ "hit", "Xsux" },
	{ "hmn", "Latn Hmng" },
	{ "ho", "Latn" },
	{ "hne", "Deva" },
	{ "hr", "Latn" },
	{ "hsb", "Latn" },
	{ "hsn", "Hans Hani" },
	{ "ht", "Latn" },
	{ "hu", "Latn" },
	{ "hup", "Latn" },
	{ "hus", "Latn" },
	{ "hy", "Armn" },
	{ "hz", "Latn" },
	{ "ia", "Latn" },
	{ "iba", "Latn" },
	{ "ibb", "Latn" },
	{ "id", "Latn Arab" },
	{ "ie", "Latn" },
	{ "ig", "Latn" },
	{ "ii", "Yiii Latn" },
	{ "ik", "Latn" },
	{ "ilo", "Latn" },
	{ "inh", "Cyrl" },
	{ "io", "Latn" },
	{ "is", "Latn" },
	{ "it", "Latn" },
	{ "iu", "Cans Latn" },
	{ "izh", "Latn" },
	{ "ja", "Jpan Hani Hira Kana" },
	{ "jam", "Latn" },
	{ "jbo", "Latn" },
	{ "jgo", "Latn" },
	{ "jmc", "Latn" },
	{ "jpr", "Hebr" },
	{ "jrb", "Hebr" },
	{ "jut", "Latn" },
	{ "jv", "Latn Java" },
	{ "ka", "Geor" },
	{ "kaa", "Cyrl Latn" },
	{ "kab", "Latn" },
	{ "kac", "Latn" },
	{ "kaj", "Latn" },
	{ "kam", "Latn" },
	{ "kaw", "Bali Java Kawi" },
	{ "kbd", "Cyrl" },
	{ "kbl", "Arab" },
	{ "kcg", "Latn" },
	{ "kde", "Latn" },
	{ "kea", "Latn" },
	{ "ken", "Latn" },
	{ "kfo", "Latn" },
	{ "kg", "Latn" },
	{ "kgp", "Latn" },
	{ "kha", "Latn Beng" },
	{ "kho", "Brah Khar" },
	{ "khq", "Latn" },
	{ "khw", "Arab" },
	{ "ki", "Latn" },
	{ "kiu", "Latn" },
	{ "kj", "Latn" },
	{ "kk", "Cyrl Arab" },
	{ "kkj", "Latn" },
	{ "kl", "Latn" },
	{ "kln", "Latn" },
	{ "km", "Khmr" },
	{ "kmb", "Latn" },
	{ "kn", "Knda" },
	{ "ko", "Kore Hang Hani" },
	{ "koi", "Cyrl" },
	{ "kok", "Deva Latn" },
	{ "kos", "Latn" },
	{ "kpe", "Latn" },
	{ "kr", "Latn" },
	{ "krc", "Cyrl" },
	{ "kri", "Latn" },
	{ "krj", "Latn" },
	{ "krl", "Latn" },
	{ "kru", "Deva" },
	{ "ks", "Arab Deva" },
	{ "ksb", "Latn" },
	{ "ksf", "Latn" },
	{ "ksh", "Latn" },
	{ "ku", "Latn Arab Cyrl" },
	{ "kum", "Cyrl" },
	{ "kut", "Latn" },
	{ "kv", "Cyrl Perm" },
	{ "kw", "Latn" },
	{ "ky", "Cyrl Arab Latn" },
	{ "la", "Latn" },
	{ "lad", "Hebr" },
	{ "lag", "Latn" },
	{ "lah", "Arab" },
	{ "lam", "Latn" },
	{ "lb", "Latn" },
	{ "lez", "Cyrl Aghb" },
	{ "lfn", "Latn Cyrl" },
	{ "lg", "Latn" },
	{ "li", "Latn" },
	{ "lij", "Latn" },
	{ "liv", "Latn" },
	{ "lkt", "Latn" },
	{ "lmo", "Latn" },
	{ "ln", "Latn" },
	{ "lo", "Laoo" },
	{ "lol", "Latn" },
	{ "lou", "Latn" },
	{ "loz", "Latn" },
	{ "lrc", "Arab" },
	{ "lt", "Latn" },
	{ "ltg", "Latn" },
	{ "lu", "Latn" },
	{ "lua", "Latn" },
	{ "lui", "Latn" },
	{ "lun", "Latn" },
	{ "luo", "Latn" },
	{ "lus", "Beng" },
	{ "luy", "Latn" },
	{ "lv", "Latn" },
	{ "lzh", "Hant" },
	{ "lzz", "Latn Geor" },
	{ "mad", "Latn" },
	{ "maf", "Latn" },
	{ "mag", "Deva" },
	{ "mai", "Deva Tirh" },
	{ "mak", "Latn Bugi" },
	{ "man", "Latn Nkoo" },
	{ "mas", "Latn" },
	{ "mde", "Arab Latn" },
	{ "mdf", "Cyrl" },
	{ "mdr", "Latn Bugi" },
	{ "men", "Latn Mend" },
	{ "mer", "Latn" },
	{ "mfe", "Latn" },
	{ "mg", "Latn" },
	{ "mga", "Latn" },
	{ "mgh", "Latn" },
	{ "mgo", "Latn" },
	{ "mh", "Latn" },
	{ "mhr", "Cyrl" },
	{ "mi", "Latn" },
	{ "mic", "Latn" },
	{ "min", "Latn" },
	{ "miq", "Latn" },
	{ "mjw", "Latn" },
	{ "mk", "Cyrl" },
	{ "ml", "Mlym" },
	{ "mn", "Cyrl Mong Phag" },
	{ "mnc", "Mong" },
	{ "mni", "Beng Mtei" },
	{ "mnw", "Mymr" },
	{ "mos", "Latn" },
	{ "moh", "Latn" },
	{ "mr", "Deva Modi" },
	{ "mrj", "Cyrl" },
	{ "ms", "Latn Arab" },
	{ "mt", "Latn" },
	{ "mua", "Latn" },
	{ "mus", "Latn" },
	{ "mwl", "Latn" },
	{ "mwr", "Deva" },
	{ "mwv", "Latn" },
	{ "my", "Mymr" },
	{ "mye", "Latn" },
	{ "myv", "Cyrl" },
	{ "mzn", "Arab" },
	{ "na", "Latn" },
	{ "nah", "Latn" },
	{ "nan", "Hans Hant Hani" },
	{ "nap", "Latn" },
	{ "naq", "Latn" },
	{ "nan", "Latn Hans Hant Hani" },
	{ "nb", "Latn" },
	{ "nd", "Latn" },
	{ "nds", "Latn" },
	{ "ne", "Deva" },
	{ "new", "Deva" },
	{ "nhn", "Latn" },
	{ "ng", "Latn" },
	{ "nia", "Latn" },
	{ "niu", "Latn" },
	{ "njo", "Latn" },
	{ "nl", "Latn" },
	{ "nmg", "Latn" },
	{ "nn", "Latn" },
	{ "nnh", "Latn" },
	{ "nog", "Cyrl" },
	{ "non", "Runr" },
	{ "nov", "Latn" },
	{ "nqo", "Nkoo" },
	{ "nr", "Latn" },
	{ "nso", "Latn" },
	{ "nus", "Latn" },
	{ "nv", "Latn" },
	{ "nwc", "Ranj Newa Deva Sidd Brah" },
	{ "ny", "Latn" },
	{ "nym", "Latn" },
	{ "nyn", "Latn" },
	{ "nyo", "Latn" },
	{ "nzi", "Latn" },
	{ "oc", "Latn" },
	{ "oj", "Cans Latn" },
	{ "om", "Latn Ethi" },
	{ "or", "Orya" },
	{ "os", "Cyrl" },
	{ "osa", "Osge Latn" },
	{ "ota", "Arab" },
	{ "pa", "Guru Arab" },
	{ "pag", "Latn" },
	{ "pal", "Phli Phlp" },
	{ "pam", "Latn" },
	{ "pap", "Latn" },
	{ "pau", "Latn" },
	{ "pcd", "Latn" },
	{ "pcm", "Latn" },
	{ "pdc", "Latn" },
	{ "pdt", "Latn" },
	{ "peo", "Xpeo" },
	{ "pfl", "Latn" },
	{ "phn", "Phnx" },
	{ "pi", "Latn Deva Mymr Sinh Thai" },
	{ "pl", "Latn" },
	{ "pms", "Latn" },
	{ "pnt", "Grek Cyrl Latn" },
	{ "pon", "Latn" },
	{ "pr", "Latn" },
	{ "prg", "Latn" },
	{ "pro", "Latn" },
	{ "prs", "Arab" },
	{ "ps", "Arab" },
	{ "pt", "Latn" },
	{ "qu", "Latn" },
	{ "quc", "Latn" },
	{ "qug", "Latn" },
	{ "quy", "Latn" },
	{ "quz", "Latn" },
	{ "raj", "Deva" },
	{ "rap", "Latn" },
	{ "rar", "Latn" },
	{ "rgn", "Latn" },
	{ "rif", "Latn Tfng" },
	{ "rm", "Latn" },
	{ "rn", "Latn" },
	{ "ro", "Latn Cyrl" },
	{ "rof", "Latn" },
	{ "rom", "Latn Cyrl" },
	{ "rtm", "Latn" },
	{ "ru", "Cyrl" },
	{ "rue", "Cyrl" },
	{ "rug", "Latn" },
	{ "rup", "Latn" },
	{ "rw", "Latn" },
	{ "rwk", "Latn" },
	{ "sa", "Deva Gran Shrd Sidd Sinh" },
	{ "sad", "Latn" },
	{ "sah", "Cyrl" },
	{ "sam", "Samr Hebr" },
	{ "saq", "Latn" },
	{ "sas", "Latn" },
	{ "sat", "Olck Beng Deva Latn Orya" },
	{ "saz", "Saur" },
	{ "sba", "Latn" },
	{ "sbp", "Latn" },
	{ "sc", "Latn" },
	{ "scn", "Latn" },
	{ "sco", "Latn" },
	{ "sd", "Arab Deva Khoj Sind" },
	{ "sdc", "Latn" },
	{ "sdh", "Arab" },
	{ "se", "Latn Cyrl Arab" },
	{ "see", "Latn" },
	{ "seh", "Latn" },
	{ "sei", "Latn" },
	{ "sel", "Cyrl" },
	{ "ses", "Latn" },
	{ "sg", "Latn" },
	{ "sga", "Latn Ogam" },
	{ "sgs", "Latn" },
	{ "sh", "Latn Cyrl" },
	{ "shi", "Tfng Latn Arab" },
	{ "shn", "Mymr" },
	{ "shs", "Latn Dupl" },
	{ "shu", "Arab" },
	{ "si", "Sinh" },
	{ "sid", "Latn" },
	{ "sk", "Latn" },
	{ "sl", "Latn" },
	{ "sli", "Latn" },
	{ "sly", "Latn" },
	{ "sm", "Latn" },
	{ "sma", "Latn" },
	{ "smj", "Latn" },
	{ "smn", "Latn" },
	{ "sms", "Latn" },
	{ "sn", "Latn" },
	{ "snk", "Latn" },
	{ "so", "Latn Arab Osma" },
	{ "sog", "Sogd Syrc Mani" },
	{ "son", "Latn Arab" },
	{ "sq", "Latn Elba" },
	{ "sr", "Cyrl Latn" },
	{ "srn", "Latn" },
	{ "srr", "Latn" },
	{ "ss", "Latn" },
	{ "ssy", "Latn" },
	{ "st", "Latn" },
	{ "stq", "Latn" },
	{ "su", "Latn Sund" },
	{ "suk", "Latn" },
	{ "sus", "Latn Arab" },
	{ "sux", "Xsux" },
	{ "sv", "Latn" },
	{ "sw", "Latn" },
	{ "swb", "Arab Latn" },
	{ "swc", "Arab Latn" },
	{ "syc", "Syrc" },
	{ "syr", "Syrc" },
	{ "szl", "Latn" },
	{ "ta", "Taml" },
	{ "tcy", "Knda" },
	{ "te", "Telu" },
	{ "tem", "Latn" },
	{ "teo", "Latn" },
	{ "ter", "Latn" },
	{ "tet", "Latn" },
	{ "tg", "Cyrl Arab Latn" },
	{ "th", "Thai" },
	{ "the", "Deva" },
	{ "ti", "Ethi" },
	{ "tig", "Ethi" },
	{ "tiv", "Latn" },
	{ "tk", "Latn Arab Cyrl" },
	{ "tkl", "Latn" },
	{ "tkr", "Latn Cyrl" },
	{ "tl", "Latn Arab Blis" },
	{ "tlh", "Piqd Latn" },
	{ "tli", "Latn" },
	{ "tly", "Latn Arab Cyrl" },
	{ "tmh", "Latn" },
	{ "tn", "Latn" },
	{ "to", "Latn" },
	{ "tog", "Latn" },
	{ "tpi", "Latn" },
	{ "tr", "Latn Arab" },
	{ "tru", "Latn Syrc" },
	{ "trv", "Latn" },
	{ "ts", "Latn" },
	{ "tsd", "Grek" },
	{ "tsi", "Latn" },
	{ "tt", "Cyrl" },
	{ "ttt", "Latn Arab Cyrl" },
	{ "tum", "Latn" },
	{ "tvl", "Latn" },
	{ "tw", "Latn" },
	{ "twq", "Latn" },
	{ "ty", "Latn" },
	{ "tyv", "Cyrl" },
	{ "tzm", "Latn Tfng" },
	{ "udm", "Cyrl Latn" },
	{ "ug", "Arab Cyrl Latn" },
	{ "uga", "Ugar" },
	{ "uk", "Cyrl" },
	{ "umb", "Latn" },
	{ "unm", "Latn" },
	{ "ur", "Arab" },
	{ "uz", "Latn Cyrl Arab" },
	{ "vai", "Vaii Latn" },
	{ "ve", "Latn" },
	{ "vec", "Latn" },
	{ "vep", "Latn" },
	{ "vi", "Latn Hani" },
	{ "vls", "Latn" },
	{ "vmf", "Latn" },
	{ "vo", "Latn" },
	{ "vot", "Latn" },
	{ "vro", "Latn" },
	{ "vun", "Latn" },
	{ "wa", "Latn" },
	{ "wae", "Latn" },
	{ "wal", "Ethi" },
	{ "war", "Latn" },
	{ "was", "Latn" },
	{ "wbp", "Latn" },
	{ "wo", "Latn Arab" },
	{ "wuu", "Hans Hani" },
	{ "xal", "Cyrl" },
	{ "xh", "Latn" },
	{ "xmf", "Geor" },
	{ "xog", "Latn" },
	{ "yao", "Latn" },
	{ "yap", "Latn" },
	{ "yav", "Latn" },
	{ "ybb", "Latn" },
	{ "yi", "Hebr" },
	{ "yo", "Latn" },
	{ "yrl", "Latn" },
	{ "yue", "Hant Hans Hani" },
	{ "yuw", "Latn" },
	{ "za", "Latn Hans Hani" },
	{ "zap", "Latn" },
	{ "zbl", "Blis" },
	{ "zea", "Latn" },
	{ "zen", "Tfng" },
	{ "zgh", "Tfng" },
	{ "zh", "Hans Hant Hani Bopo Latn Phag" },
	{ "zu", "Latn" },
	{ "zun", "Latn" },
	{ "zza", "Latn" },
	{ nullptr, nullptr },
};

// Additional regional variants.
// Variant name, supported languages.

static const char *locale_variants[][2] = {
	{ "valencia", "ca" },
	{ "iqtelif", "tt" },
	{ "saaho", "aa" },
	{ "tradnl", "es" },
	{ nullptr, nullptr },
};

// Script names and codes (excludes typographic variants, special codes, reserved codes and aliases for combined scripts).
// Reference:
// - https://en.wikipedia.org/wiki/ISO_15924

static const char *script_list[][2] = {
	{ "Adlam", "Adlm" },
	{ "Afaka", "Afak" },
	{ "Caucasian Albanian", "Aghb" },
	{ "Ahom", "Ahom" },
	{ "Arabic", "Arab" },
	{ "Imperial Aramaic", "Armi" },
	{ "Armenian", "Armn" },
	{ "Avestan", "Avst" },
	{ "Balinese", "Bali" },
	{ "Bamum", "Bamu" },
	{ "Bassa Vah", "Bass" },
	{ "Batak", "Batk" },
	{ "Bengali", "Beng" },
	{ "Bhaiksuki", "Bhks" },
	{ "Blissymbols", "Blis" },
	{ "Bopomofo", "Bopo" },
	{ "Brahmi", "Brah" },
	{ "Braille", "Brai" },
	{ "Buginese", "Bugi" },
	{ "Buhid", "Buhd" },
	{ "Chakma", "Cakm" },
	{ "Unified Canadian Aboriginal", "Cans" },
	{ "Carian", "Cari" },
	{ "Cham", "Cham" },
	{ "Cherokee", "Cher" },
	{ "Chorasmian", "Chrs" },
	{ "Cirth", "Cirt" },
	{ "Coptic", "Copt" },
	{ "Cypro-Minoan", "Cpmn" },
	{ "Cypriot", "Cprt" },
	{ "Cyrillic", "Cyrl" },
	{ "Devanagari", "Deva" },
	{ "Dives Akuru", "Diak" },
	{ "Dogra", "Dogr" },
	{ "Deseret", "Dsrt" },
	{ "Duployan", "Dupl" },
	{ "Egyptian demotic", "Egyd" },
	{ "Egyptian hieratic", "Egyh" },
	{ "Egyptian hieroglyphs", "Egyp" },
	{ "Elbasan", "Elba" },
	{ "Elymaic", "Elym" },
	{ "Ethiopic", "Ethi" },
	{ "Khutsuri", "Geok" },
	{ "Georgian", "Geor" },
	{ "Glagolitic", "Glag" },
	{ "Gunjala Gondi", "Gong" },
	{ "Masaram Gondi", "Gonm" },
	{ "Gothic", "Goth" },
	{ "Grantha", "Gran" },
	{ "Greek", "Grek" },
	{ "Gujarati", "Gujr" },
	{ "Gurmukhi", "Guru" },
	{ "Hangul", "Hang" },
	{ "Han", "Hani" },
	{ "Hanunoo", "Hano" },
	{ "Simplified Han", "Hans" },
	{ "Traditional Han", "Hant" },
	{ "Hatran", "Hatr" },
	{ "Hebrew", "Hebr" },
	{ "Hiragana", "Hira" },
	{ "Anatolian Hieroglyphs", "Hluw" },
	{ "Pahawh Hmong", "Hmng" },
	{ "Nyiakeng Puachue Hmong", "Hmnp" },
	{ "Old Hungarian", "Hung" },
	{ "Indus", "Inds" },
	{ "Old Italic", "Ital" },
	{ "Javanese", "Java" },
	{ "Jurchen", "Jurc" },
	{ "Kayah Li", "Kali" },
	{ "Katakana", "Kana" },
	{ "Kawi", "Kawi" },
	{ "Kharoshthi", "Khar" },
	{ "Khmer", "Khmr" },
	{ "Khojki", "Khoj" },
	{ "Khitan large script", "Kitl" },
	{ "Khitan small script", "Kits" },
	{ "Kannada", "Knda" },
	{ "Kpelle", "Kpel" },
	{ "Kaithi", "Kthi" },
	{ "Tai Tham", "Lana" },
	{ "Lao", "Laoo" },
	{ "Latin", "Latn" },
	{ "Leke", "Leke" },
	{ "Lepcha", "Lepc" },
	{ "Limbu", "Limb" },
	{ "Linear A", "Lina" },
	{ "Linear B", "Linb" },
	{ "Lisu", "Lisu" },
	{ "Loma", "Loma" },
	{ "Lycian", "Lyci" },
	{ "Lydian", "Lydi" },
	{ "Mahajani", "Mahj" },
	{ "Makasar", "Maka" },
	{ "Mandaic", "Mand" },
	{ "Manichaean", "Mani" },
	{ "Marchen", "Marc" },
	{ "Mayan Hieroglyphs", "Maya" },
	{ "Medefaidrin", "Medf" },
	{ "Mende Kikakui", "Mend" },
	{ "Meroitic Cursive", "Merc" },
	{ "Meroitic Hieroglyphs", "Mero" },
	{ "Malayalam", "Mlym" },
	{ "Modi", "Modi" },
	{ "Mongolian", "Mong" },
	{ "Moon", "Moon" },
	{ "Mro", "Mroo" },
	{ "Meitei Mayek", "Mtei" },
	{ "Multani", "Mult" },
	{ "Myanmar / Burmese", "Mymr" },
	{ "Nag Mundari", "Nagm" },
	{ "Nandinagari", "Nand" },
	{ "Old North Arabian", "Narb" },
	{ "Nabataean", "Nbat" },
	{ "Newa", "Newa" },
	{ "Naxi Dongba", "Nkdb" },
	{ "Nakhi Geba", "Nkgb" },
	{ "N'ko", "Nkoo" },
	{ "Nüshu", "Nshu" },
	{ "Ogham", "Ogam" },
	{ "Ol Chiki", "Olck" },
	{ "Old Turkic", "Orkh" },
	{ "Oriya", "Orya" },
	{ "Osage", "Osge" },
	{ "Osmanya", "Osma" },
	{ "Old Uyghur", "Ougr" },
	{ "Palmyrene", "Palm" },
	{ "Pau Cin Hau", "Pauc" },
	{ "Proto-Cuneiform", "Pcun" },
	{ "Proto-Elamite", "Pelm" },
	{ "Old Permic", "Perm" },
	{ "Phags-pa", "Phag" },
	{ "Inscriptional Pahlavi", "Phli" },
	{ "Psalter Pahlavi", "Phlp" },
	{ "Book Pahlavi", "Phlv" },
	{ "Phoenician", "Phnx" },
	{ "Klingon", "Piqd" },
	{ "Miao", "Plrd" },
	{ "Inscriptional Parthian", "Prti" },
	{ "Proto-Sinaitic", "Psin" },
	{ "Ranjana", "Ranj" },
	{ "Rejang", "Rjng" },
	{ "Hanifi Rohingya", "Rohg" },
	{ "Rongorongo", "Roro" },
	{ "Runic", "Runr" },
	{ "Samaritan", "Samr" },
	{ "Sarati", "Sara" },
	{ "Old South Arabian", "Sarb" },
	{ "Saurashtra", "Saur" },
	{ "SignWriting", "Sgnw" },
	{ "Shavian", "Shaw" },
	{ "Sharada", "Shrd" },
	{ "Shuishu", "Shui" },
	{ "Siddham", "Sidd" },
	{ "Khudawadi", "Sind" },
	{ "Sinhala", "Sinh" },
	{ "Sogdian", "Sogd" },
	{ "Old Sogdian", "Sogo" },
	{ "Sora Sompeng", "Sora" },
	{ "Soyombo", "Soyo" },
	{ "Sundanese", "Sund" },
	{ "Syloti Nagri", "Sylo" },
	{ "Syriac", "Syrc" },
	{ "Tagbanwa", "Tagb" },
	{ "Takri", "Takr" },
	{ "Tai Le", "Tale" },
	{ "New Tai Lue", "Talu" },
	{ "Tamil", "Taml" },
	{ "Tangut", "Tang" },
	{ "Tai Viet", "Tavt" },
	{ "Telugu", "Telu" },
	{ "Tengwar", "Teng" },
	{ "Tifinagh", "Tfng" },
	{ "Tagalog", "Tglg" },
	{ "Thaana", "Thaa" },
	{ "Thai", "Thai" },
	{ "Tibetan", "Tibt" },
	{ "Tirhuta", "Tirh" },
	{ "Tangsa", "Tnsa" },
	{ "Toto", "Toto" },
	{ "Ugaritic", "Ugar" },
	{ "Vai", "Vaii" },
	{ "Visible Speech", "Visp" },
	{ "Vithkuqi", "Vith" },
	{ "Warang Citi", "Wara" },
	{ "Wancho", "Wcho" },
	{ "Woleai", "Wole" },
	{ "Old Persian", "Xpeo" },
	{ "Cuneiform", "Xsux" },
	{ "Yezidi", "Yezi" },
	{ "Yi", "Yiii" },
	{ "Zanabazar Square", "Zanb" },
	{ nullptr, nullptr }
};

// Plural rules.
// Reference:
// - https://cgit.git.savannah.gnu.org/cgit/gettext.git/tree/gettext-tools/src/plural-table.c
// - https://github.com/unicode-org/cldr/blob/release-47/common/supplemental/plurals.xml
static const char *plural_rules[][2] = {
	{ "bm bo dz hnj id ig ii in ja jbo jv jw kde kea km ko lkt lo ms my nqo osa root sah ses sg su th to tpi vi wo yo yue zh", "nplurals=1; plural=0;" },
	{ "af an asa ast az bal bem bez bg brx ca ce cgg chr ckb da de dv ee el en eo es et eu fi fo fur fy gl gsw ha haw he hu ia io it jgo ji jmc ka kaj kcg kk kkj kl ks ksb ku ky lb lg lij mas mgo ml mn mr nah nb nd ne nl nn nnh no nr ny nyn om or os pap ps pt rm rof rwk saq sc sd sdh seh sn so sq ss ssy st sv sw syr ta te teo tig tk tn tr ts ug ur uz ve vo vun wae xh xog yi", "nplurals=2; plural=(n != 1);" },
	{ "ak bho csw ff fr guw hy kab ln mg nso pa pt_BR si ti wa", "nplurals=2; plural=(n > 1);" },
	{ "is mk", "nplurals=2; plural=(n%10==1 && n%100!=11);" },
	{ "tzm", "nplurals=2; plural=(n<=1 || (n>=11 && n<=99));" },
	{ "am as bn doi fa gu hi kn pcm zu", "nplurals=2; plural=(n==0 || n==1);" },
	{ "ceb fil tl", "nplurals=2; plural=(n==1 || n==2 || n==3 || (n%10!=4 && n%10!=6 && n%10!=9));" },
	{ "prg", "nplurals=3; plural=(n%10==0 || (n%100>=11 && n%100<=19) ? 0 : n%10==1 && n%100!=11 ? 1 : 2);" },
	{ "lv", "nplurals=3; plural=(n%10==1 && n%100!=11 ? 0 : n != 0 ? 1 : 2);" },
	{ "lt", "nplurals=3; plural=(n%10==1 && n%100!=11 ? 0 : n%10>=2 && (n%100<10 || n%100>=20) ? 1 : 2);" },
	{ "be hr ru sr uk", "nplurals=3; plural=(n%10==1 && n%100!=11 ? 0 : n%10>=2 && n%10<=4 && (n%100<10 || n%100>=20) ? 1 : 2);" },
	{ "bs sh", "nplurals=3; plural=(n%10==1 && n%100!=11 ? 0 : n%10>=2 && n%10<=4 && (n%100<12 || n%100>14) ? 1 : 2);" },
	{ "lag", "nplurals=3; plural=(n==0 ? 0 : (n==0 || n==1) && n!=0 ? 1 : 2);" },
	{ "blo ksh", "nplurals=3; plural=(n==0 ? 0 : n==1 ? 1 : 2);" },
	{ "shi", "nplurals=3; plural=(n==0 || n==1 ? 0 : n>=2 && n<=10 ? 1 : 2);" },
	{ "pl", "nplurals=3; plural=(n==1 ? 0 : n%10>=2 && n%10<=4 && (n%100<10 || n%100>=20) ? 1 : 2);" },
	{ "mo", "nplurals=3; plural=(n==1 ? 0 : n==0 || (n!=1 && n%100>=1 && n%100<=19) ? 1 : 2);" },
	{ "iu iw naq sat se sma smi smj smn sms", "nplurals=3; plural=(n==1 ? 0 : n==2 ? 1 : 2);" },
	{ "cs sk", "nplurals=3; plural=(n==1) ? 0 : (n>=2 && n<=4) ? 1 : 2;" },
	{ "ro", "nplurals=3; plural=n==1 ? 0 : (n==0 || (n%100 > 0 && n%100 < 20)) ? 1 : 2;" },
	{ "ga", "nplurals=3; plural=n==1 ? 0 : n==2 ? 1 : 2;" },
	{ "sl", "nplurals=4; plural=(n%100==1 ? 0 : n%100==2 ? 1 : n%100==3 || n%100==4 ? 2 : 3);" },
	{ "dsb hsb", "nplurals=4; plural=(n%100==1 ? 0 : n%100==2 ? 1 : n%100>=3 && n%100<=4 ? 2 : 3);" },
	{ "gv", "nplurals=4; plural=(n%10==1 ? 0 : n%10==2 ? 1 : n%100==0 || n%100==20 || n%100==40 || n%100==60 || n%100==80 ? 2 : 3);" },
	{ "gd", "nplurals=4; plural=(n==1 || n==11 ? 0 : n==2 || n==12 ? 1 : (n>=3 && n<=10) || (n>=13 && n<=19) ? 2 : 3);" },
	{ "br", "nplurals=5; plural=(n%10==1 && n%100!=11 && n%100!=71 && n%100!=91 ? 0 : n%10==2 && n%100!=12 && n%100!=72 && n%100!=92 ? 1 : ((n%10>=3 && n%10<=4) || n%10==9) && (n%100<10 || n%100>19) && (n%100<70 || n%100>79) && (n%100<90 || n%100>99) ? 2 : n!=0 && n%1000000==0 ? 3 : 4);" },
	{ "mt", "nplurals=5; plural=(n==1 ? 0 : n==2 ? 1 : n==0 || (n%100>=3 && n%100<=10) ? 2 : n%100>=11 && n%100<=19 ? 3 : 4);" },
	{ "kw", "nplurals=6; plural=(n==0 ? 0 : n==1 ? 1 : n%100==2 || n%100==22 || n%100==42 || n%100==62 || n%100==82 || (n%1000==0 && ((n%100000>=1000 && n%100000<=20000) || n%100000==40000 || n%100000==60000 || n%100000==80000)) || (n!=0 && n%1000000==100000) ? 2 : n%100==3 || n%100==23 || n%100==43 || n%100==63 || n%100==83 ? 3 : n!=1 && (n%100==1 || n%100==21 || n%100==41 || n%100==61 || n%100==81) ? 4 : 5);" },
	{ "ar ars", "nplurals=6; plural=(n==0 ? 0 : n==1 ? 1 : n==2 ? 2 : n%100>=3 && n%100<=10 ? 3 : n%100>=11 && n%100<=99 ? 4 : 5);" },
	{ "cy", "nplurals=6; plural=(n==0 ? 0 : n==1 ? 1 : n==2 ? 2 : n==3 ? 3 : n==6 ? 4 : 5);" },
	{ nullptr, nullptr }
};

static const struct NumSystemData {
	const char *locales;
	const char32_t *digits;
	const char32_t *percent_sign;
	const char32_t *exp_l;
	const char32_t *exp_u;
} num_system_data[] = {
	// Eastern Arabic numerals.
	{ "ar ar_AE ar_BH ar_DJ ar_EG ar_ER ar_IL ar_IQ ar_JO ar_KM ar_KW ar_LB ar_MR ar_OM ar_PS ar_QA ar_SA ar_SD ar_SO ar_SS ar_SY ar_TD ar_YE ckb ckb_IQ ckb_IR sd sd_PK sd_Arab sd_Arab_PK", U"٠١٢٣٤٥٦٧٨٩٫", U"٪", U"اس", U"اس" },
	// Persian and Urdu numerals.
	{ "fa fa_AF fa_IR ks ks_IN ks_Arab ks_Arab_IN lrc lrc_IQ lrc_IR mzn mzn_IR pa_PK pa_Arab pa_Arab_PK ps ps_AF ps_PK ur_IN uz_AF uz_Arab uz_Arab_AF", U"۰۱۲۳۴۵۶۷۸۹٫", U"٪", U"اس", U"اس" },
	// Bengali numerals.
	{ "as as_IN bn bn_BD bn_IN mni mni_IN mni_Beng mni_Beng_IN", U"০১২৩৪৫৬৭৮৯.", U"%", U"e", U"E" },
	// Devanagari numerals.
	{ "mr mr_IN ne ne_IN ne_NP sa sa_IN", U"०१२३४५६७८९.", U"%", U"e", U"E" },
	// Dzongkha numerals.
	{ "dz dz_BT", U"༠༡༢༣༤༥༦༧༨༩.", U"%", U"e", U"E" },
	// Santali numerals.
	{ "sat sat_IN sat_Olck sat_Olck_IN", U"᱐᱑᱒᱓᱔᱕᱖᱗᱘᱙.", U"%", U"e", U"E" },
	// Burmese numerals.
	{ "my my_MM", U"၀၁၂၃၄၅၆၇၈၉.", U"%", U"e", U"E" },
	// Chakma numerals.
	{ "ccp ccp_BD ccp_IN", U"𑄶𑄷𑄸𑄹𑄺𑄻𑄼𑄽𑄾𑄿.", U"%", U"e", U"E" },
	// Adlam numerals.
	{ "ff ff_Adlm_BF ff_Adlm_CM ff_Adlm_GH ff_Adlm_GM ff_Adlm_GN ff_Adlm_GW ff_Adlm_LR ff_Adlm_MR ff_Adlm_NE ff_Adlm_NG ff_Adlm_SL ff_Adlm_SN", U"𞥐𞥑𞥒𞥓𞥔𞥕𞥖𞥗𞥘𞥙.", U"%", U"𞤉", U"𞤉" },
	// End sentinel.
	{ nullptr, nullptr, nullptr, nullptr, nullptr }
};
