#ifndef CAPSTONE_RISCV_H
#define CAPSTONE_RISCV_H

/* Capstone Disassembly Engine */
/* RISC-V Backend By Rodrigo Cortes Porto <porto703@gmail.com> &
   Shawn Chang <citypw@gmail.com>, HardenedLinux@2018 */

#ifdef __cplusplus
extern "C" {
#endif

#if !defined(_MSC_VER) || !defined(_KERNEL_MODE)
#include <stdint.h>
#endif

#include "platform.h"
#include "cs_operand.h"

// GCC MIPS toolchain has a default macro called "mips" which breaks
// compilation
//#undef riscv

#ifdef _MSC_VER
#pragma warning(disable : 4201)
#endif

//> Operand type for instruction's operands
typedef enum riscv_op_type {
	RISCV_OP_INVALID = CS_OP_INVALID, // = CS_OP_INVALID (Uninitialized).
	RISCV_OP_REG = CS_OP_REG, // = CS_OP_REG (Register operand).
	RISCV_OP_IMM = CS_OP_IMM, // = CS_OP_IMM (Immediate operand).
	RISCV_OP_MEM = CS_OP_MEM, // = CS_OP_MEM (Memory operand).
	RISCV_OP_FP = CS_OP_FP, // = CS_OP_FP (FP immediate operand).
	RISCV_OP_CSR = CS_OP_SPECIAL // =  Control and Status Register.
} riscv_op_type;

// Instruction's operand referring to memory
// This is associated with RISCV_OP_MEM operand type above
typedef struct riscv_op_mem {
	unsigned int base; // base register
	int64_t disp; // displacement/offset value
} riscv_op_mem;

// Instruction operand
typedef struct cs_riscv_op {
	riscv_op_type type; // operand type
	union {
		unsigned int reg; // register value for REG operand
		int64_t imm; // immediate value for IMM operand
		double dimm; // immeidate double value for FP operands
		riscv_op_mem mem; // base/disp value for MEM operand
		uint16_t csr; // CSR system register (12-bit max in RISCV)
	};
	cs_ac_type
		access; ///< How is this operand accessed? (READ, WRITE or READ|WRITE)
} cs_riscv_op;

#define NUM_RISCV_OPS 8

// Instruction structure
typedef struct cs_riscv {
	// Does this instruction need effective address or not.
	bool need_effective_addr;
	// Number of operands of this instruction,
	// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_riscv_op operands[NUM_RISCV_OPS]; // operands for this instruction.
} cs_riscv;

//> RISCV registers
typedef enum riscv_reg {
	// generated content <RISCVGenCSRegEnum.inc> begin
	// clang-format off

	RISCV_REG_INVALID = 0,
	RISCV_REG_FFLAGS = 1,
	RISCV_REG_FRM = 2,
	RISCV_REG_SSP = 3,
	RISCV_REG_VL = 4,
	RISCV_REG_VLENB = 5,
	RISCV_REG_VTYPE = 6,
	RISCV_REG_VXRM = 7,
	RISCV_REG_VXSAT = 8,
	RISCV_REG_DUMMY_REG_PAIR_WITH_X0 = 9,
	RISCV_REG_V0 = 10,
	RISCV_REG_V1 = 11,
	RISCV_REG_V2 = 12,
	RISCV_REG_V3 = 13,
	RISCV_REG_V4 = 14,
	RISCV_REG_V5 = 15,
	RISCV_REG_V6 = 16,
	RISCV_REG_V7 = 17,
	RISCV_REG_V8 = 18,
	RISCV_REG_V9 = 19,
	RISCV_REG_V10 = 20,
	RISCV_REG_V11 = 21,
	RISCV_REG_V12 = 22,
	RISCV_REG_V13 = 23,
	RISCV_REG_V14 = 24,
	RISCV_REG_V15 = 25,
	RISCV_REG_V16 = 26,
	RISCV_REG_V17 = 27,
	RISCV_REG_V18 = 28,
	RISCV_REG_V19 = 29,
	RISCV_REG_V20 = 30,
	RISCV_REG_V21 = 31,
	RISCV_REG_V22 = 32,
	RISCV_REG_V23 = 33,
	RISCV_REG_V24 = 34,
	RISCV_REG_V25 = 35,
	RISCV_REG_V26 = 36,
	RISCV_REG_V27 = 37,
	RISCV_REG_V28 = 38,
	RISCV_REG_V29 = 39,
	RISCV_REG_V30 = 40,
	RISCV_REG_V31 = 41,
	RISCV_REG_X0 = 42,
	RISCV_REG_X1 = 43,
	RISCV_REG_X2 = 44,
	RISCV_REG_X3 = 45,
	RISCV_REG_X4 = 46,
	RISCV_REG_X5 = 47,
	RISCV_REG_X6 = 48,
	RISCV_REG_X7 = 49,
	RISCV_REG_X8 = 50,
	RISCV_REG_X9 = 51,
	RISCV_REG_X10 = 52,
	RISCV_REG_X11 = 53,
	RISCV_REG_X12 = 54,
	RISCV_REG_X13 = 55,
	RISCV_REG_X14 = 56,
	RISCV_REG_X15 = 57,
	RISCV_REG_X16 = 58,
	RISCV_REG_X17 = 59,
	RISCV_REG_X18 = 60,
	RISCV_REG_X19 = 61,
	RISCV_REG_X20 = 62,
	RISCV_REG_X21 = 63,
	RISCV_REG_X22 = 64,
	RISCV_REG_X23 = 65,
	RISCV_REG_X24 = 66,
	RISCV_REG_X25 = 67,
	RISCV_REG_X26 = 68,
	RISCV_REG_X27 = 69,
	RISCV_REG_X28 = 70,
	RISCV_REG_X29 = 71,
	RISCV_REG_X30 = 72,
	RISCV_REG_X31 = 73,
	RISCV_REG_F0_D = 74,
	RISCV_REG_F1_D = 75,
	RISCV_REG_F2_D = 76,
	RISCV_REG_F3_D = 77,
	RISCV_REG_F4_D = 78,
	RISCV_REG_F5_D = 79,
	RISCV_REG_F6_D = 80,
	RISCV_REG_F7_D = 81,
	RISCV_REG_F8_D = 82,
	RISCV_REG_F9_D = 83,
	RISCV_REG_F10_D = 84,
	RISCV_REG_F11_D = 85,
	RISCV_REG_F12_D = 86,
	RISCV_REG_F13_D = 87,
	RISCV_REG_F14_D = 88,
	RISCV_REG_F15_D = 89,
	RISCV_REG_F16_D = 90,
	RISCV_REG_F17_D = 91,
	RISCV_REG_F18_D = 92,
	RISCV_REG_F19_D = 93,
	RISCV_REG_F20_D = 94,
	RISCV_REG_F21_D = 95,
	RISCV_REG_F22_D = 96,
	RISCV_REG_F23_D = 97,
	RISCV_REG_F24_D = 98,
	RISCV_REG_F25_D = 99,
	RISCV_REG_F26_D = 100,
	RISCV_REG_F27_D = 101,
	RISCV_REG_F28_D = 102,
	RISCV_REG_F29_D = 103,
	RISCV_REG_F30_D = 104,
	RISCV_REG_F31_D = 105,
	RISCV_REG_F0_F = 106,
	RISCV_REG_F1_F = 107,
	RISCV_REG_F2_F = 108,
	RISCV_REG_F3_F = 109,
	RISCV_REG_F4_F = 110,
	RISCV_REG_F5_F = 111,
	RISCV_REG_F6_F = 112,
	RISCV_REG_F7_F = 113,
	RISCV_REG_F8_F = 114,
	RISCV_REG_F9_F = 115,
	RISCV_REG_F10_F = 116,
	RISCV_REG_F11_F = 117,
	RISCV_REG_F12_F = 118,
	RISCV_REG_F13_F = 119,
	RISCV_REG_F14_F = 120,
	RISCV_REG_F15_F = 121,
	RISCV_REG_F16_F = 122,
	RISCV_REG_F17_F = 123,
	RISCV_REG_F18_F = 124,
	RISCV_REG_F19_F = 125,
	RISCV_REG_F20_F = 126,
	RISCV_REG_F21_F = 127,
	RISCV_REG_F22_F = 128,
	RISCV_REG_F23_F = 129,
	RISCV_REG_F24_F = 130,
	RISCV_REG_F25_F = 131,
	RISCV_REG_F26_F = 132,
	RISCV_REG_F27_F = 133,
	RISCV_REG_F28_F = 134,
	RISCV_REG_F29_F = 135,
	RISCV_REG_F30_F = 136,
	RISCV_REG_F31_F = 137,
	RISCV_REG_F0_H = 138,
	RISCV_REG_F1_H = 139,
	RISCV_REG_F2_H = 140,
	RISCV_REG_F3_H = 141,
	RISCV_REG_F4_H = 142,
	RISCV_REG_F5_H = 143,
	RISCV_REG_F6_H = 144,
	RISCV_REG_F7_H = 145,
	RISCV_REG_F8_H = 146,
	RISCV_REG_F9_H = 147,
	RISCV_REG_F10_H = 148,
	RISCV_REG_F11_H = 149,
	RISCV_REG_F12_H = 150,
	RISCV_REG_F13_H = 151,
	RISCV_REG_F14_H = 152,
	RISCV_REG_F15_H = 153,
	RISCV_REG_F16_H = 154,
	RISCV_REG_F17_H = 155,
	RISCV_REG_F18_H = 156,
	RISCV_REG_F19_H = 157,
	RISCV_REG_F20_H = 158,
	RISCV_REG_F21_H = 159,
	RISCV_REG_F22_H = 160,
	RISCV_REG_F23_H = 161,
	RISCV_REG_F24_H = 162,
	RISCV_REG_F25_H = 163,
	RISCV_REG_F26_H = 164,
	RISCV_REG_F27_H = 165,
	RISCV_REG_F28_H = 166,
	RISCV_REG_F29_H = 167,
	RISCV_REG_F30_H = 168,
	RISCV_REG_F31_H = 169,
	RISCV_REG_X0_PAIR = 170,
	RISCV_REG_V0M2 = 171,
	RISCV_REG_V0M4 = 172,
	RISCV_REG_V0M8 = 173,
	RISCV_REG_V2M2 = 174,
	RISCV_REG_V4M2 = 175,
	RISCV_REG_V4M4 = 176,
	RISCV_REG_V6M2 = 177,
	RISCV_REG_V8M2 = 178,
	RISCV_REG_V8M4 = 179,
	RISCV_REG_V8M8 = 180,
	RISCV_REG_V10M2 = 181,
	RISCV_REG_V12M2 = 182,
	RISCV_REG_V12M4 = 183,
	RISCV_REG_V14M2 = 184,
	RISCV_REG_V16M2 = 185,
	RISCV_REG_V16M4 = 186,
	RISCV_REG_V16M8 = 187,
	RISCV_REG_V18M2 = 188,
	RISCV_REG_V20M2 = 189,
	RISCV_REG_V20M4 = 190,
	RISCV_REG_V22M2 = 191,
	RISCV_REG_V24M2 = 192,
	RISCV_REG_V24M4 = 193,
	RISCV_REG_V24M8 = 194,
	RISCV_REG_V26M2 = 195,
	RISCV_REG_V28M2 = 196,
	RISCV_REG_V28M4 = 197,
	RISCV_REG_V30M2 = 198,
	RISCV_REG_X2_X3 = 199,
	RISCV_REG_X4_X5 = 200,
	RISCV_REG_X6_X7 = 201,
	RISCV_REG_X8_X9 = 202,
	RISCV_REG_X10_X11 = 203,
	RISCV_REG_X12_X13 = 204,
	RISCV_REG_X14_X15 = 205,
	RISCV_REG_X16_X17 = 206,
	RISCV_REG_X18_X19 = 207,
	RISCV_REG_X20_X21 = 208,
	RISCV_REG_X22_X23 = 209,
	RISCV_REG_X24_X25 = 210,
	RISCV_REG_X26_X27 = 211,
	RISCV_REG_X28_X29 = 212,
	RISCV_REG_X30_X31 = 213,
	RISCV_REG_V1_V2 = 214,
	RISCV_REG_V2_V3 = 215,
	RISCV_REG_V3_V4 = 216,
	RISCV_REG_V4_V5 = 217,
	RISCV_REG_V5_V6 = 218,
	RISCV_REG_V6_V7 = 219,
	RISCV_REG_V7_V8 = 220,
	RISCV_REG_V8_V9 = 221,
	RISCV_REG_V9_V10 = 222,
	RISCV_REG_V10_V11 = 223,
	RISCV_REG_V11_V12 = 224,
	RISCV_REG_V12_V13 = 225,
	RISCV_REG_V13_V14 = 226,
	RISCV_REG_V14_V15 = 227,
	RISCV_REG_V15_V16 = 228,
	RISCV_REG_V16_V17 = 229,
	RISCV_REG_V17_V18 = 230,
	RISCV_REG_V18_V19 = 231,
	RISCV_REG_V19_V20 = 232,
	RISCV_REG_V20_V21 = 233,
	RISCV_REG_V21_V22 = 234,
	RISCV_REG_V22_V23 = 235,
	RISCV_REG_V23_V24 = 236,
	RISCV_REG_V24_V25 = 237,
	RISCV_REG_V25_V26 = 238,
	RISCV_REG_V26_V27 = 239,
	RISCV_REG_V27_V28 = 240,
	RISCV_REG_V28_V29 = 241,
	RISCV_REG_V29_V30 = 242,
	RISCV_REG_V30_V31 = 243,
	RISCV_REG_V0_V1 = 244,
	RISCV_REG_V2M2_V4M2 = 245,
	RISCV_REG_V4M2_V6M2 = 246,
	RISCV_REG_V6M2_V8M2 = 247,
	RISCV_REG_V8M2_V10M2 = 248,
	RISCV_REG_V10M2_V12M2 = 249,
	RISCV_REG_V12M2_V14M2 = 250,
	RISCV_REG_V14M2_V16M2 = 251,
	RISCV_REG_V16M2_V18M2 = 252,
	RISCV_REG_V18M2_V20M2 = 253,
	RISCV_REG_V20M2_V22M2 = 254,
	RISCV_REG_V22M2_V24M2 = 255,
	RISCV_REG_V24M2_V26M2 = 256,
	RISCV_REG_V26M2_V28M2 = 257,
	RISCV_REG_V28M2_V30M2 = 258,
	RISCV_REG_V0M2_V2M2 = 259,
	RISCV_REG_V4M4_V8M4 = 260,
	RISCV_REG_V8M4_V12M4 = 261,
	RISCV_REG_V12M4_V16M4 = 262,
	RISCV_REG_V16M4_V20M4 = 263,
	RISCV_REG_V20M4_V24M4 = 264,
	RISCV_REG_V24M4_V28M4 = 265,
	RISCV_REG_V0M4_V4M4 = 266,
	RISCV_REG_V1_V2_V3 = 267,
	RISCV_REG_V2_V3_V4 = 268,
	RISCV_REG_V3_V4_V5 = 269,
	RISCV_REG_V4_V5_V6 = 270,
	RISCV_REG_V5_V6_V7 = 271,
	RISCV_REG_V6_V7_V8 = 272,
	RISCV_REG_V7_V8_V9 = 273,
	RISCV_REG_V8_V9_V10 = 274,
	RISCV_REG_V9_V10_V11 = 275,
	RISCV_REG_V10_V11_V12 = 276,
	RISCV_REG_V11_V12_V13 = 277,
	RISCV_REG_V12_V13_V14 = 278,
	RISCV_REG_V13_V14_V15 = 279,
	RISCV_REG_V14_V15_V16 = 280,
	RISCV_REG_V15_V16_V17 = 281,
	RISCV_REG_V16_V17_V18 = 282,
	RISCV_REG_V17_V18_V19 = 283,
	RISCV_REG_V18_V19_V20 = 284,
	RISCV_REG_V19_V20_V21 = 285,
	RISCV_REG_V20_V21_V22 = 286,
	RISCV_REG_V21_V22_V23 = 287,
	RISCV_REG_V22_V23_V24 = 288,
	RISCV_REG_V23_V24_V25 = 289,
	RISCV_REG_V24_V25_V26 = 290,
	RISCV_REG_V25_V26_V27 = 291,
	RISCV_REG_V26_V27_V28 = 292,
	RISCV_REG_V27_V28_V29 = 293,
	RISCV_REG_V28_V29_V30 = 294,
	RISCV_REG_V29_V30_V31 = 295,
	RISCV_REG_V0_V1_V2 = 296,
	RISCV_REG_V2M2_V4M2_V6M2 = 297,
	RISCV_REG_V4M2_V6M2_V8M2 = 298,
	RISCV_REG_V6M2_V8M2_V10M2 = 299,
	RISCV_REG_V8M2_V10M2_V12M2 = 300,
	RISCV_REG_V10M2_V12M2_V14M2 = 301,
	RISCV_REG_V12M2_V14M2_V16M2 = 302,
	RISCV_REG_V14M2_V16M2_V18M2 = 303,
	RISCV_REG_V16M2_V18M2_V20M2 = 304,
	RISCV_REG_V18M2_V20M2_V22M2 = 305,
	RISCV_REG_V20M2_V22M2_V24M2 = 306,
	RISCV_REG_V22M2_V24M2_V26M2 = 307,
	RISCV_REG_V24M2_V26M2_V28M2 = 308,
	RISCV_REG_V26M2_V28M2_V30M2 = 309,
	RISCV_REG_V0M2_V2M2_V4M2 = 310,
	RISCV_REG_V1_V2_V3_V4 = 311,
	RISCV_REG_V2_V3_V4_V5 = 312,
	RISCV_REG_V3_V4_V5_V6 = 313,
	RISCV_REG_V4_V5_V6_V7 = 314,
	RISCV_REG_V5_V6_V7_V8 = 315,
	RISCV_REG_V6_V7_V8_V9 = 316,
	RISCV_REG_V7_V8_V9_V10 = 317,
	RISCV_REG_V8_V9_V10_V11 = 318,
	RISCV_REG_V9_V10_V11_V12 = 319,
	RISCV_REG_V10_V11_V12_V13 = 320,
	RISCV_REG_V11_V12_V13_V14 = 321,
	RISCV_REG_V12_V13_V14_V15 = 322,
	RISCV_REG_V13_V14_V15_V16 = 323,
	RISCV_REG_V14_V15_V16_V17 = 324,
	RISCV_REG_V15_V16_V17_V18 = 325,
	RISCV_REG_V16_V17_V18_V19 = 326,
	RISCV_REG_V17_V18_V19_V20 = 327,
	RISCV_REG_V18_V19_V20_V21 = 328,
	RISCV_REG_V19_V20_V21_V22 = 329,
	RISCV_REG_V20_V21_V22_V23 = 330,
	RISCV_REG_V21_V22_V23_V24 = 331,
	RISCV_REG_V22_V23_V24_V25 = 332,
	RISCV_REG_V23_V24_V25_V26 = 333,
	RISCV_REG_V24_V25_V26_V27 = 334,
	RISCV_REG_V25_V26_V27_V28 = 335,
	RISCV_REG_V26_V27_V28_V29 = 336,
	RISCV_REG_V27_V28_V29_V30 = 337,
	RISCV_REG_V28_V29_V30_V31 = 338,
	RISCV_REG_V0_V1_V2_V3 = 339,
	RISCV_REG_V2M2_V4M2_V6M2_V8M2 = 340,
	RISCV_REG_V4M2_V6M2_V8M2_V10M2 = 341,
	RISCV_REG_V6M2_V8M2_V10M2_V12M2 = 342,
	RISCV_REG_V8M2_V10M2_V12M2_V14M2 = 343,
	RISCV_REG_V10M2_V12M2_V14M2_V16M2 = 344,
	RISCV_REG_V12M2_V14M2_V16M2_V18M2 = 345,
	RISCV_REG_V14M2_V16M2_V18M2_V20M2 = 346,
	RISCV_REG_V16M2_V18M2_V20M2_V22M2 = 347,
	RISCV_REG_V18M2_V20M2_V22M2_V24M2 = 348,
	RISCV_REG_V20M2_V22M2_V24M2_V26M2 = 349,
	RISCV_REG_V22M2_V24M2_V26M2_V28M2 = 350,
	RISCV_REG_V24M2_V26M2_V28M2_V30M2 = 351,
	RISCV_REG_V0M2_V2M2_V4M2_V6M2 = 352,
	RISCV_REG_V1_V2_V3_V4_V5 = 353,
	RISCV_REG_V2_V3_V4_V5_V6 = 354,
	RISCV_REG_V3_V4_V5_V6_V7 = 355,
	RISCV_REG_V4_V5_V6_V7_V8 = 356,
	RISCV_REG_V5_V6_V7_V8_V9 = 357,
	RISCV_REG_V6_V7_V8_V9_V10 = 358,
	RISCV_REG_V7_V8_V9_V10_V11 = 359,
	RISCV_REG_V8_V9_V10_V11_V12 = 360,
	RISCV_REG_V9_V10_V11_V12_V13 = 361,
	RISCV_REG_V10_V11_V12_V13_V14 = 362,
	RISCV_REG_V11_V12_V13_V14_V15 = 363,
	RISCV_REG_V12_V13_V14_V15_V16 = 364,
	RISCV_REG_V13_V14_V15_V16_V17 = 365,
	RISCV_REG_V14_V15_V16_V17_V18 = 366,
	RISCV_REG_V15_V16_V17_V18_V19 = 367,
	RISCV_REG_V16_V17_V18_V19_V20 = 368,
	RISCV_REG_V17_V18_V19_V20_V21 = 369,
	RISCV_REG_V18_V19_V20_V21_V22 = 370,
	RISCV_REG_V19_V20_V21_V22_V23 = 371,
	RISCV_REG_V20_V21_V22_V23_V24 = 372,
	RISCV_REG_V21_V22_V23_V24_V25 = 373,
	RISCV_REG_V22_V23_V24_V25_V26 = 374,
	RISCV_REG_V23_V24_V25_V26_V27 = 375,
	RISCV_REG_V24_V25_V26_V27_V28 = 376,
	RISCV_REG_V25_V26_V27_V28_V29 = 377,
	RISCV_REG_V26_V27_V28_V29_V30 = 378,
	RISCV_REG_V27_V28_V29_V30_V31 = 379,
	RISCV_REG_V0_V1_V2_V3_V4 = 380,
	RISCV_REG_V1_V2_V3_V4_V5_V6 = 381,
	RISCV_REG_V2_V3_V4_V5_V6_V7 = 382,
	RISCV_REG_V3_V4_V5_V6_V7_V8 = 383,
	RISCV_REG_V4_V5_V6_V7_V8_V9 = 384,
	RISCV_REG_V5_V6_V7_V8_V9_V10 = 385,
	RISCV_REG_V6_V7_V8_V9_V10_V11 = 386,
	RISCV_REG_V7_V8_V9_V10_V11_V12 = 387,
	RISCV_REG_V8_V9_V10_V11_V12_V13 = 388,
	RISCV_REG_V9_V10_V11_V12_V13_V14 = 389,
	RISCV_REG_V10_V11_V12_V13_V14_V15 = 390,
	RISCV_REG_V11_V12_V13_V14_V15_V16 = 391,
	RISCV_REG_V12_V13_V14_V15_V16_V17 = 392,
	RISCV_REG_V13_V14_V15_V16_V17_V18 = 393,
	RISCV_REG_V14_V15_V16_V17_V18_V19 = 394,
	RISCV_REG_V15_V16_V17_V18_V19_V20 = 395,
	RISCV_REG_V16_V17_V18_V19_V20_V21 = 396,
	RISCV_REG_V17_V18_V19_V20_V21_V22 = 397,
	RISCV_REG_V18_V19_V20_V21_V22_V23 = 398,
	RISCV_REG_V19_V20_V21_V22_V23_V24 = 399,
	RISCV_REG_V20_V21_V22_V23_V24_V25 = 400,
	RISCV_REG_V21_V22_V23_V24_V25_V26 = 401,
	RISCV_REG_V22_V23_V24_V25_V26_V27 = 402,
	RISCV_REG_V23_V24_V25_V26_V27_V28 = 403,
	RISCV_REG_V24_V25_V26_V27_V28_V29 = 404,
	RISCV_REG_V25_V26_V27_V28_V29_V30 = 405,
	RISCV_REG_V26_V27_V28_V29_V30_V31 = 406,
	RISCV_REG_V0_V1_V2_V3_V4_V5 = 407,
	RISCV_REG_V1_V2_V3_V4_V5_V6_V7 = 408,
	RISCV_REG_V2_V3_V4_V5_V6_V7_V8 = 409,
	RISCV_REG_V3_V4_V5_V6_V7_V8_V9 = 410,
	RISCV_REG_V4_V5_V6_V7_V8_V9_V10 = 411,
	RISCV_REG_V5_V6_V7_V8_V9_V10_V11 = 412,
	RISCV_REG_V6_V7_V8_V9_V10_V11_V12 = 413,
	RISCV_REG_V7_V8_V9_V10_V11_V12_V13 = 414,
	RISCV_REG_V8_V9_V10_V11_V12_V13_V14 = 415,
	RISCV_REG_V9_V10_V11_V12_V13_V14_V15 = 416,
	RISCV_REG_V10_V11_V12_V13_V14_V15_V16 = 417,
	RISCV_REG_V11_V12_V13_V14_V15_V16_V17 = 418,
	RISCV_REG_V12_V13_V14_V15_V16_V17_V18 = 419,
	RISCV_REG_V13_V14_V15_V16_V17_V18_V19 = 420,
	RISCV_REG_V14_V15_V16_V17_V18_V19_V20 = 421,
	RISCV_REG_V15_V16_V17_V18_V19_V20_V21 = 422,
	RISCV_REG_V16_V17_V18_V19_V20_V21_V22 = 423,
	RISCV_REG_V17_V18_V19_V20_V21_V22_V23 = 424,
	RISCV_REG_V18_V19_V20_V21_V22_V23_V24 = 425,
	RISCV_REG_V19_V20_V21_V22_V23_V24_V25 = 426,
	RISCV_REG_V20_V21_V22_V23_V24_V25_V26 = 427,
	RISCV_REG_V21_V22_V23_V24_V25_V26_V27 = 428,
	RISCV_REG_V22_V23_V24_V25_V26_V27_V28 = 429,
	RISCV_REG_V23_V24_V25_V26_V27_V28_V29 = 430,
	RISCV_REG_V24_V25_V26_V27_V28_V29_V30 = 431,
	RISCV_REG_V25_V26_V27_V28_V29_V30_V31 = 432,
	RISCV_REG_V0_V1_V2_V3_V4_V5_V6 = 433,
	RISCV_REG_V1_V2_V3_V4_V5_V6_V7_V8 = 434,
	RISCV_REG_V2_V3_V4_V5_V6_V7_V8_V9 = 435,
	RISCV_REG_V3_V4_V5_V6_V7_V8_V9_V10 = 436,
	RISCV_REG_V4_V5_V6_V7_V8_V9_V10_V11 = 437,
	RISCV_REG_V5_V6_V7_V8_V9_V10_V11_V12 = 438,
	RISCV_REG_V6_V7_V8_V9_V10_V11_V12_V13 = 439,
	RISCV_REG_V7_V8_V9_V10_V11_V12_V13_V14 = 440,
	RISCV_REG_V8_V9_V10_V11_V12_V13_V14_V15 = 441,
	RISCV_REG_V9_V10_V11_V12_V13_V14_V15_V16 = 442,
	RISCV_REG_V10_V11_V12_V13_V14_V15_V16_V17 = 443,
	RISCV_REG_V11_V12_V13_V14_V15_V16_V17_V18 = 444,
	RISCV_REG_V12_V13_V14_V15_V16_V17_V18_V19 = 445,
	RISCV_REG_V13_V14_V15_V16_V17_V18_V19_V20 = 446,
	RISCV_REG_V14_V15_V16_V17_V18_V19_V20_V21 = 447,
	RISCV_REG_V15_V16_V17_V18_V19_V20_V21_V22 = 448,
	RISCV_REG_V16_V17_V18_V19_V20_V21_V22_V23 = 449,
	RISCV_REG_V17_V18_V19_V20_V21_V22_V23_V24 = 450,
	RISCV_REG_V18_V19_V20_V21_V22_V23_V24_V25 = 451,
	RISCV_REG_V19_V20_V21_V22_V23_V24_V25_V26 = 452,
	RISCV_REG_V20_V21_V22_V23_V24_V25_V26_V27 = 453,
	RISCV_REG_V21_V22_V23_V24_V25_V26_V27_V28 = 454,
	RISCV_REG_V22_V23_V24_V25_V26_V27_V28_V29 = 455,
	RISCV_REG_V23_V24_V25_V26_V27_V28_V29_V30 = 456,
	RISCV_REG_V24_V25_V26_V27_V28_V29_V30_V31 = 457,
	RISCV_REG_V0_V1_V2_V3_V4_V5_V6_V7 = 458,
	RISCV_REG_ENDING, // 459

	// clang-format on
	// generated content <RISCVGenCSRegEnum.inc> end
} riscv_reg;

//> RISCV instruction
typedef enum riscv_insn {
	// generated content <RISCVGenCSInsnEnum.inc> begin
	// clang-format off

	RISCV_INS_INVALID,
	RISCV_INS_ADD,
	RISCV_INS_CALL,
	RISCV_INS_FLD,
	RISCV_INS_FLH,
	RISCV_INS_FLW,
	RISCV_INS_FSD,
	RISCV_INS_FSH,
	RISCV_INS_FSW,
	RISCV_INS_JUMP,
	RISCV_INS_LA,
	RISCV_INS_LA_TLSDESC,
	RISCV_INS_LA_TLS_GD,
	RISCV_INS_LA_TLS_IE,
	RISCV_INS_LB,
	RISCV_INS_LBU,
	RISCV_INS_LD,
	RISCV_INS_LGA,
	RISCV_INS_LH,
	RISCV_INS_LHU,
	RISCV_INS_LI,
	RISCV_INS_LLA,
	RISCV_INS_LW,
	RISCV_INS_LWU,
	RISCV_INS_SB,
	RISCV_INS_SD,
	RISCV_INS_SEXT_B,
	RISCV_INS_SEXT_H,
	RISCV_INS_SH,
	RISCV_INS_SW,
	RISCV_INS_TAIL,
	RISCV_INS_JALR,
	RISCV_INS_VMSGEU_VI,
	RISCV_INS_VMSGEU_VX,
	RISCV_INS_VMSGE_VI,
	RISCV_INS_VMSGE_VX,
	RISCV_INS_VMSLTU_VI,
	RISCV_INS_VMSLT_VI,
	RISCV_INS_ZEXT_H,
	RISCV_INS_ZEXT_W,
	RISCV_INS_ADDI,
	RISCV_INS_ADDIW,
	RISCV_INS_ADDW,
	RISCV_INS_ADD_UW,
	RISCV_INS_AES32DSI,
	RISCV_INS_AES32DSMI,
	RISCV_INS_AES32ESI,
	RISCV_INS_AES32ESMI,
	RISCV_INS_AES64DS,
	RISCV_INS_AES64DSM,
	RISCV_INS_AES64ES,
	RISCV_INS_AES64ESM,
	RISCV_INS_AES64IM,
	RISCV_INS_AES64KS1I,
	RISCV_INS_AES64KS2,
	RISCV_INS_AMOADD_D,
	RISCV_INS_AMOADD_D_AQ,
	RISCV_INS_AMOADD_D_AQRL,
	RISCV_INS_AMOADD_D_RL,
	RISCV_INS_AMOADD_W,
	RISCV_INS_AMOADD_W_AQ,
	RISCV_INS_AMOADD_W_AQRL,
	RISCV_INS_AMOADD_W_RL,
	RISCV_INS_AMOAND_D,
	RISCV_INS_AMOAND_D_AQ,
	RISCV_INS_AMOAND_D_AQRL,
	RISCV_INS_AMOAND_D_RL,
	RISCV_INS_AMOAND_W,
	RISCV_INS_AMOAND_W_AQ,
	RISCV_INS_AMOAND_W_AQRL,
	RISCV_INS_AMOAND_W_RL,
	RISCV_INS_AMOCAS_D,
	RISCV_INS_AMOCAS_D_AQ,
	RISCV_INS_AMOCAS_D_AQRL,
	RISCV_INS_AMOCAS_D_RL,
	RISCV_INS_AMOCAS_Q,
	RISCV_INS_AMOCAS_Q_AQ,
	RISCV_INS_AMOCAS_Q_AQRL,
	RISCV_INS_AMOCAS_Q_RL,
	RISCV_INS_AMOCAS_W,
	RISCV_INS_AMOCAS_W_AQ,
	RISCV_INS_AMOCAS_W_AQRL,
	RISCV_INS_AMOCAS_W_RL,
	RISCV_INS_AMOMAXU_D,
	RISCV_INS_AMOMAXU_D_AQ,
	RISCV_INS_AMOMAXU_D_AQRL,
	RISCV_INS_AMOMAXU_D_RL,
	RISCV_INS_AMOMAXU_W,
	RISCV_INS_AMOMAXU_W_AQ,
	RISCV_INS_AMOMAXU_W_AQRL,
	RISCV_INS_AMOMAXU_W_RL,
	RISCV_INS_AMOMAX_D,
	RISCV_INS_AMOMAX_D_AQ,
	RISCV_INS_AMOMAX_D_AQRL,
	RISCV_INS_AMOMAX_D_RL,
	RISCV_INS_AMOMAX_W,
	RISCV_INS_AMOMAX_W_AQ,
	RISCV_INS_AMOMAX_W_AQRL,
	RISCV_INS_AMOMAX_W_RL,
	RISCV_INS_AMOMINU_D,
	RISCV_INS_AMOMINU_D_AQ,
	RISCV_INS_AMOMINU_D_AQRL,
	RISCV_INS_AMOMINU_D_RL,
	RISCV_INS_AMOMINU_W,
	RISCV_INS_AMOMINU_W_AQ,
	RISCV_INS_AMOMINU_W_AQRL,
	RISCV_INS_AMOMINU_W_RL,
	RISCV_INS_AMOMIN_D,
	RISCV_INS_AMOMIN_D_AQ,
	RISCV_INS_AMOMIN_D_AQRL,
	RISCV_INS_AMOMIN_D_RL,
	RISCV_INS_AMOMIN_W,
	RISCV_INS_AMOMIN_W_AQ,
	RISCV_INS_AMOMIN_W_AQRL,
	RISCV_INS_AMOMIN_W_RL,
	RISCV_INS_AMOOR_D,
	RISCV_INS_AMOOR_D_AQ,
	RISCV_INS_AMOOR_D_AQRL,
	RISCV_INS_AMOOR_D_RL,
	RISCV_INS_AMOOR_W,
	RISCV_INS_AMOOR_W_AQ,
	RISCV_INS_AMOOR_W_AQRL,
	RISCV_INS_AMOOR_W_RL,
	RISCV_INS_AMOSWAP_D,
	RISCV_INS_AMOSWAP_D_AQ,
	RISCV_INS_AMOSWAP_D_AQRL,
	RISCV_INS_AMOSWAP_D_RL,
	RISCV_INS_AMOSWAP_W,
	RISCV_INS_AMOSWAP_W_AQ,
	RISCV_INS_AMOSWAP_W_AQRL,
	RISCV_INS_AMOSWAP_W_RL,
	RISCV_INS_AMOXOR_D,
	RISCV_INS_AMOXOR_D_AQ,
	RISCV_INS_AMOXOR_D_AQRL,
	RISCV_INS_AMOXOR_D_RL,
	RISCV_INS_AMOXOR_W,
	RISCV_INS_AMOXOR_W_AQ,
	RISCV_INS_AMOXOR_W_AQRL,
	RISCV_INS_AMOXOR_W_RL,
	RISCV_INS_AND,
	RISCV_INS_ANDI,
	RISCV_INS_ANDN,
	RISCV_INS_AUIPC,
	RISCV_INS_BCLR,
	RISCV_INS_BCLRI,
	RISCV_INS_BEQ,
	RISCV_INS_BEXT,
	RISCV_INS_BEXTI,
	RISCV_INS_BGE,
	RISCV_INS_BGEU,
	RISCV_INS_BINV,
	RISCV_INS_BINVI,
	RISCV_INS_BLT,
	RISCV_INS_BLTU,
	RISCV_INS_BNE,
	RISCV_INS_BREV8,
	RISCV_INS_BSET,
	RISCV_INS_BSETI,
	RISCV_INS_CBO_CLEAN,
	RISCV_INS_CBO_FLUSH,
	RISCV_INS_CBO_INVAL,
	RISCV_INS_CBO_ZERO,
	RISCV_INS_CLMUL,
	RISCV_INS_CLMULH,
	RISCV_INS_CLMULR,
	RISCV_INS_CLZ,
	RISCV_INS_CLZW,
	RISCV_INS_CMOP_1,
	RISCV_INS_CMOP_11,
	RISCV_INS_CMOP_13,
	RISCV_INS_CMOP_15,
	RISCV_INS_CMOP_3,
	RISCV_INS_CMOP_5,
	RISCV_INS_CMOP_7,
	RISCV_INS_CMOP_9,
	RISCV_INS_CM_JALT,
	RISCV_INS_CM_JT,
	RISCV_INS_CM_MVA01S,
	RISCV_INS_CM_MVSA01,
	RISCV_INS_CM_POP,
	RISCV_INS_CM_POPRET,
	RISCV_INS_CM_POPRETZ,
	RISCV_INS_CM_PUSH,
	RISCV_INS_CPOP,
	RISCV_INS_CPOPW,
	RISCV_INS_CSRRC,
	RISCV_INS_CSRRCI,
	RISCV_INS_CSRRS,
	RISCV_INS_CSRRSI,
	RISCV_INS_CSRRW,
	RISCV_INS_CSRRWI,
	RISCV_INS_CTZ,
	RISCV_INS_CTZW,
	RISCV_INS_CV_ABS,
	RISCV_INS_CV_ABS_B,
	RISCV_INS_CV_ABS_H,
	RISCV_INS_CV_ADDN,
	RISCV_INS_CV_ADDNR,
	RISCV_INS_CV_ADDRN,
	RISCV_INS_CV_ADDRNR,
	RISCV_INS_CV_ADDUN,
	RISCV_INS_CV_ADDUNR,
	RISCV_INS_CV_ADDURN,
	RISCV_INS_CV_ADDURNR,
	RISCV_INS_CV_ADD_B,
	RISCV_INS_CV_ADD_DIV2,
	RISCV_INS_CV_ADD_DIV4,
	RISCV_INS_CV_ADD_DIV8,
	RISCV_INS_CV_ADD_H,
	RISCV_INS_CV_ADD_SCI_B,
	RISCV_INS_CV_ADD_SCI_H,
	RISCV_INS_CV_ADD_SC_B,
	RISCV_INS_CV_ADD_SC_H,
	RISCV_INS_CV_AND_B,
	RISCV_INS_CV_AND_H,
	RISCV_INS_CV_AND_SCI_B,
	RISCV_INS_CV_AND_SCI_H,
	RISCV_INS_CV_AND_SC_B,
	RISCV_INS_CV_AND_SC_H,
	RISCV_INS_CV_AVGU_B,
	RISCV_INS_CV_AVGU_H,
	RISCV_INS_CV_AVGU_SCI_B,
	RISCV_INS_CV_AVGU_SCI_H,
	RISCV_INS_CV_AVGU_SC_B,
	RISCV_INS_CV_AVGU_SC_H,
	RISCV_INS_CV_AVG_B,
	RISCV_INS_CV_AVG_H,
	RISCV_INS_CV_AVG_SCI_B,
	RISCV_INS_CV_AVG_SCI_H,
	RISCV_INS_CV_AVG_SC_B,
	RISCV_INS_CV_AVG_SC_H,
	RISCV_INS_CV_BCLR,
	RISCV_INS_CV_BCLRR,
	RISCV_INS_CV_BEQIMM,
	RISCV_INS_CV_BITREV,
	RISCV_INS_CV_BNEIMM,
	RISCV_INS_CV_BSET,
	RISCV_INS_CV_BSETR,
	RISCV_INS_CV_CLB,
	RISCV_INS_CV_CLIP,
	RISCV_INS_CV_CLIPR,
	RISCV_INS_CV_CLIPU,
	RISCV_INS_CV_CLIPUR,
	RISCV_INS_CV_CMPEQ_B,
	RISCV_INS_CV_CMPEQ_H,
	RISCV_INS_CV_CMPEQ_SCI_B,
	RISCV_INS_CV_CMPEQ_SCI_H,
	RISCV_INS_CV_CMPEQ_SC_B,
	RISCV_INS_CV_CMPEQ_SC_H,
	RISCV_INS_CV_CMPGEU_B,
	RISCV_INS_CV_CMPGEU_H,
	RISCV_INS_CV_CMPGEU_SCI_B,
	RISCV_INS_CV_CMPGEU_SCI_H,
	RISCV_INS_CV_CMPGEU_SC_B,
	RISCV_INS_CV_CMPGEU_SC_H,
	RISCV_INS_CV_CMPGE_B,
	RISCV_INS_CV_CMPGE_H,
	RISCV_INS_CV_CMPGE_SCI_B,
	RISCV_INS_CV_CMPGE_SCI_H,
	RISCV_INS_CV_CMPGE_SC_B,
	RISCV_INS_CV_CMPGE_SC_H,
	RISCV_INS_CV_CMPGTU_B,
	RISCV_INS_CV_CMPGTU_H,
	RISCV_INS_CV_CMPGTU_SCI_B,
	RISCV_INS_CV_CMPGTU_SCI_H,
	RISCV_INS_CV_CMPGTU_SC_B,
	RISCV_INS_CV_CMPGTU_SC_H,
	RISCV_INS_CV_CMPGT_B,
	RISCV_INS_CV_CMPGT_H,
	RISCV_INS_CV_CMPGT_SCI_B,
	RISCV_INS_CV_CMPGT_SCI_H,
	RISCV_INS_CV_CMPGT_SC_B,
	RISCV_INS_CV_CMPGT_SC_H,
	RISCV_INS_CV_CMPLEU_B,
	RISCV_INS_CV_CMPLEU_H,
	RISCV_INS_CV_CMPLEU_SCI_B,
	RISCV_INS_CV_CMPLEU_SCI_H,
	RISCV_INS_CV_CMPLEU_SC_B,
	RISCV_INS_CV_CMPLEU_SC_H,
	RISCV_INS_CV_CMPLE_B,
	RISCV_INS_CV_CMPLE_H,
	RISCV_INS_CV_CMPLE_SCI_B,
	RISCV_INS_CV_CMPLE_SCI_H,
	RISCV_INS_CV_CMPLE_SC_B,
	RISCV_INS_CV_CMPLE_SC_H,
	RISCV_INS_CV_CMPLTU_B,
	RISCV_INS_CV_CMPLTU_H,
	RISCV_INS_CV_CMPLTU_SCI_B,
	RISCV_INS_CV_CMPLTU_SCI_H,
	RISCV_INS_CV_CMPLTU_SC_B,
	RISCV_INS_CV_CMPLTU_SC_H,
	RISCV_INS_CV_CMPLT_B,
	RISCV_INS_CV_CMPLT_H,
	RISCV_INS_CV_CMPLT_SCI_B,
	RISCV_INS_CV_CMPLT_SCI_H,
	RISCV_INS_CV_CMPLT_SC_B,
	RISCV_INS_CV_CMPLT_SC_H,
	RISCV_INS_CV_CMPNE_B,
	RISCV_INS_CV_CMPNE_H,
	RISCV_INS_CV_CMPNE_SCI_B,
	RISCV_INS_CV_CMPNE_SCI_H,
	RISCV_INS_CV_CMPNE_SC_B,
	RISCV_INS_CV_CMPNE_SC_H,
	RISCV_INS_CV_CNT,
	RISCV_INS_CV_CPLXCONJ,
	RISCV_INS_CV_CPLXMUL_I,
	RISCV_INS_CV_CPLXMUL_I_DIV2,
	RISCV_INS_CV_CPLXMUL_I_DIV4,
	RISCV_INS_CV_CPLXMUL_I_DIV8,
	RISCV_INS_CV_CPLXMUL_R,
	RISCV_INS_CV_CPLXMUL_R_DIV2,
	RISCV_INS_CV_CPLXMUL_R_DIV4,
	RISCV_INS_CV_CPLXMUL_R_DIV8,
	RISCV_INS_CV_DOTSP_B,
	RISCV_INS_CV_DOTSP_H,
	RISCV_INS_CV_DOTSP_SCI_B,
	RISCV_INS_CV_DOTSP_SCI_H,
	RISCV_INS_CV_DOTSP_SC_B,
	RISCV_INS_CV_DOTSP_SC_H,
	RISCV_INS_CV_DOTUP_B,
	RISCV_INS_CV_DOTUP_H,
	RISCV_INS_CV_DOTUP_SCI_B,
	RISCV_INS_CV_DOTUP_SCI_H,
	RISCV_INS_CV_DOTUP_SC_B,
	RISCV_INS_CV_DOTUP_SC_H,
	RISCV_INS_CV_DOTUSP_B,
	RISCV_INS_CV_DOTUSP_H,
	RISCV_INS_CV_DOTUSP_SCI_B,
	RISCV_INS_CV_DOTUSP_SCI_H,
	RISCV_INS_CV_DOTUSP_SC_B,
	RISCV_INS_CV_DOTUSP_SC_H,
	RISCV_INS_CV_ELW,
	RISCV_INS_CV_EXTBS,
	RISCV_INS_CV_EXTBZ,
	RISCV_INS_CV_EXTHS,
	RISCV_INS_CV_EXTHZ,
	RISCV_INS_CV_EXTRACT,
	RISCV_INS_CV_EXTRACTR,
	RISCV_INS_CV_EXTRACTU,
	RISCV_INS_CV_EXTRACTUR,
	RISCV_INS_CV_EXTRACTU_B,
	RISCV_INS_CV_EXTRACTU_H,
	RISCV_INS_CV_EXTRACT_B,
	RISCV_INS_CV_EXTRACT_H,
	RISCV_INS_CV_FF1,
	RISCV_INS_CV_FL1,
	RISCV_INS_CV_INSERT,
	RISCV_INS_CV_INSERTR,
	RISCV_INS_CV_INSERT_B,
	RISCV_INS_CV_INSERT_H,
	RISCV_INS_CV_LBU,
	RISCV_INS_CV_LB,
	RISCV_INS_CV_LHU,
	RISCV_INS_CV_LH,
	RISCV_INS_CV_LW,
	RISCV_INS_CV_MAC,
	RISCV_INS_CV_MACHHSN,
	RISCV_INS_CV_MACHHSRN,
	RISCV_INS_CV_MACHHUN,
	RISCV_INS_CV_MACHHURN,
	RISCV_INS_CV_MACSN,
	RISCV_INS_CV_MACSRN,
	RISCV_INS_CV_MACUN,
	RISCV_INS_CV_MACURN,
	RISCV_INS_CV_MAX,
	RISCV_INS_CV_MAXU,
	RISCV_INS_CV_MAXU_B,
	RISCV_INS_CV_MAXU_H,
	RISCV_INS_CV_MAXU_SCI_B,
	RISCV_INS_CV_MAXU_SCI_H,
	RISCV_INS_CV_MAXU_SC_B,
	RISCV_INS_CV_MAXU_SC_H,
	RISCV_INS_CV_MAX_B,
	RISCV_INS_CV_MAX_H,
	RISCV_INS_CV_MAX_SCI_B,
	RISCV_INS_CV_MAX_SCI_H,
	RISCV_INS_CV_MAX_SC_B,
	RISCV_INS_CV_MAX_SC_H,
	RISCV_INS_CV_MIN,
	RISCV_INS_CV_MINU,
	RISCV_INS_CV_MINU_B,
	RISCV_INS_CV_MINU_H,
	RISCV_INS_CV_MINU_SCI_B,
	RISCV_INS_CV_MINU_SCI_H,
	RISCV_INS_CV_MINU_SC_B,
	RISCV_INS_CV_MINU_SC_H,
	RISCV_INS_CV_MIN_B,
	RISCV_INS_CV_MIN_H,
	RISCV_INS_CV_MIN_SCI_B,
	RISCV_INS_CV_MIN_SCI_H,
	RISCV_INS_CV_MIN_SC_B,
	RISCV_INS_CV_MIN_SC_H,
	RISCV_INS_CV_MSU,
	RISCV_INS_CV_MULHHSN,
	RISCV_INS_CV_MULHHSRN,
	RISCV_INS_CV_MULHHUN,
	RISCV_INS_CV_MULHHURN,
	RISCV_INS_CV_MULSN,
	RISCV_INS_CV_MULSRN,
	RISCV_INS_CV_MULUN,
	RISCV_INS_CV_MULURN,
	RISCV_INS_CV_OR_B,
	RISCV_INS_CV_OR_H,
	RISCV_INS_CV_OR_SCI_B,
	RISCV_INS_CV_OR_SCI_H,
	RISCV_INS_CV_OR_SC_B,
	RISCV_INS_CV_OR_SC_H,
	RISCV_INS_CV_PACK,
	RISCV_INS_CV_PACKHI_B,
	RISCV_INS_CV_PACKLO_B,
	RISCV_INS_CV_PACK_H,
	RISCV_INS_CV_ROR,
	RISCV_INS_CV_SB,
	RISCV_INS_CV_SDOTSP_B,
	RISCV_INS_CV_SDOTSP_H,
	RISCV_INS_CV_SDOTSP_SCI_B,
	RISCV_INS_CV_SDOTSP_SCI_H,
	RISCV_INS_CV_SDOTSP_SC_B,
	RISCV_INS_CV_SDOTSP_SC_H,
	RISCV_INS_CV_SDOTUP_B,
	RISCV_INS_CV_SDOTUP_H,
	RISCV_INS_CV_SDOTUP_SCI_B,
	RISCV_INS_CV_SDOTUP_SCI_H,
	RISCV_INS_CV_SDOTUP_SC_B,
	RISCV_INS_CV_SDOTUP_SC_H,
	RISCV_INS_CV_SDOTUSP_B,
	RISCV_INS_CV_SDOTUSP_H,
	RISCV_INS_CV_SDOTUSP_SCI_B,
	RISCV_INS_CV_SDOTUSP_SCI_H,
	RISCV_INS_CV_SDOTUSP_SC_B,
	RISCV_INS_CV_SDOTUSP_SC_H,
	RISCV_INS_CV_SHUFFLE2_B,
	RISCV_INS_CV_SHUFFLE2_H,
	RISCV_INS_CV_SHUFFLEI0_SCI_B,
	RISCV_INS_CV_SHUFFLEI1_SCI_B,
	RISCV_INS_CV_SHUFFLEI2_SCI_B,
	RISCV_INS_CV_SHUFFLEI3_SCI_B,
	RISCV_INS_CV_SHUFFLE_B,
	RISCV_INS_CV_SHUFFLE_H,
	RISCV_INS_CV_SHUFFLE_SCI_H,
	RISCV_INS_CV_SH,
	RISCV_INS_CV_SLET,
	RISCV_INS_CV_SLETU,
	RISCV_INS_CV_SLL_B,
	RISCV_INS_CV_SLL_H,
	RISCV_INS_CV_SLL_SCI_B,
	RISCV_INS_CV_SLL_SCI_H,
	RISCV_INS_CV_SLL_SC_B,
	RISCV_INS_CV_SLL_SC_H,
	RISCV_INS_CV_SRA_B,
	RISCV_INS_CV_SRA_H,
	RISCV_INS_CV_SRA_SCI_B,
	RISCV_INS_CV_SRA_SCI_H,
	RISCV_INS_CV_SRA_SC_B,
	RISCV_INS_CV_SRA_SC_H,
	RISCV_INS_CV_SRL_B,
	RISCV_INS_CV_SRL_H,
	RISCV_INS_CV_SRL_SCI_B,
	RISCV_INS_CV_SRL_SCI_H,
	RISCV_INS_CV_SRL_SC_B,
	RISCV_INS_CV_SRL_SC_H,
	RISCV_INS_CV_SUBN,
	RISCV_INS_CV_SUBNR,
	RISCV_INS_CV_SUBRN,
	RISCV_INS_CV_SUBRNR,
	RISCV_INS_CV_SUBROTMJ,
	RISCV_INS_CV_SUBROTMJ_DIV2,
	RISCV_INS_CV_SUBROTMJ_DIV4,
	RISCV_INS_CV_SUBROTMJ_DIV8,
	RISCV_INS_CV_SUBUN,
	RISCV_INS_CV_SUBUNR,
	RISCV_INS_CV_SUBURN,
	RISCV_INS_CV_SUBURNR,
	RISCV_INS_CV_SUB_B,
	RISCV_INS_CV_SUB_DIV2,
	RISCV_INS_CV_SUB_DIV4,
	RISCV_INS_CV_SUB_DIV8,
	RISCV_INS_CV_SUB_H,
	RISCV_INS_CV_SUB_SCI_B,
	RISCV_INS_CV_SUB_SCI_H,
	RISCV_INS_CV_SUB_SC_B,
	RISCV_INS_CV_SUB_SC_H,
	RISCV_INS_CV_SW,
	RISCV_INS_CV_XOR_B,
	RISCV_INS_CV_XOR_H,
	RISCV_INS_CV_XOR_SCI_B,
	RISCV_INS_CV_XOR_SCI_H,
	RISCV_INS_CV_XOR_SC_B,
	RISCV_INS_CV_XOR_SC_H,
	RISCV_INS_CZERO_EQZ,
	RISCV_INS_CZERO_NEZ,
	RISCV_INS_C_ADD,
	RISCV_INS_C_ADDI,
	RISCV_INS_C_ADDI16SP,
	RISCV_INS_C_ADDI4SPN,
	RISCV_INS_C_ADDIW,
	RISCV_INS_C_ADDW,
	RISCV_INS_C_AND,
	RISCV_INS_C_ANDI,
	RISCV_INS_C_BEQZ,
	RISCV_INS_C_BNEZ,
	RISCV_INS_C_EBREAK,
	RISCV_INS_C_FLD,
	RISCV_INS_C_FLDSP,
	RISCV_INS_C_FLW,
	RISCV_INS_C_FLWSP,
	RISCV_INS_C_FSD,
	RISCV_INS_C_FSDSP,
	RISCV_INS_C_FSW,
	RISCV_INS_C_FSWSP,
	RISCV_INS_C_J,
	RISCV_INS_C_JAL,
	RISCV_INS_C_JALR,
	RISCV_INS_C_JR,
	RISCV_INS_C_LBU,
	RISCV_INS_C_LD,
	RISCV_INS_C_LDSP,
	RISCV_INS_C_LH,
	RISCV_INS_C_LHU,
	RISCV_INS_C_LI,
	RISCV_INS_C_LUI,
	RISCV_INS_C_LW,
	RISCV_INS_C_LWSP,
	RISCV_INS_C_MUL,
	RISCV_INS_C_MV,
	RISCV_INS_C_NOP,
	RISCV_INS_C_NOT,
	RISCV_INS_C_OR,
	RISCV_INS_C_SB,
	RISCV_INS_C_SD,
	RISCV_INS_C_SDSP,
	RISCV_INS_C_SEXT_B,
	RISCV_INS_C_SEXT_H,
	RISCV_INS_C_SH,
	RISCV_INS_C_SLLI,
	RISCV_INS_C_SLLI64,
	RISCV_INS_C_SRAI,
	RISCV_INS_C_SRAI64,
	RISCV_INS_C_SRLI,
	RISCV_INS_C_SRLI64,
	RISCV_INS_C_SSPOPCHK,
	RISCV_INS_C_SSPUSH,
	RISCV_INS_C_SUB,
	RISCV_INS_C_SUBW,
	RISCV_INS_C_SW,
	RISCV_INS_C_SWSP,
	RISCV_INS_C_UNIMP,
	RISCV_INS_C_XOR,
	RISCV_INS_C_ZEXT_B,
	RISCV_INS_C_ZEXT_H,
	RISCV_INS_C_ZEXT_W,
	RISCV_INS_DIV,
	RISCV_INS_DIVU,
	RISCV_INS_DIVUW,
	RISCV_INS_DIVW,
	RISCV_INS_DRET,
	RISCV_INS_EBREAK,
	RISCV_INS_ECALL,
	RISCV_INS_FADD_D,
	RISCV_INS_FADD_H,
	RISCV_INS_FADD_S,
	RISCV_INS_FCLASS_D,
	RISCV_INS_FCLASS_H,
	RISCV_INS_FCLASS_S,
	RISCV_INS_FCVTMOD_W_D,
	RISCV_INS_FCVT_BF16_S,
	RISCV_INS_FCVT_D_H,
	RISCV_INS_FCVT_D_L,
	RISCV_INS_FCVT_D_LU,
	RISCV_INS_FCVT_D_S,
	RISCV_INS_FCVT_D_W,
	RISCV_INS_FCVT_D_WU,
	RISCV_INS_FCVT_H_D,
	RISCV_INS_FCVT_H_L,
	RISCV_INS_FCVT_H_LU,
	RISCV_INS_FCVT_H_S,
	RISCV_INS_FCVT_H_W,
	RISCV_INS_FCVT_H_WU,
	RISCV_INS_FCVT_LU_D,
	RISCV_INS_FCVT_LU_H,
	RISCV_INS_FCVT_LU_S,
	RISCV_INS_FCVT_L_D,
	RISCV_INS_FCVT_L_H,
	RISCV_INS_FCVT_L_S,
	RISCV_INS_FCVT_S_BF16,
	RISCV_INS_FCVT_S_D,
	RISCV_INS_FCVT_S_H,
	RISCV_INS_FCVT_S_L,
	RISCV_INS_FCVT_S_LU,
	RISCV_INS_FCVT_S_W,
	RISCV_INS_FCVT_S_WU,
	RISCV_INS_FCVT_WU_D,
	RISCV_INS_FCVT_WU_H,
	RISCV_INS_FCVT_WU_S,
	RISCV_INS_FCVT_W_D,
	RISCV_INS_FCVT_W_H,
	RISCV_INS_FCVT_W_S,
	RISCV_INS_FDIV_D,
	RISCV_INS_FDIV_H,
	RISCV_INS_FDIV_S,
	RISCV_INS_FENCE,
	RISCV_INS_FENCE_I,
	RISCV_INS_FENCE_TSO,
	RISCV_INS_FEQ_D,
	RISCV_INS_FEQ_H,
	RISCV_INS_FEQ_S,
	RISCV_INS_FLEQ_D,
	RISCV_INS_FLEQ_H,
	RISCV_INS_FLEQ_S,
	RISCV_INS_FLE_D,
	RISCV_INS_FLE_H,
	RISCV_INS_FLE_S,
	RISCV_INS_FLI_D,
	RISCV_INS_FLI_H,
	RISCV_INS_FLI_S,
	RISCV_INS_FLTQ_D,
	RISCV_INS_FLTQ_H,
	RISCV_INS_FLTQ_S,
	RISCV_INS_FLT_D,
	RISCV_INS_FLT_H,
	RISCV_INS_FLT_S,
	RISCV_INS_FMADD_D,
	RISCV_INS_FMADD_H,
	RISCV_INS_FMADD_S,
	RISCV_INS_FMAXM_D,
	RISCV_INS_FMAXM_H,
	RISCV_INS_FMAXM_S,
	RISCV_INS_FMAX_D,
	RISCV_INS_FMAX_H,
	RISCV_INS_FMAX_S,
	RISCV_INS_FMINM_D,
	RISCV_INS_FMINM_H,
	RISCV_INS_FMINM_S,
	RISCV_INS_FMIN_D,
	RISCV_INS_FMIN_H,
	RISCV_INS_FMIN_S,
	RISCV_INS_FMSUB_D,
	RISCV_INS_FMSUB_H,
	RISCV_INS_FMSUB_S,
	RISCV_INS_FMUL_D,
	RISCV_INS_FMUL_H,
	RISCV_INS_FMUL_S,
	RISCV_INS_FMVH_X_D,
	RISCV_INS_FMVP_D_X,
	RISCV_INS_FMV_D_X,
	RISCV_INS_FMV_H_X,
	RISCV_INS_FMV_W_X,
	RISCV_INS_FMV_X_D,
	RISCV_INS_FMV_X_H,
	RISCV_INS_FMV_X_W,
	RISCV_INS_FNMADD_D,
	RISCV_INS_FNMADD_H,
	RISCV_INS_FNMADD_S,
	RISCV_INS_FNMSUB_D,
	RISCV_INS_FNMSUB_H,
	RISCV_INS_FNMSUB_S,
	RISCV_INS_FROUNDNX_D,
	RISCV_INS_FROUNDNX_H,
	RISCV_INS_FROUNDNX_S,
	RISCV_INS_FROUND_D,
	RISCV_INS_FROUND_H,
	RISCV_INS_FROUND_S,
	RISCV_INS_FSGNJN_D,
	RISCV_INS_FSGNJN_H,
	RISCV_INS_FSGNJN_S,
	RISCV_INS_FSGNJX_D,
	RISCV_INS_FSGNJX_H,
	RISCV_INS_FSGNJX_S,
	RISCV_INS_FSGNJ_D,
	RISCV_INS_FSGNJ_H,
	RISCV_INS_FSGNJ_S,
	RISCV_INS_FSQRT_D,
	RISCV_INS_FSQRT_H,
	RISCV_INS_FSQRT_S,
	RISCV_INS_FSUB_D,
	RISCV_INS_FSUB_H,
	RISCV_INS_FSUB_S,
	RISCV_INS_HFENCE_GVMA,
	RISCV_INS_HFENCE_VVMA,
	RISCV_INS_HINVAL_GVMA,
	RISCV_INS_HINVAL_VVMA,
	RISCV_INS_HLVX_HU,
	RISCV_INS_HLVX_WU,
	RISCV_INS_HLV_B,
	RISCV_INS_HLV_BU,
	RISCV_INS_HLV_D,
	RISCV_INS_HLV_H,
	RISCV_INS_HLV_HU,
	RISCV_INS_HLV_W,
	RISCV_INS_HLV_WU,
	RISCV_INS_HSV_B,
	RISCV_INS_HSV_D,
	RISCV_INS_HSV_H,
	RISCV_INS_HSV_W,
	RISCV_INS_JAL,
	RISCV_INS_LR_D,
	RISCV_INS_LR_D_AQ,
	RISCV_INS_LR_D_AQRL,
	RISCV_INS_LR_D_RL,
	RISCV_INS_LR_W,
	RISCV_INS_LR_W_AQ,
	RISCV_INS_LR_W_AQRL,
	RISCV_INS_LR_W_RL,
	RISCV_INS_LUI,
	RISCV_INS_MAX,
	RISCV_INS_MAXU,
	RISCV_INS_MIN,
	RISCV_INS_MINU,
	RISCV_INS_MOP_R_0,
	RISCV_INS_MOP_R_1,
	RISCV_INS_MOP_R_10,
	RISCV_INS_MOP_R_11,
	RISCV_INS_MOP_R_12,
	RISCV_INS_MOP_R_13,
	RISCV_INS_MOP_R_14,
	RISCV_INS_MOP_R_15,
	RISCV_INS_MOP_R_16,
	RISCV_INS_MOP_R_17,
	RISCV_INS_MOP_R_18,
	RISCV_INS_MOP_R_19,
	RISCV_INS_MOP_R_2,
	RISCV_INS_MOP_R_20,
	RISCV_INS_MOP_R_21,
	RISCV_INS_MOP_R_22,
	RISCV_INS_MOP_R_23,
	RISCV_INS_MOP_R_24,
	RISCV_INS_MOP_R_25,
	RISCV_INS_MOP_R_26,
	RISCV_INS_MOP_R_27,
	RISCV_INS_MOP_R_28,
	RISCV_INS_MOP_R_29,
	RISCV_INS_MOP_R_3,
	RISCV_INS_MOP_R_30,
	RISCV_INS_MOP_R_31,
	RISCV_INS_MOP_R_4,
	RISCV_INS_MOP_R_5,
	RISCV_INS_MOP_R_6,
	RISCV_INS_MOP_R_7,
	RISCV_INS_MOP_R_8,
	RISCV_INS_MOP_R_9,
	RISCV_INS_MOP_RR_0,
	RISCV_INS_MOP_RR_1,
	RISCV_INS_MOP_RR_2,
	RISCV_INS_MOP_RR_3,
	RISCV_INS_MOP_RR_4,
	RISCV_INS_MOP_RR_5,
	RISCV_INS_MOP_RR_6,
	RISCV_INS_MOP_RR_7,
	RISCV_INS_MRET,
	RISCV_INS_MUL,
	RISCV_INS_MULH,
	RISCV_INS_MULHSU,
	RISCV_INS_MULHU,
	RISCV_INS_MULW,
	RISCV_INS_OR,
	RISCV_INS_ORC_B,
	RISCV_INS_ORI,
	RISCV_INS_ORN,
	RISCV_INS_PACK,
	RISCV_INS_PACKH,
	RISCV_INS_PACKW,
	RISCV_INS_PREFETCH_I,
	RISCV_INS_PREFETCH_R,
	RISCV_INS_PREFETCH_W,
	RISCV_INS_REM,
	RISCV_INS_REMU,
	RISCV_INS_REMUW,
	RISCV_INS_REMW,
	RISCV_INS_REV8,
	RISCV_INS_ROL,
	RISCV_INS_ROLW,
	RISCV_INS_ROR,
	RISCV_INS_RORI,
	RISCV_INS_RORIW,
	RISCV_INS_RORW,
	RISCV_INS_SC_D,
	RISCV_INS_SC_D_AQ,
	RISCV_INS_SC_D_AQRL,
	RISCV_INS_SC_D_RL,
	RISCV_INS_SC_W,
	RISCV_INS_SC_W_AQ,
	RISCV_INS_SC_W_AQRL,
	RISCV_INS_SC_W_RL,
	RISCV_INS_SFENCE_INVAL_IR,
	RISCV_INS_SFENCE_VMA,
	RISCV_INS_SFENCE_W_INVAL,
	RISCV_INS_SH1ADD,
	RISCV_INS_SH1ADD_UW,
	RISCV_INS_SH2ADD,
	RISCV_INS_SH2ADD_UW,
	RISCV_INS_SH3ADD,
	RISCV_INS_SH3ADD_UW,
	RISCV_INS_SHA256SIG0,
	RISCV_INS_SHA256SIG1,
	RISCV_INS_SHA256SUM0,
	RISCV_INS_SHA256SUM1,
	RISCV_INS_SHA512SIG0,
	RISCV_INS_SHA512SIG0H,
	RISCV_INS_SHA512SIG0L,
	RISCV_INS_SHA512SIG1,
	RISCV_INS_SHA512SIG1H,
	RISCV_INS_SHA512SIG1L,
	RISCV_INS_SHA512SUM0,
	RISCV_INS_SHA512SUM0R,
	RISCV_INS_SHA512SUM1,
	RISCV_INS_SHA512SUM1R,
	RISCV_INS_SINVAL_VMA,
	RISCV_INS_SLL,
	RISCV_INS_SLLI,
	RISCV_INS_SLLIW,
	RISCV_INS_SLLI_UW,
	RISCV_INS_SLLW,
	RISCV_INS_SLT,
	RISCV_INS_SLTI,
	RISCV_INS_SLTIU,
	RISCV_INS_SLTU,
	RISCV_INS_SM3P0,
	RISCV_INS_SM3P1,
	RISCV_INS_SM4ED,
	RISCV_INS_SM4KS,
	RISCV_INS_SRA,
	RISCV_INS_SRAI,
	RISCV_INS_SRAIW,
	RISCV_INS_SRAW,
	RISCV_INS_SRET,
	RISCV_INS_SRL,
	RISCV_INS_SRLI,
	RISCV_INS_SRLIW,
	RISCV_INS_SRLW,
	RISCV_INS_SSAMOSWAP_D,
	RISCV_INS_SSAMOSWAP_D_AQ,
	RISCV_INS_SSAMOSWAP_D_AQRL,
	RISCV_INS_SSAMOSWAP_D_RL,
	RISCV_INS_SSAMOSWAP_W,
	RISCV_INS_SSAMOSWAP_W_AQ,
	RISCV_INS_SSAMOSWAP_W_AQRL,
	RISCV_INS_SSAMOSWAP_W_RL,
	RISCV_INS_SSPOPCHK,
	RISCV_INS_SSPUSH,
	RISCV_INS_SSRDP,
	RISCV_INS_SUB,
	RISCV_INS_SUBW,
	RISCV_INS_TH_VMAQASU_VV,
	RISCV_INS_TH_VMAQASU_VX,
	RISCV_INS_TH_VMAQAUS_VX,
	RISCV_INS_TH_VMAQAU_VV,
	RISCV_INS_TH_VMAQAU_VX,
	RISCV_INS_TH_VMAQA_VV,
	RISCV_INS_TH_VMAQA_VX,
	RISCV_INS_TH_ADDSL,
	RISCV_INS_TH_DCACHE_CALL,
	RISCV_INS_TH_DCACHE_CIALL,
	RISCV_INS_TH_DCACHE_CIPA,
	RISCV_INS_TH_DCACHE_CISW,
	RISCV_INS_TH_DCACHE_CIVA,
	RISCV_INS_TH_DCACHE_CPA,
	RISCV_INS_TH_DCACHE_CPAL1,
	RISCV_INS_TH_DCACHE_CSW,
	RISCV_INS_TH_DCACHE_CVA,
	RISCV_INS_TH_DCACHE_CVAL1,
	RISCV_INS_TH_DCACHE_IALL,
	RISCV_INS_TH_DCACHE_IPA,
	RISCV_INS_TH_DCACHE_ISW,
	RISCV_INS_TH_DCACHE_IVA,
	RISCV_INS_TH_EXT,
	RISCV_INS_TH_EXTU,
	RISCV_INS_TH_FF0,
	RISCV_INS_TH_FF1,
	RISCV_INS_TH_FLRD,
	RISCV_INS_TH_FLRW,
	RISCV_INS_TH_FLURD,
	RISCV_INS_TH_FLURW,
	RISCV_INS_TH_FSRD,
	RISCV_INS_TH_FSRW,
	RISCV_INS_TH_FSURD,
	RISCV_INS_TH_FSURW,
	RISCV_INS_TH_ICACHE_IALL,
	RISCV_INS_TH_ICACHE_IALLS,
	RISCV_INS_TH_ICACHE_IPA,
	RISCV_INS_TH_ICACHE_IVA,
	RISCV_INS_TH_L2CACHE_CALL,
	RISCV_INS_TH_L2CACHE_CIALL,
	RISCV_INS_TH_L2CACHE_IALL,
	RISCV_INS_TH_LBIA,
	RISCV_INS_TH_LBIB,
	RISCV_INS_TH_LBUIA,
	RISCV_INS_TH_LBUIB,
	RISCV_INS_TH_LDD,
	RISCV_INS_TH_LDIA,
	RISCV_INS_TH_LDIB,
	RISCV_INS_TH_LHIA,
	RISCV_INS_TH_LHIB,
	RISCV_INS_TH_LHUIA,
	RISCV_INS_TH_LHUIB,
	RISCV_INS_TH_LRB,
	RISCV_INS_TH_LRBU,
	RISCV_INS_TH_LRD,
	RISCV_INS_TH_LRH,
	RISCV_INS_TH_LRHU,
	RISCV_INS_TH_LRW,
	RISCV_INS_TH_LRWU,
	RISCV_INS_TH_LURB,
	RISCV_INS_TH_LURBU,
	RISCV_INS_TH_LURD,
	RISCV_INS_TH_LURH,
	RISCV_INS_TH_LURHU,
	RISCV_INS_TH_LURW,
	RISCV_INS_TH_LURWU,
	RISCV_INS_TH_LWD,
	RISCV_INS_TH_LWIA,
	RISCV_INS_TH_LWIB,
	RISCV_INS_TH_LWUD,
	RISCV_INS_TH_LWUIA,
	RISCV_INS_TH_LWUIB,
	RISCV_INS_TH_MULA,
	RISCV_INS_TH_MULAH,
	RISCV_INS_TH_MULAW,
	RISCV_INS_TH_MULS,
	RISCV_INS_TH_MULSH,
	RISCV_INS_TH_MULSW,
	RISCV_INS_TH_MVEQZ,
	RISCV_INS_TH_MVNEZ,
	RISCV_INS_TH_REV,
	RISCV_INS_TH_REVW,
	RISCV_INS_TH_SBIA,
	RISCV_INS_TH_SBIB,
	RISCV_INS_TH_SDD,
	RISCV_INS_TH_SDIA,
	RISCV_INS_TH_SDIB,
	RISCV_INS_TH_SFENCE_VMAS,
	RISCV_INS_TH_SHIA,
	RISCV_INS_TH_SHIB,
	RISCV_INS_TH_SRB,
	RISCV_INS_TH_SRD,
	RISCV_INS_TH_SRH,
	RISCV_INS_TH_SRRI,
	RISCV_INS_TH_SRRIW,
	RISCV_INS_TH_SRW,
	RISCV_INS_TH_SURB,
	RISCV_INS_TH_SURD,
	RISCV_INS_TH_SURH,
	RISCV_INS_TH_SURW,
	RISCV_INS_TH_SWD,
	RISCV_INS_TH_SWIA,
	RISCV_INS_TH_SWIB,
	RISCV_INS_TH_SYNC,
	RISCV_INS_TH_SYNC_I,
	RISCV_INS_TH_SYNC_IS,
	RISCV_INS_TH_SYNC_S,
	RISCV_INS_TH_TST,
	RISCV_INS_TH_TSTNBZ,
	RISCV_INS_UNIMP,
	RISCV_INS_UNZIP,
	RISCV_INS_VAADDU_VV,
	RISCV_INS_VAADDU_VX,
	RISCV_INS_VAADD_VV,
	RISCV_INS_VAADD_VX,
	RISCV_INS_VADC_VIM,
	RISCV_INS_VADC_VVM,
	RISCV_INS_VADC_VXM,
	RISCV_INS_VADD_VI,
	RISCV_INS_VADD_VV,
	RISCV_INS_VADD_VX,
	RISCV_INS_VAESDF_VS,
	RISCV_INS_VAESDF_VV,
	RISCV_INS_VAESDM_VS,
	RISCV_INS_VAESDM_VV,
	RISCV_INS_VAESEF_VS,
	RISCV_INS_VAESEF_VV,
	RISCV_INS_VAESEM_VS,
	RISCV_INS_VAESEM_VV,
	RISCV_INS_VAESKF1_VI,
	RISCV_INS_VAESKF2_VI,
	RISCV_INS_VAESZ_VS,
	RISCV_INS_VANDN_VV,
	RISCV_INS_VANDN_VX,
	RISCV_INS_VAND_VI,
	RISCV_INS_VAND_VV,
	RISCV_INS_VAND_VX,
	RISCV_INS_VASUBU_VV,
	RISCV_INS_VASUBU_VX,
	RISCV_INS_VASUB_VV,
	RISCV_INS_VASUB_VX,
	RISCV_INS_VBREV8_V,
	RISCV_INS_VBREV_V,
	RISCV_INS_VCLMULH_VV,
	RISCV_INS_VCLMULH_VX,
	RISCV_INS_VCLMUL_VV,
	RISCV_INS_VCLMUL_VX,
	RISCV_INS_VCLZ_V,
	RISCV_INS_VCOMPRESS_VM,
	RISCV_INS_VCPOP_M,
	RISCV_INS_VCPOP_V,
	RISCV_INS_VCTZ_V,
	RISCV_INS_SF_VC_FV,
	RISCV_INS_SF_VC_FVV,
	RISCV_INS_SF_VC_FVW,
	RISCV_INS_SF_VC_I,
	RISCV_INS_SF_VC_IV,
	RISCV_INS_SF_VC_IVV,
	RISCV_INS_SF_VC_IVW,
	RISCV_INS_SF_VC_VV,
	RISCV_INS_SF_VC_VVV,
	RISCV_INS_SF_VC_VVW,
	RISCV_INS_SF_VC_V_FV,
	RISCV_INS_SF_VC_V_FVV,
	RISCV_INS_SF_VC_V_FVW,
	RISCV_INS_SF_VC_V_I,
	RISCV_INS_SF_VC_V_IV,
	RISCV_INS_SF_VC_V_IVV,
	RISCV_INS_SF_VC_V_IVW,
	RISCV_INS_SF_VC_V_VV,
	RISCV_INS_SF_VC_V_VVV,
	RISCV_INS_SF_VC_V_VVW,
	RISCV_INS_SF_VC_V_X,
	RISCV_INS_SF_VC_V_XV,
	RISCV_INS_SF_VC_V_XVV,
	RISCV_INS_SF_VC_V_XVW,
	RISCV_INS_SF_VC_X,
	RISCV_INS_SF_VC_XV,
	RISCV_INS_SF_VC_XVV,
	RISCV_INS_SF_VC_XVW,
	RISCV_INS_VDIVU_VV,
	RISCV_INS_VDIVU_VX,
	RISCV_INS_VDIV_VV,
	RISCV_INS_VDIV_VX,
	RISCV_INS_VFADD_VF,
	RISCV_INS_VFADD_VV,
	RISCV_INS_VFCLASS_V,
	RISCV_INS_VFCVT_F_XU_V,
	RISCV_INS_VFCVT_F_X_V,
	RISCV_INS_VFCVT_RTZ_XU_F_V,
	RISCV_INS_VFCVT_RTZ_X_F_V,
	RISCV_INS_VFCVT_XU_F_V,
	RISCV_INS_VFCVT_X_F_V,
	RISCV_INS_VFDIV_VF,
	RISCV_INS_VFDIV_VV,
	RISCV_INS_VFIRST_M,
	RISCV_INS_VFMACC_VF,
	RISCV_INS_VFMACC_VV,
	RISCV_INS_VFMADD_VF,
	RISCV_INS_VFMADD_VV,
	RISCV_INS_VFMAX_VF,
	RISCV_INS_VFMAX_VV,
	RISCV_INS_VFMERGE_VFM,
	RISCV_INS_VFMIN_VF,
	RISCV_INS_VFMIN_VV,
	RISCV_INS_VFMSAC_VF,
	RISCV_INS_VFMSAC_VV,
	RISCV_INS_VFMSUB_VF,
	RISCV_INS_VFMSUB_VV,
	RISCV_INS_VFMUL_VF,
	RISCV_INS_VFMUL_VV,
	RISCV_INS_VFMV_F_S,
	RISCV_INS_VFMV_S_F,
	RISCV_INS_VFMV_V_F,
	RISCV_INS_VFNCVTBF16_F_F_W,
	RISCV_INS_VFNCVT_F_F_W,
	RISCV_INS_VFNCVT_F_XU_W,
	RISCV_INS_VFNCVT_F_X_W,
	RISCV_INS_VFNCVT_ROD_F_F_W,
	RISCV_INS_VFNCVT_RTZ_XU_F_W,
	RISCV_INS_VFNCVT_RTZ_X_F_W,
	RISCV_INS_VFNCVT_XU_F_W,
	RISCV_INS_VFNCVT_X_F_W,
	RISCV_INS_VFNMACC_VF,
	RISCV_INS_VFNMACC_VV,
	RISCV_INS_VFNMADD_VF,
	RISCV_INS_VFNMADD_VV,
	RISCV_INS_VFNMSAC_VF,
	RISCV_INS_VFNMSAC_VV,
	RISCV_INS_VFNMSUB_VF,
	RISCV_INS_VFNMSUB_VV,
	RISCV_INS_SF_VFNRCLIP_XU_F_QF,
	RISCV_INS_SF_VFNRCLIP_X_F_QF,
	RISCV_INS_VFRDIV_VF,
	RISCV_INS_VFREC7_V,
	RISCV_INS_VFREDMAX_VS,
	RISCV_INS_VFREDMIN_VS,
	RISCV_INS_VFREDOSUM_VS,
	RISCV_INS_VFREDUSUM_VS,
	RISCV_INS_VFRSQRT7_V,
	RISCV_INS_VFRSUB_VF,
	RISCV_INS_VFSGNJN_VF,
	RISCV_INS_VFSGNJN_VV,
	RISCV_INS_VFSGNJX_VF,
	RISCV_INS_VFSGNJX_VV,
	RISCV_INS_VFSGNJ_VF,
	RISCV_INS_VFSGNJ_VV,
	RISCV_INS_VFSLIDE1DOWN_VF,
	RISCV_INS_VFSLIDE1UP_VF,
	RISCV_INS_VFSQRT_V,
	RISCV_INS_VFSUB_VF,
	RISCV_INS_VFSUB_VV,
	RISCV_INS_VFWADD_VF,
	RISCV_INS_VFWADD_VV,
	RISCV_INS_VFWADD_WF,
	RISCV_INS_VFWADD_WV,
	RISCV_INS_VFWCVTBF16_F_F_V,
	RISCV_INS_VFWCVT_F_F_V,
	RISCV_INS_VFWCVT_F_XU_V,
	RISCV_INS_VFWCVT_F_X_V,
	RISCV_INS_VFWCVT_RTZ_XU_F_V,
	RISCV_INS_VFWCVT_RTZ_X_F_V,
	RISCV_INS_VFWCVT_XU_F_V,
	RISCV_INS_VFWCVT_X_F_V,
	RISCV_INS_VFWMACCBF16_VF,
	RISCV_INS_VFWMACCBF16_VV,
	RISCV_INS_SF_VFWMACC_4X4X4,
	RISCV_INS_VFWMACC_VF,
	RISCV_INS_VFWMACC_VV,
	RISCV_INS_VFWMSAC_VF,
	RISCV_INS_VFWMSAC_VV,
	RISCV_INS_VFWMUL_VF,
	RISCV_INS_VFWMUL_VV,
	RISCV_INS_VFWNMACC_VF,
	RISCV_INS_VFWNMACC_VV,
	RISCV_INS_VFWNMSAC_VF,
	RISCV_INS_VFWNMSAC_VV,
	RISCV_INS_VFWREDOSUM_VS,
	RISCV_INS_VFWREDUSUM_VS,
	RISCV_INS_VFWSUB_VF,
	RISCV_INS_VFWSUB_VV,
	RISCV_INS_VFWSUB_WF,
	RISCV_INS_VFWSUB_WV,
	RISCV_INS_VGHSH_VV,
	RISCV_INS_VGMUL_VV,
	RISCV_INS_VID_V,
	RISCV_INS_VIOTA_M,
	RISCV_INS_VL1RE16_V,
	RISCV_INS_VL1RE32_V,
	RISCV_INS_VL1RE64_V,
	RISCV_INS_VL1RE8_V,
	RISCV_INS_VL2RE16_V,
	RISCV_INS_VL2RE32_V,
	RISCV_INS_VL2RE64_V,
	RISCV_INS_VL2RE8_V,
	RISCV_INS_VL4RE16_V,
	RISCV_INS_VL4RE32_V,
	RISCV_INS_VL4RE64_V,
	RISCV_INS_VL4RE8_V,
	RISCV_INS_VL8RE16_V,
	RISCV_INS_VL8RE32_V,
	RISCV_INS_VL8RE64_V,
	RISCV_INS_VL8RE8_V,
	RISCV_INS_VLE16FF_V,
	RISCV_INS_VLE16_V,
	RISCV_INS_VLE32FF_V,
	RISCV_INS_VLE32_V,
	RISCV_INS_VLE64FF_V,
	RISCV_INS_VLE64_V,
	RISCV_INS_VLE8FF_V,
	RISCV_INS_VLE8_V,
	RISCV_INS_VLM_V,
	RISCV_INS_VLOXEI16_V,
	RISCV_INS_VLOXEI32_V,
	RISCV_INS_VLOXEI64_V,
	RISCV_INS_VLOXEI8_V,
	RISCV_INS_VLOXSEG2EI16_V,
	RISCV_INS_VLOXSEG2EI32_V,
	RISCV_INS_VLOXSEG2EI64_V,
	RISCV_INS_VLOXSEG2EI8_V,
	RISCV_INS_VLOXSEG3EI16_V,
	RISCV_INS_VLOXSEG3EI32_V,
	RISCV_INS_VLOXSEG3EI64_V,
	RISCV_INS_VLOXSEG3EI8_V,
	RISCV_INS_VLOXSEG4EI16_V,
	RISCV_INS_VLOXSEG4EI32_V,
	RISCV_INS_VLOXSEG4EI64_V,
	RISCV_INS_VLOXSEG4EI8_V,
	RISCV_INS_VLOXSEG5EI16_V,
	RISCV_INS_VLOXSEG5EI32_V,
	RISCV_INS_VLOXSEG5EI64_V,
	RISCV_INS_VLOXSEG5EI8_V,
	RISCV_INS_VLOXSEG6EI16_V,
	RISCV_INS_VLOXSEG6EI32_V,
	RISCV_INS_VLOXSEG6EI64_V,
	RISCV_INS_VLOXSEG6EI8_V,
	RISCV_INS_VLOXSEG7EI16_V,
	RISCV_INS_VLOXSEG7EI32_V,
	RISCV_INS_VLOXSEG7EI64_V,
	RISCV_INS_VLOXSEG7EI8_V,
	RISCV_INS_VLOXSEG8EI16_V,
	RISCV_INS_VLOXSEG8EI32_V,
	RISCV_INS_VLOXSEG8EI64_V,
	RISCV_INS_VLOXSEG8EI8_V,
	RISCV_INS_VLSE16_V,
	RISCV_INS_VLSE32_V,
	RISCV_INS_VLSE64_V,
	RISCV_INS_VLSE8_V,
	RISCV_INS_VLSEG2E16FF_V,
	RISCV_INS_VLSEG2E16_V,
	RISCV_INS_VLSEG2E32FF_V,
	RISCV_INS_VLSEG2E32_V,
	RISCV_INS_VLSEG2E64FF_V,
	RISCV_INS_VLSEG2E64_V,
	RISCV_INS_VLSEG2E8FF_V,
	RISCV_INS_VLSEG2E8_V,
	RISCV_INS_VLSEG3E16FF_V,
	RISCV_INS_VLSEG3E16_V,
	RISCV_INS_VLSEG3E32FF_V,
	RISCV_INS_VLSEG3E32_V,
	RISCV_INS_VLSEG3E64FF_V,
	RISCV_INS_VLSEG3E64_V,
	RISCV_INS_VLSEG3E8FF_V,
	RISCV_INS_VLSEG3E8_V,
	RISCV_INS_VLSEG4E16FF_V,
	RISCV_INS_VLSEG4E16_V,
	RISCV_INS_VLSEG4E32FF_V,
	RISCV_INS_VLSEG4E32_V,
	RISCV_INS_VLSEG4E64FF_V,
	RISCV_INS_VLSEG4E64_V,
	RISCV_INS_VLSEG4E8FF_V,
	RISCV_INS_VLSEG4E8_V,
	RISCV_INS_VLSEG5E16FF_V,
	RISCV_INS_VLSEG5E16_V,
	RISCV_INS_VLSEG5E32FF_V,
	RISCV_INS_VLSEG5E32_V,
	RISCV_INS_VLSEG5E64FF_V,
	RISCV_INS_VLSEG5E64_V,
	RISCV_INS_VLSEG5E8FF_V,
	RISCV_INS_VLSEG5E8_V,
	RISCV_INS_VLSEG6E16FF_V,
	RISCV_INS_VLSEG6E16_V,
	RISCV_INS_VLSEG6E32FF_V,
	RISCV_INS_VLSEG6E32_V,
	RISCV_INS_VLSEG6E64FF_V,
	RISCV_INS_VLSEG6E64_V,
	RISCV_INS_VLSEG6E8FF_V,
	RISCV_INS_VLSEG6E8_V,
	RISCV_INS_VLSEG7E16FF_V,
	RISCV_INS_VLSEG7E16_V,
	RISCV_INS_VLSEG7E32FF_V,
	RISCV_INS_VLSEG7E32_V,
	RISCV_INS_VLSEG7E64FF_V,
	RISCV_INS_VLSEG7E64_V,
	RISCV_INS_VLSEG7E8FF_V,
	RISCV_INS_VLSEG7E8_V,
	RISCV_INS_VLSEG8E16FF_V,
	RISCV_INS_VLSEG8E16_V,
	RISCV_INS_VLSEG8E32FF_V,
	RISCV_INS_VLSEG8E32_V,
	RISCV_INS_VLSEG8E64FF_V,
	RISCV_INS_VLSEG8E64_V,
	RISCV_INS_VLSEG8E8FF_V,
	RISCV_INS_VLSEG8E8_V,
	RISCV_INS_VLSSEG2E16_V,
	RISCV_INS_VLSSEG2E32_V,
	RISCV_INS_VLSSEG2E64_V,
	RISCV_INS_VLSSEG2E8_V,
	RISCV_INS_VLSSEG3E16_V,
	RISCV_INS_VLSSEG3E32_V,
	RISCV_INS_VLSSEG3E64_V,
	RISCV_INS_VLSSEG3E8_V,
	RISCV_INS_VLSSEG4E16_V,
	RISCV_INS_VLSSEG4E32_V,
	RISCV_INS_VLSSEG4E64_V,
	RISCV_INS_VLSSEG4E8_V,
	RISCV_INS_VLSSEG5E16_V,
	RISCV_INS_VLSSEG5E32_V,
	RISCV_INS_VLSSEG5E64_V,
	RISCV_INS_VLSSEG5E8_V,
	RISCV_INS_VLSSEG6E16_V,
	RISCV_INS_VLSSEG6E32_V,
	RISCV_INS_VLSSEG6E64_V,
	RISCV_INS_VLSSEG6E8_V,
	RISCV_INS_VLSSEG7E16_V,
	RISCV_INS_VLSSEG7E32_V,
	RISCV_INS_VLSSEG7E64_V,
	RISCV_INS_VLSSEG7E8_V,
	RISCV_INS_VLSSEG8E16_V,
	RISCV_INS_VLSSEG8E32_V,
	RISCV_INS_VLSSEG8E64_V,
	RISCV_INS_VLSSEG8E8_V,
	RISCV_INS_VLUXEI16_V,
	RISCV_INS_VLUXEI32_V,
	RISCV_INS_VLUXEI64_V,
	RISCV_INS_VLUXEI8_V,
	RISCV_INS_VLUXSEG2EI16_V,
	RISCV_INS_VLUXSEG2EI32_V,
	RISCV_INS_VLUXSEG2EI64_V,
	RISCV_INS_VLUXSEG2EI8_V,
	RISCV_INS_VLUXSEG3EI16_V,
	RISCV_INS_VLUXSEG3EI32_V,
	RISCV_INS_VLUXSEG3EI64_V,
	RISCV_INS_VLUXSEG3EI8_V,
	RISCV_INS_VLUXSEG4EI16_V,
	RISCV_INS_VLUXSEG4EI32_V,
	RISCV_INS_VLUXSEG4EI64_V,
	RISCV_INS_VLUXSEG4EI8_V,
	RISCV_INS_VLUXSEG5EI16_V,
	RISCV_INS_VLUXSEG5EI32_V,
	RISCV_INS_VLUXSEG5EI64_V,
	RISCV_INS_VLUXSEG5EI8_V,
	RISCV_INS_VLUXSEG6EI16_V,
	RISCV_INS_VLUXSEG6EI32_V,
	RISCV_INS_VLUXSEG6EI64_V,
	RISCV_INS_VLUXSEG6EI8_V,
	RISCV_INS_VLUXSEG7EI16_V,
	RISCV_INS_VLUXSEG7EI32_V,
	RISCV_INS_VLUXSEG7EI64_V,
	RISCV_INS_VLUXSEG7EI8_V,
	RISCV_INS_VLUXSEG8EI16_V,
	RISCV_INS_VLUXSEG8EI32_V,
	RISCV_INS_VLUXSEG8EI64_V,
	RISCV_INS_VLUXSEG8EI8_V,
	RISCV_INS_VMACC_VV,
	RISCV_INS_VMACC_VX,
	RISCV_INS_VMADC_VI,
	RISCV_INS_VMADC_VIM,
	RISCV_INS_VMADC_VV,
	RISCV_INS_VMADC_VVM,
	RISCV_INS_VMADC_VX,
	RISCV_INS_VMADC_VXM,
	RISCV_INS_VMADD_VV,
	RISCV_INS_VMADD_VX,
	RISCV_INS_VMANDN_MM,
	RISCV_INS_VMAND_MM,
	RISCV_INS_VMAXU_VV,
	RISCV_INS_VMAXU_VX,
	RISCV_INS_VMAX_VV,
	RISCV_INS_VMAX_VX,
	RISCV_INS_VMERGE_VIM,
	RISCV_INS_VMERGE_VVM,
	RISCV_INS_VMERGE_VXM,
	RISCV_INS_VMFEQ_VF,
	RISCV_INS_VMFEQ_VV,
	RISCV_INS_VMFGE_VF,
	RISCV_INS_VMFGT_VF,
	RISCV_INS_VMFLE_VF,
	RISCV_INS_VMFLE_VV,
	RISCV_INS_VMFLT_VF,
	RISCV_INS_VMFLT_VV,
	RISCV_INS_VMFNE_VF,
	RISCV_INS_VMFNE_VV,
	RISCV_INS_VMINU_VV,
	RISCV_INS_VMINU_VX,
	RISCV_INS_VMIN_VV,
	RISCV_INS_VMIN_VX,
	RISCV_INS_VMNAND_MM,
	RISCV_INS_VMNOR_MM,
	RISCV_INS_VMORN_MM,
	RISCV_INS_VMOR_MM,
	RISCV_INS_VMSBC_VV,
	RISCV_INS_VMSBC_VVM,
	RISCV_INS_VMSBC_VX,
	RISCV_INS_VMSBC_VXM,
	RISCV_INS_VMSBF_M,
	RISCV_INS_VMSEQ_VI,
	RISCV_INS_VMSEQ_VV,
	RISCV_INS_VMSEQ_VX,
	RISCV_INS_VMSGTU_VI,
	RISCV_INS_VMSGTU_VX,
	RISCV_INS_VMSGT_VI,
	RISCV_INS_VMSGT_VX,
	RISCV_INS_VMSIF_M,
	RISCV_INS_VMSLEU_VI,
	RISCV_INS_VMSLEU_VV,
	RISCV_INS_VMSLEU_VX,
	RISCV_INS_VMSLE_VI,
	RISCV_INS_VMSLE_VV,
	RISCV_INS_VMSLE_VX,
	RISCV_INS_VMSLTU_VV,
	RISCV_INS_VMSLTU_VX,
	RISCV_INS_VMSLT_VV,
	RISCV_INS_VMSLT_VX,
	RISCV_INS_VMSNE_VI,
	RISCV_INS_VMSNE_VV,
	RISCV_INS_VMSNE_VX,
	RISCV_INS_VMSOF_M,
	RISCV_INS_VMULHSU_VV,
	RISCV_INS_VMULHSU_VX,
	RISCV_INS_VMULHU_VV,
	RISCV_INS_VMULHU_VX,
	RISCV_INS_VMULH_VV,
	RISCV_INS_VMULH_VX,
	RISCV_INS_VMUL_VV,
	RISCV_INS_VMUL_VX,
	RISCV_INS_VMV1R_V,
	RISCV_INS_VMV2R_V,
	RISCV_INS_VMV4R_V,
	RISCV_INS_VMV8R_V,
	RISCV_INS_VMV_S_X,
	RISCV_INS_VMV_V_I,
	RISCV_INS_VMV_V_V,
	RISCV_INS_VMV_V_X,
	RISCV_INS_VMV_X_S,
	RISCV_INS_VMXNOR_MM,
	RISCV_INS_VMXOR_MM,
	RISCV_INS_VNCLIPU_WI,
	RISCV_INS_VNCLIPU_WV,
	RISCV_INS_VNCLIPU_WX,
	RISCV_INS_VNCLIP_WI,
	RISCV_INS_VNCLIP_WV,
	RISCV_INS_VNCLIP_WX,
	RISCV_INS_VNMSAC_VV,
	RISCV_INS_VNMSAC_VX,
	RISCV_INS_VNMSUB_VV,
	RISCV_INS_VNMSUB_VX,
	RISCV_INS_VNSRA_WI,
	RISCV_INS_VNSRA_WV,
	RISCV_INS_VNSRA_WX,
	RISCV_INS_VNSRL_WI,
	RISCV_INS_VNSRL_WV,
	RISCV_INS_VNSRL_WX,
	RISCV_INS_VOR_VI,
	RISCV_INS_VOR_VV,
	RISCV_INS_VOR_VX,
	RISCV_INS_SF_VQMACCSU_2X8X2,
	RISCV_INS_SF_VQMACCSU_4X8X4,
	RISCV_INS_SF_VQMACCUS_2X8X2,
	RISCV_INS_SF_VQMACCUS_4X8X4,
	RISCV_INS_SF_VQMACCU_2X8X2,
	RISCV_INS_SF_VQMACCU_4X8X4,
	RISCV_INS_SF_VQMACC_2X8X2,
	RISCV_INS_SF_VQMACC_4X8X4,
	RISCV_INS_VREDAND_VS,
	RISCV_INS_VREDMAXU_VS,
	RISCV_INS_VREDMAX_VS,
	RISCV_INS_VREDMINU_VS,
	RISCV_INS_VREDMIN_VS,
	RISCV_INS_VREDOR_VS,
	RISCV_INS_VREDSUM_VS,
	RISCV_INS_VREDXOR_VS,
	RISCV_INS_VREMU_VV,
	RISCV_INS_VREMU_VX,
	RISCV_INS_VREM_VV,
	RISCV_INS_VREM_VX,
	RISCV_INS_VREV8_V,
	RISCV_INS_VRGATHEREI16_VV,
	RISCV_INS_VRGATHER_VI,
	RISCV_INS_VRGATHER_VV,
	RISCV_INS_VRGATHER_VX,
	RISCV_INS_VROL_VV,
	RISCV_INS_VROL_VX,
	RISCV_INS_VROR_VI,
	RISCV_INS_VROR_VV,
	RISCV_INS_VROR_VX,
	RISCV_INS_VRSUB_VI,
	RISCV_INS_VRSUB_VX,
	RISCV_INS_VS1R_V,
	RISCV_INS_VS2R_V,
	RISCV_INS_VS4R_V,
	RISCV_INS_VS8R_V,
	RISCV_INS_VSADDU_VI,
	RISCV_INS_VSADDU_VV,
	RISCV_INS_VSADDU_VX,
	RISCV_INS_VSADD_VI,
	RISCV_INS_VSADD_VV,
	RISCV_INS_VSADD_VX,
	RISCV_INS_VSBC_VVM,
	RISCV_INS_VSBC_VXM,
	RISCV_INS_VSE16_V,
	RISCV_INS_VSE32_V,
	RISCV_INS_VSE64_V,
	RISCV_INS_VSE8_V,
	RISCV_INS_VSETIVLI,
	RISCV_INS_VSETVL,
	RISCV_INS_VSETVLI,
	RISCV_INS_VSEXT_VF2,
	RISCV_INS_VSEXT_VF4,
	RISCV_INS_VSEXT_VF8,
	RISCV_INS_VSHA2CH_VV,
	RISCV_INS_VSHA2CL_VV,
	RISCV_INS_VSHA2MS_VV,
	RISCV_INS_VSLIDE1DOWN_VX,
	RISCV_INS_VSLIDE1UP_VX,
	RISCV_INS_VSLIDEDOWN_VI,
	RISCV_INS_VSLIDEDOWN_VX,
	RISCV_INS_VSLIDEUP_VI,
	RISCV_INS_VSLIDEUP_VX,
	RISCV_INS_VSLL_VI,
	RISCV_INS_VSLL_VV,
	RISCV_INS_VSLL_VX,
	RISCV_INS_VSM3C_VI,
	RISCV_INS_VSM3ME_VV,
	RISCV_INS_VSM4K_VI,
	RISCV_INS_VSM4R_VS,
	RISCV_INS_VSM4R_VV,
	RISCV_INS_VSMUL_VV,
	RISCV_INS_VSMUL_VX,
	RISCV_INS_VSM_V,
	RISCV_INS_VSOXEI16_V,
	RISCV_INS_VSOXEI32_V,
	RISCV_INS_VSOXEI64_V,
	RISCV_INS_VSOXEI8_V,
	RISCV_INS_VSOXSEG2EI16_V,
	RISCV_INS_VSOXSEG2EI32_V,
	RISCV_INS_VSOXSEG2EI64_V,
	RISCV_INS_VSOXSEG2EI8_V,
	RISCV_INS_VSOXSEG3EI16_V,
	RISCV_INS_VSOXSEG3EI32_V,
	RISCV_INS_VSOXSEG3EI64_V,
	RISCV_INS_VSOXSEG3EI8_V,
	RISCV_INS_VSOXSEG4EI16_V,
	RISCV_INS_VSOXSEG4EI32_V,
	RISCV_INS_VSOXSEG4EI64_V,
	RISCV_INS_VSOXSEG4EI8_V,
	RISCV_INS_VSOXSEG5EI16_V,
	RISCV_INS_VSOXSEG5EI32_V,
	RISCV_INS_VSOXSEG5EI64_V,
	RISCV_INS_VSOXSEG5EI8_V,
	RISCV_INS_VSOXSEG6EI16_V,
	RISCV_INS_VSOXSEG6EI32_V,
	RISCV_INS_VSOXSEG6EI64_V,
	RISCV_INS_VSOXSEG6EI8_V,
	RISCV_INS_VSOXSEG7EI16_V,
	RISCV_INS_VSOXSEG7EI32_V,
	RISCV_INS_VSOXSEG7EI64_V,
	RISCV_INS_VSOXSEG7EI8_V,
	RISCV_INS_VSOXSEG8EI16_V,
	RISCV_INS_VSOXSEG8EI32_V,
	RISCV_INS_VSOXSEG8EI64_V,
	RISCV_INS_VSOXSEG8EI8_V,
	RISCV_INS_VSRA_VI,
	RISCV_INS_VSRA_VV,
	RISCV_INS_VSRA_VX,
	RISCV_INS_VSRL_VI,
	RISCV_INS_VSRL_VV,
	RISCV_INS_VSRL_VX,
	RISCV_INS_VSSE16_V,
	RISCV_INS_VSSE32_V,
	RISCV_INS_VSSE64_V,
	RISCV_INS_VSSE8_V,
	RISCV_INS_VSSEG2E16_V,
	RISCV_INS_VSSEG2E32_V,
	RISCV_INS_VSSEG2E64_V,
	RISCV_INS_VSSEG2E8_V,
	RISCV_INS_VSSEG3E16_V,
	RISCV_INS_VSSEG3E32_V,
	RISCV_INS_VSSEG3E64_V,
	RISCV_INS_VSSEG3E8_V,
	RISCV_INS_VSSEG4E16_V,
	RISCV_INS_VSSEG4E32_V,
	RISCV_INS_VSSEG4E64_V,
	RISCV_INS_VSSEG4E8_V,
	RISCV_INS_VSSEG5E16_V,
	RISCV_INS_VSSEG5E32_V,
	RISCV_INS_VSSEG5E64_V,
	RISCV_INS_VSSEG5E8_V,
	RISCV_INS_VSSEG6E16_V,
	RISCV_INS_VSSEG6E32_V,
	RISCV_INS_VSSEG6E64_V,
	RISCV_INS_VSSEG6E8_V,
	RISCV_INS_VSSEG7E16_V,
	RISCV_INS_VSSEG7E32_V,
	RISCV_INS_VSSEG7E64_V,
	RISCV_INS_VSSEG7E8_V,
	RISCV_INS_VSSEG8E16_V,
	RISCV_INS_VSSEG8E32_V,
	RISCV_INS_VSSEG8E64_V,
	RISCV_INS_VSSEG8E8_V,
	RISCV_INS_VSSRA_VI,
	RISCV_INS_VSSRA_VV,
	RISCV_INS_VSSRA_VX,
	RISCV_INS_VSSRL_VI,
	RISCV_INS_VSSRL_VV,
	RISCV_INS_VSSRL_VX,
	RISCV_INS_VSSSEG2E16_V,
	RISCV_INS_VSSSEG2E32_V,
	RISCV_INS_VSSSEG2E64_V,
	RISCV_INS_VSSSEG2E8_V,
	RISCV_INS_VSSSEG3E16_V,
	RISCV_INS_VSSSEG3E32_V,
	RISCV_INS_VSSSEG3E64_V,
	RISCV_INS_VSSSEG3E8_V,
	RISCV_INS_VSSSEG4E16_V,
	RISCV_INS_VSSSEG4E32_V,
	RISCV_INS_VSSSEG4E64_V,
	RISCV_INS_VSSSEG4E8_V,
	RISCV_INS_VSSSEG5E16_V,
	RISCV_INS_VSSSEG5E32_V,
	RISCV_INS_VSSSEG5E64_V,
	RISCV_INS_VSSSEG5E8_V,
	RISCV_INS_VSSSEG6E16_V,
	RISCV_INS_VSSSEG6E32_V,
	RISCV_INS_VSSSEG6E64_V,
	RISCV_INS_VSSSEG6E8_V,
	RISCV_INS_VSSSEG7E16_V,
	RISCV_INS_VSSSEG7E32_V,
	RISCV_INS_VSSSEG7E64_V,
	RISCV_INS_VSSSEG7E8_V,
	RISCV_INS_VSSSEG8E16_V,
	RISCV_INS_VSSSEG8E32_V,
	RISCV_INS_VSSSEG8E64_V,
	RISCV_INS_VSSSEG8E8_V,
	RISCV_INS_VSSUBU_VV,
	RISCV_INS_VSSUBU_VX,
	RISCV_INS_VSSUB_VV,
	RISCV_INS_VSSUB_VX,
	RISCV_INS_VSUB_VV,
	RISCV_INS_VSUB_VX,
	RISCV_INS_VSUXEI16_V,
	RISCV_INS_VSUXEI32_V,
	RISCV_INS_VSUXEI64_V,
	RISCV_INS_VSUXEI8_V,
	RISCV_INS_VSUXSEG2EI16_V,
	RISCV_INS_VSUXSEG2EI32_V,
	RISCV_INS_VSUXSEG2EI64_V,
	RISCV_INS_VSUXSEG2EI8_V,
	RISCV_INS_VSUXSEG3EI16_V,
	RISCV_INS_VSUXSEG3EI32_V,
	RISCV_INS_VSUXSEG3EI64_V,
	RISCV_INS_VSUXSEG3EI8_V,
	RISCV_INS_VSUXSEG4EI16_V,
	RISCV_INS_VSUXSEG4EI32_V,
	RISCV_INS_VSUXSEG4EI64_V,
	RISCV_INS_VSUXSEG4EI8_V,
	RISCV_INS_VSUXSEG5EI16_V,
	RISCV_INS_VSUXSEG5EI32_V,
	RISCV_INS_VSUXSEG5EI64_V,
	RISCV_INS_VSUXSEG5EI8_V,
	RISCV_INS_VSUXSEG6EI16_V,
	RISCV_INS_VSUXSEG6EI32_V,
	RISCV_INS_VSUXSEG6EI64_V,
	RISCV_INS_VSUXSEG6EI8_V,
	RISCV_INS_VSUXSEG7EI16_V,
	RISCV_INS_VSUXSEG7EI32_V,
	RISCV_INS_VSUXSEG7EI64_V,
	RISCV_INS_VSUXSEG7EI8_V,
	RISCV_INS_VSUXSEG8EI16_V,
	RISCV_INS_VSUXSEG8EI32_V,
	RISCV_INS_VSUXSEG8EI64_V,
	RISCV_INS_VSUXSEG8EI8_V,
	RISCV_INS_VT_MASKC,
	RISCV_INS_VT_MASKCN,
	RISCV_INS_VWADDU_VV,
	RISCV_INS_VWADDU_VX,
	RISCV_INS_VWADDU_WV,
	RISCV_INS_VWADDU_WX,
	RISCV_INS_VWADD_VV,
	RISCV_INS_VWADD_VX,
	RISCV_INS_VWADD_WV,
	RISCV_INS_VWADD_WX,
	RISCV_INS_VWMACCSU_VV,
	RISCV_INS_VWMACCSU_VX,
	RISCV_INS_VWMACCUS_VX,
	RISCV_INS_VWMACCU_VV,
	RISCV_INS_VWMACCU_VX,
	RISCV_INS_VWMACC_VV,
	RISCV_INS_VWMACC_VX,
	RISCV_INS_VWMULSU_VV,
	RISCV_INS_VWMULSU_VX,
	RISCV_INS_VWMULU_VV,
	RISCV_INS_VWMULU_VX,
	RISCV_INS_VWMUL_VV,
	RISCV_INS_VWMUL_VX,
	RISCV_INS_VWREDSUMU_VS,
	RISCV_INS_VWREDSUM_VS,
	RISCV_INS_VWSLL_VI,
	RISCV_INS_VWSLL_VV,
	RISCV_INS_VWSLL_VX,
	RISCV_INS_VWSUBU_VV,
	RISCV_INS_VWSUBU_VX,
	RISCV_INS_VWSUBU_WV,
	RISCV_INS_VWSUBU_WX,
	RISCV_INS_VWSUB_VV,
	RISCV_INS_VWSUB_VX,
	RISCV_INS_VWSUB_WV,
	RISCV_INS_VWSUB_WX,
	RISCV_INS_VXOR_VI,
	RISCV_INS_VXOR_VV,
	RISCV_INS_VXOR_VX,
	RISCV_INS_VZEXT_VF2,
	RISCV_INS_VZEXT_VF4,
	RISCV_INS_VZEXT_VF8,
	RISCV_INS_WFI,
	RISCV_INS_WRS_NTO,
	RISCV_INS_WRS_STO,
	RISCV_INS_XNOR,
	RISCV_INS_XOR,
	RISCV_INS_XORI,
	RISCV_INS_XPERM4,
	RISCV_INS_XPERM8,
	RISCV_INS_ZIP,

	// clang-format on
	// generated content <RISCVGenCSInsnEnum.inc> end
	RISCV_INS_ENDING,

	RISCV_INS_ALIAS_BEGIN,
	// generated content <RISCVGenCSAliasEnum.inc> begin
	// clang-format off

	RISCV_INS_ALIAS_NOP, // Real instr.: RISCV_ADDI
	RISCV_INS_ALIAS_LI, // Real instr.: RISCV_ADDI
	RISCV_INS_ALIAS_MV, // Real instr.: RISCV_ADDI
	RISCV_INS_ALIAS_NOT, // Real instr.: RISCV_XORI
	RISCV_INS_ALIAS_NEG, // Real instr.: RISCV_SUB
	RISCV_INS_ALIAS_NEGW, // Real instr.: RISCV_SUBW
	RISCV_INS_ALIAS_SEXT_W, // Real instr.: RISCV_ADDIW
	RISCV_INS_ALIAS_SEQZ, // Real instr.: RISCV_SLTIU
	RISCV_INS_ALIAS_SNEZ, // Real instr.: RISCV_SLTU
	RISCV_INS_ALIAS_SLTZ, // Real instr.: RISCV_SLT
	RISCV_INS_ALIAS_SGTZ, // Real instr.: RISCV_SLT
	RISCV_INS_ALIAS_BEQZ, // Real instr.: RISCV_BEQ
	RISCV_INS_ALIAS_BNEZ, // Real instr.: RISCV_BNE
	RISCV_INS_ALIAS_BLEZ, // Real instr.: RISCV_BGE
	RISCV_INS_ALIAS_BGEZ, // Real instr.: RISCV_BGE
	RISCV_INS_ALIAS_BLTZ, // Real instr.: RISCV_BLT
	RISCV_INS_ALIAS_BGTZ, // Real instr.: RISCV_BLT
	RISCV_INS_ALIAS_J, // Real instr.: RISCV_JAL
	RISCV_INS_ALIAS_JAL, // Real instr.: RISCV_JAL
	RISCV_INS_ALIAS_JR, // Real instr.: RISCV_JALR
	RISCV_INS_ALIAS_JALR, // Real instr.: RISCV_JALR
	RISCV_INS_ALIAS_RET, // Real instr.: RISCV_JALR
	RISCV_INS_ALIAS_FENCE, // Real instr.: RISCV_FENCE
	RISCV_INS_ALIAS_PAUSE, // Real instr.: RISCV_FENCE
	RISCV_INS_ALIAS_RDINSTRET, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_RDCYCLE, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_RDTIME, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_RDINSTRETH, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_RDCYCLEH, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_RDTIMEH, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_CSRR, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_CSRW, // Real instr.: RISCV_CSRRW
	RISCV_INS_ALIAS_CSRS, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_CSRC, // Real instr.: RISCV_CSRRC
	RISCV_INS_ALIAS_CSRWI, // Real instr.: RISCV_CSRRWI
	RISCV_INS_ALIAS_CSRSI, // Real instr.: RISCV_CSRRSI
	RISCV_INS_ALIAS_CSRCI, // Real instr.: RISCV_CSRRCI
	RISCV_INS_ALIAS_SFENCE_VMA, // Real instr.: RISCV_SFENCE_VMA
	RISCV_INS_ALIAS_HFENCE_GVMA, // Real instr.: RISCV_HFENCE_GVMA
	RISCV_INS_ALIAS_HFENCE_VVMA, // Real instr.: RISCV_HFENCE_VVMA
	RISCV_INS_ALIAS_NTL_P1, // Real instr.: RISCV_ADD
	RISCV_INS_ALIAS_NTL_PALL, // Real instr.: RISCV_ADD
	RISCV_INS_ALIAS_NTL_S1, // Real instr.: RISCV_ADD
	RISCV_INS_ALIAS_NTL_ALL, // Real instr.: RISCV_ADD
	RISCV_INS_ALIAS_LPAD, // Real instr.: RISCV_AUIPC
	RISCV_INS_ALIAS_FMV_S, // Real instr.: RISCV_FSGNJ_S
	RISCV_INS_ALIAS_FABS_S, // Real instr.: RISCV_FSGNJX_S
	RISCV_INS_ALIAS_FNEG_S, // Real instr.: RISCV_FSGNJN_S
	RISCV_INS_ALIAS_FRCSR, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_FSCSR, // Real instr.: RISCV_CSRRW
	RISCV_INS_ALIAS_FRRM, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_FSRM, // Real instr.: RISCV_CSRRW
	RISCV_INS_ALIAS_FSRMI, // Real instr.: RISCV_CSRRWI
	RISCV_INS_ALIAS_FRFLAGS, // Real instr.: RISCV_CSRRS
	RISCV_INS_ALIAS_FSFLAGS, // Real instr.: RISCV_CSRRW
	RISCV_INS_ALIAS_FSFLAGSI, // Real instr.: RISCV_CSRRWI
	RISCV_INS_ALIAS_FMV_D, // Real instr.: RISCV_FSGNJ_D
	RISCV_INS_ALIAS_FABS_D, // Real instr.: RISCV_FSGNJX_D
	RISCV_INS_ALIAS_FNEG_D, // Real instr.: RISCV_FSGNJN_D
	RISCV_INS_ALIAS_FMV_H, // Real instr.: RISCV_FSGNJ_H
	RISCV_INS_ALIAS_FABS_H, // Real instr.: RISCV_FSGNJX_H
	RISCV_INS_ALIAS_FNEG_H, // Real instr.: RISCV_FSGNJN_H
	RISCV_INS_ALIAS_ZEXT_W, // Real instr.: RISCV_ADD_UW
	RISCV_INS_ALIAS_VL1R_V, // Real instr.: RISCV_VL1RE8_V
	RISCV_INS_ALIAS_VL2R_V, // Real instr.: RISCV_VL2RE8_V
	RISCV_INS_ALIAS_VL4R_V, // Real instr.: RISCV_VL4RE8_V
	RISCV_INS_ALIAS_VL8R_V, // Real instr.: RISCV_VL8RE8_V
	RISCV_INS_ALIAS_VNEG_V, // Real instr.: RISCV_VRSUB_VX
	RISCV_INS_ALIAS_VWCVT_X_X_V, // Real instr.: RISCV_VWADD_VX
	RISCV_INS_ALIAS_VWCVTU_X_X_V, // Real instr.: RISCV_VWADDU_VX
	RISCV_INS_ALIAS_VNOT_V, // Real instr.: RISCV_VXOR_VI
	RISCV_INS_ALIAS_VNCVT_X_X_W, // Real instr.: RISCV_VNSRL_WX
	RISCV_INS_ALIAS_VFNEG_V, // Real instr.: RISCV_VFSGNJN_VV
	RISCV_INS_ALIAS_VFABS_V, // Real instr.: RISCV_VFSGNJX_VV
	RISCV_INS_ALIAS_VMMV_M, // Real instr.: RISCV_VMAND_MM
	RISCV_INS_ALIAS_VMCLR_M, // Real instr.: RISCV_VMXOR_MM
	RISCV_INS_ALIAS_VMSET_M, // Real instr.: RISCV_VMXNOR_MM
	RISCV_INS_ALIAS_VMNOT_M, // Real instr.: RISCV_VMNAND_MM
	RISCV_INS_ALIAS_C_NTL_P1, // Real instr.: RISCV_C_ADD_HINT
	RISCV_INS_ALIAS_C_NTL_PALL, // Real instr.: RISCV_C_ADD_HINT
	RISCV_INS_ALIAS_C_NTL_S1, // Real instr.: RISCV_C_ADD_HINT
	RISCV_INS_ALIAS_C_NTL_ALL, // Real instr.: RISCV_C_ADD_HINT
	RISCV_INS_ALIAS_CV_MULS, // Real instr.: RISCV_CV_MULSN
	RISCV_INS_ALIAS_CV_MULHHS, // Real instr.: RISCV_CV_MULHHSN
	RISCV_INS_ALIAS_CV_MULU, // Real instr.: RISCV_CV_MULUN
	RISCV_INS_ALIAS_CV_MULHHU, // Real instr.: RISCV_CV_MULHHUN

	// clang-format on
	// generated content <RISCVGenCSAliasEnum.inc> end
	RISCV_INS_ALIAS_END,
} riscv_insn;

//> Group of RISCV instructions
typedef enum riscv_insn_group {
	RISCV_GRP_INVALID = 0, ///< = CS_GRP_INVALID

	// Generic groups
	// all jump instructions (conditional+direct+indirect jumps)
	RISCV_GRP_JUMP, ///< = CS_GRP_JUMP
	// all call instructions
	RISCV_GRP_CALL, ///< = CS_GRP_CALL
	// all return instructions
	RISCV_GRP_RET, ///< = CS_GRP_RET
	// all interrupt instructions (int+syscall)
	RISCV_GRP_INT, ///< = CS_GRP_INT
	// all interrupt return instructions
	RISCV_GRP_IRET, ///< = CS_GRP_IRET
	// all privileged instructions
	RISCV_GRP_PRIVILEGE, ///< = CS_GRP_PRIVILEGE
	// all relative branching instructions
	RISCV_GRP_BRANCH_RELATIVE, ///< = CS_GRP_BRANCH_RELATIVE

	// Architecture-specific groups
	// generated content <RISCVGenCSFeatureEnum.inc> begin
	// clang-format off

	RISCV_FEATURE_HASSTDEXTI = 128,
	RISCV_FEATURE_HASSTDEXTZICBOM,
	RISCV_FEATURE_HASSTDEXTZICBOP,
	RISCV_FEATURE_HASSTDEXTZICBOZ,
	RISCV_FEATURE_HASSTDEXTZICSR,
	RISCV_FEATURE_HASSTDEXTZICOND,
	RISCV_FEATURE_HASSTDEXTZIFENCEI,
	RISCV_FEATURE_HASSTDEXTZIHINTPAUSE,
	RISCV_FEATURE_HASSTDEXTZIHINTNTL,
	RISCV_FEATURE_HASSTDEXTZIMOP,
	RISCV_FEATURE_HASSTDEXTZICFILP,
	RISCV_FEATURE_HASSTDEXTZICFISS,
	RISCV_FEATURE_HASSTDEXTM,
	RISCV_FEATURE_HASSTDEXTMORZMMUL,
	RISCV_FEATURE_HASSTDEXTA,
	RISCV_FEATURE_HASSTDEXTZTSO,
	RISCV_FEATURE_HASSTDEXTZACAS,
	RISCV_FEATURE_HASSTDEXTZAWRS,
	RISCV_FEATURE_HASSTDEXTF,
	RISCV_FEATURE_HASSTDEXTD,
	RISCV_FEATURE_HASSTDEXTZFHMIN,
	RISCV_FEATURE_HASSTDEXTZFH,
	RISCV_FEATURE_HASSTDEXTZFBFMIN,
	RISCV_FEATURE_HASHALFFPLOADSTOREMOVE,
	RISCV_FEATURE_HASSTDEXTZFA,
	RISCV_FEATURE_HASSTDEXTZFINX,
	RISCV_FEATURE_HASSTDEXTZDINX,
	RISCV_FEATURE_HASSTDEXTZHINXMIN,
	RISCV_FEATURE_HASSTDEXTZHINX,
	RISCV_FEATURE_HASSTDEXTC,
	RISCV_FEATURE_HASRVCHINTS,
	RISCV_FEATURE_HASSTDEXTCORZCA,
	RISCV_FEATURE_HASSTDEXTZCB,
	RISCV_FEATURE_HASSTDEXTCORZCD,
	RISCV_FEATURE_HASSTDEXTZCMP,
	RISCV_FEATURE_HASSTDEXTZCMT,
	RISCV_FEATURE_HASSTDEXTCORZCFORZCE,
	RISCV_FEATURE_HASSTDEXTZCMOP,
	RISCV_FEATURE_HASSTDEXTZBA,
	RISCV_FEATURE_HASSTDEXTZBB,
	RISCV_FEATURE_HASSTDEXTZBC,
	RISCV_FEATURE_HASSTDEXTZBS,
	RISCV_FEATURE_HASSTDEXTZBKB,
	RISCV_FEATURE_HASSTDEXTZBKX,
	RISCV_FEATURE_HASSTDEXTZBBORZBKB,
	RISCV_FEATURE_HASSTDEXTZBKC,
	RISCV_FEATURE_HASSTDEXTZBCORZBKC,
	RISCV_FEATURE_HASSTDEXTZKND,
	RISCV_FEATURE_HASSTDEXTZKNE,
	RISCV_FEATURE_HASSTDEXTZKNDORZKNE,
	RISCV_FEATURE_HASSTDEXTZKNH,
	RISCV_FEATURE_HASSTDEXTZKSED,
	RISCV_FEATURE_HASSTDEXTZKSH,
	RISCV_FEATURE_HASSTDEXTZKR,
	RISCV_FEATURE_HASSTDEXTZVFBFMIN,
	RISCV_FEATURE_HASSTDEXTZVFBFWMA,
	RISCV_FEATURE_HASSTDEXTZFHORZVFH,
	RISCV_FEATURE_HASSTDEXTZVKB,
	RISCV_FEATURE_HASSTDEXTZVBB,
	RISCV_FEATURE_HASSTDEXTZVBC,
	RISCV_FEATURE_HASSTDEXTZVKG,
	RISCV_FEATURE_HASSTDEXTZVKNED,
	RISCV_FEATURE_HASSTDEXTZVKNHA,
	RISCV_FEATURE_HASSTDEXTZVKNHB,
	RISCV_FEATURE_HASSTDEXTZVKNHAORZVKNHB,
	RISCV_FEATURE_HASSTDEXTZVKSED,
	RISCV_FEATURE_HASSTDEXTZVKSH,
	RISCV_FEATURE_HASVINSTRUCTIONS,
	RISCV_FEATURE_HASVINSTRUCTIONSI64,
	RISCV_FEATURE_HASVINSTRUCTIONSANYF,
	RISCV_FEATURE_HASVINSTRUCTIONSF16MINIMAL,
	RISCV_FEATURE_HASSTDEXTH,
	RISCV_FEATURE_HASSTDEXTSVINVAL,
	RISCV_FEATURE_HASVENDORXVENTANACONDOPS,
	RISCV_FEATURE_HASVENDORXTHEADBA,
	RISCV_FEATURE_HASVENDORXTHEADBB,
	RISCV_FEATURE_HASVENDORXTHEADBS,
	RISCV_FEATURE_HASVENDORXTHEADCONDMOV,
	RISCV_FEATURE_HASVENDORXTHEADCMO,
	RISCV_FEATURE_HASVENDORXTHEADFMEMIDX,
	RISCV_FEATURE_HASVENDORXTHEADMAC,
	RISCV_FEATURE_HASVENDORXTHEADMEMIDX,
	RISCV_FEATURE_HASVENDORXTHEADMEMPAIR,
	RISCV_FEATURE_HASVENDORXTHEADSYNC,
	RISCV_FEATURE_HASVENDORXTHEADVDOT,
	RISCV_FEATURE_HASVENDORXSFVCP,
	RISCV_FEATURE_HASVENDORXSFVQMACCDOD,
	RISCV_FEATURE_HASVENDORXSFVQMACCQOQ,
	RISCV_FEATURE_HASVENDORXSFVFWMACCQQQ,
	RISCV_FEATURE_HASVENDORXSFVFNRCLIPXFQF,
	RISCV_FEATURE_HASVENDORXCVELW,
	RISCV_FEATURE_HASVENDORXCVBITMANIP,
	RISCV_FEATURE_HASVENDORXCVMAC,
	RISCV_FEATURE_HASVENDORXCVMEM,
	RISCV_FEATURE_HASVENDORXCVALU,
	RISCV_FEATURE_HASVENDORXCVSIMD,
	RISCV_FEATURE_HASVENDORXCVBI,
	RISCV_FEATURE_ISRV64,
	RISCV_FEATURE_ISRV32,
	RISCV_FEATURE_ISRVE,

	// clang-format on
	// generated content <RISCVGenCSFeatureEnum.inc> end

	RISCV_GRP_ENDING,
} riscv_insn_group;

#ifdef __cplusplus
}
#endif

#endif
