import type { Module } from '@swc/core';
import type { PluginContext, I18nextToolkitConfig, Logger, ASTVisitorHooks, ScopeInfo } from '../../types';
import { ExpressionResolver } from '../parsers/expression-resolver';
/**
 * AST visitor class that traverses JavaScript/TypeScript syntax trees to extract translation keys.
 *
 * This class implements a manual recursive walker that:
 * - Maintains scope information for tracking useTranslation and getFixedT calls
 * - Extracts keys from t() function calls with various argument patterns
 * - Handles JSX Trans components with complex children serialization
 * - Supports both string literals and selector API for type-safe keys
 * - Processes pluralization and context variants
 * - Manages namespace resolution from multiple sources
 *
 * The visitor respects configuration options for separators, function names,
 * component names, and other extraction settings.
 *
 * @example
 * ```typescript
 * const visitors = new ASTVisitors(config, pluginContext, logger)
 * visitors.visit(parsedAST)
 *
 * // The pluginContext will now contain all extracted keys
 * ```
 */
export declare class ASTVisitors {
    private readonly pluginContext;
    private readonly config;
    private readonly logger;
    private hooks;
    get objectKeys(): Set<string>;
    private readonly scopeManager;
    private readonly expressionResolver;
    private readonly callExpressionHandler;
    private readonly jsxHandler;
    private currentFile;
    private currentCode;
    /**
     * Creates a new AST visitor instance.
     *
     * @param config - Toolkit configuration with extraction settings
     * @param pluginContext - Context for adding discovered translation keys
     * @param logger - Logger for warnings and debug information
     */
    constructor(config: Omit<I18nextToolkitConfig, 'plugins'>, pluginContext: PluginContext, logger: Logger, hooks?: ASTVisitorHooks, expressionResolver?: ExpressionResolver);
    /**
     * Main entry point for AST traversal.
     * Creates a root scope and begins the recursive walk through the syntax tree.
     *
     * @param node - The root module node to traverse
     */
    visit(node: Module): void;
    /**
     * Recursively walks through AST nodes, handling scoping and visiting logic.
     *
     * This is the core traversal method that:
     * 1. Manages function scopes (enter/exit)
     * 2. Dispatches to specific handlers based on node type
     * 3. Recursively processes child nodes
     * 4. Maintains proper scope cleanup
     *
     * @param node - The current AST node to process
     *
     * @private
     */
    private walk;
    /**
     * Retrieves variable information from the scope chain.
     * Searches from innermost to outermost scope.
     *
     * @param name - Variable name to look up
     * @returns Scope information if found, undefined otherwise
     *
     * @private
     */
    getVarFromScope(name: string): ScopeInfo | undefined;
    /**
     * Sets the current file path and code used by the extractor.
     * Also resets the search index for location tracking.
     */
    setCurrentFile(file: string, code: string): void;
    /**
     * Returns the currently set file path.
     *
     * @returns The current file path as a string, or `undefined` if no file has been set.
     * @remarks
     * Use this to retrieve the file context that was previously set via `setCurrentFile`.
     */
    getCurrentFile(): string;
    /**
     * @returns The full source code string for the file currently under processing.
     */
    getCurrentCode(): string;
}
//# sourceMappingURL=ast-visitors.d.ts.map