/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%     CCCC   OOO   L       OOO   RRRR   SSSSS  PPPP    AAA    CCCC  EEEEE     %
%    C      O   O  L      O   O  R   R  SS     P   P  A   A  C      E         %
%    C      O   O  L      O   O  RRRR    SSS   PPPP   AAAAA  C      EEE       %
%    C      O   O  L      O   O  R R       SS  P      A   A  C      E         %
%     CCCC   OOO   LLLLL   OOO   R  R   SSSSS  P      A   A   CCCC  EEEEE     %
%                                                                             %
%                                                                             %
%                   ImageMagick Image Colorspace Methods                      %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 July 1992                                   %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/error.h"
#include "magick/gem.h"
#include "magick/monitor.h"
#include "magick/utility.h"
#if defined(HasLCMS)
#if defined(HAVE_LCMS_LCMS_H)
#include <lcms/lcms.h>
#else
#include "lcms.h"
#endif
#endif

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+     R G B T r a n s f o r m I m a g e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RGBTransformImage() converts the reference image from RGB to an alternate
%  colorspace.  The transformation matrices are not the standard ones: the
%  weights are rescaled to normalized the range of the transformed values to
%  be [0..MaxRGB].
%
%  The format of the RGBTransformImage method is:
%
%      unsigned int RGBTransformImage(Image *image,
%        const ColorspaceType colorspace)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o colorspace: the colorspace to transform the image to.
%
%
*/
MagickExport unsigned int RGBTransformImage(Image *image,
  const ColorspaceType colorspace)
{
#define RGBTransformImageTag  "RGBTransform/Image"
#define X  0
#define Y  ((MaxMap+1))
#define Z  (2*(MaxMap+1))

  double
    *x_map,
    *y_map,
    *z_map;

  DoublePixelPacket
    pixel;

  long
    y;

  IndexPacket
    *indexes;

  LongPixelPacket
    quantum;

  PrimaryInfo
    primary_info;

  register long
    i,
    x;

  register PixelPacket
    *q;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if ((colorspace == RGBColorspace) || (colorspace == TransparentColorspace))
    return(True);
  switch (colorspace)
  {
    case CMYKColorspace:
    {
      Quantum
        black,
        cyan,
        magenta,
        yellow;

      /*
        Convert RGB to CMYK colorspace.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      image->colorspace=CMYKColorspace;
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) image->columns; x++)
        {
          cyan=(Quantum) (MaxRGB-q->red);
          magenta=(Quantum) (MaxRGB-q->green);
          yellow=(Quantum) (MaxRGB-q->blue);
          black=(Quantum)
            (cyan < magenta ? Min(cyan,yellow) : Min(magenta,yellow));
          q->red=cyan;
          q->green=magenta;
          q->blue=yellow;
          indexes[x]=q->opacity;
          q->opacity=(Quantum) black;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      return(True);
    }
    case GRAYColorspace:
    {
      IndexPacket
        index;

      /*
        Convert RGB to GRAY colorspace.
      */
      if (IsGrayImage(image,&image->exception))
        return(True);
      if (!AllocateImageColormap(image,MaxColormapSize))
        ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
          "UnableToTransformColorspace");
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) image->columns; x++)
        {
          index=PixelIntensityToQuantum(q);
          q->red=index;
          q->green=index;
          q->blue=index;
          indexes[x]=index;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      return(True);
    }
    case HSLColorspace:
    {
      double
        hue,
        luminosity,
        saturation;

      /*
        Transform image from RGB to HSL.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          TransformHSL(q->red,q->green,q->blue,&hue,&saturation,&luminosity);
          q->red=(Quantum) (MaxRGB*hue+0.5);
          q->green=(Quantum) (MaxRGB*saturation+0.5);
          q->blue=(Quantum) (MaxRGB*luminosity+0.5);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      image->colorspace=RGBColorspace;
      return(True);
    }
    case HWBColorspace:
    {
      double
        blackness,
        hue,
        whiteness;

      /*
        Transform image from RGB to HWB.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          TransformHWB(q->red,q->green,q->blue,&hue,&whiteness,&blackness);
          q->red=(Quantum) (MaxRGB*hue+0.5);
          q->green=(Quantum) (MaxRGB*whiteness+0.5);
          q->blue=(Quantum) (MaxRGB*blackness+0.5);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      image->colorspace=RGBColorspace;
      return(True);
    }
    default:
      break;
  }
  /*
    Allocate the tables.
  */
  x_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  y_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  z_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  if ((x_map == (double *) NULL) || (y_map == (double *) NULL) ||
      (z_map == (double *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToTransformColorspace");
  memset(&primary_info,0,sizeof(primary_info));
  switch (colorspace)
  {
    case GRAYColorspace:
    {
      /*
        Initialize GRAY tables:

          G = 0.29900*R+0.58700*G+0.11400*B
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.299*i;
        y_map[i+X]=0.587*i;
        z_map[i+X]=0.114*i;
        x_map[i+Y]=0.299*i;
        y_map[i+Y]=0.587*i;
        z_map[i+Y]=0.114*i;
        x_map[i+Z]=0.299*i;
        y_map[i+Z]=0.587*i;
        z_map[i+Z]=0.114*i;
      }
      break;
    }
    case OHTAColorspace:
    {
      /*
        Initialize OHTA tables:

          I1 = 0.33333*R+0.33334*G+0.33333*B
          I2 = 0.50000*R+0.00000*G-0.50000*B
          I3 =-0.25000*R+0.50000*G-0.25000*B

        I and Q, normally -0.5 through 0.5, are normalized to the range 0
        through MaxRGB.
      */
      primary_info.y=(MaxMap+1)/2;
      primary_info.z=(MaxMap+1)/2;
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.33333*i;
        y_map[i+X]=0.33334*i;
        z_map[i+X]=0.33333*i;
        x_map[i+Y]=0.5*i;
        y_map[i+Y]=0.0;
        z_map[i+Y]=(-0.5)*i;
        x_map[i+Z]=(-0.25)*i;
        y_map[i+Z]=0.5*i;
        z_map[i+Z]=(-0.25)*i;
      }
      break;
    }
    case sRGBColorspace:
    {
      /*
        Initialize sRGB tables:

          Y =  0.29900*R+0.58700*G+0.11400*B
          C1= -0.29900*R-0.58700*G+0.88600*B
          C2=  0.70100*R-0.58700*G-0.11400*B

        sRGB is scaled by 1.3584.  C1 zero is 156 and C2 is at 137.
      */
      primary_info.y=ScaleQuantumToMap(ScaleCharToQuantum(156));
      primary_info.z=ScaleQuantumToMap(ScaleCharToQuantum(137));
      for (i=0; i <= (long) (0.018*MaxMap); i++)
      {
        x_map[i+X]=0.003962014134275617*i;
        y_map[i+X]=0.007778268551236748*i;
        z_map[i+X]=0.001510600706713781*i;
        x_map[i+Y]=(-0.002426619775463276)*i;
        y_map[i+Y]=(-0.004763965913702149)*i;
        z_map[i+Y]=0.007190585689165425*i;
        x_map[i+Z]=0.006927257754597858*i;
        y_map[i+Z]=(-0.005800713697502058)*i;
        z_map[i+Z]=(-0.0011265440570958)*i;
      }
      for ( ; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.2201118963486454*(1.099*i-0.099);
        y_map[i+X]=0.4321260306242638*(1.099*i-0.099);
        z_map[i+X]=0.08392226148409894*(1.099*i-0.099);
        x_map[i+Y]=(-0.1348122097479598)*(1.099*i-0.099);
        y_map[i+Y]=(-0.2646647729834528)*(1.099*i-0.099);
        z_map[i+Y]=0.3994769827314126*(1.099*i-0.099);
        x_map[i+Z]=0.3848476530332144*(1.099*i-0.099);
        y_map[i+Z]=(-0.3222618720834477)*(1.099*i-0.099);
        z_map[i+Z]=(-0.06258578094976668)*(1.099*i-0.099);
      }
      break;
    }
    case XYZColorspace:
    {
      /*
        Initialize CIE XYZ tables:

          X = 0.412453*X+0.357580*Y+0.180423*Z
          Y = 0.212671*X+0.715160*Y+0.072169*Z
          Z = 0.019334*X+0.119193*Y+0.950227*Z
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.412453*i;
        y_map[i+X]=0.35758*i;
        z_map[i+X]=0.180423*i;
        x_map[i+Y]=0.212671*i;
        y_map[i+Y]=0.71516*i;
        z_map[i+Y]=0.072169*i;
        x_map[i+Z]=0.019334*i;
        y_map[i+Z]=0.119193*i;
        z_map[i+Z]=0.950227*i;
      }
      break;
    }
    case YCbCrColorspace:
    {
      /*
        Initialize YCbCr tables:

          Y =  0.299000*R+0.587000*G+0.114000*B
          Cb= -0.168736*R-0.331264*G+0.500000*B
          Cr=  0.500000*R-0.418688*G-0.081316*B

        Cb and Cr, normally -0.5 through 0.5, are normalized to the range 0
        through MaxRGB.
      */
      primary_info.y=(MaxMap+1)/2;
      primary_info.z=(MaxMap+1)/2;
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.299*i;
        y_map[i+X]=0.587*i;
        z_map[i+X]=0.114*i;
        x_map[i+Y]=(-0.16873)*i;
        y_map[i+Y]=(-0.331264)*i;
        z_map[i+Y]=0.500000*i;
        x_map[i+Z]=0.500000*i;
        y_map[i+Z]=(-0.418688)*i;
        z_map[i+Z]=(-0.081316)*i;
      }
      break;
    }
    case YCCColorspace:
    {
      /*
        Initialize YCC tables:

          Y =  0.29900*R+0.58700*G+0.11400*B
          C1= -0.29900*R-0.58700*G+0.88600*B
          C2=  0.70100*R-0.58700*G-0.11400*B

        YCC is scaled by 1.3584.  C1 zero is 156 and C2 is at 137.
      */
      primary_info.y=ScaleQuantumToMap(ScaleCharToQuantum(156));
      primary_info.z=ScaleQuantumToMap(ScaleCharToQuantum(137));
      for (i=0; i <= (long) (0.018*MaxMap); i++)
      {
        x_map[i+X]=0.003962014134275617*i;
        y_map[i+X]=0.007778268551236748*i;
        z_map[i+X]=0.001510600706713781*i;
        x_map[i+Y]=(-0.002426619775463276)*i;
        y_map[i+Y]=(-0.004763965913702149)*i;
        z_map[i+Y]=0.007190585689165425*i;
        x_map[i+Z]=0.006927257754597858*i;
        y_map[i+Z]=(-0.005800713697502058)*i;
        z_map[i+Z]=(-0.0011265440570958)*i;
      }
      for ( ; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.2201118963486454*(1.099*i-0.099);
        y_map[i+X]=0.4321260306242638*(1.099*i-0.099);
        z_map[i+X]=0.08392226148409894*(1.099*i-0.099);
        x_map[i+Y]=(-0.1348122097479598)*(1.099*i-0.099);
        y_map[i+Y]=(-0.2646647729834528)*(1.099*i-0.099);
        z_map[i+Y]=0.3994769827314126*(1.099*i-0.099);
        x_map[i+Z]=0.3848476530332144*(1.099*i-0.099);
        y_map[i+Z]=(-0.3222618720834477)*(1.099*i-0.099);
        z_map[i+Z]=(-0.06258578094976668)*(1.099*i-0.099);
      }
      break;
    }
    case YIQColorspace:
    {
      /*
        Initialize YIQ tables:

          Y = 0.29900*R+0.58700*G+0.11400*B
          I = 0.59600*R-0.27400*G-0.32200*B
          Q = 0.21100*R-0.52300*G+0.31200*B

        I and Q, normally -0.5 through 0.5, are normalized to the range 0
        through MaxRGB.
      */
      primary_info.y=(MaxMap+1)/2;
      primary_info.z=(MaxMap+1)/2;
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.299*i;
        y_map[i+X]=0.587*i;
        z_map[i+X]=0.114*i;
        x_map[i+Y]=0.596*i;
        y_map[i+Y]=(-0.274)*i;
        z_map[i+Y]=(-0.322)*i;
        x_map[i+Z]=0.211*i;
        y_map[i+Z]=(-0.523)*i;
        z_map[i+Z]=0.312*i;
      }
      break;
    }
    case YPbPrColorspace:
    {
      /*
        Initialize YPbPr tables:

          Y =  0.299000*R+0.587000*G+0.114000*B
          Pb= -0.168736*R-0.331264*G+0.500000*B
          Pr=  0.500000*R-0.418688*G-0.081312*B

        Pb and Pr, normally -0.5 through 0.5, are normalized to the range 0
        through MaxRGB.
      */
      primary_info.y=(MaxMap+1)/2;
      primary_info.z=(MaxMap+1)/2;
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.299*i;
        y_map[i+X]=0.587*i;
        z_map[i+X]=0.114*i;
        x_map[i+Y]=(-0.168736)*i;
        y_map[i+Y]=(-0.331264)*i;
        z_map[i+Y]=0.5*i;
        x_map[i+Z]=0.5*i;
        y_map[i+Z]=(-0.418688)*i;
        z_map[i+Z]=(-0.081312)*i;
      }
      break;
    }
    case YUVColorspace:
    default:
    {
      /*
        Initialize YUV tables:

          Y =  0.29900*R+0.58700*G+0.11400*B
          U = -0.14740*R-0.28950*G+0.43690*B
          V =  0.61500*R-0.51500*G-0.10000*B

        U and V, normally -0.5 through 0.5, are normalized to the range 0
        through MaxRGB.  Note that U = 0.493*(B-Y), V = 0.877*(R-Y).
      */
      primary_info.y=(MaxMap+1)/2;
      primary_info.z=(MaxMap+1)/2;
      for (i=0; i <= (long) MaxMap; i++)
      {
        x_map[i+X]=0.299*i;
        y_map[i+X]=0.587*i;
        z_map[i+X]=0.114*i;
        x_map[i+Y]=(-0.1474)*i;
        y_map[i+Y]=(-0.2895)*i;
        z_map[i+Y]=0.4369*i;
        x_map[i+Z]=0.615*i;
        y_map[i+Z]=(-0.515)*i;
        z_map[i+Z]=(-0.1)*i;
      }
      break;
    }
  }
  /*
    Convert from RGB.
  */
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      ExceptionInfo
        *exception;

      /*
        Convert DirectClass image.
      */
      exception=(&image->exception);
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=0; x < (long) image->columns; x++)
        {
          pixel.red=x_map[ScaleQuantumToMap(q->red)+X]+
            y_map[ScaleQuantumToMap(q->green)+X]+
            z_map[ScaleQuantumToMap(q->blue)+X]+primary_info.x;
          quantum.red=(unsigned long) ((pixel.red < 0) ? 0 :
            (pixel.red > MaxMap) ? MaxMap : pixel.red+0.5);
          pixel.green=x_map[ScaleQuantumToMap(q->red)+Y]+
            y_map[ScaleQuantumToMap(q->green)+Y]+
            z_map[ScaleQuantumToMap(q->blue)+Y]+primary_info.y;
          quantum.green=(unsigned long) ((pixel.green < 0) ? 0 :
            (pixel.green > MaxMap) ? MaxMap : pixel.green+0.5);
          pixel.blue=x_map[ScaleQuantumToMap(q->red)+Z]+
            y_map[ScaleQuantumToMap(q->green)+Z]+
            z_map[ScaleQuantumToMap(q->blue)+Z]+primary_info.z;
          quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0 :
            (pixel.blue > MaxMap) ? MaxMap : pixel.blue+0.5);
          q->red=ScaleMapToQuantum(quantum.red);
          q->green=ScaleMapToQuantum(quantum.green);
          q->blue=ScaleMapToQuantum(quantum.blue);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(RGBTransformImageTag,y,image->rows,exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Convert PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        pixel.red=x_map[ScaleQuantumToMap(image->colormap[i].red)+X]+
          y_map[ScaleQuantumToMap(image->colormap[i].green)+X]+
          z_map[ScaleQuantumToMap(image->colormap[i].blue)+X]+primary_info.x;
        quantum.red=(unsigned long) ((pixel.red < 0) ? 0.0 :
          (pixel.red > MaxMap) ? MaxMap : pixel.red+0.5);
        pixel.green=x_map[ScaleQuantumToMap(image->colormap[i].red)+Y]+
          y_map[ScaleQuantumToMap(image->colormap[i].green)+Y]+
          z_map[ScaleQuantumToMap(image->colormap[i].blue)+Y]+primary_info.y;
        quantum.green=(unsigned long) ((pixel.green < 0) ? 0.0 :
          (pixel.green > MaxMap) ? MaxMap : pixel.green+0.5);
        pixel.blue=x_map[ScaleQuantumToMap(image->colormap[i].red)+Z]+
          y_map[ScaleQuantumToMap(image->colormap[i].green)+Z]+
          z_map[ScaleQuantumToMap(image->colormap[i].blue)+Z]+primary_info.z;
        quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0.0 :
          (pixel.blue > MaxMap) ? MaxMap : pixel.blue+0.5);
        image->colormap[i].red=ScaleMapToQuantum(quantum.red);
        image->colormap[i].green=ScaleMapToQuantum(quantum.green);
        image->colormap[i].blue=ScaleMapToQuantum(quantum.blue);
      }
      SyncImage(image);
      break;
    }
  }
  /*
    Free allocate memory.
  */
  LiberateMemory((void **) &z_map);
  LiberateMemory((void **) &y_map);
  LiberateMemory((void **) &x_map);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P r o f i l e I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ProfileImage() adds or removes a ICC, IPTC, or generic profile from an
%  image.  If the profile is NULL, it is removed from the image otherwise
%  added.  Use a name of '*' and a profile of NULL to remove all profiles
%  from the image. Ownership of the profile is transferred to ImageMagick
%  (it should not be altered or deallocated) unless the clone option is set
%  to True.
%
%  The format of the ProfileImage method is:
%
%      unsigned int ProfileImage(Image *image,const char *name,
%        const unsigned char *profile,const size_t length,unsigned int clone)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o name: Name of profile to add or remove: ICC, IPTC, or generic profile.
%
%    o profile: The profile.
%
%    o length: The length of the profile.
%
%    o clone: If set True, then copy the profile rather than taking
%             ownership of it.
%
%
*/
MagickExport unsigned int ProfileImage(Image *image,const char *name,
  const unsigned char *profile,const size_t length,unsigned int clone)
{
  register long
    i,
    j;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (name == (const char *) NULL)
    ThrowBinaryException(OptionError,"NoProfileNameWasGiven",
      "UnableToAddOrRemoveProfile");
  if ((profile == (const unsigned char *) NULL) || (length == 0))
    {
      /*
        Remove an ICM, IPTC, or generic profile from the image.
      */
      if (GlobExpression("icc",name) || GlobExpression("icm",name))
        {
          if (image->color_profile.length != 0)
            LiberateMemory((void **) &image->color_profile.info);
          image->color_profile.length=0;
          image->color_profile.info=(unsigned char *) NULL;
        }
      if (GlobExpression("iptc",name) || GlobExpression("8bim",name))
        {
          if (image->iptc_profile.length != 0)
            LiberateMemory((void **) &image->iptc_profile.info);
          image->iptc_profile.length=0;
          image->iptc_profile.info=(unsigned char *) NULL;
        }
      for (i=0; i < (long) image->generic_profiles; i++)
      {
        if (!GlobExpression(image->generic_profile[i].name,name))
          continue;
        if (image->generic_profile[i].name != (char *) NULL)
          LiberateMemory((void **) &image->generic_profile[i].name);
        if (image->generic_profile[i].length != 0)
          LiberateMemory((void **) &image->generic_profile[i].info);
        image->generic_profiles--;
        for (j=i; j < (long) image->generic_profiles; j++)
          image->generic_profile[j]=image->generic_profile[j+1];
        i--;
      }
      return(True);
    }
  /*
    Add a ICC, IPTC, or generic profile to the image.
  */
  if ((LocaleCompare("icc",name) == 0) || (LocaleCompare("icm",name) == 0))
    {
      if (image->color_profile.length != 0)
        {
#if defined(HasLCMS)
          typedef struct _ProfilePacket
          {
            unsigned short
              red,
              green,
              blue,
              opacity;
          } ProfilePacket;

          ColorspaceType
            colorspace;

          cmsHPROFILE
            image_profile,
            transform_profile;

          cmsHTRANSFORM
            transform;

          int
            intent;

          long
            y;

          ProfilePacket
            alpha,
            beta;

          register long
            x;

          register PixelPacket
            *q;

          /*
            Transform pixel colors as defined by the color profiles.
          */
          image_profile=cmsOpenProfileFromMem(image->color_profile.info,
            image->color_profile.length);
          transform_profile=
            cmsOpenProfileFromMem((unsigned char *) profile,length);
          if ((image_profile == (cmsHPROFILE) NULL) ||
              (transform_profile == (cmsHPROFILE) NULL))
            ThrowBinaryException(ResourceLimitError,"UnableToManageColor",
              "UnableToOpenColorProfile");
          switch (cmsGetColorSpace(transform_profile))
          {
            case icSigCmykData: colorspace=CMYKColorspace; break;
            case icSigYCbCrData: colorspace=YCbCrColorspace; break;
            case icSigLuvData: colorspace=YUVColorspace; break;
            case icSigGrayData: colorspace=GRAYColorspace; break;
            case icSigRgbData: colorspace=RGBColorspace; break;
            default: colorspace=RGBColorspace; break;
          }
          switch (image->rendering_intent)
          {
            case AbsoluteIntent: intent=INTENT_ABSOLUTE_COLORIMETRIC; break;
            case PerceptualIntent: intent=INTENT_PERCEPTUAL; break;
            case RelativeIntent: intent=INTENT_RELATIVE_COLORIMETRIC; break;
            case SaturationIntent: intent=INTENT_SATURATION; break;
            default: intent=INTENT_PERCEPTUAL; break;
          }
          if (image->colorspace == CMYKColorspace)
            {
              if (colorspace == CMYKColorspace)
                transform=cmsCreateTransform(image_profile,TYPE_CMYK_16,
                  transform_profile,TYPE_CMYK_16,intent,0);
              else
                transform=cmsCreateTransform(image_profile,TYPE_CMYK_16,
                  transform_profile,TYPE_RGBA_16,intent,0);
            }
          else
            {
              if (colorspace == CMYKColorspace)
                transform=cmsCreateTransform(image_profile,TYPE_RGBA_16,
                  transform_profile,TYPE_CMYK_16,intent,0);
              else
                transform=cmsCreateTransform(image_profile,TYPE_RGBA_16,
                  transform_profile,TYPE_RGBA_16,intent,0);
            }
          if (transform == (cmsHTRANSFORM) NULL)
            ThrowBinaryException(ResourceLimitError,"UnableToManageColor",
              "UnableToCreateColorTransform");
          for (y=0; y < (long) image->rows; y++)
          {
            q=GetImagePixels(image,0,y,image->columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) image->columns; x++)
            {
              alpha.red=ScaleQuantumToShort(q->red);
              alpha.green=ScaleQuantumToShort(q->green);
              alpha.blue=ScaleQuantumToShort(q->blue);
              alpha.opacity=ScaleQuantumToShort(q->opacity);
              cmsDoTransform(transform,&alpha,&beta,1);
              q->red=ScaleShortToQuantum(beta.red);
              q->green=ScaleShortToQuantum(beta.green);
              q->blue=ScaleShortToQuantum(beta.blue);
              q->opacity=ScaleShortToQuantum(beta.opacity);
              q++;
            }
            if (!SyncImagePixels(image))
              break;
          }
          cmsDeleteTransform(transform);
          cmsCloseProfile(image_profile);
          cmsCloseProfile(transform_profile);
          if (colorspace == CMYKColorspace)
            image->colorspace=CMYKColorspace;
          else
            image->colorspace=RGBColorspace;
#else
          ThrowBinaryException(MissingDelegateError,"LCMSLibraryIsNotAvailable",
            image->filename);
#endif
          LiberateMemory((void **) &image->color_profile.info);
        }
      if (!clone)
        {
          image->color_profile.length=length;
          image->color_profile.info=(unsigned char *) profile;
        }
      else
        {
          image->color_profile.info=(unsigned char *) AcquireMemory(length);
          if (image->color_profile.info == (unsigned char *) NULL)
            ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
              "UnableToAddICMProfile");
          image->color_profile.length=length;
          (void) memcpy(image->color_profile.info,profile,length);
        }
      return(True);
    }
  if ((LocaleCompare("iptc",name) == 0) || (LocaleCompare("8bim",name) == 0))
    {
      if (image->iptc_profile.length != 0)
        LiberateMemory((void **) &image->iptc_profile.info);
      if (!clone)
        {
          image->iptc_profile.length=length;
          image->iptc_profile.info=(unsigned char *) profile;
        }
      else
        {
          image->iptc_profile.info=(unsigned char *) AcquireMemory(length);
          if (image->iptc_profile.info == (unsigned char *) NULL)
            ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
              "UnableToAddIPTCProfile");
          image->iptc_profile.length=length;
          (void) memcpy(image->iptc_profile.info,profile,length);
        }
      return(True);
    }
  for (i=0; i < (long) image->generic_profiles; i++)
    if (LocaleCompare(image->generic_profile[i].name,name) == 0)
      break;
  if (i == (long) image->generic_profiles)
    {
      if (image->generic_profile == (ProfileInfo *) NULL)
        image->generic_profile=(ProfileInfo *)
          AcquireMemory((i+1)*sizeof(ProfileInfo));
      else
        ReacquireMemory((void **) &image->generic_profile,
          (i+1)*sizeof(ProfileInfo));
      if (image->generic_profile == (ProfileInfo *) NULL)
        ThrowBinaryException(ResourceLimitWarning,"MemoryAllocationFailed",
          (char *) NULL)
      image->generic_profiles++;
      image->generic_profile[i].length=0;
      image->generic_profile[i].info=(unsigned char *) NULL;
      image->generic_profile[i].name=AllocateString(name);
    }
  if (image->generic_profile[i].length != 0)
    LiberateMemory((void **) &image->generic_profile[i].info);
  if (!clone)
    {
      image->generic_profile[i].length=length;
      image->generic_profile[i].info=(unsigned char *) profile;
    }
  else
    {
      image->generic_profile[i].info=(unsigned char *) AcquireMemory(length);
      if (image->generic_profile[i].info == (unsigned char *) NULL)
        ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
          "UnableToAddGenericProfile");
      image->generic_profile[i].length=length;
      (void) memcpy(image->generic_profile[i].info,profile,length);
    }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   S e t I m a g e C o l o r s p a c e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  SetImageColorspace() sets the colorspace member of the Image structure.
%
%  The format of the SetImageColorspace method is:
%
%      unsigned int SetImageColorspace(Image *image,
%        const ColorspaceType colorspace)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o colorspace: The colorspace.
%
%
*/
MagickExport unsigned int SetImageColorspace(Image *image,
  const ColorspaceType colorspace)
{
  if (image->colorspace == colorspace)
    return(True);
  if ((colorspace == RGBColorspace) || (colorspace == TransparentColorspace))
    return(TransformRGBImage(image,image->colorspace));
  if ((image->colorspace != RGBColorspace) &&
      (image->colorspace != TransparentColorspace) &&
      (image->colorspace != GRAYColorspace))
    (void) TransformRGBImage(image,image->colorspace);
  return(RGBTransformImage(image,colorspace));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+     T r a n s f o r m R G B I m a g e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  TransformRGBImage() converts the reference image from an alternate
%  colorspace to RGB.  The transformation matrices are not the standard ones:
%  the weights are rescaled to normalize the range of the transformed values to
%  be [0..MaxRGB].
%
%  The format of the TransformRGBImage method is:
%
%      unsigned int TransformRGBImage(Image *image,
%        const ColorspaceType colorspace)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o colorspace: the colorspace to transform the image to.
%
%
*/
MagickExport unsigned int TransformRGBImage(Image *image,
  const ColorspaceType colorspace)
{
#define B  (2*(MaxMap+1))
#define G  (MaxMap+1)
#define R  0
#define MaxYCC  ((unsigned long) (1.3584*MaxMap+0.5))
#define TransformRGBImageTag  "Transform/Image"

  static const unsigned char
    sRGBMap[351] =
    {
      0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
      19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 29, 30, 31, 32, 33,
      34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
      50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 65, 66,
      67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82,
      83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 95, 96, 97, 98, 99,
      100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112,
      114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126,
      127, 128, 129, 130, 131, 132, 133, 135, 136, 137, 138, 139, 140,
      141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153,
      154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166,
      167, 168, 169, 170, 171, 172, 173, 174, 175, 175, 176, 177, 178,
      179, 180, 181, 182, 183, 184, 185, 186, 187, 187, 188, 189, 190,
      191, 192, 193, 194, 194, 195, 196, 197, 198, 199, 199, 200, 201,
      202, 203, 203, 204, 205, 206, 207, 207, 208, 209, 210, 210, 211,
      212, 213, 213, 214, 215, 215, 216, 217, 218, 218, 219, 220, 220,
      221, 222, 222, 223, 223, 224, 225, 225, 226, 227, 227, 228, 228,
      229, 229, 230, 230, 231, 232, 232, 233, 233, 234, 234, 235, 235,
      235, 236, 236, 237, 237, 238, 238, 238, 239, 239, 240, 240, 240,
      241, 241, 242, 242, 242, 243, 243, 243, 243, 244, 244, 244, 245,
      245, 245, 245, 246, 246, 246, 247, 247, 247, 247, 247, 248, 248,
      248, 248, 249, 249, 249, 249, 249, 249, 250, 250, 250, 250, 250,
      250, 251, 251, 251, 251, 251, 251, 252, 252, 252, 252, 252, 252,
      252, 252, 252, 253, 253, 253, 253, 253, 253, 253, 253, 253, 254,
      254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 255, 255,
      255, 255, 255, 255, 255
    },
    YCCMap[351] =  /* Photo CD information beyond 100% white, Gamma 2.2 */
    {
      0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
      19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 32, 33, 34, 35,
      36, 37, 38, 39, 40, 41, 42, 43, 45, 46, 47, 48, 49, 50, 51, 52,
      53, 54, 56, 57, 58, 59, 60, 61, 62, 63, 64, 66, 67, 68, 69, 70,
      71, 72, 73, 74, 76, 77, 78, 79, 80, 81, 82, 83, 84, 86, 87, 88,
      89, 90, 91, 92, 93, 94, 95, 97, 98, 99, 100, 101, 102, 103, 104,
      105, 106, 107, 108, 110, 111, 112, 113, 114, 115, 116, 117, 118,
      119, 120, 121, 122, 123, 124, 125, 126, 127, 129, 130, 131, 132,
      133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145,
      146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158,
      159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171,
      172, 173, 174, 175, 176, 176, 177, 178, 179, 180, 181, 182, 183,
      184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 193, 194, 195,
      196, 197, 198, 199, 200, 201, 201, 202, 203, 204, 205, 206, 207,
      207, 208, 209, 210, 211, 211, 212, 213, 214, 215, 215, 216, 217,
      218, 218, 219, 220, 221, 221, 222, 223, 224, 224, 225, 226, 226,
      227, 228, 228, 229, 230, 230, 231, 232, 232, 233, 234, 234, 235,
      236, 236, 237, 237, 238, 238, 239, 240, 240, 241, 241, 242, 242,
      243, 243, 244, 244, 245, 245, 245, 246, 246, 247, 247, 247, 248,
      248, 248, 249, 249, 249, 249, 250, 250, 250, 250, 251, 251, 251,
      251, 251, 252, 252, 252, 252, 252, 253, 253, 253, 253, 253, 253,
      253, 253, 253, 253, 253, 253, 253, 254, 254, 254, 254, 254, 254,
      254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255
    };

  double
    *blue_map,
    *green_map,
    *red_map;

  DoublePixelPacket
    pixel;

  long
    y;

  LongPixelPacket
    quantum;

  register long
    x;

  register long
    i;

  register PixelPacket
    *q;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if ((colorspace == RGBColorspace) || (colorspace == GRAYColorspace) ||
      (colorspace == TransparentColorspace))
    return(True);
  switch (colorspace)
  {
    case CMYKColorspace:
    {
      IndexPacket
        *indexes;

      /*
        Transform image from CMYK to RGB.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) image->columns; x++)
        {
          q->red=(Quantum)
            (((double) (MaxRGB-q->red)*(MaxRGB-q->opacity))/MaxRGB);
          q->green=(Quantum)
            (((double) (MaxRGB-q->green)*(MaxRGB-q->opacity))/MaxRGB);
          q->blue=(Quantum)
            (((double) (MaxRGB-q->blue)*(MaxRGB-q->opacity))/MaxRGB);
          q->opacity=image->matte ? indexes[x] : OpaqueOpacity;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      image->colorspace=RGBColorspace;
      return(True);
    }
    case HSLColorspace:
    {
      double
        hue,
        luminosity,
        saturation;

      /*
        Transform image from HSL to RGB.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          hue=(double) q->red/MaxRGB;
          saturation=(double) q->green/MaxRGB;
          luminosity=(double) q->blue/MaxRGB;
          HSLTransform(hue,saturation,luminosity,&q->red,&q->green,&q->blue);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      image->colorspace=RGBColorspace;
      return(True);
    }
    case HWBColorspace:
    {
      double
        blackness,
        hue,
        whiteness;

      /*
        Transform image from HWB to RGB.
      */
      if (image->storage_class == PseudoClass)
        {
          SyncImage(image);
          image->storage_class=DirectClass;
        }
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          hue=(double) q->red/MaxRGB;
          whiteness=(double) q->green/MaxRGB;
          blackness=(double) q->blue/MaxRGB;
          HWBTransform(hue,whiteness,blackness,&q->red,&q->green,&q->blue);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
      }
      image->colorspace=RGBColorspace;
      return(True);
    }
    default:
      break;
  }
  /*
    Allocate the tables.
  */
  red_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  green_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  blue_map=(double *) AcquireMemory(3*(MaxMap+1)*sizeof(double));
  if ((red_map == (double *) NULL) || (green_map == (double *) NULL) ||
      (blue_map == (double *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToSetImageColorspace");
  switch (colorspace)
  {
    case OHTAColorspace:
    {
      /*
        Initialize OHTA tables:

          R = I1+1.00000*I2-0.66668*I3
          G = I1+0.00000*I2+1.33333*I3
          B = I1-1.00000*I2-0.66668*I3

        I and Q, normally -0.5 through 0.5, must be normalized to the range 0
        through MaxRGB.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=i;
        green_map[i+R]=0.5*(2.0*i-MaxMap);
        blue_map[i+R]=(-0.33334)*(2.0*i-MaxMap);
        red_map[i+G]=i;
        green_map[i+G]=0.0;
        blue_map[i+G]=0.666665*(2.0*i-MaxMap);
        red_map[i+B]=i;
        green_map[i+B]=(-0.5)*(2.0*i-MaxMap);
        blue_map[i+B]=(-0.33334)*(2.0*i-MaxMap);
      }
      break;
    }
    case sRGBColorspace:
    {
      /*
        Initialize sRGB tables:

          R = Y            +1.032096*C2
          G = Y-0.326904*C1-0.704445*C2
          B = Y+1.685070*C1

        sRGB is scaled by 1.3584.  C1 zero is 156 and C2 is at 137.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=1.40200*i;
        green_map[i+R]=0.0;
        blue_map[i+R]=1.88000*(i-(long)
          ScaleQuantumToMap(ScaleCharToQuantum(137)));
        red_map[i+G]=1.40200*i;
        green_map[i+G]=(-0.444066)*(i-(long)
          ScaleQuantumToMap(ScaleCharToQuantum(156)));
        blue_map[i+G]=(-0.95692)*(i-(long)
          ScaleQuantumToMap(ScaleCharToQuantum(137)));
        red_map[i+B]=1.40200*i;
        green_map[i+B]=2.28900*(i-(long)
          ScaleQuantumToMap(ScaleCharToQuantum(156)));
        blue_map[i+B]=0.0;
      }
      break;
    }
    case XYZColorspace:
    {
      /*
        Initialize CIE XYZ tables:

          R =  3.240479*R-1.537150*G-0.498535*B
          G = -0.969256*R+1.875992*G+0.041556*B
          B =  0.055648*R-0.204043*G+1.057311*B
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=3.240479*i;
        green_map[i+R]=(-1.537150)*i;
        blue_map[i+R]=(-0.498535)*i;
        red_map[i+G]=(-0.969256)*i;
        green_map[i+G]=1.875992*i;
        blue_map[i+G]=0.041556*i;
        red_map[i+B]=0.055648*i;
        green_map[i+B]=(-0.204043)*i;
        blue_map[i+B]=1.057311*i;
      }
      break;
    }
    case YCbCrColorspace:
    {
      /*
        Initialize YCbCr tables:

          R = Y            +1.402000*Cr
          G = Y-0.344136*Cb-0.714136*Cr
          B = Y+1.772000*Cb

        Cb and Cr, normally -0.5 through 0.5, must be normalized to the range 0
        through MaxRGB.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=i;
        green_map[i+R]=0.0;
        blue_map[i+R]=(1.402000*0.5)*(2.0*i-MaxMap);
        red_map[i+G]=i;
        green_map[i+G]=(-0.344136*0.5)*(2.0*i-MaxMap);
        blue_map[i+G]=(-0.714136*0.5)*(2.0*i-MaxMap);
        red_map[i+B]=i;
        green_map[i+B]=(1.772000*0.5)*(2.0*i-MaxMap);
        blue_map[i+B]=0.0;
      }
      break;
    }
    case YCCColorspace:
    {
      /*
        Initialize YCC tables:

          R = Y            +1.340762*C2
          G = Y-0.317038*C1-0.682243*C2
          B = Y+1.632639*C1

        YCC is scaled by 1.3584.  C1 zero is 156 and C2 is at 137.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=1.3584*i;
        green_map[i+R]=0.0;
        blue_map[i+R]=1.8215*(i-(double)
          ScaleQuantumToMap(ScaleCharToQuantum(137)));
        red_map[i+G]=1.3584*i;
        green_map[i+G]=(-0.4302726)*(i-(double)
          ScaleQuantumToMap(ScaleCharToQuantum(156)));
        blue_map[i+G]=(-0.9271435)*(i-(double)
          ScaleQuantumToMap(ScaleCharToQuantum(137)));
        red_map[i+B]=1.3584*i;
        green_map[i+B]=2.2179*(i-(double)
          ScaleQuantumToMap(ScaleCharToQuantum(156)));
        blue_map[i+B]=0.0;
      }
      break;
    }
    case YIQColorspace:
    {
      /*
        Initialize YIQ tables:

          R = Y+0.95620*I+0.62140*Q
          G = Y-0.27270*I-0.64680*Q
          B = Y-1.10370*I+1.70060*Q

        I and Q, normally -0.5 through 0.5, must be normalized to the range 0
        through MaxRGB.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=i;
        green_map[i+R]=0.4781*(2.0*i-MaxMap);
        blue_map[i+R]=0.3107*(2.0*i-MaxMap);
        red_map[i+G]=i;
        green_map[i+G]=(-0.13635)*(2.0*i-MaxMap);
        blue_map[i+G]=(-0.3234)*(2.0*i-MaxMap);
        red_map[i+B]=i;
        green_map[i+B]=(-0.55185)*(2.0*i-MaxMap);
        blue_map[i+B]=0.8503*(2.0*i-MaxMap);
      }
      break;
    }
    case YPbPrColorspace:
    {
      /*
        Initialize YPbPr tables:

          R = Y            +1.402000*C2
          G = Y-0.344136*C1+0.714136*C2
          B = Y+1.772000*C1

        Pb and Pr, normally -0.5 through 0.5, must be normalized to the range 0
        through MaxRGB.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=i;
        green_map[i+R]=0.0;
        blue_map[i+R]=0.701*(2.0*i-MaxMap);
        red_map[i+G]=i;
        green_map[i+G]=(-0.172068)*(2.0*i-MaxMap);
        blue_map[i+G]=0.357068*(2.0*i-MaxMap);
        red_map[i+B]=i;
        green_map[i+B]=0.886*(2.0*i-MaxMap);
        blue_map[i+B]=0.0;
      }
      break;
    }
    case YUVColorspace:
    default:
    {
      /*
        Initialize YUV tables:

          R = Y          +1.13980*V
          G = Y-0.39380*U-0.58050*V
          B = Y+2.02790*U

        U and V, normally -0.5 through 0.5, must be normalized to the range 0
        through MaxRGB.
      */
      for (i=0; i <= (long) MaxMap; i++)
      {
        red_map[i+R]=i;
        green_map[i+R]=0.0;
        blue_map[i+R]=0.5699*(2.0*i-MaxMap);
        red_map[i+G]=i;
        green_map[i+G]=(-0.1969)*(2.0*i-MaxMap);
        blue_map[i+G]=(-0.29025)*(2.0*i-MaxMap);
        red_map[i+B]=i;
        green_map[i+B]=1.01395*(2.0*i-MaxMap);
        blue_map[i+B]=0;
      }
      break;
    }
  }
  /*
    Convert to RGB.
  */
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      ExceptionInfo
        *exception;

      /*
        Convert DirectClass image.
      */
      exception=(&image->exception);
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=0; x < (long) image->columns; x++)
        {
          pixel.red=red_map[ScaleQuantumToMap(q->red)+R]+
            green_map[ScaleQuantumToMap(q->green)+R]+
            blue_map[ScaleQuantumToMap(q->blue)+R];
          pixel.green=red_map[ScaleQuantumToMap(q->red)+G]+
            green_map[ScaleQuantumToMap(q->green)+G]+
            blue_map[ScaleQuantumToMap(q->blue)+G];
          pixel.blue=red_map[ScaleQuantumToMap(q->red)+B]+
            green_map[ScaleQuantumToMap(q->green)+B]+
            blue_map[ScaleQuantumToMap(q->blue)+B];
          switch (colorspace)
          {
            case sRGBColorspace:
            case YCCColorspace:
            {
              quantum.red=(unsigned long) ((pixel.red < 0) ? 0 :
                (pixel.red > MaxYCC) ? MaxYCC : pixel.red+0.5);
              quantum.green=(unsigned long) ((pixel.green < 0) ? 0 :
                (pixel.green > MaxYCC) ? MaxYCC : pixel.green+0.5);
              quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0 :
                (pixel.blue > MaxYCC) ? MaxYCC : pixel.blue+0.5);
              if (colorspace == sRGBColorspace)
                {
                  q->red=ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.red)]);
                  q->green=ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.green)]);
                  q->blue=ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.blue)]);
                  break;
                }
              q->red=ScaleToQuantum(YCCMap[ScaleQuantum(quantum.red)]);
              q->green=ScaleToQuantum(YCCMap[ScaleQuantum(quantum.green)]);
              q->blue=ScaleToQuantum(YCCMap[ScaleQuantum(quantum.blue)]);
              break;
            }
            default:
            {
              quantum.red=(unsigned long) ((pixel.red < 0) ? 0 :
                (pixel.red > MaxMap) ? MaxMap : pixel.red+0.5);
              quantum.green=(unsigned long) ((pixel.green < 0) ? 0 :
                (pixel.green > MaxMap) ? MaxMap : pixel.green+0.5);
              quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0 :
                (pixel.blue > MaxMap) ? MaxMap : pixel.blue+0.5);
              q->red=ScaleMapToQuantum(quantum.red);
              q->green=ScaleMapToQuantum(quantum.green);
              q->blue=ScaleMapToQuantum(quantum.blue);
              break;
            }
          }
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(TransformRGBImageTag,y,image->rows,exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Convert PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        pixel.red=red_map[ScaleQuantumToMap(image->colormap[i].red)+R]+
          green_map[ScaleQuantumToMap(image->colormap[i].green)+R]+
          blue_map[ScaleQuantumToMap(image->colormap[i].blue)+R];
        pixel.green=red_map[ScaleQuantumToMap(image->colormap[i].red)+G]+
          green_map[ScaleQuantumToMap(image->colormap[i].green)+G]+
          blue_map[ScaleQuantumToMap(image->colormap[i].blue)+G];
        pixel.blue=red_map[ScaleQuantumToMap(image->colormap[i].red)+B]+
          green_map[ScaleQuantumToMap(image->colormap[i].green)+B]+
          blue_map[ScaleQuantumToMap(image->colormap[i].blue)+B];
        switch (colorspace)
        {
          case sRGBColorspace:
          case YCCColorspace:
          {
            quantum.red=(unsigned long) ((pixel.red < 0) ? 0 :
              (pixel.red > MaxYCC) ? MaxYCC : pixel.red+0.5);
            quantum.green=(unsigned long) ((pixel.green < 0) ? 0 :
              (pixel.green > MaxYCC) ? MaxYCC : pixel.green+0.5);
            quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0 :
              (pixel.blue > MaxYCC) ? MaxYCC : pixel.blue+0.5);
            if (colorspace == sRGBColorspace)
              {
                image->colormap[i].red=
                  ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.red)]);
                image->colormap[i].green=
                  ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.green)]);
                image->colormap[i].blue=(Quantum)
                  ScaleToQuantum(sRGBMap[ScaleQuantum(quantum.blue)]);
                break;
              }
            image->colormap[i].red=
              ScaleToQuantum(YCCMap[ScaleQuantum(quantum.red)]);
            image->colormap[i].green=
              ScaleToQuantum(YCCMap[ScaleQuantum(quantum.green)]);
            image->colormap[i].blue=
              ScaleToQuantum(YCCMap[ScaleQuantum(quantum.blue)]);
            break;
          }
          default:
          {
            quantum.red=(unsigned long) ((pixel.red < 0) ? 0 :
              (pixel.red > MaxMap) ? MaxMap : pixel.red+0.5);
            quantum.green=(unsigned long) ((pixel.green < 0) ? 0 :
              (pixel.green > MaxMap) ? MaxMap : pixel.green+0.5);
            quantum.blue=(unsigned long) ((pixel.blue < 0) ? 0 :
              (pixel.blue > MaxMap) ? MaxMap : pixel.blue+0.5);
            image->colormap[i].red=ScaleMapToQuantum(quantum.red);
            image->colormap[i].green=ScaleMapToQuantum(quantum.green);
            image->colormap[i].blue=ScaleMapToQuantum(quantum.blue);
            break;
          }
        }
      }
      SyncImage(image);
      break;
    }
  }
  image->colorspace=RGBColorspace;
  /*
    Free allocate memory.
  */
  LiberateMemory((void **) &blue_map);
  LiberateMemory((void **) &green_map);
  LiberateMemory((void **) &red_map);
  return(True);
}
