/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * template - argus client template.
 *
 *  This module must define these routines:
 *
 *   (void) usage (void);
 *                    this routine should print the standard usage message
 *                    for the specific application.
 *
 *          ArgusClientInit ();
 *                    this is the application specific init
 *                    routine, which is called after all parsing
 *                    initialization is done, prior to reading the
 *                    first argus(1) datum.
 *
 *   (void) ArgusClientTimeout ();
 *                    this routine is called every second, when
 *                    the argus client is connected to a remote
 *                    data source using the -S flag.
 *
 *          RaProcessRecord ((struct ArgusRecord *) argus);
 *                    this routine is called from the main library
 *                    for all Argus Records read from the stream.
 *                    The template suggests calling protocol specific
 *                    subroutines, but this is just a suggestion.
 *
 *   (void) RaParseComplete (int);
 *                    this routine will be called after all the
 *                    monitor data has been read.
 *
 *
 *  These modules can optionally extend the common routines functions:
 *
 *          parse_arg (argc, argv)
 *                    this routine can process client specific command
 *                    line options, specified with appOptstring.
 *
 *
 *
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

#include <argus_client.h>


int RaInitialized = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized))
      RaInitialized++;

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusClientInit: returning\n");
#endif
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaParseComplete: returning\n");
#endif
}

void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusClientTimeout: returning\n");
#endif
}

void
parse_arg (int argc, char**argv)
{ 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "parse_arg: returning\n");
#endif
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Ratemplate Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a                print record summaries on termination.\n");
   fprintf (stderr, "         -A                print application bytes.\n");
   fprintf (stderr, "         -b                dump packet-matching code.\n");
   fprintf (stderr, "         -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -D <level>        specify debug level\n");
   fprintf (stderr, "         -E <file>         write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -f <flowfile>     read flow model from <flowfile>.\n");
   fprintf (stderr, "         -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                print help.\n");
   fprintf (stderr, "         -n                don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>       print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -q                quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R                print out response data when availabile.\n");
   fprintf (stderr, "         -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "                                               -%%d{yMhdms}\n");
   fprintf (stderr, "         -T <secs>         attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u                print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>         write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z                print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>        print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessRecord (0x%x) returning\n", argus);
#endif
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessManRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessTCPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessICMPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessICMPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessUDPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessUDPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessIPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessIPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessARPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessARPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessNonIPRecord (0x%x) returning\n", argus);
#endif
}

int
RaSendArgusRecord(struct ArgusRecordStore *argus)
{
   int retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaSendArgusRecord (0x%x) returning\n", argus);
#endif

   return (retn);
}
