///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_stream.hpp>

#include "LAMMPSTextDumpWriter.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSTextDumpWriter, LAMMPSDumpWriter)

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool LAMMPSTextDumpWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	using namespace boost::iostreams;

	MsgLogger() << "Opening text LAMMPS dump file" << filepath << "for writing." << endl;

	bool isGZipped = filepath.endsWith(QLatin1String(".gz"));
	std::ios_base::openmode openmode = std::ios_base::out;
	if(isGZipped) openmode |= std::ios_base::binary;
	stream<file_sink> file_stream(filepath.toUtf8().constData(), openmode);
	if(!file_stream->is_open())
		throw Exception(tr("Failed to open LAMMPS dump file %1 for writing.").arg(filepath));

	// Automatically compress gzipped dump files.
	filtering_stream<output> output_stream;
	if(isGZipped) output_stream.push(gzip_compressor());
	output_stream.push(file_stream);

	ProgressIndicator progress(QString(), exportFrames.size() * 100, suppressDialogs);

	Q_FOREACH(TimeTicks time, exportFrames) {
		int frame = time / dataset->animationSettings()->ticksPerFrame();
		progress.setLabelText(tr("Writing LAMMPS dump file (frame %1)").arg(frame));
		if(progress.isCanceled()) return false;

		// Extract the atoms to be exported from the scene.
		PipelineFlowState flowState = retrieveAtoms(dataset, time);
		AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
		if(atoms == NULL)
			throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));

		DataRecordWriterHelper helper(&channelMapping(), atoms);

		TimeInterval interval;
		AffineTransformation simCell = atoms->simulationCell()->cellMatrix();

		FloatType xlo = simCell.getTranslation().X;
		FloatType ylo = simCell.getTranslation().Y;
		FloatType zlo = simCell.getTranslation().Z;
		FloatType xhi = simCell.column(0).X + xlo;
		FloatType yhi = simCell.column(1).Y + ylo;
		FloatType zhi = simCell.column(2).Z + zlo;
		FloatType xy = simCell.column(1).X;
		FloatType xz = simCell.column(2).X;
		FloatType yz = simCell.column(2).Y;

		xlo += min((FloatType)0.0, min(xy, min(xz, xy+xz)));
		xhi += max((FloatType)0.0, max(xy, max(xz, xy+xz)));
		ylo += min((FloatType)0.0, yz);
		yhi += max((FloatType)0.0, yz);

		int numAtoms = atoms->atomsCount();
		output_stream << "ITEM: TIMESTEP" << endl;
		output_stream << frame << endl;
		output_stream << "ITEM: NUMBER OF ATOMS" << endl;
		output_stream << numAtoms << endl;
		if(xy != 0 || xz != 0 || yz != 0) {
			output_stream << "ITEM: BOX BOUNDS xy xz yz";
			if(atoms->simulationCell()->periodicity()[0]) output_stream << " pp"; else output_stream << " ff";
			if(atoms->simulationCell()->periodicity()[1]) output_stream << " pp"; else output_stream << " ff";
			if(atoms->simulationCell()->periodicity()[2]) output_stream << " pp"; else output_stream << " ff";
			output_stream << endl;
			output_stream << xlo << " " << xhi << " " << xy << endl;
			output_stream << ylo << " " << yhi << " " << xz << endl;
			output_stream << zlo << " " << zhi << " " << yz << endl;
		}
		else {
			output_stream << "ITEM: BOX BOUNDS";
			if(atoms->simulationCell()->periodicity()[0]) output_stream << " pp"; else output_stream << " ff";
			if(atoms->simulationCell()->periodicity()[1]) output_stream << " pp"; else output_stream << " ff";
			if(atoms->simulationCell()->periodicity()[2]) output_stream << " pp"; else output_stream << " ff";
			output_stream << endl;
			output_stream << xlo << " " << xhi << endl;
			output_stream << ylo << " " << yhi << endl;
			output_stream << zlo << " " << zhi << endl;
		}
		output_stream << "ITEM: ATOMS";

		// Write column names to dump file.
		for(int i=0; i<channelMapping().columnCount(); i++) {
			switch(channelMapping().getChannelId(i)) {
			case DataChannel::PositionChannel:
				if(channelMapping().getVectorComponent(i) == 0) output_stream << " x";
				else if(channelMapping().getVectorComponent(i) == 1) output_stream << " y";
				else if(channelMapping().getVectorComponent(i) == 2) output_stream << " z";
				else output_stream << " position";
				break;
			case DataChannel::VelocityChannel:
				if(channelMapping().getVectorComponent(i) == 0) output_stream << " vx";
				else if(channelMapping().getVectorComponent(i) == 1) output_stream << " vy";
				else if(channelMapping().getVectorComponent(i) == 2) output_stream << " vz";
				else output_stream << " velocity";
				break;
			case DataChannel::ForceChannel:
				if(channelMapping().getVectorComponent(i) == 0) output_stream << " fx";
				else if(channelMapping().getVectorComponent(i) == 1) output_stream << " fy";
				else if(channelMapping().getVectorComponent(i) == 2) output_stream << " fz";
				else output_stream << " force";
				break;
			case DataChannel::PeriodicImageChannel:
				if(channelMapping().getVectorComponent(i) == 0) output_stream << " ix";
				else if(channelMapping().getVectorComponent(i) == 1) output_stream << " iy";
				else if(channelMapping().getVectorComponent(i) == 2) output_stream << " iz";
				else output_stream << " periodicimage";
				break;
			case DataChannel::AtomIndexChannel: output_stream << " id"; break;
			case DataChannel::AtomTypeChannel: output_stream << " type"; break;
			case DataChannel::MassChannel: output_stream << " mass"; break;
			case DataChannel::CNATypeChannel: output_stream << " c_cna"; break;
			case DataChannel::PotentialEnergyChannel: output_stream << " c_epot"; break;
			case DataChannel::KineticEnergyChannel: output_stream << " c_kpot"; break;
			case DataChannel::SelectionChannel: output_stream << " selection"; break;
			case DataChannel::StressTensorChannel:
				if(channelMapping().getVectorComponent(i) == 0) output_stream << " c_stress[1]";
				else if(channelMapping().getVectorComponent(i) == 1) output_stream << " c_stress[2]";
				else if(channelMapping().getVectorComponent(i) == 2) output_stream << " c_stress[3]";
				else if(channelMapping().getVectorComponent(i) == 3) output_stream << " c_stress[4]";
				else if(channelMapping().getVectorComponent(i) == 4) output_stream << " c_stress[5]";
				else if(channelMapping().getVectorComponent(i) == 5) output_stream << " c_stress[6]";
				else output_stream << " stress";
				break;
			default:
				QString columnName = channelMapping().getChannelName(i);
				columnName.remove(QRegExp("[^A-Za-z\\d_]"));
				output_stream << " " << columnName.toStdString();
				if(channelMapping().getChannelId(i) == DataChannel::UserDataChannel) {
					if(channelMapping().getVectorComponent(i) > 0)
						output_stream << "." << channelMapping().getVectorComponent(i);
				}
				else {
					QStringList componentNames = DataChannel::standardChannelComponentNames(channelMapping().getChannelId(i));
					if(channelMapping().getVectorComponent(i) < componentNames.size()) {
						output_stream << "." << componentNames[channelMapping().getVectorComponent(i)].toStdString();
					}
				}

			}
		}
		output_stream << endl;

		int progressStartValue = progress.value();
		for(int i=0; i<numAtoms; i++) {

			// Update progress indicator from time to time.
			if((i % 4096) == 0) {
				progress.setValue(i * 100 / numAtoms + progressStartValue);
				progress.isCanceled();
			}

			helper.writeAtom(i, output_stream);
			output_stream << endl;
		}
	}
	return true;
}

};	// End of namespace AtomViz
