///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ColorControllerUI, ReferenceParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
ColorControllerUI::ColorControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField) 
	: ReferenceParameterUI(parentEditor, refField)
{
	_label = new QLabel(referenceField().displayName() + ":");
	_colorPicker = new ColorPickerWidget();
	_colorPicker->setObjectName("colorButton");
	connect(_colorPicker, SIGNAL(colorChanged()), this, SLOT(onColorPickerChanged()));
	
	// Update the displayed color when the animation time has changed.
	connect(&ANIM_MANAGER, SIGNAL(timeChanged(TimeTicks)), this, SLOT(updateUI()));
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
ColorControllerUI::~ColorControllerUI()
{
	// Release GUI controls. 
	delete label();
	delete colorPicker();
}

/******************************************************************************
* This method is called when parameter object has been assigned to the reference field of the editable object 
* this parameter UI is bound to. It is also called when the editable object itself has
* been replaced in the editor. This implementation of the method automatically
* disables the UI widgets when the reference field of the edited object does not contain a
* controller. 
******************************************************************************/
void ColorControllerUI::resetUI()
{
	ReferenceParameterUI::resetUI();	
	
	if(colorPicker()) {
		if(parameterObject()) {
			colorPicker()->setEnabled(isEnabled());
		}
		else {
			colorPicker()->setEnabled(false);
			colorPicker()->setColor(Color(1,1,1));
		}
	}
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void ColorControllerUI::updateUI()
{
	VectorController* ctrl = dynamic_object_cast<VectorController>(parameterObject());
	if(ctrl != NULL && colorPicker()) {
		Vector3 val = ctrl->getCurrentValue();
		colorPicker()->setColor(Color(val));	
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void ColorControllerUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	ReferenceParameterUI::setEnabled(enabled);
	if(colorPicker()) colorPicker()->setEnabled(parameterObject() != NULL && isEnabled());
}

/******************************************************************************
* Is called when the user has changed the color.
******************************************************************************/
void ColorControllerUI::onColorPickerChanged()
{
	VectorController* ctrl = dynamic_object_cast<VectorController>(parameterObject());
	if(ctrl == NULL) return;

	ViewportSuspender noVPUpdate;

	UNDO_MANAGER.beginCompoundOperation(tr("Change Color"));		
	ctrl->setCurrentValue(colorPicker()->color());
	UNDO_MANAGER.endCompoundOperation();
}

};

