use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * Test declaration.
 */
package class TestDecl extends NamedDecl {
	SStr name;
	Scope scope;
	SFreeOptions options;
	STestBody body;

	init(Scope scope, SStr name, SFreeOptions options, STestBody body) {
		init() {
			name = name;
			scope = scope;
			options = options;
			body = body;
		}
	}

	// Create.
	Named doCreate() {
		TestFn fn(scope, name, body);
		options.transform(fileScope(scope, name.pos), fn);
		fn;
	}

	// Update.
	Named? update(Scope scope) {
		var found = scope.find(Name(name.v));
		if (found as TestFn) {
			found.update(scope, body);
		} else if (found.empty) {
			var c = create();
			resolve();
			return c;
		}

		return null;
	}
}

/**
 * Test function.
 */
package class TestFn extends BSRawFn {
	Scope scope;
	STestBody body;

	init(Scope scope, SStr name, STestBody body) {
		init(Value(named{TestResult}), name, ValParam[], null) {
			scope = scope;
			body = body;
		}
	}

	// Update ourselves.
	void update(Scope scope, STestBody body) {
		this.scope = scope.withPos(pos);
		this.body = body;
		reset();
	}

	FnBody createBody() {
		TestBody body(this, scope.withPos(pos));

		this.body.transform(body);

		// Return the result.
		body.add(LocalVarAccess(SrcPos(), body.resultVar));

		// print(body.toS);
		body;
	}
}

/**
 * Custom body block that stores the result variable for easier access.
 */
package class TestBody extends FnBody {
	init(BSRawFn owner, Scope scope) {
		super(owner, scope);

		Var r(this, Value(named{TestResult}), SStr("test-result"), Actuals());

		init {
			resultVar = r.var;
		}

		add(r);
	}

	LocalVar resultVar;
}

// Find the 'result' variable.
package LocalVar findResult(Block block) {
	BlockLookup lookup = block.lookup;
	do {
		if (test = lookup.block as TestBody) {
			return test.resultVar;
		}

		if (p = lookup.parent as BlockLookup) {
			lookup = p;
		} else {
			throw SyntaxError(block.pos, "Must be used inside a test function!");
		}
	}
}

// Captured expression/text pair.
class ExprText on Compiler {
	Expr expr;
	Str text;

	init(Expr expr, Str text) {
		init {
			expr = expr;
			text = text;
		}
	}
}

// Add catch block for a test.
private CatchBlock createCatch(SrcPos pos, Block parent, LocalVarAccess result, Str exprText) on Compiler {
	CatchBlock block(pos, parent, named{Exception}, SStr("error"));
	ExprBlock sub(pos, block);
	block.expr = sub;

	sub.add(namedExpr(block, pos, "++*", namedExpr(block, pos, "crashed", result)));

	ConcatExpr msg(pos, block);
	msg.append(StrLiteral(pos, "Crashed: ${exprText} ==> got exception "));
	msg.append(namedExpr(block, pos, "error", Actuals()));
	sub.add(FnCall(pos, block.scope, named{print<Str>}, Actuals(msg)));

	block;
}


// Expression for a single check expression.
package Expr checkExpr(SrcPos pos, Block block, ExprText lhs, OpInfo op, ExprText rhs) on Compiler {
	LocalVar resultVar = findResult(block);
	LocalVarAccess result(pos, resultVar);

	TryBlock sub(pos, block);

	// Increment 'total'.
	sub.add(namedExpr(sub, pos, "++*", namedExpr(sub, pos, "total", result)));

	// Put lhs and rhs into an operator to find out what it means. This is so that any automatic
	// casts will be applied as needed.
	Value lhsType = lhs.expr.result.type;
	Value rhsType = rhs.expr.result.type;
	if (call = Operator(sub, lhs.expr, op, rhs.expr).meaning as FnCall) {
		var toExec = call.function;
		lhsType = toExec.params[0].asRef(false);
		rhsType = toExec.params[1].asRef(false);
	}

	// Evaluate lhs and rhs and save them.
	Var lhsVar(sub, lhsType, SStr(" l", pos), lhs.expr);
	sub.add(lhsVar);
	LocalVarAccess l(pos, lhsVar.var);

	Var rhsVar(sub, rhsType, SStr(" r", pos), rhs.expr);
	sub.add(rhsVar);
	LocalVarAccess r(pos, rhsVar.var);

	// Check if they are equal.
	BoolCondition cond(mkOperator(sub, l, op, r));
	If check(sub, cond);
	sub.add(check);

	ExprBlock fail(pos, check);
	check.fail(fail);

	fail.add(namedExpr(fail, pos, "++*", namedExpr(fail, pos, "failed", result)));

	Str exprText = "${lhs.text.trimWhitespace} ${op.name} ${rhs.text.trimWhitespace}";

	// Generate a nice message on failure.
	ConcatExpr msg(pos, fail);
	msg.append(StrLiteral(pos, "Failed: ${exprText} ==> "));
	msg.append(l);
	msg.append(StrLiteral(pos, " ${op.name} "));
	msg.append(r);

	fail.add(FnCall(pos, fail.scope, named{print<Str>}, Actuals(msg)));

	sub.addCatch(createCatch(pos, block, result, exprText));

	sub;
}

package Str reverseName(Str original) {
	if (original == "==")
		return "!=";
	else if (original == "!=")
		return "==";
	else if (original == "<")
		return ">=";
	else if (original == ">")
		return "<=";
	else if (original == "<=")
		return ">";
	else if (original == ">=")
		return "<";
	else if (original == "is")
		return "!is";
	else if (original == "!is")
		return is;
	return "!" + original;
}

package Expr checkExpr(SrcPos pos, Block block, Condition cond, Str text) on Compiler {
	LocalVar resultVar = findResult(block);
	LocalVarAccess result(pos, resultVar);

	TryBlock sub(pos, block);

	// Increment 'total'.
	sub.add(namedExpr(sub, pos, "++*", namedExpr(sub, pos, "total", result)));

	// Check if the condition evaluates to true.
	If check(sub, cond);
	sub.add(check);

	ExprBlock fail(pos, check);
	check.fail(fail);

	fail.add(namedExpr(fail, pos, "++*", namedExpr(fail, pos, "failed", result)));

	// Generate a message. In some cases, we could be more specific (e.g. for downcasts).
	text = text.trimWhitespace;
	StrLiteral msg(pos, "Failed: ${text} ==> the check was false.");
	fail.add(FnCall(pos, fail.scope, named{print<Str>}, Actuals(msg)));

	sub.addCatch(createCatch(pos, block, result, text));

	sub;
}

package Expr checkExpr(SrcPos pos, Block block, ExprText expr) on Compiler {
	LocalVar resultVar = findResult(block);
	LocalVarAccess result(pos, resultVar);

	TryBlock sub(pos, block);

	// Increment 'total'.
	sub.add(namedExpr(sub, pos, "++*", namedExpr(sub, pos, "total", result)));

	// Run the expression.
	sub.add(expr.expr);

	// If we got here, it did not throw.
	sub.addCatch(createCatch(pos, block, result, expr.text));

	sub;
}

package Expr checkExpr(SrcPos pos, Block block, ExprText expr, SrcName exception) on Compiler {
	LocalVar resultVar = findResult(block);
	LocalVarAccess result(pos, resultVar);

	TryBlock sub(pos, block);

	// Increment 'total'.
	sub.add(namedExpr(sub, pos, "++*", namedExpr(sub, pos, "total", result)));

	// Run the expression.
	sub.add(expr.expr);

	// If we got here, it did not throw.
	sub.add(namedExpr(sub, pos, "++*", namedExpr(sub, pos, "failed", result)));

	Str text = expr.text.trimWhitespace;
	StrLiteral msg(pos, "Failed: ${text} ==> did not throw ${exception} as expected.");
	sub.add(FnCall(pos, sub.scope, named{print<Str>}, Actuals(msg)));

	// Add catch handler for the expected case.
	{
		CatchBlock catch(pos, block, exception, null);
		catch.expr = Expr(pos);
		sub.addCatch(catch);
	}

	// Add catch handler for other exceptions.
	{
		CatchBlock catch(pos, block, named{Exception}, SStr("error"));
		ExprBlock handler(pos, catch);
		catch.expr = handler;
		sub.addCatch(catch);

		handler.add(namedExpr(sub, pos, "++*", namedExpr(handler, pos, "failed", result)));

		ConcatExpr msg(pos, block);
		msg.append(StrLiteral(pos, "Crashed: ${text} ==> did not get exception ${exception} as expected, got "));
		msg.append(namedExpr(handler, pos, "error", Actuals()));
		handler.add(FnCall(pos, handler.scope, named{print<Str>}, Actuals(msg)));
	}

	// print(sub.toS);

	sub;
}

package Expr abortExpr(SrcPos pos, Block block) on Compiler {
	LocalVar resultVar = findResult(block);
	LocalVarAccess result(pos, resultVar);

	ExprBlock sub(pos, block);
	AssignOpInfo assignOp(SStr("=", pos), 0, false);
	sub.add(mkOperator(sub, namedExpr(sub, pos, "aborted", result), assignOp, BoolLiteral(pos, true)));

	sub.add(Return(pos, sub, result));

	sub;
}
