// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program shows a multitude of ways DOLFIN can
// crash or produce the wrong results using interpolation.

#include <dolfin.h>
#include "generated_code/Elements.h"

using namespace dolfin;

class Vector2D: public Function
{
public:
  Vector2D(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 3.0 + 0.1*x[0] + 0.01*x[1];
    values[1] = 5.0 + 0.1*x[0] + 0.01*x[1];
    values[0] = x[0] + x[1];
    values[1] = x[0] + x[1];
  }
};

int main()
{
  UnitSquare mesh(1, 1);

  // from a VectorElement("CG", "triangle", 1)
  Elements::CoefficientSpace_v1 V1(mesh);
  // from a VectorElement("CG", "triangle", 2)
  Elements::CoefficientSpace_v2 V2(mesh);
  

  // Ideally, I'd like to say "make f the interpolant of s" where f already has a function space attached:
  /*
  Vector2D s;
  Function f(V2);
  f.interpolate(s);
  */

  
  // But with the currently available signatures, I wanted to do this:
  /*
  Vector2D s; // No function space necessary
  Function f(V2);
  s.interpolate(f.vector(), V2);
  */
  // But this gives:
  /*
terminate called after throwing an instance of 'std::runtime_error'
  what():  *** Assertion (_function_space) [at dolfin/function/Function.cpp:134 in function_space()]
  */
  // For some reason there's an assertion for a function space in s


  // This works: (sqrt(2^2+2^2) = 2.83 in the corner)
  /*
  Vector2D s(V2);
  Function f(V2);
  s.interpolate(f.vector(), V2);
  */


  // This error should be detected, but it only produces wrong results:
  /*
  Vector2D s(V1);
  Function f(V1);
  s.interpolate(f.vector(), V2);
  */


  // This might be ok (not sure how to interpret s(V1)), but produces wrong results:
  /*
  Vector2D s(V1);
  Function f(V2);
  s.interpolate(f.vector(), V2);
  */


  // This should fail with an error message, but produces wrong results:
  /*
  Vector2D s(V2);
  Function f(V2);
  s.interpolate(f.vector(), V1);
  */

  File file("f.pvd");
  file << f;

  return 0;
}

