/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.utilities.io.buffer;

import java.io.PrintStream;
import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.Formatter;
import java.util.IdentityHashMap;
import java.util.Objects;
import java.util.Set;

public final class DumpUtility {
    private final PrintStream printStream;
    private final CharSequence linePrefix;
    private final Set<Object> dumpedSet = Collections.newSetFromMap(new IdentityHashMap());
    private static final CharsetEncoder ASCII_ENCODER = StandardCharsets.US_ASCII.newEncoder();

    public DumpUtility(PrintStream printStream) {
        this(printStream, "");
    }

    public DumpUtility(PrintStream printStream, CharSequence linePrefix) {
        this.printStream = Objects.requireNonNull(printStream, "printStream");
        this.linePrefix = Objects.requireNonNull(linePrefix, "linePrefix");
    }

    private boolean wasDumped(Object o) {
        if (this.dumpedSet.contains(o)) {
            this.printStream.format("%s    ==> %s%n", this.linePrefix, DumpUtility.getObjectId(o));
            return true;
        }
        this.dumpedSet.add(o);
        this.printStream.format("%s[%s]%n", this.linePrefix, DumpUtility.getObjectId(o));
        return false;
    }

    public static void dumpBuffer(IntBuffer buffer, PrintStream printStream) {
        new DumpUtility(printStream).dumpBuffer(buffer);
    }

    public static void dumpBuffer(LongBuffer buffer, PrintStream printStream) {
        new DumpUtility(printStream).dumpBuffer(buffer);
    }

    public static void dumpBuffer(ByteBuffer buffer, PrintStream printStream) {
        new DumpUtility(printStream).dumpBuffer(buffer);
    }

    public void dumpBuffer(IntBuffer buffer) {
        String byteBufferFieldName;
        Objects.requireNonNull(buffer, "buffer");
        if (this.wasDumped(buffer)) {
            return;
        }
        String bufferClassName = buffer.getClass().getName();
        if (bufferClassName.equals("java.nio.ByteBufferAsInfBufferB") || bufferClassName.equals("java.nio.ByteBufferAsIntBufferL") || bufferClassName.equals("java.nio.ByteBufferAsIntBufferRB") || bufferClassName.equals("java.nio.ByteBufferAsIntBufferRL")) {
            byteBufferFieldName = "bb";
        } else if (bufferClassName.equals("java.nio.DirectIntBufferU") || bufferClassName.equals("java.nio.DirectIntBufferS") || bufferClassName.equals("java.nio.DirectIntBufferRU") || bufferClassName.equals("java.nio.DirectIntBufferRS")) {
            byteBufferFieldName = "att";
        } else {
            throw new UnsupportedOperationException(String.format("LongBuffer type not supported: %s", bufferClassName));
        }
        this.dumpBuffer(this.getFieldValue(ByteBuffer.class, buffer, byteBufferFieldName));
    }

    public void dumpBuffer(LongBuffer buffer) {
        String byteBufferFieldName;
        Objects.requireNonNull(buffer, "buffer");
        if (this.wasDumped(buffer)) {
            return;
        }
        String bufferClassName = buffer.getClass().getName();
        if (bufferClassName.equals("java.nio.ByteBufferAsLongBufferB") || bufferClassName.equals("java.nio.ByteBufferAsLongBufferL") || bufferClassName.equals("java.nio.ByteBufferAsLongBufferRB") || bufferClassName.equals("java.nio.ByteBufferAsLongBufferRL")) {
            byteBufferFieldName = "bb";
        } else if (bufferClassName.equals("java.nio.DirectLongBufferU") || bufferClassName.equals("java.nio.DirectLongBufferS") || bufferClassName.equals("java.nio.DirectLongBufferRU") || bufferClassName.equals("java.nio.DirectLongBufferRS")) {
            byteBufferFieldName = "att";
        } else {
            throw new UnsupportedOperationException(String.format("LongBuffer type not supported: %s", bufferClassName));
        }
        this.dumpBuffer(this.getFieldValue(ByteBuffer.class, buffer, byteBufferFieldName));
    }

    public void dumpBuffer(ByteBuffer buffer) {
        Objects.requireNonNull(buffer, "buffer");
        if (this.wasDumped(buffer)) {
            return;
        }
        this.printStream.format("%s    ByteOrder=%s; capacity=%d (0x%<X); limit=%d (0x%<X); position=%d (0x%<X)", this.linePrefix, buffer.order(), buffer.capacity(), buffer.limit(), buffer.position());
        if (buffer.isDirect()) {
            try {
                long address = this.getFieldValue(Long.class, buffer, "address", true);
                this.printStream.format("; address=0x%X%n", address);
            }
            catch (FieldInaccessibleException e) {
                this.printStream.println();
            }
        } else {
            this.printStream.println();
        }
        ByteBuffer view = buffer.asReadOnlyBuffer();
        view.clear();
        this.dumpBufferInternal(view);
    }

    public void dump(byte[] bytes) {
        Objects.requireNonNull(bytes, "bytes");
        this.dumpBufferInternal(ByteBuffer.wrap(bytes));
    }

    private void dumpBufferInternal(ByteBuffer view) {
        int segmentSize = 32;
        int dumpFormatMax = 10 + 8 * (segmentSize / 4) + (segmentSize / 4 - 1) + (segmentSize / 16 - 1);
        int charFormatMax = segmentSize + (segmentSize / 16 - 1);
        StringBuilder dumpBuilder = new StringBuilder(128);
        Formatter dumpFormatter = new Formatter(dumpBuilder);
        StringBuilder charBuilder = new StringBuilder(128);
        byte[][] segments = new byte[2][segmentSize];
        int activeIndex = 0;
        boolean previousSegmentSame = false;
        while (view.hasRemaining()) {
            if (!previousSegmentSame) {
                this.flushDumpLine(this.printStream, dumpBuilder, charBuilder);
            }
            int offset = view.position();
            byte[] activeSegment = segments[activeIndex];
            int segmentLength = Math.min(activeSegment.length, view.remaining());
            view.get(activeSegment, 0, segmentLength);
            if (offset != 0) {
                if (view.remaining() != 0 && Arrays.equals(activeSegment, segments[1 - activeIndex])) {
                    if (previousSegmentSame) continue;
                    dumpFormatter.format("%08X", offset);
                    previousSegmentSame = true;
                    continue;
                }
                if (previousSegmentSame) {
                    dumpFormatter.format("-%08X  duplicates above", offset - 1);
                    dumpBuilder.append(new String(new char[dumpFormatMax - dumpBuilder.length()]).replace('\u0000', ' '));
                    charBuilder.append(new String(new char[charFormatMax]).replace('\u0000', ' '));
                    this.flushDumpLine(this.printStream, dumpBuilder, charBuilder);
                    previousSegmentSame = false;
                }
            }
            dumpFormatter.format("%08X  ", offset);
            for (int i = 0; i < segmentLength; ++i) {
                if (i != 0) {
                    DumpUtility.addGroupSpace(i, dumpBuilder, charBuilder);
                }
                byte b = activeSegment[i];
                dumpFormatter.format("%02X", b & 0xFF);
                charBuilder.append(ASCII_ENCODER.canEncode((char)b) && !Character.isISOControl(b) ? (char)b : (char)'.');
            }
            activeIndex = 1 - activeIndex;
        }
        int segmentOffset = view.position() % segmentSize;
        if (segmentOffset != 0) {
            for (int i = segmentOffset; i < segmentSize; ++i) {
                DumpUtility.addGroupSpace(i, dumpBuilder, charBuilder);
                dumpBuilder.append("  ");
                charBuilder.append(' ');
            }
        }
        this.flushDumpLine(this.printStream, dumpBuilder, charBuilder);
    }

    private static void addGroupSpace(int i, StringBuilder dumpBuilder, StringBuilder charBuilder) {
        if (i % 4 == 0) {
            dumpBuilder.append(' ');
        }
        if (i % 16 == 0) {
            dumpBuilder.append(' ');
            charBuilder.append(' ');
        }
    }

    private void flushDumpLine(PrintStream out, StringBuilder dumpBuilder, StringBuilder charBuilder) {
        if (dumpBuilder.length() != 0) {
            out.append(this.linePrefix).append(dumpBuilder).append("  ").append('*').append(charBuilder).append('*');
            out.println();
            dumpBuilder.setLength(0);
            charBuilder.setLength(0);
        }
    }

    private <V, T> V getFieldValue(Class<V> expectedType, T instance, String fieldName) {
        return this.getFieldValue(expectedType, instance, fieldName, false);
    }

    private <V, T> V getFieldValue(Class<V> expectedType, T instance, String fieldName, boolean quiet) {
        V fieldValue;
        try {
            Class<?> fieldHoldingClass = instance.getClass();
            NoSuchFieldException firstFault = null;
            Field fieldDef = null;
            do {
                try {
                    fieldDef = fieldHoldingClass.getDeclaredField(fieldName);
                }
                catch (NoSuchFieldException e) {
                    if (firstFault == null) {
                        firstFault = e;
                    }
                    if ((fieldHoldingClass = fieldHoldingClass.getSuperclass()) != null) continue;
                    throw firstFault;
                }
            } while (fieldDef == null);
            fieldDef.setAccessible(true);
            fieldValue = expectedType.cast(fieldDef.get(instance));
            if (!quiet) {
                if (fieldValue == null) {
                    throw new AssertionError((Object)String.format("%s.%s is null; expecting %s instance", instance.getClass().getSimpleName(), fieldName, expectedType.getSimpleName()));
                }
                this.printStream.format("%s    %s.%s -> %s%n", this.linePrefix, instance.getClass().getSimpleName(), fieldName, DumpUtility.getObjectId(fieldValue));
            }
        }
        catch (IllegalAccessException | NoSuchFieldException e) {
            throw new FieldInaccessibleException(String.format("Unable to get '%s' field from %s instance: %s", fieldName, instance.getClass().getName(), e), e);
        }
        return fieldValue;
    }

    private static String getObjectId(Object o) {
        if (o == null) {
            return "null@0";
        }
        return o.getClass().getName() + "@" + Integer.toHexString(System.identityHashCode(o));
    }

    public static final class FieldInaccessibleException
    extends UnsupportedOperationException {
        private static final long serialVersionUID = -2136579828792539023L;

        public FieldInaccessibleException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

