# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
import json
from dataclasses import dataclass, field
from typing import List

from dataclasses_json import config
from gravitino.exceptions.base import IllegalArgumentException

from .base_response import BaseResponse
from ..job.job_template_dto import JobTemplateDTO


@dataclass
class JobTemplateListResponse(BaseResponse):
    """Represents a response containing a list of job templates."""

    _job_templates: List[JobTemplateDTO] = field(
        metadata=config(field_name="jobTemplates")
    )

    def validate(self):
        """Validates the response data.

        Raises:
            IllegalArgumentException: If the job templates list is not set or contains invalid items.
        """
        super().validate()

        if self._job_templates is None:
            raise IllegalArgumentException("jobTemplates must not be None")

        for job_template in self._job_templates:
            if job_template is not None:
                job_template.validate()

    def job_templates(self) -> List[JobTemplateDTO]:
        """Returns the list of job templates from the response."""
        return self._job_templates

    @classmethod
    def from_json(
        cls, s: str, infer_missing: bool = False, **kwargs
    ) -> "JobTemplateListResponse":
        """Deserialize JSON string into a JobTemplateListResponse object."""

        data = json.loads(s)

        # Deserialize each job template using JobTemplateDTO.from_json
        job_templates = [
            JobTemplateDTO.from_json(
                json.dumps(item), infer_missing=infer_missing, **kwargs
            )
            for item in data.get("jobTemplates", [])
        ]

        return cls(_job_templates=job_templates, _code=data["code"])
