/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.util.concurrent.RateLimiter;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.UUID;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import org.apache.cassandra.exceptions.RequestFailureReason;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintDiagnostics;
import org.apache.cassandra.hints.HintMessage;
import org.apache.cassandra.hints.HintsReader;
import org.apache.cassandra.hints.InputPosition;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.metrics.HintsServiceMetrics;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.utils.MonotonicClock;
import org.apache.cassandra.utils.concurrent.Condition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDispatcher
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(HintsDispatcher.class);
    private final HintsReader reader;
    final UUID hostId;
    final InetAddressAndPort address;
    private final int messagingVersion;
    private final BooleanSupplier abortRequested;
    private InputPosition currentPagePosition = null;

    private HintsDispatcher(HintsReader reader, UUID hostId, InetAddressAndPort address, int messagingVersion, BooleanSupplier abortRequested) {
        this.reader = reader;
        this.hostId = hostId;
        this.address = address;
        this.messagingVersion = messagingVersion;
        this.abortRequested = abortRequested;
    }

    static HintsDispatcher create(File file, RateLimiter rateLimiter, InetAddressAndPort address, UUID hostId, BooleanSupplier abortRequested) {
        int messagingVersion = MessagingService.instance().versions.get(address);
        HintsDispatcher dispatcher = new HintsDispatcher(HintsReader.open(file, rateLimiter), hostId, address, messagingVersion, abortRequested);
        HintDiagnostics.dispatcherCreated(dispatcher);
        return dispatcher;
    }

    @Override
    public void close() {
        HintDiagnostics.dispatcherClosed(this);
        this.reader.close();
    }

    void seek(InputPosition position) {
        this.reader.seek(position);
    }

    boolean dispatch() {
        for (HintsReader.Page page : this.reader) {
            this.currentPagePosition = page.position;
            if (this.dispatch(page) == Action.CONTINUE) continue;
            return false;
        }
        return true;
    }

    InputPosition dispatchPosition() {
        return this.currentPagePosition;
    }

    private Action dispatch(HintsReader.Page page) {
        HintDiagnostics.dispatchPage(this);
        return this.sendHintsAndAwait(page);
    }

    private Action sendHintsAndAwait(HintsReader.Page page) {
        Action action;
        ArrayList<Callback> callbacks = new ArrayList<Callback>();
        Action action2 = action = this.reader.descriptor().messagingVersion() == this.messagingVersion ? this.sendHints(page.buffersIterator(), callbacks, this::sendEncodedHint) : this.sendHints(page.hintsIterator(), callbacks, this::sendHint);
        if (action == Action.ABORT) {
            return action;
        }
        long success = 0L;
        long failures = 0L;
        long timeouts = 0L;
        for (Callback cb : callbacks) {
            Callback.Outcome outcome = cb.await();
            if (outcome == Callback.Outcome.SUCCESS) {
                ++success;
                continue;
            }
            if (outcome == Callback.Outcome.FAILURE) {
                ++failures;
                continue;
            }
            if (outcome != Callback.Outcome.TIMEOUT) continue;
            ++timeouts;
        }
        this.updateMetrics(success, failures, timeouts);
        if (failures > 0L || timeouts > 0L) {
            HintDiagnostics.pageFailureResult(this, success, failures, timeouts);
            return Action.ABORT;
        }
        HintDiagnostics.pageSuccessResult(this, success, failures, timeouts);
        return Action.CONTINUE;
    }

    private void updateMetrics(long success, long failures, long timeouts) {
        HintsServiceMetrics.hintsSucceeded.mark(success);
        HintsServiceMetrics.hintsFailed.mark(failures);
        HintsServiceMetrics.hintsTimedOut.mark(timeouts);
    }

    private <T> Action sendHints(Iterator<T> hints, Collection<Callback> callbacks, Function<T, Callback> sendFunction) {
        while (hints.hasNext()) {
            if (this.abortRequested.getAsBoolean()) {
                HintDiagnostics.abortRequested(this);
                return Action.ABORT;
            }
            callbacks.add(sendFunction.apply(hints.next()));
        }
        return Action.CONTINUE;
    }

    private Callback sendHint(Hint hint) {
        Callback callback = new Callback(hint.creationTime);
        Message<HintMessage> message = Message.out(Verb.HINT_REQ, new HintMessage(this.hostId, hint));
        MessagingService.instance().sendWithCallback(message, this.address, callback);
        return callback;
    }

    private Callback sendEncodedHint(ByteBuffer hint) {
        HintMessage.Encoded message = new HintMessage.Encoded(this.hostId, hint, this.messagingVersion);
        Callback callback = new Callback(message.getHintCreationTime());
        MessagingService.instance().sendWithCallback(Message.out(Verb.HINT_REQ, message), this.address, callback);
        return callback;
    }

    static final class Callback
    implements RequestCallback {
        private final long start = MonotonicClock.Global.approxTime.now();
        private final Condition condition = Condition.newOneTimeCondition();
        private volatile Outcome outcome;
        private final long hintCreationNanoTime;

        private Callback(long hintCreationTimeMillisSinceEpoch) {
            this.hintCreationNanoTime = MonotonicClock.Global.approxTime.translate().fromMillisSinceEpoch(hintCreationTimeMillisSinceEpoch);
        }

        Outcome await() {
            boolean timedOut;
            try {
                timedOut = !this.condition.awaitUntil(Verb.HINT_REQ.expiresAtNanos(this.start));
            }
            catch (InterruptedException e) {
                logger.warn("Hint dispatch was interrupted", (Throwable)e);
                return Outcome.INTERRUPTED;
            }
            return timedOut ? Outcome.TIMEOUT : this.outcome;
        }

        @Override
        public boolean invokeOnFailure() {
            return true;
        }

        @Override
        public void onFailure(InetAddressAndPort from, RequestFailureReason failureReason) {
            this.outcome = Outcome.FAILURE;
            this.condition.signalAll();
        }

        public void onResponse(Message msg) {
            HintsServiceMetrics.updateDelayMetrics(msg.from(), MonotonicClock.Global.approxTime.now() - this.hintCreationNanoTime);
            this.outcome = Outcome.SUCCESS;
            this.condition.signalAll();
        }

        static enum Outcome {
            SUCCESS,
            TIMEOUT,
            FAILURE,
            INTERRUPTED;

        }
    }

    private static enum Action {
        CONTINUE,
        ABORT;

    }
}

