/*
 * Decompiled with CFR 0.152.
 */
package opennlp.tools.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import opennlp.tools.commons.Internal;
import opennlp.tools.models.ModelType;
import opennlp.tools.util.model.BaseModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DownloadUtil {
    private static final Logger logger = LoggerFactory.getLogger(DownloadUtil.class);
    private static final String BASE_URL = System.getProperty("OPENNLP_DOWNLOAD_BASE_URL", "https://dlcdn.apache.org/opennlp/");
    private static final String MODEL_URI_PATH = System.getProperty("OPENNLP_DOWNLOAD_MODEL_PATH", "models/ud-models-1.3/");
    private static final String OPENNLP_DOWNLOAD_HOME = "OPENNLP_DOWNLOAD_HOME";
    private static Map<String, Map<ModelType, String>> availableModels;

    static boolean existsModel(String language, ModelType modelType) throws IOException {
        Map<ModelType, String> modelsByLanguage = DownloadUtil.getAvailableModels().get(language);
        if (modelsByLanguage == null) {
            return false;
        }
        String url = modelsByLanguage.get(modelType);
        if (url != null) {
            boolean exists;
            String filename;
            Path homeDirectory = DownloadUtil.getDownloadHome();
            Path localFile = homeDirectory.resolve(filename = url.substring(url.lastIndexOf("/") + 1));
            if (Files.exists(localFile, new LinkOption[0])) {
                DownloadUtil.validateModel(new URL(url + ".sha512"), localFile);
                exists = true;
            } else {
                exists = false;
            }
            return exists;
        }
        return false;
    }

    public static <T extends BaseModel> T downloadModel(String language, ModelType modelType, Class<T> type) throws IOException {
        String url;
        if (DownloadUtil.getAvailableModels().containsKey(language) && (url = DownloadUtil.getAvailableModels().get(language).get(modelType)) != null) {
            return DownloadUtil.downloadModel(new URL(url), type);
        }
        throw new IOException("There is no model available: " + language + " " + modelType.getName());
    }

    public static <T extends BaseModel> T downloadModel(URL url, Class<T> type) throws IOException {
        String filename;
        Path localFile;
        Path homeDirectory = DownloadUtil.getDownloadHome();
        if (!Files.isDirectory(homeDirectory, new LinkOption[0])) {
            try {
                Files.createDirectories(homeDirectory, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        if (!Files.exists(localFile = homeDirectory.resolve(filename = url.toString().substring(url.toString().lastIndexOf("/") + 1)), new LinkOption[0])) {
            logger.debug("Downloading model to {}.", (Object)localFile);
            try (InputStream in = url.openStream();){
                Files.copy(in, localFile, StandardCopyOption.REPLACE_EXISTING);
            }
            DownloadUtil.validateModel(new URL(String.valueOf(url) + ".sha512"), localFile);
            logger.debug("Download complete.");
        } else {
            logger.debug("Model file '{}' already exists. Skipping download.", (Object)filename);
        }
        try {
            return (T)((BaseModel)type.getConstructor(Path.class).newInstance(localFile));
        }
        catch (Exception e) {
            throw new IOException("Could not initialize Model of type " + type.getTypeName(), e);
        }
    }

    public static Map<String, Map<ModelType, String>> getAvailableModels() {
        if (availableModels == null) {
            try {
                availableModels = new DownloadParser(new URL(BASE_URL + MODEL_URI_PATH)).getAvailableModels();
            }
            catch (MalformedURLException e) {
                throw new RuntimeException(e);
            }
        }
        return Collections.unmodifiableMap(availableModels);
    }

    private static void validateModel(URL sha512, Path downloadedModel) throws IOException {
        String expectedChecksum;
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(sha512.openStream()));){
            expectedChecksum = reader.readLine();
            if (expectedChecksum != null) {
                expectedChecksum = expectedChecksum.split("\\s")[0].trim();
            }
        }
        String actualChecksum = DownloadUtil.calculateSHA512(downloadedModel);
        if (!actualChecksum.equalsIgnoreCase(expectedChecksum)) {
            throw new IOException("SHA512 checksum validation failed for " + String.valueOf(downloadedModel.getFileName()) + ". Expected: " + expectedChecksum + ", but got: " + actualChecksum);
        }
    }

    private static String calculateSHA512(Path file) throws IOException {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-512");
            try (InputStream fis = Files.newInputStream(file, new OpenOption[0]);
                 DigestInputStream dis = new DigestInputStream(fis, digest);){
                byte[] buffer = new byte[4096];
                while (dis.read(buffer) != -1) {
                }
            }
            return DownloadUtil.byteArrayToHexString(digest.digest());
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException("SHA-512 algorithm not found", e);
        }
    }

    private static String byteArrayToHexString(byte[] bytes) {
        try (Formatter formatter = new Formatter();){
            for (byte b : bytes) {
                formatter.format("%02x", b);
            }
            String string = formatter.toString();
            return string;
        }
    }

    private static Path getDownloadHome() {
        return Paths.get(System.getProperty(OPENNLP_DOWNLOAD_HOME, System.getProperty("user.home")), new String[0]).resolve(".opennlp");
    }

    @Internal
    static class DownloadParser {
        private static final Pattern LINK_PATTERN = Pattern.compile("<a href=\\\"(.*?)\\\">(.*?)</a>", 34);
        private final URL indexUrl;

        DownloadParser(URL indexUrl) {
            Objects.requireNonNull(indexUrl);
            this.indexUrl = indexUrl;
        }

        Map<String, Map<ModelType, String>> getAvailableModels() {
            Matcher matcher = LINK_PATTERN.matcher(this.fetchPageIndex());
            ArrayList<String> links = new ArrayList<String>();
            while (matcher.find()) {
                links.add(matcher.group(1));
            }
            return this.toMap(links);
        }

        private Map<String, Map<ModelType, String>> toMap(List<String> links) {
            HashMap<String, Map<ModelType, String>> result = new HashMap<String, Map<ModelType, String>>();
            for (String link : links) {
                if (!link.endsWith(".bin")) continue;
                if (link.contains("de-ud")) {
                    this.addModel("de", link, result);
                    continue;
                }
                if (link.contains("en-ud")) {
                    this.addModel("en", link, result);
                    continue;
                }
                if (link.contains("it-ud")) {
                    this.addModel("it", link, result);
                    continue;
                }
                if (link.contains("nl-ud")) {
                    this.addModel("nl", link, result);
                    continue;
                }
                if (link.contains("fr-ud")) {
                    this.addModel("fr", link, result);
                    continue;
                }
                if (link.contains("af-ud")) {
                    this.addModel("af", link, result);
                    continue;
                }
                if (link.contains("bg-ud")) {
                    this.addModel("bg", link, result);
                    continue;
                }
                if (link.contains("ca-ud")) {
                    this.addModel("ca", link, result);
                    continue;
                }
                if (link.contains("cs-ud")) {
                    this.addModel("cs", link, result);
                    continue;
                }
                if (link.contains("hr-ud")) {
                    this.addModel("hr", link, result);
                    continue;
                }
                if (link.contains("da-ud")) {
                    this.addModel("da", link, result);
                    continue;
                }
                if (link.contains("el-ud")) {
                    this.addModel("el", link, result);
                    continue;
                }
                if (link.contains("es-ud")) {
                    this.addModel("es", link, result);
                    continue;
                }
                if (link.contains("et-ud")) {
                    this.addModel("et", link, result);
                    continue;
                }
                if (link.contains("eu-ud")) {
                    this.addModel("eu", link, result);
                    continue;
                }
                if (link.contains("fa-ud")) {
                    this.addModel("fa", link, result);
                    continue;
                }
                if (link.contains("fi-ud")) {
                    this.addModel("fi", link, result);
                    continue;
                }
                if (link.contains("ga-ud")) {
                    this.addModel("ga", link, result);
                    continue;
                }
                if (link.contains("hy-ud")) {
                    this.addModel("hy", link, result);
                    continue;
                }
                if (link.contains("id-ud")) {
                    this.addModel("id", link, result);
                    continue;
                }
                if (link.contains("is-ud")) {
                    this.addModel("is", link, result);
                    continue;
                }
                if (link.contains("ka-ud")) {
                    this.addModel("ka", link, result);
                    continue;
                }
                if (link.contains("kk-ud")) {
                    this.addModel("kk", link, result);
                    continue;
                }
                if (link.contains("ko-ud")) {
                    this.addModel("ko", link, result);
                    continue;
                }
                if (link.contains("lv-ud")) {
                    this.addModel("lv", link, result);
                    continue;
                }
                if (link.contains("no-ud")) {
                    this.addModel("no", link, result);
                    continue;
                }
                if (link.contains("pl-ud")) {
                    this.addModel("pl", link, result);
                    continue;
                }
                if (link.contains("pt-ud")) {
                    this.addModel("pt", link, result);
                    continue;
                }
                if (link.contains("ro-ud")) {
                    this.addModel("ro", link, result);
                    continue;
                }
                if (link.contains("ru-ud")) {
                    this.addModel("ru", link, result);
                    continue;
                }
                if (link.contains("sr-ud")) {
                    this.addModel("sr", link, result);
                    continue;
                }
                if (link.contains("sk-ud")) {
                    this.addModel("sk", link, result);
                    continue;
                }
                if (link.contains("sl-ud")) {
                    this.addModel("sl", link, result);
                    continue;
                }
                if (link.contains("sv-ud")) {
                    this.addModel("sv", link, result);
                    continue;
                }
                if (link.contains("tr-ud")) {
                    this.addModel("tr", link, result);
                    continue;
                }
                if (!link.contains("uk-ud")) continue;
                this.addModel("uk", link, result);
            }
            return result;
        }

        private void addModel(String locale, String link, Map<String, Map<ModelType, String>> result) {
            Map models = result.getOrDefault(locale, new HashMap());
            String url = String.valueOf(this.indexUrl.toString().endsWith("/") ? this.indexUrl : String.valueOf(this.indexUrl) + "/") + link;
            if (link.contains("sentence")) {
                models.put(ModelType.SENTENCE_DETECTOR, url);
            } else if (link.contains("tokens")) {
                models.put(ModelType.TOKENIZER, url);
            } else if (link.contains("lemma")) {
                models.put(ModelType.LEMMATIZER, url);
            } else if (link.contains("pos")) {
                models.put(ModelType.POS, url);
            }
            result.putIfAbsent(locale, models);
        }

        private String fetchPageIndex() {
            StringBuilder html = new StringBuilder();
            try (BufferedReader br = new BufferedReader(new InputStreamReader(this.indexUrl.openStream(), StandardCharsets.UTF_8));){
                String line;
                while ((line = br.readLine()) != null) {
                    html.append(line);
                }
            }
            catch (IOException e) {
                logger.error("Could not read page index from {}", (Object)this.indexUrl, (Object)e);
            }
            return html.toString();
        }
    }
}

