/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds;

import com.google.common.base.Preconditions;
import com.google.common.net.HostAndPort;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.ServiceException;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.TreeMap;
import java.util.UUID;
import javax.management.ObjectName;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.ConfigRedactor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.conf.ConfigurationException;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.ha.SCMNodeInfo;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.ipc.RpcException;
import org.apache.hadoop.ipc.RpcNoSuchMethodException;
import org.apache.hadoop.ipc.RpcNoSuchProtocolException;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.DNS;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.ozone.conf.OzoneServiceConfig;
import org.apache.hadoop.ozone.ha.ConfUtils;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.SizeInBytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Stable
public final class HddsUtils {
    private static final Logger LOG = LoggerFactory.getLogger(HddsUtils.class);
    public static final ByteString REDACTED = ByteString.copyFromUtf8((String)"<redacted>");
    private static final int ONE_MB = SizeInBytes.valueOf((String)"1m").getSizeInt();
    private static final int NO_PORT = -1;

    private HddsUtils() {
    }

    public static Collection<InetSocketAddress> getScmAddressForClients(ConfigurationSource conf) {
        if (HddsUtils.getScmServiceId(conf) != null) {
            List<SCMNodeInfo> scmNodeInfoList = SCMNodeInfo.buildNodeInfo(conf);
            HashSet<InetSocketAddress> scmAddressList = new HashSet<InetSocketAddress>(scmNodeInfoList.size());
            for (SCMNodeInfo scmNodeInfo : scmNodeInfoList) {
                if (scmNodeInfo.getScmClientAddress() == null) {
                    throw new ConfigurationException("Ozone scm client address is not set for SCM service-id " + scmNodeInfo.getServiceId() + "node-id" + scmNodeInfo.getNodeId());
                }
                scmAddressList.add(NetUtils.createSocketAddr((String)scmNodeInfo.getScmClientAddress()));
            }
            return scmAddressList;
        }
        String address = conf.getTrimmed("ozone.scm.client.address");
        int port = -1;
        if (address == null) {
            Collection scmAddresses = conf.getTrimmedStringCollection("ozone.scm.names");
            if (scmAddresses.isEmpty()) {
                throw new ConfigurationException("Ozone scm client address is not set. Configure one of these config ozone.scm.client.address, ozone.scm.names");
            }
            if (scmAddresses.size() > 1) {
                throw new ConfigurationException("For non-HA SCM ozone.scm.names should be set with single address");
            }
            address = (String)scmAddresses.iterator().next();
            port = conf.getInt("ozone.scm.client.port", 9860);
        } else {
            port = HddsUtils.getHostPort(address).orElse(conf.getInt("ozone.scm.client.port", 9860));
        }
        return Collections.singletonList(NetUtils.createSocketAddr((String)(HddsUtils.getHostName(address).get() + ":" + port)));
    }

    public static Optional<String> getHostNameFromConfigKeys(ConfigurationSource conf, String ... keys) {
        for (String key : keys) {
            String value = conf.getTrimmed(key);
            Optional<String> hostName = HddsUtils.getHostName(value);
            if (!hostName.isPresent()) continue;
            return hostName;
        }
        return Optional.empty();
    }

    public static Optional<String> getHostName(String value) {
        if (value == null || value.isEmpty()) {
            return Optional.empty();
        }
        String hostname = value.replaceAll("\\:[0-9]+$", "");
        if (hostname.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(hostname);
    }

    public static OptionalInt getHostPort(String value) {
        if (value == null || value.isEmpty()) {
            return OptionalInt.empty();
        }
        int port = HostAndPort.fromString((String)value).getPortOrDefault(-1);
        if (port == -1) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(port);
    }

    public static OptionalInt getNumberFromConfigKeys(ConfigurationSource conf, String ... keys) {
        for (String key : keys) {
            String value = conf.getTrimmed(key);
            if (value == null) continue;
            return OptionalInt.of(Integer.parseInt(value));
        }
        return OptionalInt.empty();
    }

    public static OptionalInt getPortNumberFromConfigKeys(ConfigurationSource conf, String ... keys) {
        for (String key : keys) {
            String value = conf.getTrimmed(key);
            OptionalInt hostPort = HddsUtils.getHostPort(value);
            if (!hostPort.isPresent()) continue;
            return hostPort;
        }
        return OptionalInt.empty();
    }

    public static Collection<InetSocketAddress> getSCMAddressForDatanodes(ConfigurationSource conf) {
        if (HddsUtils.getScmServiceId(conf) != null) {
            List<SCMNodeInfo> scmNodeInfoList = SCMNodeInfo.buildNodeInfo(conf);
            HashSet<InetSocketAddress> scmAddressList = new HashSet<InetSocketAddress>(scmNodeInfoList.size());
            for (SCMNodeInfo scmNodeInfo : scmNodeInfoList) {
                scmAddressList.add(NetUtils.createSocketAddr((String)scmNodeInfo.getScmDatanodeAddress()));
            }
            return scmAddressList;
        }
        Collection names = conf.getTrimmedStringCollection("ozone.scm.names");
        if (names.isEmpty()) {
            throw new IllegalArgumentException("ozone.scm.names need to be a set of valid DNS names or IP addresses. Empty address list found.");
        }
        HashSet<InetSocketAddress> addresses = new HashSet<InetSocketAddress>(names.size());
        for (String address : names) {
            Optional<String> hostname = HddsUtils.getHostName(address);
            if (!hostname.isPresent()) {
                throw new IllegalArgumentException("Invalid hostname for SCM: " + address);
            }
            int port = HddsUtils.getHostPort(address).orElse(conf.getInt("ozone.scm.datanode.port", 9861));
            InetSocketAddress addr = NetUtils.createSocketAddr((String)hostname.get(), (int)port);
            addresses.add(addr);
        }
        if (addresses.size() > 1) {
            LOG.warn("When SCM HA is configured, configure {} appended with serviceId and nodeId. {} is deprecated.", (Object)"ozone.scm.address", (Object)"ozone.scm.names");
        }
        return addresses;
    }

    public static InetSocketAddress getReconAddresses(ConfigurationSource conf) {
        String name = conf.get("ozone.recon.address");
        if (StringUtils.isEmpty((CharSequence)name)) {
            return null;
        }
        Optional<String> hostname = HddsUtils.getHostName(name);
        if (!hostname.isPresent()) {
            throw new IllegalArgumentException("Invalid hostname for Recon: " + name);
        }
        int port = HddsUtils.getHostPort(name).orElse(9891);
        return NetUtils.createSocketAddr((String)hostname.get(), (int)port);
    }

    public static String getHostName(ConfigurationSource conf) throws UnknownHostException {
        String name = conf.get("hdds.datanode.hostname");
        if (name == null) {
            String dnsInterface = conf.get("hadoop.security.dns.interface");
            String nameServer = conf.get("hadoop.security.dns.nameserver");
            boolean fallbackToHosts = false;
            if (dnsInterface == null) {
                dnsInterface = conf.get("hdds.datanode.dns.interface");
                dnsInterface = conf.get("hdds.datanode.dns.interface");
                nameServer = conf.get("hdds.datanode.dns.nameserver");
            } else {
                fallbackToHosts = true;
            }
            name = DNS.getDefaultHost((String)dnsInterface, (String)nameServer, (boolean)fallbackToHosts);
        }
        return name;
    }

    public static InetSocketAddress getDatanodeRpcAddress(ConfigurationSource conf) {
        String host = HddsUtils.getHostNameFromConfigKeys(conf, "hdds.datanode.client.bind.host").orElse("0.0.0.0");
        int port = HddsUtils.getPortNumberFromConfigKeys(conf, "hdds.datanode.client.address").orElse(conf.getInt("hdds.datanode.client.port", 19864));
        return NetUtils.createSocketAddr((String)(host + ":" + port));
    }

    public static boolean isReadOnly(ContainerProtos.ContainerCommandRequestProtoOrBuilder proto) {
        switch (proto.getCmdType()) {
            case ReadContainer: 
            case ReadChunk: 
            case ListBlock: 
            case GetBlock: 
            case GetSmallFile: 
            case ListContainer: 
            case ListChunk: 
            case GetCommittedBlockLength: {
                return true;
            }
            case CloseContainer: 
            case WriteChunk: 
            case UpdateContainer: 
            case CompactChunk: 
            case CreateContainer: 
            case DeleteChunk: 
            case DeleteContainer: 
            case DeleteBlock: 
            case PutBlock: 
            case PutSmallFile: 
            case StreamInit: 
            case StreamWrite: 
            case FinalizeBlock: {
                return false;
            }
            case Echo: {
                return proto.getEcho().hasReadOnly() && proto.getEcho().getReadOnly();
            }
        }
        return false;
    }

    public static boolean isOpenToWriteState(ContainerProtos.ContainerDataProto.State state) {
        return state == ContainerProtos.ContainerDataProto.State.OPEN || state == ContainerProtos.ContainerDataProto.State.RECOVERING;
    }

    public static boolean requireBlockToken(ContainerProtos.Type cmdType) {
        switch (cmdType) {
            case ReadChunk: 
            case GetBlock: 
            case GetSmallFile: 
            case GetCommittedBlockLength: 
            case WriteChunk: 
            case DeleteChunk: 
            case DeleteBlock: 
            case PutBlock: 
            case PutSmallFile: 
            case FinalizeBlock: {
                return true;
            }
        }
        return false;
    }

    public static boolean requireContainerToken(ContainerProtos.Type cmdType) {
        switch (cmdType) {
            case ReadContainer: 
            case ListBlock: 
            case CloseContainer: 
            case UpdateContainer: 
            case CreateContainer: 
            case DeleteContainer: {
                return true;
            }
        }
        return false;
    }

    public static BlockID getBlockID(ContainerProtos.ContainerCommandRequestProtoOrBuilder msg) {
        ContainerProtos.DatanodeBlockID blockID = null;
        switch (msg.getCmdType()) {
            case DeleteBlock: {
                if (!msg.hasDeleteBlock()) break;
                blockID = msg.getDeleteBlock().getBlockID();
                break;
            }
            case DeleteChunk: {
                if (!msg.hasDeleteChunk()) break;
                blockID = msg.getDeleteChunk().getBlockID();
                break;
            }
            case GetBlock: {
                if (!msg.hasGetBlock()) break;
                blockID = msg.getGetBlock().getBlockID();
                break;
            }
            case GetCommittedBlockLength: {
                if (!msg.hasGetCommittedBlockLength()) break;
                blockID = msg.getGetCommittedBlockLength().getBlockID();
                break;
            }
            case GetSmallFile: {
                if (!msg.hasGetSmallFile()) break;
                blockID = msg.getGetSmallFile().getBlock().getBlockID();
                break;
            }
            case ListChunk: {
                if (!msg.hasListChunk()) break;
                blockID = msg.getListChunk().getBlockID();
                break;
            }
            case PutBlock: {
                if (!msg.hasPutBlock()) break;
                blockID = msg.getPutBlock().getBlockData().getBlockID();
                break;
            }
            case PutSmallFile: {
                if (!msg.hasPutSmallFile()) break;
                blockID = msg.getPutSmallFile().getBlock().getBlockData().getBlockID();
                break;
            }
            case ReadChunk: {
                if (!msg.hasReadChunk()) break;
                blockID = msg.getReadChunk().getBlockID();
                break;
            }
            case WriteChunk: {
                if (!msg.hasWriteChunk()) break;
                blockID = msg.getWriteChunk().getBlockID();
                break;
            }
            case FinalizeBlock: {
                if (!msg.hasFinalizeBlock()) break;
                blockID = msg.getFinalizeBlock().getBlockID();
                break;
            }
        }
        return blockID != null ? BlockID.getFromProtobuf(blockID) : null;
    }

    public static ObjectName registerWithJmxProperties(String serviceName, String mBeanName, Map<String, String> jmxProperties, Object mBean) {
        try {
            Method registerMethod = MBeans.class.getMethod("register", String.class, String.class, Map.class, Object.class);
            return (ObjectName)registerMethod.invoke(null, serviceName, mBeanName, jmxProperties, mBean);
        }
        catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Registering MBean {} without additional properties {}", (Object)mBeanName, jmxProperties);
            }
            return MBeans.register((String)serviceName, (String)mBeanName, (Object)mBean);
        }
    }

    public static long getTime() {
        return System.currentTimeMillis();
    }

    public static void validatePath(Path path, Path ancestor) {
        Preconditions.checkNotNull((Object)path, (Object)"Path should not be null");
        Preconditions.checkNotNull((Object)ancestor, (Object)"Ancestor should not be null");
        Preconditions.checkArgument((boolean)path.normalize().startsWith(ancestor.normalize()), (String)"Path should be a descendant of %s", (Object)ancestor);
    }

    public static File createDir(String dirPath) {
        File dirFile = new File(dirPath);
        if (!dirFile.mkdirs() && !dirFile.exists()) {
            throw new IllegalArgumentException("Unable to create path: " + dirFile);
        }
        return dirFile;
    }

    public static String format(List<String> nodes) {
        StringBuilder sb = new StringBuilder();
        for (String node : nodes) {
            String[] x = node.split(":");
            sb.append(String.format("{ HostName : %s, Ratis Port : %s, Role : %s } ", x[0], x[1], x[2]));
        }
        return sb.toString();
    }

    public static long getShutDownTimeOut(ConfigurationSource conf) {
        return ((OzoneServiceConfig)conf.getObject(OzoneServiceConfig.class)).getServiceShutdownTimeout();
    }

    public static int roundupMb(long bytes) {
        return (int)Math.ceil((double)bytes / (double)ONE_MB);
    }

    public static Throwable getUnwrappedException(Exception ex) {
        Throwable cause;
        Throwable t = ex;
        if (ex instanceof ServiceException) {
            t = ex.getCause();
        }
        if (t instanceof RemoteException) {
            t = ((RemoteException)t).unwrapRemoteException();
        }
        while (!(t == null || t instanceof RpcException || t instanceof AccessControlException || t instanceof SecretManager.InvalidToken || (cause = t.getCause()) == null || cause instanceof RemoteException)) {
            t = cause;
        }
        return t;
    }

    public static boolean shouldNotFailoverOnRpcException(Throwable exception) {
        if (exception instanceof RpcException) {
            if (exception instanceof RpcNoSuchMethodException || exception instanceof RpcNoSuchProtocolException || exception instanceof RPC.VersionMismatch) {
                return true;
            }
            if (exception.getMessage().contains("RPC response exceeds maximum data length") || exception.getMessage().contains("RPC response has invalid length")) {
                return true;
            }
        }
        return exception instanceof InvalidProtocolBufferException;
    }

    public static ContainerProtos.ContainerCommandRequestProto processForDebug(ContainerProtos.ContainerCommandRequestProto msg) {
        if (msg == null) {
            return null;
        }
        if (msg.hasWriteChunk() || msg.hasPutSmallFile()) {
            ContainerProtos.ContainerCommandRequestProto.Builder builder = msg.toBuilder();
            if (msg.hasWriteChunk()) {
                builder.getWriteChunkBuilder().setData(REDACTED);
            }
            if (msg.hasPutSmallFile()) {
                builder.getPutSmallFileBuilder().setData(REDACTED);
            }
            return builder.build();
        }
        return msg;
    }

    public static ContainerProtos.ContainerCommandResponseProto processForDebug(ContainerProtos.ContainerCommandResponseProto msg) {
        if (msg == null) {
            return null;
        }
        if (msg.hasReadChunk() || msg.hasGetSmallFile()) {
            ContainerProtos.ContainerCommandResponseProto.Builder builder = msg.toBuilder();
            if (msg.hasReadChunk()) {
                if (msg.getReadChunk().hasData()) {
                    builder.getReadChunkBuilder().setData(REDACTED);
                }
                if (msg.getReadChunk().hasDataBuffers()) {
                    builder.getReadChunkBuilder().getDataBuffersBuilder().clearBuffers().addBuffers(REDACTED);
                }
            }
            if (msg.hasGetSmallFile()) {
                if (msg.getGetSmallFile().getData().hasData()) {
                    builder.getGetSmallFileBuilder().getDataBuilder().setData(REDACTED);
                }
                if (msg.getGetSmallFile().getData().hasDataBuffers()) {
                    builder.getGetSmallFileBuilder().getDataBuilder().getDataBuffersBuilder().clearBuffers().addBuffers(REDACTED);
                }
            }
            return builder.build();
        }
        return msg;
    }

    public static Map<String, String> processForLogging(OzoneConfiguration conf) {
        Map<String, String> ozoneProps = conf.getOzoneProperties();
        ConfigRedactor redactor = new ConfigRedactor((Configuration)conf);
        TreeMap<String, String> sortedOzoneProps = new TreeMap<String, String>();
        for (Map.Entry<String, String> entry : ozoneProps.entrySet()) {
            String value = redactor.redact(entry.getKey(), entry.getValue());
            if (value != null) {
                value = value.trim();
            }
            sortedOzoneProps.put(entry.getKey(), value);
        }
        return sortedOzoneProps;
    }

    @Nonnull
    public static String threadNamePrefix(@Nullable Object id) {
        return id != null && !"".equals(id) ? id + "-" : "";
    }

    public static UUID fromProtobuf(HddsProtos.UUID uuid) {
        Objects.requireNonNull(uuid, "HddsProtos.UUID can't be null to transform to java UUID.");
        return new UUID(uuid.getMostSigBits(), uuid.getLeastSigBits());
    }

    public static HddsProtos.UUID toProtobuf(UUID uuid) {
        Objects.requireNonNull(uuid, "UUID can't be null to transform to protobuf UUID.");
        return HddsProtos.UUID.newBuilder().setMostSigBits(uuid.getMostSignificantBits()).setLeastSigBits(uuid.getLeastSignificantBits()).build();
    }

    @Nonnull
    public static String formatStackTrace(@Nullable StackTraceElement[] elements, int startIndex) {
        if (elements != null && elements.length > startIndex) {
            StringBuilder sb = new StringBuilder();
            for (int line = startIndex; line < elements.length; ++line) {
                sb.append(elements[line]).append("\n");
            }
            return sb.toString();
        }
        return "";
    }

    @Nullable
    public static StackTraceElement[] getStackTrace(@Nonnull Logger logger) {
        return logger.isDebugEnabled() ? Thread.currentThread().getStackTrace() : null;
    }

    public static void reportLeak(Class<?> clazz, String stackTrace, Logger log) {
        String warning = String.format("%s is not closed properly", clazz.getSimpleName());
        if (stackTrace != null && log.isDebugEnabled()) {
            String debugMessage = String.format("%nStackTrace for unclosed instance: %s", stackTrace);
            warning = warning.concat(debugMessage);
        }
        log.warn(warning);
    }

    public static String getScmServiceId(ConfigurationSource conf) {
        String localScmServiceId = conf.getTrimmed("ozone.scm.default.service.id");
        if (localScmServiceId == null) {
            Collection scmServiceIds = conf.getTrimmedStringCollection("ozone.scm.service.ids");
            if (scmServiceIds.size() > 1) {
                throw new ConfigurationException("When multiple SCM Service Ids are configured,ozone.scm.default.service.id need to be defined");
            }
            if (scmServiceIds.size() == 1) {
                localScmServiceId = (String)scmServiceIds.iterator().next();
            }
        }
        return localScmServiceId;
    }

    public static Collection<String> getSCMNodeIds(ConfigurationSource conf, String scmServiceId) {
        String key = ConfUtils.addSuffix("ozone.scm.nodes", scmServiceId);
        return conf.getTrimmedStringCollection(key);
    }

    public static Collection<String> getSCMNodeIds(ConfigurationSource configuration) {
        String scmServiceId = HddsUtils.getScmServiceId(configuration);
        return HddsUtils.getSCMNodeIds(configuration, scmServiceId);
    }
}

