/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.symmetric;

import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.SequenceWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.crypto.spec.SecretKeySpec;
import org.apache.hadoop.hdds.security.symmetric.ManagedSecretKey;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalSecretKeyStore
implements SecretKeyStore {
    private static final Set<PosixFilePermission> SECRET_KEYS_PERMISSIONS = Sets.newHashSet((Object[])new PosixFilePermission[]{PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE});
    private static final Logger LOG = LoggerFactory.getLogger(LocalSecretKeyStore.class);
    private final Path secretKeysFile;
    private final ObjectMapper mapper;

    public LocalSecretKeyStore(Path secretKeysFile) {
        this.secretKeysFile = Objects.requireNonNull(secretKeysFile);
        this.mapper = new ObjectMapper().registerModule((Module)new JavaTimeModule()).configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public synchronized List<ManagedSecretKey> load() {
        if (!this.secretKeysFile.toFile().exists()) {
            return Collections.emptyList();
        }
        ObjectReader reader = this.mapper.readerFor(ManagedSecretKeyDto.class);
        try (MappingIterator iterator = reader.readValues(this.secretKeysFile.toFile());){
            List dtos = iterator.readAll();
            List<ManagedSecretKey> result = dtos.stream().map(ManagedSecretKeyDto::toObject).collect(Collectors.toList());
            LOG.info("Loaded {} from {}", result, (Object)this.secretKeysFile);
            List<ManagedSecretKey> list = result;
            return list;
        }
        catch (IOException e) {
            throw new IllegalStateException("Error reading SecretKeys from " + this.secretKeysFile, e);
        }
    }

    @Override
    public synchronized void save(Collection<ManagedSecretKey> secretKeys) {
        this.createSecretKeyFiles();
        List dtos = secretKeys.stream().map(ManagedSecretKeyDto::new).collect(Collectors.toList());
        try (SequenceWriter writer = this.mapper.writer().writeValues(this.secretKeysFile.toFile());){
            writer.init(true);
            writer.writeAll(dtos);
        }
        catch (IOException e) {
            throw new IllegalStateException("Error saving SecretKeys to file " + this.secretKeysFile, e);
        }
        LOG.info("Saved {} to file {}", secretKeys, (Object)this.secretKeysFile);
    }

    private void createSecretKeyFiles() {
        try {
            if (!Files.exists(this.secretKeysFile, new LinkOption[0])) {
                Path parent = this.secretKeysFile.getParent();
                if (parent != null && !Files.exists(parent, new LinkOption[0])) {
                    Files.createDirectories(parent, new FileAttribute[0]);
                }
                Files.createFile(this.secretKeysFile, new FileAttribute[0]);
            }
            Files.setPosixFilePermissions(this.secretKeysFile, SECRET_KEYS_PERMISSIONS);
        }
        catch (IOException e) {
            throw new IllegalStateException("Error setting secret keys file permission: " + this.secretKeysFile, e);
        }
    }

    private static class ManagedSecretKeyDto {
        private UUID id;
        private Instant creationTime;
        private Instant expiryTime;
        private String algorithm;
        private byte[] encoded;

        ManagedSecretKeyDto() {
        }

        ManagedSecretKeyDto(ManagedSecretKey object) {
            this.id = object.getId();
            this.creationTime = object.getCreationTime();
            this.expiryTime = object.getExpiryTime();
            this.algorithm = object.getSecretKey().getAlgorithm();
            this.encoded = object.getSecretKey().getEncoded();
        }

        public ManagedSecretKey toObject() {
            SecretKeySpec secretKey = new SecretKeySpec(this.encoded, this.algorithm);
            return new ManagedSecretKey(this.id, this.creationTime, this.expiryTime, secretKey);
        }

        public UUID getId() {
            return this.id;
        }

        public void setId(UUID id) {
            this.id = id;
        }

        public Instant getCreationTime() {
            return this.creationTime;
        }

        public void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public Instant getExpiryTime() {
            return this.expiryTime;
        }

        public void setExpiryTime(Instant expiryTime) {
            this.expiryTime = expiryTime;
        }

        public String getAlgorithm() {
            return this.algorithm;
        }

        public void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        public byte[] getEncoded() {
            return this.encoded;
        }

        public void setEncoded(byte[] encoded) {
            this.encoded = encoded;
        }
    }
}

