/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.util.sketch;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Random;
import org.apache.spark.util.sketch.CountMinSketch;
import org.apache.spark.util.sketch.IncompatibleMergeException;
import org.apache.spark.util.sketch.Murmur3_x86_32;
import org.apache.spark.util.sketch.Platform;
import org.apache.spark.util.sketch.Utils;

class CountMinSketchImpl
extends CountMinSketch
implements Serializable {
    private static final long PRIME_MODULUS = Integer.MAX_VALUE;
    private int depth;
    private int width;
    private long[][] table;
    private long[] hashA;
    private long totalCount;
    private double eps;
    private double confidence;

    private CountMinSketchImpl() {
    }

    CountMinSketchImpl(int depth, int width, int seed) {
        if (depth <= 0 || width <= 0) {
            throw new IllegalArgumentException("Depth and width must be both positive");
        }
        this.depth = depth;
        this.width = width;
        this.eps = 2.0 / (double)width;
        this.confidence = 1.0 - 1.0 / Math.pow(2.0, depth);
        this.initTablesWith(depth, width, seed);
    }

    CountMinSketchImpl(double eps, double confidence, int seed) {
        if (eps <= 0.0) {
            throw new IllegalArgumentException("Relative error must be positive");
        }
        if (confidence <= 0.0 || confidence >= 1.0) {
            throw new IllegalArgumentException("Confidence must be within range (0.0, 1.0)");
        }
        this.eps = eps;
        this.confidence = confidence;
        this.width = (int)Math.ceil(2.0 / eps);
        this.depth = (int)Math.ceil(-Math.log1p(-confidence) / Math.log(2.0));
        this.initTablesWith(this.depth, this.width, seed);
    }

    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if (!(other instanceof CountMinSketchImpl)) {
            return false;
        }
        CountMinSketchImpl that = (CountMinSketchImpl)other;
        return this.depth == that.depth && this.width == that.width && this.totalCount == that.totalCount && Arrays.equals(this.hashA, that.hashA) && Arrays.deepEquals((Object[])this.table, (Object[])that.table);
    }

    public int hashCode() {
        int hash = this.depth;
        hash = hash * 31 + this.width;
        hash = hash * 31 + (int)(this.totalCount ^ this.totalCount >>> 32);
        hash = hash * 31 + Arrays.hashCode(this.hashA);
        hash = hash * 31 + Arrays.deepHashCode((Object[])this.table);
        return hash;
    }

    private void initTablesWith(int depth, int width, int seed) {
        this.table = new long[depth][width];
        this.hashA = new long[depth];
        Random r = new Random(seed);
        for (int i = 0; i < depth; ++i) {
            this.hashA[i] = r.nextInt(Integer.MAX_VALUE);
        }
    }

    @Override
    public double relativeError() {
        return this.eps;
    }

    @Override
    public double confidence() {
        return this.confidence;
    }

    @Override
    public int depth() {
        return this.depth;
    }

    @Override
    public int width() {
        return this.width;
    }

    @Override
    public long totalCount() {
        return this.totalCount;
    }

    @Override
    public void add(Object item) {
        this.add(item, 1L);
    }

    @Override
    public void add(Object item, long count) {
        if (item instanceof String) {
            String str = (String)item;
            this.addString(str, count);
        } else if (item instanceof byte[]) {
            byte[] bytes = (byte[])item;
            this.addBinary(bytes, count);
        } else {
            this.addLong(Utils.integralToLong(item), count);
        }
    }

    @Override
    public void addString(String item) {
        this.addString(item, 1L);
    }

    @Override
    public void addString(String item, long count) {
        this.addBinary(Utils.getBytesFromUTF8String(item), count);
    }

    @Override
    public void addLong(long item) {
        this.addLong(item, 1L);
    }

    @Override
    public void addLong(long item, long count) {
        if (count < 0L) {
            throw new IllegalArgumentException("Negative increments not implemented");
        }
        for (int i = 0; i < this.depth; ++i) {
            long[] lArray = this.table[i];
            int n = this.hash(item, i);
            lArray[n] = lArray[n] + count;
        }
        this.totalCount += count;
    }

    @Override
    public void addBinary(byte[] item) {
        this.addBinary(item, 1L);
    }

    @Override
    public void addBinary(byte[] item, long count) {
        if (count < 0L) {
            throw new IllegalArgumentException("Negative increments not implemented");
        }
        int[] buckets = CountMinSketchImpl.getHashBuckets(item, this.depth, this.width);
        for (int i = 0; i < this.depth; ++i) {
            long[] lArray = this.table[i];
            int n = buckets[i];
            lArray[n] = lArray[n] + count;
        }
        this.totalCount += count;
    }

    private int hash(long item, int count) {
        long hash = this.hashA[count] * item;
        hash += hash >> 32;
        return (int)(hash &= Integer.MAX_VALUE) % this.width;
    }

    private static int[] getHashBuckets(String key, int hashCount, int max) {
        return CountMinSketchImpl.getHashBuckets(Utils.getBytesFromUTF8String(key), hashCount, max);
    }

    private static int[] getHashBuckets(byte[] b, int hashCount, int max) {
        int[] result = new int[hashCount];
        int hash1 = Murmur3_x86_32.hashUnsafeBytes(b, Platform.BYTE_ARRAY_OFFSET, b.length, 0);
        int hash2 = Murmur3_x86_32.hashUnsafeBytes(b, Platform.BYTE_ARRAY_OFFSET, b.length, hash1);
        for (int i = 0; i < hashCount; ++i) {
            result[i] = Math.abs((hash1 + i * hash2) % max);
        }
        return result;
    }

    @Override
    public long estimateCount(Object item) {
        if (item instanceof String) {
            String str = (String)item;
            return this.estimateCountForStringItem(str);
        }
        if (item instanceof byte[]) {
            byte[] bytes = (byte[])item;
            return this.estimateCountForBinaryItem(bytes);
        }
        return this.estimateCountForLongItem(Utils.integralToLong(item));
    }

    private long estimateCountForLongItem(long item) {
        long res = Long.MAX_VALUE;
        for (int i = 0; i < this.depth; ++i) {
            res = Math.min(res, this.table[i][this.hash(item, i)]);
        }
        return res;
    }

    private long estimateCountForStringItem(String item) {
        long res = Long.MAX_VALUE;
        int[] buckets = CountMinSketchImpl.getHashBuckets(item, this.depth, this.width);
        for (int i = 0; i < this.depth; ++i) {
            res = Math.min(res, this.table[i][buckets[i]]);
        }
        return res;
    }

    private long estimateCountForBinaryItem(byte[] item) {
        long res = Long.MAX_VALUE;
        int[] buckets = CountMinSketchImpl.getHashBuckets(item, this.depth, this.width);
        for (int i = 0; i < this.depth; ++i) {
            res = Math.min(res, this.table[i][buckets[i]]);
        }
        return res;
    }

    @Override
    public CountMinSketch mergeInPlace(CountMinSketch other) throws IncompatibleMergeException {
        if (other == null) {
            throw new IncompatibleMergeException("Cannot merge null estimator");
        }
        if (!(other instanceof CountMinSketchImpl)) {
            throw new IncompatibleMergeException("Cannot merge estimator of class " + other.getClass().getName());
        }
        CountMinSketchImpl that = (CountMinSketchImpl)other;
        if (this.depth != that.depth) {
            throw new IncompatibleMergeException("Cannot merge estimators of different depth");
        }
        if (this.width != that.width) {
            throw new IncompatibleMergeException("Cannot merge estimators of different width");
        }
        if (!Arrays.equals(this.hashA, that.hashA)) {
            throw new IncompatibleMergeException("Cannot merge estimators of different seed");
        }
        for (int i = 0; i < this.table.length; ++i) {
            for (int j = 0; j < this.table[i].length; ++j) {
                this.table[i][j] = this.table[i][j] + that.table[i][j];
            }
        }
        this.totalCount += that.totalCount;
        return this;
    }

    @Override
    public void writeTo(OutputStream out) throws IOException {
        int i;
        DataOutputStream dos = new DataOutputStream(out);
        dos.writeInt(CountMinSketch.Version.V1.getVersionNumber());
        dos.writeLong(this.totalCount);
        dos.writeInt(this.depth);
        dos.writeInt(this.width);
        for (i = 0; i < this.depth; ++i) {
            dos.writeLong(this.hashA[i]);
        }
        for (i = 0; i < this.depth; ++i) {
            for (int j = 0; j < this.width; ++j) {
                dos.writeLong(this.table[i][j]);
            }
        }
    }

    @Override
    public byte[] toByteArray() throws IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream();){
            this.writeTo(out);
            byte[] byArray = out.toByteArray();
            return byArray;
        }
    }

    public static CountMinSketchImpl readFrom(InputStream in) throws IOException {
        CountMinSketchImpl sketch = new CountMinSketchImpl();
        sketch.readFrom0(in);
        return sketch;
    }

    private void readFrom0(InputStream in) throws IOException {
        int i;
        DataInputStream dis = new DataInputStream(in);
        int version = dis.readInt();
        if (version != CountMinSketch.Version.V1.getVersionNumber()) {
            throw new IOException("Unexpected Count-Min Sketch version number (" + version + ")");
        }
        this.totalCount = dis.readLong();
        this.depth = dis.readInt();
        this.width = dis.readInt();
        this.eps = 2.0 / (double)this.width;
        this.confidence = 1.0 - 1.0 / Math.pow(2.0, this.depth);
        this.hashA = new long[this.depth];
        for (i = 0; i < this.depth; ++i) {
            this.hashA[i] = dis.readLong();
        }
        this.table = new long[this.depth][this.width];
        for (i = 0; i < this.depth; ++i) {
            for (int j = 0; j < this.width; ++j) {
                this.table[i][j] = dis.readLong();
            }
        }
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        this.writeTo(out);
    }

    private void readObject(ObjectInputStream in) throws IOException {
        this.readFrom0(in);
    }
}

