/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.zeppelin.common.SessionInfo;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.interpreter.InterpreterSettingManager;
import org.apache.zeppelin.interpreter.ManagedInterpreterGroup;
import org.apache.zeppelin.interpreter.remote.RemoteInterpreterProcess;
import org.apache.zeppelin.notebook.Notebook;
import org.apache.zeppelin.scheduler.ExecutorFactory;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManagerService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionManagerService.class);
    private static final int RETRY = 3;
    private final Map<String, SessionInfo> sessions = new ConcurrentHashMap<String, SessionInfo>();
    private final InterpreterSettingManager interpreterSettingManager;
    private final Notebook notebook;
    private final ScheduledExecutorService sessionCheckerExecutor;
    private final ZeppelinConfiguration zConf;

    public SessionManagerService(Notebook notebook, InterpreterSettingManager interpreterSettingManager, ZeppelinConfiguration zConf) {
        this.notebook = notebook;
        this.interpreterSettingManager = interpreterSettingManager;
        this.zConf = zConf;
        this.sessionCheckerExecutor = ExecutorFactory.singleton().createOrGetScheduled("Session-Checker-Executor", 1);
        int sessionCheckerInterval = zConf.getInt(ZeppelinConfiguration.ConfVars.ZEPPELIN_SESSION_CHECK_INTERVAL);
        this.sessionCheckerExecutor.scheduleAtFixedRate(() -> {
            LOGGER.info("Start session check task");
            Iterator<Map.Entry<String, SessionInfo>> iter = this.sessions.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<String, SessionInfo> entry = iter.next();
                SessionInfo sessionInfo = null;
                try {
                    sessionInfo = this.getSessionInfo(entry.getKey());
                    if (sessionInfo == null || !sessionInfo.getState().equalsIgnoreCase("Stopped")) continue;
                    LOGGER.info("Session {} has been stopped, remove it and its associated note", (Object)entry.getKey());
                    try {
                        notebook.removeNote(sessionInfo.getNoteId(), AuthenticationInfo.ANONYMOUS);
                    }
                    catch (IOException e) {
                        LOGGER.warn("Fail to remove session note: " + sessionInfo.getNoteId(), (Throwable)e);
                    }
                    iter.remove();
                }
                catch (Exception e) {
                    LOGGER.warn("Fail to check session for session: " + entry.getKey(), (Throwable)e);
                }
            }
        }, sessionCheckerInterval, sessionCheckerInterval, TimeUnit.MILLISECONDS);
    }

    public synchronized SessionInfo createSession(String interpreter) throws Exception {
        String sessionId = null;
        int i = 0;
        while (i < 3 && this.sessions.containsKey(sessionId = interpreter + "_" + System.currentTimeMillis())) {
            try {
                Thread.sleep(1L);
            }
            catch (InterruptedException e) {
                LOGGER.error("Interrupted", (Throwable)e);
            }
        }
        if (sessionId == null) {
            throw new Exception("Unable to generate session id");
        }
        String sessionNoteId = this.notebook.createNote(this.buildNotePath(interpreter, sessionId), AuthenticationInfo.ANONYMOUS);
        SessionInfo sessionInfo = new SessionInfo(sessionId, sessionNoteId, interpreter);
        this.sessions.put(sessionId, sessionInfo);
        return sessionInfo;
    }

    private String buildNotePath(String interpreter, String sessionId) {
        return "/_ZSession/" + interpreter + "/" + sessionId;
    }

    public void stopSession(String sessionId) throws Exception {
        SessionInfo sessionInfo = this.sessions.remove(sessionId);
        if (sessionInfo == null) {
            throw new Exception("No such session: " + sessionId);
        }
        ManagedInterpreterGroup interpreterGroup = this.interpreterSettingManager.getInterpreterGroupById(sessionId);
        if (interpreterGroup == null) {
            LOGGER.info("No interpreterGroup for session: {}", (Object)sessionId);
            return;
        }
        interpreterGroup.getInterpreterSetting().closeInterpreters(sessionId);
        this.notebook.removeNote(sessionInfo.getNoteId(), AuthenticationInfo.ANONYMOUS);
    }

    public SessionInfo getSessionInfo(String sessionId) throws Exception {
        SessionInfo sessionInfo = this.sessions.get(sessionId);
        if (sessionInfo == null) {
            LOGGER.warn("No such session: {}", (Object)sessionId);
            return null;
        }
        ManagedInterpreterGroup interpreterGroup = this.interpreterSettingManager.getInterpreterGroupById(sessionId);
        if (interpreterGroup != null) {
            RemoteInterpreterProcess remoteInterpreterProcess = interpreterGroup.getRemoteInterpreterProcess();
            if (remoteInterpreterProcess == null) {
                sessionInfo.setState(SessionState.READY.name());
            } else {
                sessionInfo.setStartTime(remoteInterpreterProcess.getStartTime());
                sessionInfo.setWeburl(interpreterGroup.getWebUrl());
                if (remoteInterpreterProcess.isRunning()) {
                    sessionInfo.setState(SessionState.RUNNING.name());
                } else if (SessionState.RUNNING.name().equalsIgnoreCase(sessionInfo.getState())) {
                    sessionInfo.setState(SessionState.STOPPED.name());
                }
            }
        } else if (SessionState.RUNNING.name().equalsIgnoreCase(sessionInfo.getState())) {
            sessionInfo.setState(SessionState.STOPPED.name());
        }
        return sessionInfo;
    }

    public List<SessionInfo> listSessions() {
        ArrayList<SessionInfo> sessionList = new ArrayList<SessionInfo>();
        for (String sessionId : this.sessions.keySet()) {
            SessionInfo session = null;
            try {
                session = this.getSessionInfo(sessionId);
            }
            catch (Exception e) {
                LOGGER.warn("Fail to get sessionInfo for session: " + sessionId, (Throwable)e);
            }
            if (session == null) continue;
            sessionList.add(session);
        }
        return sessionList;
    }

    public List<SessionInfo> listSessions(String interpreter) {
        ArrayList<SessionInfo> sessionList = new ArrayList<SessionInfo>();
        for (String sessionId : this.sessions.keySet()) {
            SessionInfo status = null;
            try {
                status = this.getSessionInfo(sessionId);
            }
            catch (Exception e) {
                LOGGER.warn("Fail to get sessionInfo for session: " + sessionId, (Throwable)e);
            }
            if (status == null || !interpreter.equals(status.getInterpreter())) continue;
            sessionList.add(status);
        }
        return sessionList;
    }

    static enum SessionState {
        READY,
        RUNNING,
        STOPPED;

    }
}

