/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.ClientConfigurationFactory;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3EncryptionClient;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CryptoConfiguration;
import com.amazonaws.services.s3.model.EncryptionMaterialsProvider;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.KMSEncryptionMaterialsProvider;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.NoteParser;
import org.apache.zeppelin.notebook.repo.AbstractNotebookRepo;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3NotebookRepo
extends AbstractNotebookRepo {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3NotebookRepo.class);
    private AmazonS3 s3client;
    private String bucketName;
    private String user;
    private boolean useServerSideEncryption;
    private CannedAccessControlList objectCannedAcl;
    private String rootFolder;

    public void init(ZeppelinConfiguration zConf, NoteParser noteParser) throws IOException {
        super.init(zConf, noteParser);
        this.bucketName = zConf.getS3BucketName();
        this.user = zConf.getS3User();
        this.rootFolder = this.user + "/notebook";
        this.useServerSideEncryption = zConf.isS3ServerSideEncryption();
        if (StringUtils.isNotBlank((CharSequence)zConf.getS3CannedAcl())) {
            this.objectCannedAcl = CannedAccessControlList.valueOf((String)zConf.getS3CannedAcl());
        }
        DefaultAWSCredentialsProviderChain credentialsProvider = new DefaultAWSCredentialsProviderChain();
        CryptoConfiguration cryptoConf = new CryptoConfiguration();
        String keyRegion = zConf.getS3KMSKeyRegion();
        if (StringUtils.isNotBlank((CharSequence)keyRegion)) {
            cryptoConf.setAwsKmsRegion(Region.getRegion((Regions)Regions.fromName((String)keyRegion)));
        }
        ClientConfiguration cliConf = this.createClientConfiguration();
        String kmsKeyID = zConf.getS3KMSKeyID();
        if (kmsKeyID != null) {
            KMSEncryptionMaterialsProvider emp = new KMSEncryptionMaterialsProvider(kmsKeyID);
            this.s3client = new AmazonS3EncryptionClient((AWSCredentialsProvider)credentialsProvider, (EncryptionMaterialsProvider)emp, cliConf, cryptoConf);
        } else if (zConf.getS3EncryptionMaterialsProviderClass() != null) {
            EncryptionMaterialsProvider emp = this.createCustomProvider(zConf);
            this.s3client = new AmazonS3EncryptionClient((AWSCredentialsProvider)credentialsProvider, emp, cliConf, cryptoConf);
        } else {
            this.s3client = new AmazonS3Client((AWSCredentialsProvider)credentialsProvider, cliConf);
        }
        this.s3client.setS3ClientOptions(S3ClientOptions.builder().setPathStyleAccess(zConf.isS3PathStyleAccess()).build());
        this.s3client.setEndpoint(zConf.getS3Endpoint());
    }

    private EncryptionMaterialsProvider createCustomProvider(ZeppelinConfiguration zConf) throws IOException {
        EncryptionMaterialsProvider emp;
        String empClassname = zConf.getS3EncryptionMaterialsProviderClass();
        try {
            Object empInstance = Class.forName(empClassname).newInstance();
            if (!(empInstance instanceof EncryptionMaterialsProvider)) {
                throw new IOException("Class " + empClassname + " does not implement " + EncryptionMaterialsProvider.class.getName());
            }
            emp = (EncryptionMaterialsProvider)empInstance;
        }
        catch (Exception e) {
            throw new IOException("Unable to instantiate encryption materials provider class " + empClassname + ": " + e, e);
        }
        return emp;
    }

    private ClientConfiguration createClientConfiguration() {
        ClientConfigurationFactory configFactory = new ClientConfigurationFactory();
        ClientConfiguration config = configFactory.getConfig();
        String s3SignerOverride = this.zConf.getS3SignerOverride();
        if (StringUtils.isNotBlank((CharSequence)s3SignerOverride)) {
            config.setSignerOverride(s3SignerOverride);
        }
        return config;
    }

    public Map<String, NoteInfo> list(AuthenticationInfo subject) throws IOException {
        HashMap<String, NoteInfo> notesInfo = new HashMap<String, NoteInfo>();
        try {
            ObjectListing objectListing;
            ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(this.bucketName).withPrefix(this.user + "/notebook");
            do {
                objectListing = this.s3client.listObjects(listObjectsRequest);
                for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries()) {
                    if (!objectSummary.getKey().endsWith(".zpln")) continue;
                    try {
                        NoteInfo info = this.getNoteInfo(objectSummary.getKey());
                        notesInfo.put(info.getId(), info);
                    }
                    catch (IOException e) {
                        LOGGER.warn(e.getMessage());
                    }
                }
                listObjectsRequest.setMarker(objectListing.getNextMarker());
            } while (objectListing.isTruncated());
        }
        catch (AmazonClientException ace) {
            throw new IOException("Fail to list objects in S3", ace);
        }
        return notesInfo;
    }

    private NoteInfo getNoteInfo(String key) throws IOException {
        return new NoteInfo(this.getNoteId(key), this.getNotePath(this.rootFolder, key));
    }

    public Note get(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        S3Object s3object;
        try {
            s3object = this.s3client.getObject(new GetObjectRequest(this.bucketName, this.rootFolder + "/" + this.buildNoteFileName(noteId, notePath)));
        }
        catch (AmazonClientException ace) {
            throw new IOException("Fail to get note: " + notePath + " from S3", ace);
        }
        try (S3ObjectInputStream ins = s3object.getObjectContent();){
            String json = IOUtils.toString((InputStream)ins, (String)this.zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING));
            Note note = this.noteParser.fromJson(noteId, json);
            return note;
        }
    }

    public void save(Note note, AuthenticationInfo subject) throws IOException {
        String json = note.toJson();
        String key = this.rootFolder + "/" + this.buildNoteFileName(note);
        File file = File.createTempFile("note", "zpln");
        try {
            try (OutputStreamWriter writer = new OutputStreamWriter(new FileOutputStream(file));){
                writer.write(json);
            }
            PutObjectRequest putRequest = new PutObjectRequest(this.bucketName, key, file);
            if (this.useServerSideEncryption) {
                ObjectMetadata objectMetadata = new ObjectMetadata();
                objectMetadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
                putRequest.setMetadata(objectMetadata);
            }
            if (this.objectCannedAcl != null) {
                putRequest.withCannedAcl(this.objectCannedAcl);
            }
            this.s3client.putObject(putRequest);
        }
        catch (AmazonClientException ace) {
            throw new IOException("Fail to store note: " + note.getPath() + " in S3", ace);
        }
        finally {
            FileUtils.deleteQuietly((File)file);
        }
    }

    public void move(String noteId, String notePath, String newNotePath, AuthenticationInfo subject) throws IOException {
        String key = this.rootFolder + "/" + this.buildNoteFileName(noteId, notePath);
        String newKey = this.rootFolder + "/" + this.buildNoteFileName(noteId, newNotePath);
        this.s3client.copyObject(this.bucketName, key, this.bucketName, newKey);
        this.s3client.deleteObject(this.bucketName, key);
    }

    public void move(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        try {
            ObjectListing objectListing;
            ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(this.bucketName).withPrefix(this.rootFolder + folderPath + "/");
            do {
                objectListing = this.s3client.listObjects(listObjectsRequest);
                for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries()) {
                    if (!objectSummary.getKey().endsWith(".zpln")) continue;
                    String noteId = this.getNoteId(objectSummary.getKey());
                    String notePath = this.getNotePath(this.rootFolder, objectSummary.getKey());
                    String newNotePath = newFolderPath + notePath.substring(folderPath.length());
                    this.move(noteId, notePath, newNotePath, subject);
                }
                listObjectsRequest.setMarker(objectListing.getNextMarker());
            } while (objectListing.isTruncated());
        }
        catch (AmazonClientException ace) {
            throw new IOException("Fail to move folder: " + folderPath + " to " + newFolderPath + " in S3", ace);
        }
    }

    public void remove(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        try {
            this.s3client.deleteObject(this.bucketName, this.rootFolder + "/" + this.buildNoteFileName(noteId, notePath));
        }
        catch (AmazonClientException ace) {
            throw new IOException("Fail to remove note: " + notePath + " from S3", ace);
        }
    }

    public void remove(String folderPath, AuthenticationInfo subject) throws IOException {
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(this.bucketName).withPrefix(this.rootFolder + folderPath + "/");
        try {
            ObjectListing objects = this.s3client.listObjects(listObjectsRequest);
            do {
                for (S3ObjectSummary objectSummary : objects.getObjectSummaries()) {
                    this.s3client.deleteObject(this.bucketName, objectSummary.getKey());
                }
            } while ((objects = this.s3client.listNextBatchOfObjects(objects)).isTruncated());
        }
        catch (AmazonClientException ace) {
            throw new IOException("Unable to remove folder " + folderPath + " in S3", ace);
        }
    }

    public void close() {
        if (this.s3client != null) {
            this.s3client.shutdown();
        }
    }

    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        LOGGER.warn("Method not implemented");
        return Collections.emptyList();
    }

    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        LOGGER.warn("Method not implemented");
    }
}

