﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateTransitGatewayRouteRequest : public EC2Request {
 public:
  AWS_EC2_API CreateTransitGatewayRouteRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTransitGatewayRoute"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The CIDR range used for destination matches. Routing decisions are based on
   * the most specific match.</p>
   */
  inline const Aws::String& GetDestinationCidrBlock() const { return m_destinationCidrBlock; }
  inline bool DestinationCidrBlockHasBeenSet() const { return m_destinationCidrBlockHasBeenSet; }
  template <typename DestinationCidrBlockT = Aws::String>
  void SetDestinationCidrBlock(DestinationCidrBlockT&& value) {
    m_destinationCidrBlockHasBeenSet = true;
    m_destinationCidrBlock = std::forward<DestinationCidrBlockT>(value);
  }
  template <typename DestinationCidrBlockT = Aws::String>
  CreateTransitGatewayRouteRequest& WithDestinationCidrBlock(DestinationCidrBlockT&& value) {
    SetDestinationCidrBlock(std::forward<DestinationCidrBlockT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the transit gateway route table.</p>
   */
  inline const Aws::String& GetTransitGatewayRouteTableId() const { return m_transitGatewayRouteTableId; }
  inline bool TransitGatewayRouteTableIdHasBeenSet() const { return m_transitGatewayRouteTableIdHasBeenSet; }
  template <typename TransitGatewayRouteTableIdT = Aws::String>
  void SetTransitGatewayRouteTableId(TransitGatewayRouteTableIdT&& value) {
    m_transitGatewayRouteTableIdHasBeenSet = true;
    m_transitGatewayRouteTableId = std::forward<TransitGatewayRouteTableIdT>(value);
  }
  template <typename TransitGatewayRouteTableIdT = Aws::String>
  CreateTransitGatewayRouteRequest& WithTransitGatewayRouteTableId(TransitGatewayRouteTableIdT&& value) {
    SetTransitGatewayRouteTableId(std::forward<TransitGatewayRouteTableIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the attachment.</p>
   */
  inline const Aws::String& GetTransitGatewayAttachmentId() const { return m_transitGatewayAttachmentId; }
  inline bool TransitGatewayAttachmentIdHasBeenSet() const { return m_transitGatewayAttachmentIdHasBeenSet; }
  template <typename TransitGatewayAttachmentIdT = Aws::String>
  void SetTransitGatewayAttachmentId(TransitGatewayAttachmentIdT&& value) {
    m_transitGatewayAttachmentIdHasBeenSet = true;
    m_transitGatewayAttachmentId = std::forward<TransitGatewayAttachmentIdT>(value);
  }
  template <typename TransitGatewayAttachmentIdT = Aws::String>
  CreateTransitGatewayRouteRequest& WithTransitGatewayAttachmentId(TransitGatewayAttachmentIdT&& value) {
    SetTransitGatewayAttachmentId(std::forward<TransitGatewayAttachmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether to drop traffic that matches this route.</p>
   */
  inline bool GetBlackhole() const { return m_blackhole; }
  inline bool BlackholeHasBeenSet() const { return m_blackholeHasBeenSet; }
  inline void SetBlackhole(bool value) {
    m_blackholeHasBeenSet = true;
    m_blackhole = value;
  }
  inline CreateTransitGatewayRouteRequest& WithBlackhole(bool value) {
    SetBlackhole(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateTransitGatewayRouteRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_destinationCidrBlock;

  Aws::String m_transitGatewayRouteTableId;

  Aws::String m_transitGatewayAttachmentId;

  bool m_blackhole{false};

  bool m_dryRun{false};
  bool m_destinationCidrBlockHasBeenSet = false;
  bool m_transitGatewayRouteTableIdHasBeenSet = false;
  bool m_transitGatewayAttachmentIdHasBeenSet = false;
  bool m_blackholeHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
