#include "mqtt_select.h"
#include "esphome/core/log.h"

#include "mqtt_const.h"

#ifdef USE_MQTT
#ifdef USE_SELECT

namespace esphome::mqtt {

static const char *const TAG = "mqtt.select";

using namespace esphome::select;

MQTTSelectComponent::MQTTSelectComponent(Select *select) : select_(select) {}

void MQTTSelectComponent::setup() {
  this->subscribe(this->get_command_topic_(), [this](const std::string &topic, const std::string &state) {
    auto call = this->select_->make_call();
    call.set_option(state);
    call.perform();
  });
  this->select_->add_on_state_callback([this](size_t index) { this->publish_state(this->select_->option_at(index)); });
}

void MQTTSelectComponent::dump_config() {
  ESP_LOGCONFIG(TAG, "MQTT Select '%s':", this->select_->get_name().c_str());
  LOG_MQTT_COMPONENT(true, false);
}

MQTT_COMPONENT_TYPE(MQTTSelectComponent, "select")
const EntityBase *MQTTSelectComponent::get_entity() const { return this->select_; }

void MQTTSelectComponent::send_discovery(JsonObject root, mqtt::SendDiscoveryConfig &config) {
  const auto &traits = select_->traits;
  // https://www.home-assistant.io/integrations/select.mqtt/
  // NOLINTNEXTLINE(clang-analyzer-cplusplus.NewDeleteLeaks) false positive with ArduinoJson
  JsonArray options = root[MQTT_OPTIONS].to<JsonArray>();
  for (const auto &option : traits.get_options())
    options.add(option);

  config.command_topic = true;
}
bool MQTTSelectComponent::send_initial_state() {
  if (this->select_->has_state()) {
    auto option = this->select_->current_option();
    return this->publish_state(std::string(option.c_str(), option.size()));
  } else {
    return true;
  }
}
bool MQTTSelectComponent::publish_state(const std::string &value) {
  char topic_buf[MQTT_DEFAULT_TOPIC_MAX_LEN];
  return this->publish(this->get_state_topic_to_(topic_buf), value.data(), value.size());
}

}  // namespace esphome::mqtt

#endif
#endif  // USE_MQTT
