﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mq/MQRequest.h>
#include <aws/mq/MQ_EXPORTS.h>

#include <utility>

namespace Aws {
namespace MQ {
namespace Model {

/**
 * <p>Updates the specified configuration.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateConfigurationRequest">AWS
 * API Reference</a></p>
 */
class UpdateConfigurationRequest : public MQRequest {
 public:
  AWS_MQ_API UpdateConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateConfiguration"; }

  AWS_MQ_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique ID that Amazon MQ generates for the configuration.</p>
   */
  inline const Aws::String& GetConfigurationId() const { return m_configurationId; }
  inline bool ConfigurationIdHasBeenSet() const { return m_configurationIdHasBeenSet; }
  template <typename ConfigurationIdT = Aws::String>
  void SetConfigurationId(ConfigurationIdT&& value) {
    m_configurationIdHasBeenSet = true;
    m_configurationId = std::forward<ConfigurationIdT>(value);
  }
  template <typename ConfigurationIdT = Aws::String>
  UpdateConfigurationRequest& WithConfigurationId(ConfigurationIdT&& value) {
    SetConfigurationId(std::forward<ConfigurationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon MQ for Active MQ: The base64-encoded XML configuration. Amazon MQ for
   * RabbitMQ: the base64-encoded Cuttlefish configuration.</p>
   */
  inline const Aws::String& GetData() const { return m_data; }
  inline bool DataHasBeenSet() const { return m_dataHasBeenSet; }
  template <typename DataT = Aws::String>
  void SetData(DataT&& value) {
    m_dataHasBeenSet = true;
    m_data = std::forward<DataT>(value);
  }
  template <typename DataT = Aws::String>
  UpdateConfigurationRequest& WithData(DataT&& value) {
    SetData(std::forward<DataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateConfigurationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_configurationId;

  Aws::String m_data;

  Aws::String m_description;
  bool m_configurationIdHasBeenSet = false;
  bool m_dataHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace MQ
}  // namespace Aws
