# frozen_string_literal: true

require_relative 'table'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      module ClickHouse
        class CategoryOwnersTable < GitlabQuality::TestTooling::CodeCoverage::ClickHouse::Table
          TABLE_NAME = "category_owners"

          MissingMappingError = Class.new(StandardError)

          KNOWN_UNOWNED = %w[shared not_owned tooling].freeze

          # SQL query to get the latest ownership record for each unique category+ownership combination
          # Partitions by the full composite key to handle cases where a category has multiple ownerships
          LATEST_RECORDS_QUERY = <<~SQL
            SELECT category, group, stage, section
            FROM (
              SELECT category, group, stage, section,
                     ROW_NUMBER() OVER (PARTITION BY category, group, stage, section ORDER BY timestamp DESC) as rn
              FROM %{table_name}
            )
            WHERE rn = 1
          SQL

          # Insert only new category ownership records that don't already exist
          # This avoids needing TRUNCATE permission
          def push(data)
            return logger.warn("#{LOG_PREFIX} No data found, skipping insert!") if data.empty?

            sanitized_data = sanitize_and_filter_data(data)
            return if sanitized_data.empty?

            new_records = filter_new_records(sanitized_data)
            return if new_records.empty?

            insert_new_records(new_records, sanitized_data.size)
          rescue StandardError => e
            logger.error("#{LOG_PREFIX} Error occurred while pushing data to #{full_table_name}: #{e.message}")
            raise
          end

          # Owners of particular feature category as group, stage and section
          #
          # @param feature_category_name [String] the feature_category name
          # @return [Hash]
          def owners(feature_category_name)
            if KNOWN_UNOWNED.include?(feature_category_name)
              logger.info(
                "#{LOG_PREFIX} #{feature_category_name} is a known feature category without owner..."
              )
              return {}
            end

            records.fetch(feature_category_name)
          rescue KeyError
            raise(MissingMappingError, "Feature category '#{feature_category_name}' not found in table '#{table_name}'")
          end

          private

          def records
            @records ||= fetch_latest_records.each_with_object({}) do |record, hsh|
              hsh[record["category"]] = record.slice("group", "stage", "section")
            end
          end

          def sanitize_and_filter_data(data)
            logger.debug("#{LOG_PREFIX} Starting data export to ClickHouse")
            sanitized_data = sanitize(data)

            logger.warn("#{LOG_PREFIX} No valid data found after sanitization, skipping ClickHouse export!") if sanitized_data.empty?

            sanitized_data
          end

          def filter_new_records(sanitized_data)
            existing_records = fetch_existing_records
            # Deduplicate against latest records per category to prevent inserting duplicate historical records.
            # This ensures we only insert records with new category+ownership combinations, even if an older
            # version of the same category+ownership existed previously.
            new_records = sanitized_data.reject { |record| existing_records.include?(record_key(record)) }

            logger.info("#{LOG_PREFIX} No new records to insert, all data already exists") if new_records.empty?

            new_records
          end

          def insert_new_records(new_records, total_sanitized_count)
            client.insert_json_data(table_name, new_records)
            new_count = new_records.size
            existing_count = total_sanitized_count - new_count
            record_word = new_count == 1 ? 'record' : 'records'
            logger.info("#{LOG_PREFIX} Inserted #{new_count} new #{record_word} (#{existing_count} already existed)")
          end

          def fetch_existing_records
            fetch_latest_records.to_set { |record| record_key(record) }
          end

          def fetch_latest_records
            query = format(LATEST_RECORDS_QUERY, table_name: table_name)
            client.query(query)
          end

          def sanitized_data_record(record)
            {
              timestamp: time,
              category: record[:feature_category],
              group: record[:group],
              stage: record[:stage],
              section: record[:section]
            }
          end

          def record_key(record)
            # Create a unique key for the combination of category + ownership
            # Normalize to string keys for consistent access
            normalized = record.transform_keys(&:to_s)
            [
              normalized["category"],
              normalized["group"],
              normalized["stage"],
              normalized["section"]
            ]
          end

          # @return [Boolean] True if the record is valid, false otherwise
          def valid_record?(record)
            required_fields = %i[feature_category group stage section]

            required_fields.each do |field|
              if record[field].nil?
                logger.warn("#{LOG_PREFIX} Skipping record with nil #{field}: #{record}")
                return false
              end
            end

            true
          end
        end
      end
    end
  end
end
