"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;
var _uuid = require("uuid");
var _custom_panels = require("../../../common/constants/custom_panels");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class CustomPanelsAdaptor {
  constructor() {
    // index a panel
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });
    // update a panel
    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });
    // fetch a panel by id
    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });
    // gets list of panels stored in index
    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });
    // Create a new Panel
    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };
      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }
      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
    // Rename an existing panel
    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });
    // Clone an existing panel
    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });
    // Add filters to an existing panel
    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });
    // parses fetched saved visualization
    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        userConfigs: visualization.savedVisualization.hasOwnProperty('userConfigs') || visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs || visualization.savedVisualization.userConfigs) : {},
        subType: visualization.savedVisualization.hasOwnProperty('subType') ? visualization.savedVisualization.subType : '',
        metricType: visualization.savedVisualization.hasOwnProperty('metricType') ? visualization.savedVisualization.metricType : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });
    // gets all saved visualizations
    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });
    // gets list of savedVisualizations by Id
    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });
    // Get All Visualizations from a Panel
    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });
    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const xLeft = Math.max(bb1.x1, bb2.x1);
      const yTop = Math.max(bb1.y1, bb2.y1);
      const xRight = Math.min(bb1.x2, bb2.x2);
      const yBottom = Math.min(bb1.y2, bb2.y2);
      if (xRight < xLeft || yBottom < yTop) return 0;
      return (xRight - xLeft) * (yBottom - yTop);
    });
    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });
    // We want to check if the new visualization being added, can be placed at { x: 0, y: 0, w: 6, h: 4 };
    // To check this we try to calculate overlap between all the current visualizations and new visualization
    // if there is no overalap (i.e Total Overlap Area is 0), we place the new viz. in default position
    // else, we add it to the bottom of the panel
    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0;

      // check if we can place the new visualization at default location
      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      }

      // else place the new visualization at the bottom of the panel
      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });
    // Add Visualization in the  Panel
    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];
        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }
        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Add Multiple visualizations in a Panel
    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Edits all Visualizations in the Panel
    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];
        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({
                ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });
    // Create Sample Panels
    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _custom_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }
}
exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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