// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

#define I_KNOW_THE_INITIAL_QUARKS_PROJECTION_IS_DODGY_BUT_NEED_TO_USE_IT
#include "Rivet/Projections/InitialQuarks.hh"

namespace Rivet {


  /// @brief multiplicities in u, d, s events
  class OPAL_2001_I536266 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(OPAL_2001_I536266);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Projections
      declare(Beam(), "Beams");
      declare(ChargedFinalState(), "CFS");
      declare(InitialQuarks(), "IQF");
      book(_wDown   , "/TMP/WDOWN"   );
      book(_wUp     , "/TMP/WUP"     );
      book(_wStrange, "/TMP/WSTRANGE");
      book(_hUp, 1, 1, 1);
      book(_hDown, 1, 1, 2);
      book(_hStrange, 1, 1, 3);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      const FinalState& cfs = apply<FinalState>(event, "CFS");
      if (cfs.size() < 2) vetoEvent;


      int flavour = 0;
      const InitialQuarks& iqf = apply<InitialQuarks>(event, "IQF");

      // If we only have two quarks (qqbar), just take the flavour.
      // If we have more than two quarks, look for the highest energetic q-qbar pair.
      if (iqf.particles().size() == 2) {
        flavour = iqf.particles().front().abspid();
      }
      else {
        map<int, double> quarkmap;
        for (const Particle& p : iqf.particles()) {
          if (quarkmap[p.pid()] < p.E()) {
            quarkmap[p.pid()] = p.E();
          }
        }
        double maxenergy = 0.;
        for (int i = 1; i <= 5; ++i) {
          if (quarkmap[i]+quarkmap[-i] > maxenergy) {
            flavour = i;
          }
        }
      }
      const size_t numParticles = cfs.particles().size();
      switch (flavour) {
      case 1:
        _wDown->fill();
        _hDown->fill(Ecm, numParticles);
        break;
      case 2:
        _wUp->fill();
        _hUp->fill(Ecm, numParticles);
        break;
      case 3:
        _wStrange->fill();
        _hStrange->fill(Ecm, numParticles);
        break;
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // calculate the averages and ratios
      if (_wUp->effNumEntries() != 0.)  scale(_hUp, 1./ *_wUp);
      if (_wDown->effNumEntries() != 0.)  scale(_hDown, 1./ *_wDown);
      if (_wStrange->effNumEntries( )!= 0.)  scale(_hStrange, 1./ *_wStrange);

      BinnedEstimatePtr<string> ratioUD;
      book(ratioUD, 2, 1, 1);
      divide(_hUp, _hDown, ratioUD);

      BinnedEstimatePtr<string> ratioSD;
      book(ratioSD, 2, 1, 2);
      divide(_hStrange, _hDown, ratioSD);

      BinnedEstimatePtr<string> ratioSU;
      book(ratioSU, 2, 1, 3);
      divide(_hStrange, _hUp, ratioSU);
    }

    /// @}

    /// @name Member variables
    /// @{
    CounterPtr _wDown, _wUp, _wStrange;
    BinnedHistoPtr<string> _hDown, _hUp, _hStrange;
    const string Ecm = "45.6";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(OPAL_2001_I536266);

}
