//    SPDX-License-Identifier: LGPL-3.0-or-later
/*
   This file is part of FlexiBLAS, a BLAS/LAPACK interface wrapper library.
   Copyright (C) 2013-2025 Martin Koehler

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <complex.h>

#include "flexiblas_config.h"
#include "flexiblas.h"
#include "flexiblas_fortran_char_len.h"
#include "flexiblas_fortran_mangle.h"
#include "helper.h"

#if __GNUC__ >= 7
#define likely(x)      __builtin_expect(!!(x), 1)
#define unlikely(x)    __builtin_expect(!!(x), 0)
#else
#define likely(x) x
#define unlikely(x) x
#endif

#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_caxpby = 0;
#endif

/* Wrapper for caxpby */
void FC_GLOBAL(caxpby,CAXPBY)
    (blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cb, blas_complex_float *cy, blasint *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy);
    hook_pos_caxpby = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.caxpby;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->caxpby.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    } else {
        fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(caxpby,CAXPBY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cb, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(caxpby,CAXPBY)))));
void FC_GLOBAL3(caxpby,CAXPBY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cb, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(caxpby,CAXPBY)))));
#else
void FC_GLOBAL2(caxpby,CAXPBY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cb, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(caxpby,CAXPBY)((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
}
void FC_GLOBAL3(caxpby,CAXPBY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cb, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(caxpby,CAXPBY)((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_caxpby_(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy);
    *(void **) &fn = current_backend->blas.caxpby;
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_caxpby(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy) __attribute__((alias("flexiblas_real_caxpby_")));
#else
void flexiblas_real_caxpby(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy)
{
    flexiblas_real_caxpby_((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_caxpby_(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy);
    hook_pos_caxpby++;
    if ( hook_pos_caxpby < __flexiblas_hooks->caxpby.nhook ) {
        *(void **) &fn = __flexiblas_hooks->caxpby.hook_function[hook_pos_caxpby];
    } else {
        hook_pos_caxpby = 0;
        *(void **) &fn = current_backend->blas.caxpby;
    }
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_caxpby(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy) __attribute__((alias("flexiblas_chain_caxpby_")));
#else
void flexiblas_chain_caxpby(void *n, void *ca, void *cx, void *incx, void *cb, void *cy, void *incy)
{
    flexiblas_chain_caxpby_((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cb, (void *) cy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_caxpy = 0;
#endif

/* Wrapper for caxpy */
void FC_GLOBAL(caxpy,CAXPY)
    (blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *ca, void *cx, void *incx, void *cy, void *incy);
    hook_pos_caxpy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.caxpy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->caxpy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(caxpy,CAXPY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(caxpy,CAXPY)))));
void FC_GLOBAL3(caxpy,CAXPY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(caxpy,CAXPY)))));
#else
void FC_GLOBAL2(caxpy,CAXPY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(caxpy,CAXPY)((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
void FC_GLOBAL3(caxpy,CAXPY)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(caxpy,CAXPY)((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_caxpy_(void *n, void *ca, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cy, void *incy);
    *(void **) &fn = current_backend->blas.caxpy;
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_caxpy(void *n, void *ca, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_real_caxpy_")));
#else
void flexiblas_real_caxpy(void *n, void *ca, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_real_caxpy_((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_caxpy_(void *n, void *ca, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx, void *cy, void *incy);
    hook_pos_caxpy++;
    if ( hook_pos_caxpy < __flexiblas_hooks->caxpy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->caxpy.hook_function[hook_pos_caxpy];
    } else {
        hook_pos_caxpy = 0;
        *(void **) &fn = current_backend->blas.caxpy;
    }
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_caxpy(void *n, void *ca, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_chain_caxpy_")));
#else
void flexiblas_chain_caxpy(void *n, void *ca, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_chain_caxpy_((void *) n, (void *) ca, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ccopy = 0;
#endif

/* Wrapper for ccopy */
void FC_GLOBAL(ccopy,CCOPY)
    (blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_ccopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ccopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ccopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ccopy,CCOPY)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(ccopy,CCOPY)))));
void FC_GLOBAL3(ccopy,CCOPY)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(ccopy,CCOPY)))));
#else
void FC_GLOBAL2(ccopy,CCOPY)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(ccopy,CCOPY)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
void FC_GLOBAL3(ccopy,CCOPY)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(ccopy,CCOPY)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ccopy_(void *n, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    *(void **) &fn = current_backend->blas.ccopy;
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ccopy(void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_real_ccopy_")));
#else
void flexiblas_real_ccopy(void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_real_ccopy_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_ccopy_(void *n, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_ccopy++;
    if ( hook_pos_ccopy < __flexiblas_hooks->ccopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ccopy.hook_function[hook_pos_ccopy];
    } else {
        hook_pos_ccopy = 0;
        *(void **) &fn = current_backend->blas.ccopy;
    }
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ccopy(void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_chain_ccopy_")));
#else
void flexiblas_chain_ccopy(void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_chain_ccopy_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cdotc = 0;
#endif

/* Wrapper for cdotc */
blas_complex_float FC_GLOBAL(cdotc,CDOTC)
    (blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);

    #ifdef FLEXIBLAS_HOOK_API
    blas_complex_float (*fn_hook) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel_hook) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_cdotc = 0;
    #endif
    blas_complex_float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cdotc;
    *(void **) &fn_intel = current_backend->blas.cdotc;

    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cdotc.hook_function[0];
    *(void **) &fn_intel_hook = __flexiblas_hooks->cdotc.hook_function[0];

    if ( fn_hook != NULL) {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn_hook((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        } else {
            fn_intel_hook( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        }
    } else {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        } else {
            fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        }
    }
    #else
    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    #endif
    return ret;
}

#ifndef __APPLE__
blas_complex_float FC_GLOBAL2(cdotc,CDOTC)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cdotc,CDOTC)))));
blas_complex_float FC_GLOBAL3(cdotc,CDOTC)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cdotc,CDOTC)))));
#else
blas_complex_float FC_GLOBAL2(cdotc,CDOTC)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    return FC_GLOBAL(cdotc,CDOTC)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
blas_complex_float FC_GLOBAL3(cdotc,CDOTC)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    return FC_GLOBAL(cdotc,CDOTC)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cdotc_(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    blas_complex_float ret;
    *(void **) &fn = current_backend->blas.cdotc;
    *(void **) &fn_intel = current_backend->blas.cdotc;

    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    *((blas_complex_float *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_real_cdotc(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_real_cdotc_")));
#else
void flexiblas_real_cdotc(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_real_cdotc_(returnvalue, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif

void flexiblas_chain_cdotc_(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    blas_complex_float ret;
    hook_pos_cdotc++;
    if ( hook_pos_cdotc < __flexiblas_hooks->cdotc.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cdotc.hook_function[hook_pos_cdotc];
        *(void **) &fn_intel = __flexiblas_hooks->cdotc.hook_function[hook_pos_cdotc];
    } else {
        hook_pos_cdotc = 0;
        *(void **) &fn = current_backend->blas.cdotc;
        *(void **) &fn_intel = current_backend->blas.cdotc;
    }
    if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
            fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    *((blas_complex_float *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cdotc(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_chain_cdotc_")));
#else
void flexiblas_chain_cdotc(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_chain_cdotc_(returnvalue, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cdotu = 0;
#endif

/* Wrapper for cdotu */
blas_complex_float FC_GLOBAL(cdotu,CDOTU)
    (blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);

    #ifdef FLEXIBLAS_HOOK_API
    blas_complex_float (*fn_hook) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel_hook) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_cdotu = 0;
    #endif
    blas_complex_float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cdotu;
    *(void **) &fn_intel = current_backend->blas.cdotu;

    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cdotu.hook_function[0];
    *(void **) &fn_intel_hook = __flexiblas_hooks->cdotu.hook_function[0];

    if ( fn_hook != NULL) {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn_hook((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        } else {
            fn_intel_hook( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        }
    } else {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        } else {
            fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
        }
    }
    #else
    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    #endif
    return ret;
}

#ifndef __APPLE__
blas_complex_float FC_GLOBAL2(cdotu,CDOTU)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cdotu,CDOTU)))));
blas_complex_float FC_GLOBAL3(cdotu,CDOTU)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cdotu,CDOTU)))));
#else
blas_complex_float FC_GLOBAL2(cdotu,CDOTU)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    return FC_GLOBAL(cdotu,CDOTU)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
blas_complex_float FC_GLOBAL3(cdotu,CDOTU)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    return FC_GLOBAL(cdotu,CDOTU)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cdotu_(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    blas_complex_float ret;
    *(void **) &fn = current_backend->blas.cdotu;
    *(void **) &fn_intel = current_backend->blas.cdotu;

    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    *((blas_complex_float *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_real_cdotu(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_real_cdotu_")));
#else
void flexiblas_real_cdotu(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_real_cdotu_(returnvalue, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif

void flexiblas_chain_cdotu_(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    blas_complex_float (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    void (*fn_intel) (blas_complex_float *ret, void *n, void *cx, void *incx, void *cy, void *incy);
    blas_complex_float ret;
    hook_pos_cdotu++;
    if ( hook_pos_cdotu < __flexiblas_hooks->cdotu.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cdotu.hook_function[hook_pos_cdotu];
        *(void **) &fn_intel = __flexiblas_hooks->cdotu.hook_function[hook_pos_cdotu];
    } else {
        hook_pos_cdotu = 0;
        *(void **) &fn = current_backend->blas.cdotu;
        *(void **) &fn_intel = current_backend->blas.cdotu;
    }
    if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
            fn_intel( &ret, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    *((blas_complex_float *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cdotu(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_chain_cdotu_")));
#else
void flexiblas_chain_cdotu(blas_complex_float *returnvalue, void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_chain_cdotu_(returnvalue, (void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgbmv = 0;
#endif

/* Wrapper for cgbmv */
void FC_GLOBAL(cgbmv,CGBMV)
    (char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cgbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgbmv,CGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cgbmv,CGBMV)))));
void FC_GLOBAL3(cgbmv,CGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cgbmv,CGBMV)))));
#else
void FC_GLOBAL2(cgbmv,CGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cgbmv,CGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(cgbmv,CGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cgbmv,CGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.cgbmv;
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_cgbmv_")));
#else
void flexiblas_real_cgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_cgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_cgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cgbmv++;
    if ( hook_pos_cgbmv < __flexiblas_hooks->cgbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgbmv.hook_function[hook_pos_cgbmv];
    } else {
        hook_pos_cgbmv = 0;
        *(void **) &fn = current_backend->blas.cgbmv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_cgbmv_")));
#else
void flexiblas_chain_cgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_cgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgeadd = 0;
#endif

/* Wrapper for cgeadd */
void FC_GLOBAL(cgeadd,CGEADD)
    (blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *b, blasint *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_cgeadd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgeadd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgeadd.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgeadd,CGEADD)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(cgeadd,CGEADD)))));
void FC_GLOBAL3(cgeadd,CGEADD)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(cgeadd,CGEADD)))));
#else
void FC_GLOBAL2(cgeadd,CGEADD)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *b, blasint *ldb)
{
    FC_GLOBAL(cgeadd,CGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
void FC_GLOBAL3(cgeadd,CGEADD)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *b, blasint *ldb)
{
    FC_GLOBAL(cgeadd,CGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    *(void **) &fn = current_backend->blas.cgeadd;
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_real_cgeadd_")));
#else
void flexiblas_real_cgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_real_cgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif

void flexiblas_chain_cgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_cgeadd++;
    if ( hook_pos_cgeadd < __flexiblas_hooks->cgeadd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgeadd.hook_function[hook_pos_cgeadd];
    } else {
        hook_pos_cgeadd = 0;
        *(void **) &fn = current_backend->blas.cgeadd;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_chain_cgeadd_")));
#else
void flexiblas_chain_cgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_chain_cgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgemm = 0;
#endif

/* Wrapper for cgemm */
void FC_GLOBAL(cgemm,CGEMM)
    (char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_cgemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    } else {
        fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    }
    #else
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgemm,CGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemm,CGEMM)))));
void FC_GLOBAL3(cgemm,CGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemm,CGEMM)))));
#else
void FC_GLOBAL2(cgemm,CGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemm,CGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
void FC_GLOBAL3(cgemm,CGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemm,CGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.cgemm;
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_cgemm_")));
#else
void flexiblas_real_cgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_cgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_cgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_cgemm++;
    if ( hook_pos_cgemm < __flexiblas_hooks->cgemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgemm.hook_function[hook_pos_cgemm];
    } else {
        hook_pos_cgemm = 0;
        *(void **) &fn = current_backend->blas.cgemm;
    }
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_cgemm_")));
#else
void flexiblas_chain_cgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_cgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgemmtr = 0;
#endif

/* Wrapper for cgemmtr */
void FC_GLOBAL(cgemmtr,CGEMMTR)
    (char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_cgemmtr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgemmtr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgemmtr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    } else {
        fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    }
    #else
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgemmtr,CGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemmtr,CGEMMTR)))));
void FC_GLOBAL3(cgemmtr,CGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemmtr,CGEMMTR)))));
#else
void FC_GLOBAL2(cgemmtr,CGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemmtr,CGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(cgemmtr,CGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemmtr,CGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.cgemmtr;
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_cgemmtr_")));
#else
void flexiblas_real_cgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_cgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_cgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_cgemmtr++;
    if ( hook_pos_cgemmtr < __flexiblas_hooks->cgemmtr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgemmtr.hook_function[hook_pos_cgemmtr];
    } else {
        hook_pos_cgemmtr = 0;
        *(void **) &fn = current_backend->blas.cgemmtr;
    }
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_cgemmtr_")));
#else
void flexiblas_chain_cgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_cgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif
#endif
/* Alternative function name for cgemmtr -- cgemmt */
#ifndef __APPLE__
void FC_GLOBAL(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemmtr,CGEMMTR)))));
void FC_GLOBAL2(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemmtr,CGEMMTR)))));
void FC_GLOBAL3(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(cgemmtr,CGEMMTR)))));
#else
void FC_GLOBAL(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemmtr,CGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL2(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemmtr,CGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(cgemmt,CGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(cgemmtr,CGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif

#ifdef FLEXIBLAS_HOOK_API
#ifndef __APPLE__
void flexiblas_real_cgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_cgemmtr")));
void flexiblas_real_cgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_cgemmtr")));
void flexiblas_chain_cgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_cgemmtr")));
void flexiblas_chain_cgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_cgemmtr")));

#else
void flexiblas_real_cgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_cgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
void flexiblas_real_cgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_cgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_cgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_cgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_cgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_cgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgemv = 0;
#endif

/* Wrapper for cgemv */
void FC_GLOBAL(cgemv,CGEMV)
    (char *trans, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cgemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgemv,CGEMV)(char *trans, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cgemv,CGEMV)))));
void FC_GLOBAL3(cgemv,CGEMV)(char *trans, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cgemv,CGEMV)))));
#else
void FC_GLOBAL2(cgemv,CGEMV)(char *trans, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cgemv,CGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(cgemv,CGEMV)(char *trans, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cgemv,CGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.cgemv;
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_cgemv_")));
#else
void flexiblas_real_cgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_cgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_cgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cgemv++;
    if ( hook_pos_cgemv < __flexiblas_hooks->cgemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgemv.hook_function[hook_pos_cgemv];
    } else {
        hook_pos_cgemv = 0;
        *(void **) &fn = current_backend->blas.cgemv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_cgemv_")));
#else
void flexiblas_chain_cgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_cgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgerc = 0;
#endif

/* Wrapper for cgerc */
void FC_GLOBAL(cgerc,CGERC)
    (blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_cgerc = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgerc;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgerc.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgerc,CGERC)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(cgerc,CGERC)))));
void FC_GLOBAL3(cgerc,CGERC)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(cgerc,CGERC)))));
#else
void FC_GLOBAL2(cgerc,CGERC)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    FC_GLOBAL(cgerc,CGERC)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(cgerc,CGERC)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    FC_GLOBAL(cgerc,CGERC)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgerc_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.cgerc;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_cgerc_")));
#else
void flexiblas_real_cgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_cgerc_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_cgerc_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_cgerc++;
    if ( hook_pos_cgerc < __flexiblas_hooks->cgerc.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgerc.hook_function[hook_pos_cgerc];
    } else {
        hook_pos_cgerc = 0;
        *(void **) &fn = current_backend->blas.cgerc;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_cgerc_")));
#else
void flexiblas_chain_cgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_cgerc_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cgeru = 0;
#endif

/* Wrapper for cgeru */
void FC_GLOBAL(cgeru,CGERU)
    (blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_cgeru = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cgeru;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cgeru.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cgeru,CGERU)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(cgeru,CGERU)))));
void FC_GLOBAL3(cgeru,CGERU)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(cgeru,CGERU)))));
#else
void FC_GLOBAL2(cgeru,CGERU)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    FC_GLOBAL(cgeru,CGERU)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(cgeru,CGERU)(blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda)
{
    FC_GLOBAL(cgeru,CGERU)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cgeru_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.cgeru;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_cgeru_")));
#else
void flexiblas_real_cgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_cgeru_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_cgeru_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_cgeru++;
    if ( hook_pos_cgeru < __flexiblas_hooks->cgeru.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cgeru.hook_function[hook_pos_cgeru];
    } else {
        hook_pos_cgeru = 0;
        *(void **) &fn = current_backend->blas.cgeru;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_cgeru_")));
#else
void flexiblas_chain_cgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_cgeru_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chbmv = 0;
#endif

/* Wrapper for chbmv */
void FC_GLOBAL(chbmv,CHBMV)
    (char *uplo, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chbmv,CHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chbmv,CHBMV)))));
void FC_GLOBAL3(chbmv,CHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chbmv,CHBMV)))));
#else
void FC_GLOBAL2(chbmv,CHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chbmv,CHBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(chbmv,CHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chbmv,CHBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chbmv;
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chbmv_")));
#else
void flexiblas_real_chbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_chbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chbmv++;
    if ( hook_pos_chbmv < __flexiblas_hooks->chbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chbmv.hook_function[hook_pos_chbmv];
    } else {
        hook_pos_chbmv = 0;
        *(void **) &fn = current_backend->blas.chbmv;
    }
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chbmv_")));
#else
void flexiblas_chain_chbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chemm = 0;
#endif

/* Wrapper for chemm */
void FC_GLOBAL(chemm,CHEMM)
    (char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chemm,CHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chemm,CHEMM)))));
void FC_GLOBAL3(chemm,CHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chemm,CHEMM)))));
#else
void FC_GLOBAL2(chemm,CHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chemm,CHEMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(chemm,CHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chemm,CHEMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chemm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chemm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chemm_")));
#else
void flexiblas_real_chemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chemm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_chemm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chemm++;
    if ( hook_pos_chemm < __flexiblas_hooks->chemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chemm.hook_function[hook_pos_chemm];
    } else {
        hook_pos_chemm = 0;
        *(void **) &fn = current_backend->blas.chemm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chemm_")));
#else
void flexiblas_chain_chemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chemm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chemv = 0;
#endif

/* Wrapper for chemv */
void FC_GLOBAL(chemv,CHEMV)
    (char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chemv,CHEMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chemv,CHEMV)))));
void FC_GLOBAL3(chemv,CHEMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chemv,CHEMV)))));
#else
void FC_GLOBAL2(chemv,CHEMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chemv,CHEMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(chemv,CHEMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chemv,CHEMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chemv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chemv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chemv_")));
#else
void flexiblas_real_chemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chemv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_chemv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chemv++;
    if ( hook_pos_chemv < __flexiblas_hooks->chemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chemv.hook_function[hook_pos_chemv];
    } else {
        hook_pos_chemv = 0;
        *(void **) &fn = current_backend->blas.chemv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chemv_")));
#else
void flexiblas_chain_chemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chemv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cher = 0;
#endif

/* Wrapper for cher */
void FC_GLOBAL(cher,CHER)
    (char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_cher = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cher;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cher.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cher,CHER)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(cher,CHER)))));
void FC_GLOBAL3(cher,CHER)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(cher,CHER)))));
#else
void FC_GLOBAL2(cher,CHER)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(cher,CHER)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(cher,CHER)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(cher,CHER)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cher_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.cher;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_cher_")));
#else
void flexiblas_real_cher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_cher_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_cher_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_cher++;
    if ( hook_pos_cher < __flexiblas_hooks->cher.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cher.hook_function[hook_pos_cher];
    } else {
        hook_pos_cher = 0;
        *(void **) &fn = current_backend->blas.cher;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_cher_")));
#else
void flexiblas_chain_cher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_cher_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cher2 = 0;
#endif

/* Wrapper for cher2 */
void FC_GLOBAL(cher2,CHER2)
    (char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_cher2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cher2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cher2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cher2,CHER2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(cher2,CHER2)))));
void FC_GLOBAL3(cher2,CHER2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(cher2,CHER2)))));
#else
void FC_GLOBAL2(cher2,CHER2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(cher2,CHER2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(cher2,CHER2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(cher2,CHER2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cher2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.cher2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_cher2_")));
#else
void flexiblas_real_cher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_cher2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_cher2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_cher2++;
    if ( hook_pos_cher2 < __flexiblas_hooks->cher2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cher2.hook_function[hook_pos_cher2];
    } else {
        hook_pos_cher2 = 0;
        *(void **) &fn = current_backend->blas.cher2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_cher2_")));
#else
void flexiblas_chain_cher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_cher2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cher2k = 0;
#endif

/* Wrapper for cher2k */
void FC_GLOBAL(cher2k,CHER2K)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cher2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cher2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cher2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cher2k,CHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cher2k,CHER2K)))));
void FC_GLOBAL3(cher2k,CHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cher2k,CHER2K)))));
#else
void FC_GLOBAL2(cher2k,CHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cher2k,CHER2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(cher2k,CHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cher2k,CHER2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cher2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.cher2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_cher2k_")));
#else
void flexiblas_real_cher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_cher2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_cher2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cher2k++;
    if ( hook_pos_cher2k < __flexiblas_hooks->cher2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cher2k.hook_function[hook_pos_cher2k];
    } else {
        hook_pos_cher2k = 0;
        *(void **) &fn = current_backend->blas.cher2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_cher2k_")));
#else
void flexiblas_chain_cher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_cher2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cherk = 0;
#endif

/* Wrapper for cherk */
void FC_GLOBAL(cherk,CHERK)
    (char *uplo, char *trans, blasint *n, blasint *k, float *alpha, blas_complex_float *a, blasint *lda, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cherk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cherk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cherk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cherk,CHERK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, blas_complex_float *a, blasint *lda, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cherk,CHERK)))));
void FC_GLOBAL3(cherk,CHERK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, blas_complex_float *a, blasint *lda, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cherk,CHERK)))));
#else
void FC_GLOBAL2(cherk,CHERK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, blas_complex_float *a, blasint *lda, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cherk,CHERK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(cherk,CHERK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, blas_complex_float *a, blasint *lda, float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cherk,CHERK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cherk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.cherk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_cherk_")));
#else
void flexiblas_real_cherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_cherk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_cherk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cherk++;
    if ( hook_pos_cherk < __flexiblas_hooks->cherk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cherk.hook_function[hook_pos_cherk];
    } else {
        hook_pos_cherk = 0;
        *(void **) &fn = current_backend->blas.cherk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_cherk_")));
#else
void flexiblas_chain_cherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_cherk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chpmv = 0;
#endif

/* Wrapper for chpmv */
void FC_GLOBAL(chpmv,CHPMV)
    (char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *ap, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chpmv,CHPMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *ap, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpmv,CHPMV)))));
void FC_GLOBAL3(chpmv,CHPMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *ap, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpmv,CHPMV)))));
#else
void FC_GLOBAL2(chpmv,CHPMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *ap, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpmv,CHPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(chpmv,CHPMV)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *ap, blas_complex_float *x, blasint *incx, blas_complex_float *beta, blas_complex_float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpmv,CHPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chpmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chpmv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chpmv_")));
#else
void flexiblas_real_chpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chpmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_chpmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpmv++;
    if ( hook_pos_chpmv < __flexiblas_hooks->chpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chpmv.hook_function[hook_pos_chpmv];
    } else {
        hook_pos_chpmv = 0;
        *(void **) &fn = current_backend->blas.chpmv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chpmv_")));
#else
void flexiblas_chain_chpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chpmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chpr = 0;
#endif

/* Wrapper for chpr */
void FC_GLOBAL(chpr,CHPR)
    (char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chpr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chpr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chpr,CHPR)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpr,CHPR)))));
void FC_GLOBAL3(chpr,CHPR)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpr,CHPR)))));
#else
void FC_GLOBAL2(chpr,CHPR)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpr,CHPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
void FC_GLOBAL3(chpr,CHPR)(char *uplo, blasint *n, float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpr,CHPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chpr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chpr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chpr_")));
#else
void flexiblas_real_chpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chpr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_chpr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpr++;
    if ( hook_pos_chpr < __flexiblas_hooks->chpr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chpr.hook_function[hook_pos_chpr];
    } else {
        hook_pos_chpr = 0;
        *(void **) &fn = current_backend->blas.chpr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chpr_")));
#else
void flexiblas_chain_chpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chpr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_chpr2 = 0;
#endif

/* Wrapper for chpr2 */
void FC_GLOBAL(chpr2,CHPR2)
    (char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.chpr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->chpr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(chpr2,CHPR2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpr2,CHPR2)))));
void FC_GLOBAL3(chpr2,CHPR2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(chpr2,CHPR2)))));
#else
void FC_GLOBAL2(chpr2,CHPR2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpr2,CHPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
void FC_GLOBAL3(chpr2,CHPR2)(char *uplo, blasint *n, blas_complex_float *alpha, blas_complex_float *x, blasint *incx, blas_complex_float *y, blasint *incy, blas_complex_float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(chpr2,CHPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_chpr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.chpr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_chpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_chpr2_")));
#else
void flexiblas_real_chpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_chpr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_chpr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_chpr2++;
    if ( hook_pos_chpr2 < __flexiblas_hooks->chpr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->chpr2.hook_function[hook_pos_chpr2];
    } else {
        hook_pos_chpr2 = 0;
        *(void **) &fn = current_backend->blas.chpr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_chpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_chpr2_")));
#else
void flexiblas_chain_chpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_chpr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cimatcopy = 0;
#endif

/* Wrapper for cimatcopy */
void FC_GLOBAL(cimatcopy,CIMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cimatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cimatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cimatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cimatcopy,CIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cimatcopy,CIMATCOPY)))));
void FC_GLOBAL3(cimatcopy,CIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(cimatcopy,CIMATCOPY)))));
#else
void FC_GLOBAL2(cimatcopy,CIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cimatcopy,CIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
void FC_GLOBAL3(cimatcopy,CIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(cimatcopy,CIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.cimatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_cimatcopy_")));
#else
void flexiblas_real_cimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_cimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif

void flexiblas_chain_cimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_cimatcopy++;
    if ( hook_pos_cimatcopy < __flexiblas_hooks->cimatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cimatcopy.hook_function[hook_pos_cimatcopy];
    } else {
        hook_pos_cimatcopy = 0;
        *(void **) &fn = current_backend->blas.cimatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_cimatcopy_")));
#else
void flexiblas_chain_cimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_cimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_comatcopy = 0;
#endif

/* Wrapper for comatcopy */
void FC_GLOBAL(comatcopy,COMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_comatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.comatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->comatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(comatcopy,COMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(comatcopy,COMATCOPY)))));
void FC_GLOBAL3(comatcopy,COMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(comatcopy,COMATCOPY)))));
#else
void FC_GLOBAL2(comatcopy,COMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(comatcopy,COMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
void FC_GLOBAL3(comatcopy,COMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(comatcopy,COMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_comatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    *(void **) &fn = current_backend->blas.comatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_comatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_real_comatcopy_")));
#else
void flexiblas_real_comatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_real_comatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif

void flexiblas_chain_comatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_comatcopy++;
    if ( hook_pos_comatcopy < __flexiblas_hooks->comatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->comatcopy.hook_function[hook_pos_comatcopy];
    } else {
        hook_pos_comatcopy = 0;
        *(void **) &fn = current_backend->blas.comatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_comatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_chain_comatcopy_")));
#else
void flexiblas_chain_comatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_chain_comatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_crotg = 0;
#endif

/* Wrapper for crotg */
void FC_GLOBAL(crotg,CROTG)
    (blas_complex_float *a, blas_complex_float *b, float *c, blas_complex_float *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *a, void *b, void *c, void *s);
    hook_pos_crotg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.crotg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->crotg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) a, (void *) b, (void *) c, (void *) s);
    } else {
        fn((void *) a, (void *) b, (void *) c, (void *) s);
    }
    #else
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(crotg,CROTG)(blas_complex_float *a, blas_complex_float *b, float *c, blas_complex_float *s) __attribute__((alias(MTS(FC_GLOBAL(crotg,CROTG)))));
void FC_GLOBAL3(crotg,CROTG)(blas_complex_float *a, blas_complex_float *b, float *c, blas_complex_float *s) __attribute__((alias(MTS(FC_GLOBAL(crotg,CROTG)))));
#else
void FC_GLOBAL2(crotg,CROTG)(blas_complex_float *a, blas_complex_float *b, float *c, blas_complex_float *s)
{
    FC_GLOBAL(crotg,CROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
void FC_GLOBAL3(crotg,CROTG)(blas_complex_float *a, blas_complex_float *b, float *c, blas_complex_float *s)
{
    FC_GLOBAL(crotg,CROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_crotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    *(void **) &fn = current_backend->blas.crotg;
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_crotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_real_crotg_")));
#else
void flexiblas_real_crotg(void *a, void *b, void *c, void *s)
{
    flexiblas_real_crotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_crotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    hook_pos_crotg++;
    if ( hook_pos_crotg < __flexiblas_hooks->crotg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->crotg.hook_function[hook_pos_crotg];
    } else {
        hook_pos_crotg = 0;
        *(void **) &fn = current_backend->blas.crotg;
    }
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_crotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_chain_crotg_")));
#else
void flexiblas_chain_crotg(void *a, void *b, void *c, void *s)
{
    flexiblas_chain_crotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cscal = 0;
#endif

/* Wrapper for cscal */
void FC_GLOBAL(cscal,CSCAL)
    (blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *ca, void *cx, void *incx);
    hook_pos_cscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) ca, (void *) cx, (void *) incx);
    } else {
        fn((void *) n, (void *) ca, (void *) cx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cscal,CSCAL)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(cscal,CSCAL)))));
void FC_GLOBAL3(cscal,CSCAL)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(cscal,CSCAL)))));
#else
void FC_GLOBAL2(cscal,CSCAL)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx)
{
    FC_GLOBAL(cscal,CSCAL)((void *) n, (void *) ca, (void *) cx, (void *) incx);
}
void FC_GLOBAL3(cscal,CSCAL)(blasint *n, blas_complex_float *ca, blas_complex_float *cx, blasint *incx)
{
    FC_GLOBAL(cscal,CSCAL)((void *) n, (void *) ca, (void *) cx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cscal_(void *n, void *ca, void *cx, void *incx)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx);
    *(void **) &fn = current_backend->blas.cscal;
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cscal(void *n, void *ca, void *cx, void *incx) __attribute__((alias("flexiblas_real_cscal_")));
#else
void flexiblas_real_cscal(void *n, void *ca, void *cx, void *incx)
{
    flexiblas_real_cscal_((void *) n, (void *) ca, (void *) cx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_cscal_(void *n, void *ca, void *cx, void *incx)
{
    void (*fn) (void *n, void *ca, void *cx, void *incx);
    hook_pos_cscal++;
    if ( hook_pos_cscal < __flexiblas_hooks->cscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cscal.hook_function[hook_pos_cscal];
    } else {
        hook_pos_cscal = 0;
        *(void **) &fn = current_backend->blas.cscal;
    }
    fn((void *) n, (void *) ca, (void *) cx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cscal(void *n, void *ca, void *cx, void *incx) __attribute__((alias("flexiblas_chain_cscal_")));
#else
void flexiblas_chain_cscal(void *n, void *ca, void *cx, void *incx)
{
    flexiblas_chain_cscal_((void *) n, (void *) ca, (void *) cx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_csrot = 0;
#endif

/* Wrapper for csrot */
void FC_GLOBAL(csrot,CSROT)
    (blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy, float *c, float *s)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s);
    hook_pos_csrot = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.csrot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->csrot.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    } else {
        fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    }
    #else
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(csrot,CSROT)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(csrot,CSROT)))));
void FC_GLOBAL3(csrot,CSROT)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(csrot,CSROT)))));
#else
void FC_GLOBAL2(csrot,CSROT)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy, float *c, float *s)
{
    FC_GLOBAL(csrot,CSROT)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
}
void FC_GLOBAL3(csrot,CSROT)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy, float *c, float *s)
{
    FC_GLOBAL(csrot,CSROT)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_csrot_(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s);
    *(void **) &fn = current_backend->blas.csrot;
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_csrot(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_real_csrot_")));
#else
void flexiblas_real_csrot(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s)
{
    flexiblas_real_csrot_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_csrot_(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s);
    hook_pos_csrot++;
    if ( hook_pos_csrot < __flexiblas_hooks->csrot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->csrot.hook_function[hook_pos_csrot];
    } else {
        hook_pos_csrot = 0;
        *(void **) &fn = current_backend->blas.csrot;
    }
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_csrot(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_chain_csrot_")));
#else
void flexiblas_chain_csrot(void *n, void *cx, void *incx, void *cy, void *incy, void *c, void *s)
{
    flexiblas_chain_csrot_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_csscal = 0;
#endif

/* Wrapper for csscal */
void FC_GLOBAL(csscal,CSSCAL)
    (blasint *n, float *sa, blas_complex_float *cx, blasint *incx)
{
    void (*fn) (void *n, void *sa, void *cx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sa, void *cx, void *incx);
    hook_pos_csscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.csscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->csscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sa, (void *) cx, (void *) incx);
    } else {
        fn((void *) n, (void *) sa, (void *) cx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) sa, (void *) cx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(csscal,CSSCAL)(blasint *n, float *sa, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(csscal,CSSCAL)))));
void FC_GLOBAL3(csscal,CSSCAL)(blasint *n, float *sa, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(csscal,CSSCAL)))));
#else
void FC_GLOBAL2(csscal,CSSCAL)(blasint *n, float *sa, blas_complex_float *cx, blasint *incx)
{
    FC_GLOBAL(csscal,CSSCAL)((void *) n, (void *) sa, (void *) cx, (void *) incx);
}
void FC_GLOBAL3(csscal,CSSCAL)(blasint *n, float *sa, blas_complex_float *cx, blasint *incx)
{
    FC_GLOBAL(csscal,CSSCAL)((void *) n, (void *) sa, (void *) cx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_csscal_(void *n, void *sa, void *cx, void *incx)
{
    void (*fn) (void *n, void *sa, void *cx, void *incx);
    *(void **) &fn = current_backend->blas.csscal;
    fn((void *) n, (void *) sa, (void *) cx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_csscal(void *n, void *sa, void *cx, void *incx) __attribute__((alias("flexiblas_real_csscal_")));
#else
void flexiblas_real_csscal(void *n, void *sa, void *cx, void *incx)
{
    flexiblas_real_csscal_((void *) n, (void *) sa, (void *) cx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_csscal_(void *n, void *sa, void *cx, void *incx)
{
    void (*fn) (void *n, void *sa, void *cx, void *incx);
    hook_pos_csscal++;
    if ( hook_pos_csscal < __flexiblas_hooks->csscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->csscal.hook_function[hook_pos_csscal];
    } else {
        hook_pos_csscal = 0;
        *(void **) &fn = current_backend->blas.csscal;
    }
    fn((void *) n, (void *) sa, (void *) cx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_csscal(void *n, void *sa, void *cx, void *incx) __attribute__((alias("flexiblas_chain_csscal_")));
#else
void flexiblas_chain_csscal(void *n, void *sa, void *cx, void *incx)
{
    flexiblas_chain_csscal_((void *) n, (void *) sa, (void *) cx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_cswap = 0;
#endif

/* Wrapper for cswap */
void FC_GLOBAL(cswap,CSWAP)
    (blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_cswap = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.cswap;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->cswap.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    } else {
        fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(cswap,CSWAP)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cswap,CSWAP)))));
void FC_GLOBAL3(cswap,CSWAP)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(cswap,CSWAP)))));
#else
void FC_GLOBAL2(cswap,CSWAP)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(cswap,CSWAP)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
void FC_GLOBAL3(cswap,CSWAP)(blasint *n, blas_complex_float *cx, blasint *incx, blas_complex_float *cy, blasint *incy)
{
    FC_GLOBAL(cswap,CSWAP)((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_cswap_(void *n, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    *(void **) &fn = current_backend->blas.cswap;
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_cswap(void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_real_cswap_")));
#else
void flexiblas_real_cswap(void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_real_cswap_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_cswap_(void *n, void *cx, void *incx, void *cy, void *incy)
{
    void (*fn) (void *n, void *cx, void *incx, void *cy, void *incy);
    hook_pos_cswap++;
    if ( hook_pos_cswap < __flexiblas_hooks->cswap.nhook ) {
        *(void **) &fn = __flexiblas_hooks->cswap.hook_function[hook_pos_cswap];
    } else {
        hook_pos_cswap = 0;
        *(void **) &fn = current_backend->blas.cswap;
    }
    fn((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_cswap(void *n, void *cx, void *incx, void *cy, void *incy) __attribute__((alias("flexiblas_chain_cswap_")));
#else
void flexiblas_chain_cswap(void *n, void *cx, void *incx, void *cy, void *incy)
{
    flexiblas_chain_cswap_((void *) n, (void *) cx, (void *) incx, (void *) cy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_csymm = 0;
#endif

/* Wrapper for csymm */
void FC_GLOBAL(csymm,CSYMM)
    (char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_csymm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.csymm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->csymm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(csymm,CSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(csymm,CSYMM)))));
void FC_GLOBAL3(csymm,CSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(csymm,CSYMM)))));
#else
void FC_GLOBAL2(csymm,CSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(csymm,CSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(csymm,CSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(csymm,CSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_csymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.csymm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_csymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_csymm_")));
#else
void flexiblas_real_csymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_csymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_csymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_csymm++;
    if ( hook_pos_csymm < __flexiblas_hooks->csymm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->csymm.hook_function[hook_pos_csymm];
    } else {
        hook_pos_csymm = 0;
        *(void **) &fn = current_backend->blas.csymm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_csymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_csymm_")));
#else
void flexiblas_chain_csymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_csymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_csyr2k = 0;
#endif

/* Wrapper for csyr2k */
void FC_GLOBAL(csyr2k,CSYR2K)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_csyr2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.csyr2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->csyr2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(csyr2k,CSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(csyr2k,CSYR2K)))));
void FC_GLOBAL3(csyr2k,CSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(csyr2k,CSYR2K)))));
#else
void FC_GLOBAL2(csyr2k,CSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(csyr2k,CSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(csyr2k,CSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(csyr2k,CSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_csyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.csyr2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_csyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_csyr2k_")));
#else
void flexiblas_real_csyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_csyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_csyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_csyr2k++;
    if ( hook_pos_csyr2k < __flexiblas_hooks->csyr2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->csyr2k.hook_function[hook_pos_csyr2k];
    } else {
        hook_pos_csyr2k = 0;
        *(void **) &fn = current_backend->blas.csyr2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_csyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_csyr2k_")));
#else
void flexiblas_chain_csyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_csyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_csyrk = 0;
#endif

/* Wrapper for csyrk */
void FC_GLOBAL(csyrk,CSYRK)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_csyrk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.csyrk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->csyrk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(csyrk,CSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(csyrk,CSYRK)))));
void FC_GLOBAL3(csyrk,CSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(csyrk,CSYRK)))));
#else
void FC_GLOBAL2(csyrk,CSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(csyrk,CSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(csyrk,CSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *beta, blas_complex_float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(csyrk,CSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_csyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.csyrk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_csyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_csyrk_")));
#else
void flexiblas_real_csyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_csyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_csyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_csyrk++;
    if ( hook_pos_csyrk < __flexiblas_hooks->csyrk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->csyrk.hook_function[hook_pos_csyrk];
    } else {
        hook_pos_csyrk = 0;
        *(void **) &fn = current_backend->blas.csyrk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_csyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_csyrk_")));
#else
void flexiblas_chain_csyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_csyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctbmv = 0;
#endif

/* Wrapper for ctbmv */
void FC_GLOBAL(ctbmv,CTBMV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctbmv,CTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctbmv,CTBMV)))));
void FC_GLOBAL3(ctbmv,CTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctbmv,CTBMV)))));
#else
void FC_GLOBAL2(ctbmv,CTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctbmv,CTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctbmv,CTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctbmv,CTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctbmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctbmv_")));
#else
void flexiblas_real_ctbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctbmv++;
    if ( hook_pos_ctbmv < __flexiblas_hooks->ctbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctbmv.hook_function[hook_pos_ctbmv];
    } else {
        hook_pos_ctbmv = 0;
        *(void **) &fn = current_backend->blas.ctbmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctbmv_")));
#else
void flexiblas_chain_ctbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctbsv = 0;
#endif

/* Wrapper for ctbsv */
void FC_GLOBAL(ctbsv,CTBSV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctbsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctbsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctbsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctbsv,CTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctbsv,CTBSV)))));
void FC_GLOBAL3(ctbsv,CTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctbsv,CTBSV)))));
#else
void FC_GLOBAL2(ctbsv,CTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctbsv,CTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctbsv,CTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctbsv,CTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctbsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctbsv_")));
#else
void flexiblas_real_ctbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctbsv++;
    if ( hook_pos_ctbsv < __flexiblas_hooks->ctbsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctbsv.hook_function[hook_pos_ctbsv];
    } else {
        hook_pos_ctbsv = 0;
        *(void **) &fn = current_backend->blas.ctbsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctbsv_")));
#else
void flexiblas_chain_ctbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctpmv = 0;
#endif

/* Wrapper for ctpmv */
void FC_GLOBAL(ctpmv,CTPMV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctpmv,CTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctpmv,CTPMV)))));
void FC_GLOBAL3(ctpmv,CTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctpmv,CTPMV)))));
#else
void FC_GLOBAL2(ctpmv,CTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctpmv,CTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctpmv,CTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctpmv,CTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctpmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctpmv_")));
#else
void flexiblas_real_ctpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctpmv++;
    if ( hook_pos_ctpmv < __flexiblas_hooks->ctpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctpmv.hook_function[hook_pos_ctpmv];
    } else {
        hook_pos_ctpmv = 0;
        *(void **) &fn = current_backend->blas.ctpmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctpmv_")));
#else
void flexiblas_chain_ctpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctpsv = 0;
#endif

/* Wrapper for ctpsv */
void FC_GLOBAL(ctpsv,CTPSV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctpsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctpsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctpsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctpsv,CTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctpsv,CTPSV)))));
void FC_GLOBAL3(ctpsv,CTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctpsv,CTPSV)))));
#else
void FC_GLOBAL2(ctpsv,CTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctpsv,CTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctpsv,CTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *ap, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctpsv,CTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctpsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctpsv_")));
#else
void flexiblas_real_ctpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctpsv++;
    if ( hook_pos_ctpsv < __flexiblas_hooks->ctpsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctpsv.hook_function[hook_pos_ctpsv];
    } else {
        hook_pos_ctpsv = 0;
        *(void **) &fn = current_backend->blas.ctpsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctpsv_")));
#else
void flexiblas_chain_ctpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctrmm = 0;
#endif

/* Wrapper for ctrmm */
void FC_GLOBAL(ctrmm,CTRMM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrmm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctrmm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctrmm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctrmm,CTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrmm,CTRMM)))));
void FC_GLOBAL3(ctrmm,CTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrmm,CTRMM)))));
#else
void FC_GLOBAL2(ctrmm,CTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrmm,CTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(ctrmm,CTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrmm,CTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctrmm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctrmm_")));
#else
void flexiblas_real_ctrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrmm++;
    if ( hook_pos_ctrmm < __flexiblas_hooks->ctrmm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctrmm.hook_function[hook_pos_ctrmm];
    } else {
        hook_pos_ctrmm = 0;
        *(void **) &fn = current_backend->blas.ctrmm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctrmm_")));
#else
void flexiblas_chain_ctrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctrmv = 0;
#endif

/* Wrapper for ctrmv */
void FC_GLOBAL(ctrmv,CTRMV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctrmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctrmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctrmv,CTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrmv,CTRMV)))));
void FC_GLOBAL3(ctrmv,CTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrmv,CTRMV)))));
#else
void FC_GLOBAL2(ctrmv,CTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrmv,CTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctrmv,CTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrmv,CTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctrmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctrmv_")));
#else
void flexiblas_real_ctrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrmv++;
    if ( hook_pos_ctrmv < __flexiblas_hooks->ctrmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctrmv.hook_function[hook_pos_ctrmv];
    } else {
        hook_pos_ctrmv = 0;
        *(void **) &fn = current_backend->blas.ctrmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctrmv_")));
#else
void flexiblas_chain_ctrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctrsm = 0;
#endif

/* Wrapper for ctrsm */
void FC_GLOBAL(ctrsm,CTRSM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrsm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctrsm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctrsm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctrsm,CTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrsm,CTRSM)))));
void FC_GLOBAL3(ctrsm,CTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrsm,CTRSM)))));
#else
void FC_GLOBAL2(ctrsm,CTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrsm,CTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(ctrsm,CTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_float *alpha, blas_complex_float *a, blasint *lda, blas_complex_float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrsm,CTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctrsm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctrsm_")));
#else
void flexiblas_real_ctrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrsm++;
    if ( hook_pos_ctrsm < __flexiblas_hooks->ctrsm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctrsm.hook_function[hook_pos_ctrsm];
    } else {
        hook_pos_ctrsm = 0;
        *(void **) &fn = current_backend->blas.ctrsm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctrsm_")));
#else
void flexiblas_chain_ctrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ctrsv = 0;
#endif

/* Wrapper for ctrsv */
void FC_GLOBAL(ctrsv,CTRSV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ctrsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ctrsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ctrsv,CTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrsv,CTRSV)))));
void FC_GLOBAL3(ctrsv,CTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ctrsv,CTRSV)))));
#else
void FC_GLOBAL2(ctrsv,CTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrsv,CTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ctrsv,CTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_float *a, blasint *lda, blas_complex_float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ctrsv,CTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ctrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ctrsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ctrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ctrsv_")));
#else
void flexiblas_real_ctrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ctrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ctrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ctrsv++;
    if ( hook_pos_ctrsv < __flexiblas_hooks->ctrsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ctrsv.hook_function[hook_pos_ctrsv];
    } else {
        hook_pos_ctrsv = 0;
        *(void **) &fn = current_backend->blas.ctrsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ctrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ctrsv_")));
#else
void flexiblas_chain_ctrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ctrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dasum = 0;
#endif

/* Wrapper for dasum */
double FC_GLOBAL(dasum,DASUM)
    (blasint *n, double *dx, blasint *incx)
{
    double (*fn) (void *n, void *dx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *dx, void *incx);
    hook_pos_dasum = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dasum;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dasum.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) dx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) dx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) dx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(dasum,DASUM)(blasint *n, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dasum,DASUM)))));
double FC_GLOBAL3(dasum,DASUM)(blasint *n, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dasum,DASUM)))));
#else
double FC_GLOBAL2(dasum,DASUM)(blasint *n, double *dx, blasint *incx)
{
    return FC_GLOBAL(dasum,DASUM)((void *) n, (void *) dx, (void *) incx);
}
double FC_GLOBAL3(dasum,DASUM)(blasint *n, double *dx, blasint *incx)
{
    return FC_GLOBAL(dasum,DASUM)((void *) n, (void *) dx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_dasum_(void *n, void *dx, void *incx)
{
    double (*fn) (void *n, void *dx, void *incx);
    double ret;
    *(void **) &fn = current_backend->blas.dasum;
    ret = fn((void *) n, (void *) dx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_dasum(void *n, void *dx, void *incx) __attribute__((alias("flexiblas_real_dasum_")));
#else
double flexiblas_real_dasum(void *n, void *dx, void *incx)
{
    return flexiblas_real_dasum_((void *) n, (void *) dx, (void *) incx);
}
#endif

double flexiblas_chain_dasum_(void *n, void *dx, void *incx)
{
    double (*fn) (void *n, void *dx, void *incx);
    double ret;
    hook_pos_dasum++;
    if ( hook_pos_dasum < __flexiblas_hooks->dasum.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dasum.hook_function[hook_pos_dasum];
    } else {
        hook_pos_dasum = 0;
        *(void **) &fn = current_backend->blas.dasum;
    }
    ret = fn((void *) n, (void *) dx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_dasum(void *n, void *dx, void *incx) __attribute__((alias("flexiblas_chain_dasum_")));
#else
double flexiblas_chain_dasum(void *n, void *dx, void *incx)
{
    return flexiblas_chain_dasum_((void *) n, (void *) dx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_daxpby = 0;
#endif

/* Wrapper for daxpby */
void FC_GLOBAL(daxpby,DAXPBY)
    (blasint *n, double *da, double *dx, blasint *incx, double *db, double *dy, blasint *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy);
    hook_pos_daxpby = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.daxpby;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->daxpby.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    } else {
        fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(daxpby,DAXPBY)(blasint *n, double *da, double *dx, blasint *incx, double *db, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(daxpby,DAXPBY)))));
void FC_GLOBAL3(daxpby,DAXPBY)(blasint *n, double *da, double *dx, blasint *incx, double *db, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(daxpby,DAXPBY)))));
#else
void FC_GLOBAL2(daxpby,DAXPBY)(blasint *n, double *da, double *dx, blasint *incx, double *db, double *dy, blasint *incy)
{
    FC_GLOBAL(daxpby,DAXPBY)((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
}
void FC_GLOBAL3(daxpby,DAXPBY)(blasint *n, double *da, double *dx, blasint *incx, double *db, double *dy, blasint *incy)
{
    FC_GLOBAL(daxpby,DAXPBY)((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_daxpby_(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy);
    *(void **) &fn = current_backend->blas.daxpby;
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_daxpby(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy) __attribute__((alias("flexiblas_real_daxpby_")));
#else
void flexiblas_real_daxpby(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy)
{
    flexiblas_real_daxpby_((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_daxpby_(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy);
    hook_pos_daxpby++;
    if ( hook_pos_daxpby < __flexiblas_hooks->daxpby.nhook ) {
        *(void **) &fn = __flexiblas_hooks->daxpby.hook_function[hook_pos_daxpby];
    } else {
        hook_pos_daxpby = 0;
        *(void **) &fn = current_backend->blas.daxpby;
    }
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_daxpby(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy) __attribute__((alias("flexiblas_chain_daxpby_")));
#else
void flexiblas_chain_daxpby(void *n, void *da, void *dx, void *incx, void *db, void *dy, void *incy)
{
    flexiblas_chain_daxpby_((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) db, (void *) dy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_daxpy = 0;
#endif

/* Wrapper for daxpy */
void FC_GLOBAL(daxpy,DAXPY)
    (blasint *n, double *da, double *dx, blasint *incx, double *dy, blasint *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *dy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *da, void *dx, void *incx, void *dy, void *incy);
    hook_pos_daxpy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.daxpy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->daxpy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    } else {
        fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(daxpy,DAXPY)(blasint *n, double *da, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(daxpy,DAXPY)))));
void FC_GLOBAL3(daxpy,DAXPY)(blasint *n, double *da, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(daxpy,DAXPY)))));
#else
void FC_GLOBAL2(daxpy,DAXPY)(blasint *n, double *da, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(daxpy,DAXPY)((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
void FC_GLOBAL3(daxpy,DAXPY)(blasint *n, double *da, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(daxpy,DAXPY)((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_daxpy_(void *n, void *da, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *dy, void *incy);
    *(void **) &fn = current_backend->blas.daxpy;
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_daxpy(void *n, void *da, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_real_daxpy_")));
#else
void flexiblas_real_daxpy(void *n, void *da, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_real_daxpy_((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_daxpy_(void *n, void *da, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *da, void *dx, void *incx, void *dy, void *incy);
    hook_pos_daxpy++;
    if ( hook_pos_daxpy < __flexiblas_hooks->daxpy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->daxpy.hook_function[hook_pos_daxpy];
    } else {
        hook_pos_daxpy = 0;
        *(void **) &fn = current_backend->blas.daxpy;
    }
    fn((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_daxpy(void *n, void *da, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_chain_daxpy_")));
#else
void flexiblas_chain_daxpy(void *n, void *da, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_chain_daxpy_((void *) n, (void *) da, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dcopy = 0;
#endif

/* Wrapper for dcopy */
void FC_GLOBAL(dcopy,DCOPY)
    (blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *dx, void *incx, void *dy, void *incy);
    hook_pos_dcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    } else {
        fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dcopy,DCOPY)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dcopy,DCOPY)))));
void FC_GLOBAL3(dcopy,DCOPY)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dcopy,DCOPY)))));
#else
void FC_GLOBAL2(dcopy,DCOPY)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(dcopy,DCOPY)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
void FC_GLOBAL3(dcopy,DCOPY)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(dcopy,DCOPY)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dcopy_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    *(void **) &fn = current_backend->blas.dcopy;
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dcopy(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_real_dcopy_")));
#else
void flexiblas_real_dcopy(void *n, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_real_dcopy_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_dcopy_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    hook_pos_dcopy++;
    if ( hook_pos_dcopy < __flexiblas_hooks->dcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dcopy.hook_function[hook_pos_dcopy];
    } else {
        hook_pos_dcopy = 0;
        *(void **) &fn = current_backend->blas.dcopy;
    }
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dcopy(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_chain_dcopy_")));
#else
void flexiblas_chain_dcopy(void *n, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_chain_dcopy_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ddot = 0;
#endif

/* Wrapper for ddot */
double FC_GLOBAL(ddot,DDOT)
    (blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    double (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *dx, void *incx, void *dy, void *incy);
    hook_pos_ddot = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ddot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ddot.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    } else {
        ret = fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    }
    #else
    ret = fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(ddot,DDOT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(ddot,DDOT)))));
double FC_GLOBAL3(ddot,DDOT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(ddot,DDOT)))));
#else
double FC_GLOBAL2(ddot,DDOT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    return FC_GLOBAL(ddot,DDOT)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
double FC_GLOBAL3(ddot,DDOT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    return FC_GLOBAL(ddot,DDOT)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_ddot_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    double (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    double ret;
    *(void **) &fn = current_backend->blas.ddot;
    ret = fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_ddot(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_real_ddot_")));
#else
double flexiblas_real_ddot(void *n, void *dx, void *incx, void *dy, void *incy)
{
    return flexiblas_real_ddot_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif

double flexiblas_chain_ddot_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    double (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    double ret;
    hook_pos_ddot++;
    if ( hook_pos_ddot < __flexiblas_hooks->ddot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ddot.hook_function[hook_pos_ddot];
    } else {
        hook_pos_ddot = 0;
        *(void **) &fn = current_backend->blas.ddot;
    }
    ret = fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_ddot(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_chain_ddot_")));
#else
double flexiblas_chain_ddot(void *n, void *dx, void *incx, void *dy, void *incy)
{
    return flexiblas_chain_ddot_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dgbmv = 0;
#endif

/* Wrapper for dgbmv */
void FC_GLOBAL(dgbmv,DGBMV)
    (char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dgbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dgbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dgbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dgbmv,DGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dgbmv,DGBMV)))));
void FC_GLOBAL3(dgbmv,DGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dgbmv,DGBMV)))));
#else
void FC_GLOBAL2(dgbmv,DGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dgbmv,DGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(dgbmv,DGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dgbmv,DGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.dgbmv;
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_dgbmv_")));
#else
void flexiblas_real_dgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_dgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_dgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dgbmv++;
    if ( hook_pos_dgbmv < __flexiblas_hooks->dgbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dgbmv.hook_function[hook_pos_dgbmv];
    } else {
        hook_pos_dgbmv = 0;
        *(void **) &fn = current_backend->blas.dgbmv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_dgbmv_")));
#else
void flexiblas_chain_dgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_dgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dgeadd = 0;
#endif

/* Wrapper for dgeadd */
void FC_GLOBAL(dgeadd,DGEADD)
    (blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *beta, double *b, blasint *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_dgeadd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dgeadd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dgeadd.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dgeadd,DGEADD)(blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *beta, double *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(dgeadd,DGEADD)))));
void FC_GLOBAL3(dgeadd,DGEADD)(blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *beta, double *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(dgeadd,DGEADD)))));
#else
void FC_GLOBAL2(dgeadd,DGEADD)(blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *beta, double *b, blasint *ldb)
{
    FC_GLOBAL(dgeadd,DGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
void FC_GLOBAL3(dgeadd,DGEADD)(blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *beta, double *b, blasint *ldb)
{
    FC_GLOBAL(dgeadd,DGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    *(void **) &fn = current_backend->blas.dgeadd;
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_real_dgeadd_")));
#else
void flexiblas_real_dgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_real_dgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif

void flexiblas_chain_dgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_dgeadd++;
    if ( hook_pos_dgeadd < __flexiblas_hooks->dgeadd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dgeadd.hook_function[hook_pos_dgeadd];
    } else {
        hook_pos_dgeadd = 0;
        *(void **) &fn = current_backend->blas.dgeadd;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_chain_dgeadd_")));
#else
void flexiblas_chain_dgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_chain_dgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dgemm = 0;
#endif

/* Wrapper for dgemm */
void FC_GLOBAL(dgemm,DGEMM)
    (char *transa, char *transb, blasint *m, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_dgemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dgemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dgemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    } else {
        fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    }
    #else
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dgemm,DGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemm,DGEMM)))));
void FC_GLOBAL3(dgemm,DGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemm,DGEMM)))));
#else
void FC_GLOBAL2(dgemm,DGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemm,DGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
void FC_GLOBAL3(dgemm,DGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemm,DGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.dgemm;
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_dgemm_")));
#else
void flexiblas_real_dgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_dgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_dgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_dgemm++;
    if ( hook_pos_dgemm < __flexiblas_hooks->dgemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dgemm.hook_function[hook_pos_dgemm];
    } else {
        hook_pos_dgemm = 0;
        *(void **) &fn = current_backend->blas.dgemm;
    }
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_dgemm_")));
#else
void flexiblas_chain_dgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_dgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dgemmtr = 0;
#endif

/* Wrapper for dgemmtr */
void FC_GLOBAL(dgemmtr,DGEMMTR)
    (char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_dgemmtr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dgemmtr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dgemmtr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    } else {
        fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    }
    #else
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dgemmtr,DGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemmtr,DGEMMTR)))));
void FC_GLOBAL3(dgemmtr,DGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemmtr,DGEMMTR)))));
#else
void FC_GLOBAL2(dgemmtr,DGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemmtr,DGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(dgemmtr,DGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemmtr,DGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.dgemmtr;
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_dgemmtr_")));
#else
void flexiblas_real_dgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_dgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_dgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_dgemmtr++;
    if ( hook_pos_dgemmtr < __flexiblas_hooks->dgemmtr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dgemmtr.hook_function[hook_pos_dgemmtr];
    } else {
        hook_pos_dgemmtr = 0;
        *(void **) &fn = current_backend->blas.dgemmtr;
    }
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_dgemmtr_")));
#else
void flexiblas_chain_dgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_dgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif
#endif
/* Alternative function name for dgemmtr -- dgemmt */
#ifndef __APPLE__
void FC_GLOBAL(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemmtr,DGEMMTR)))));
void FC_GLOBAL2(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemmtr,DGEMMTR)))));
void FC_GLOBAL3(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(dgemmtr,DGEMMTR)))));
#else
void FC_GLOBAL(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemmtr,DGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL2(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemmtr,DGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(dgemmt,DGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(dgemmtr,DGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif

#ifdef FLEXIBLAS_HOOK_API
#ifndef __APPLE__
void flexiblas_real_dgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_dgemmtr")));
void flexiblas_real_dgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_dgemmtr")));
void flexiblas_chain_dgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_dgemmtr")));
void flexiblas_chain_dgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_dgemmtr")));

#else
void flexiblas_real_dgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_dgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
void flexiblas_real_dgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_dgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_dgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_dgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_dgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_dgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dgemv = 0;
#endif

/* Wrapper for dgemv */
void FC_GLOBAL(dgemv,DGEMV)
    (char *trans, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dgemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dgemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dgemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dgemv,DGEMV)(char *trans, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dgemv,DGEMV)))));
void FC_GLOBAL3(dgemv,DGEMV)(char *trans, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dgemv,DGEMV)))));
#else
void FC_GLOBAL2(dgemv,DGEMV)(char *trans, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dgemv,DGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(dgemv,DGEMV)(char *trans, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dgemv,DGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.dgemv;
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_dgemv_")));
#else
void flexiblas_real_dgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_dgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_dgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dgemv++;
    if ( hook_pos_dgemv < __flexiblas_hooks->dgemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dgemv.hook_function[hook_pos_dgemv];
    } else {
        hook_pos_dgemv = 0;
        *(void **) &fn = current_backend->blas.dgemv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_dgemv_")));
#else
void flexiblas_chain_dgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_dgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dger = 0;
#endif

/* Wrapper for dger */
void FC_GLOBAL(dger,DGER)
    (blasint *m, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_dger = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dger;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dger.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dger,DGER)(blasint *m, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(dger,DGER)))));
void FC_GLOBAL3(dger,DGER)(blasint *m, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(dger,DGER)))));
#else
void FC_GLOBAL2(dger,DGER)(blasint *m, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda)
{
    FC_GLOBAL(dger,DGER)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(dger,DGER)(blasint *m, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda)
{
    FC_GLOBAL(dger,DGER)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dger_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.dger;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_dger_")));
#else
void flexiblas_real_dger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_dger_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_dger_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_dger++;
    if ( hook_pos_dger < __flexiblas_hooks->dger.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dger.hook_function[hook_pos_dger];
    } else {
        hook_pos_dger = 0;
        *(void **) &fn = current_backend->blas.dger;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_dger_")));
#else
void flexiblas_chain_dger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_dger_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dimatcopy = 0;
#endif

/* Wrapper for dimatcopy */
void FC_GLOBAL(dimatcopy,DIMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dimatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dimatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dimatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dimatcopy,DIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dimatcopy,DIMATCOPY)))));
void FC_GLOBAL3(dimatcopy,DIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dimatcopy,DIMATCOPY)))));
#else
void FC_GLOBAL2(dimatcopy,DIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dimatcopy,DIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
void FC_GLOBAL3(dimatcopy,DIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dimatcopy,DIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.dimatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_dimatcopy_")));
#else
void flexiblas_real_dimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_dimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif

void flexiblas_chain_dimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dimatcopy++;
    if ( hook_pos_dimatcopy < __flexiblas_hooks->dimatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dimatcopy.hook_function[hook_pos_dimatcopy];
    } else {
        hook_pos_dimatcopy = 0;
        *(void **) &fn = current_backend->blas.dimatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_dimatcopy_")));
#else
void flexiblas_chain_dimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_dimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dnrm2 = 0;
#endif

/* Wrapper for dnrm2 */
double FC_GLOBAL(dnrm2,DNRM2)
    (blasint *n, double *x, blasint *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *x, void *incx);
    hook_pos_dnrm2 = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dnrm2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dnrm2.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) x, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) x, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) x, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(dnrm2,DNRM2)(blasint *n, double *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dnrm2,DNRM2)))));
double FC_GLOBAL3(dnrm2,DNRM2)(blasint *n, double *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dnrm2,DNRM2)))));
#else
double FC_GLOBAL2(dnrm2,DNRM2)(blasint *n, double *x, blasint *incx)
{
    return FC_GLOBAL(dnrm2,DNRM2)((void *) n, (void *) x, (void *) incx);
}
double FC_GLOBAL3(dnrm2,DNRM2)(blasint *n, double *x, blasint *incx)
{
    return FC_GLOBAL(dnrm2,DNRM2)((void *) n, (void *) x, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_dnrm2_(void *n, void *x, void *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    double ret;
    *(void **) &fn = current_backend->blas.dnrm2;
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_dnrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_real_dnrm2_")));
#else
double flexiblas_real_dnrm2(void *n, void *x, void *incx)
{
    return flexiblas_real_dnrm2_((void *) n, (void *) x, (void *) incx);
}
#endif

double flexiblas_chain_dnrm2_(void *n, void *x, void *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    double ret;
    hook_pos_dnrm2++;
    if ( hook_pos_dnrm2 < __flexiblas_hooks->dnrm2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dnrm2.hook_function[hook_pos_dnrm2];
    } else {
        hook_pos_dnrm2 = 0;
        *(void **) &fn = current_backend->blas.dnrm2;
    }
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_dnrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_chain_dnrm2_")));
#else
double flexiblas_chain_dnrm2(void *n, void *x, void *incx)
{
    return flexiblas_chain_dnrm2_((void *) n, (void *) x, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_domatcopy = 0;
#endif

/* Wrapper for domatcopy */
void FC_GLOBAL(domatcopy,DOMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_domatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.domatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->domatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(domatcopy,DOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(domatcopy,DOMATCOPY)))));
void FC_GLOBAL3(domatcopy,DOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(domatcopy,DOMATCOPY)))));
#else
void FC_GLOBAL2(domatcopy,DOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(domatcopy,DOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
void FC_GLOBAL3(domatcopy,DOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(domatcopy,DOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_domatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    *(void **) &fn = current_backend->blas.domatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_domatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_real_domatcopy_")));
#else
void flexiblas_real_domatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_real_domatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif

void flexiblas_chain_domatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_domatcopy++;
    if ( hook_pos_domatcopy < __flexiblas_hooks->domatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->domatcopy.hook_function[hook_pos_domatcopy];
    } else {
        hook_pos_domatcopy = 0;
        *(void **) &fn = current_backend->blas.domatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_domatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_chain_domatcopy_")));
#else
void flexiblas_chain_domatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_chain_domatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_drot = 0;
#endif

/* Wrapper for drot */
void FC_GLOBAL(drot,DROT)
    (blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *c, double *s)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s);
    hook_pos_drot = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.drot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->drot.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    } else {
        fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    }
    #else
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(drot,DROT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(drot,DROT)))));
void FC_GLOBAL3(drot,DROT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(drot,DROT)))));
#else
void FC_GLOBAL2(drot,DROT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *c, double *s)
{
    FC_GLOBAL(drot,DROT)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
}
void FC_GLOBAL3(drot,DROT)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *c, double *s)
{
    FC_GLOBAL(drot,DROT)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_drot_(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s);
    *(void **) &fn = current_backend->blas.drot;
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_drot(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_real_drot_")));
#else
void flexiblas_real_drot(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s)
{
    flexiblas_real_drot_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_drot_(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s);
    hook_pos_drot++;
    if ( hook_pos_drot < __flexiblas_hooks->drot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->drot.hook_function[hook_pos_drot];
    } else {
        hook_pos_drot = 0;
        *(void **) &fn = current_backend->blas.drot;
    }
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_drot(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_chain_drot_")));
#else
void flexiblas_chain_drot(void *n, void *dx, void *incx, void *dy, void *incy, void *c, void *s)
{
    flexiblas_chain_drot_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_drotg = 0;
#endif

/* Wrapper for drotg */
void FC_GLOBAL(drotg,DROTG)
    (double *a, double *b, double *c, double *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *a, void *b, void *c, void *s);
    hook_pos_drotg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.drotg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->drotg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) a, (void *) b, (void *) c, (void *) s);
    } else {
        fn((void *) a, (void *) b, (void *) c, (void *) s);
    }
    #else
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(drotg,DROTG)(double *a, double *b, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(drotg,DROTG)))));
void FC_GLOBAL3(drotg,DROTG)(double *a, double *b, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(drotg,DROTG)))));
#else
void FC_GLOBAL2(drotg,DROTG)(double *a, double *b, double *c, double *s)
{
    FC_GLOBAL(drotg,DROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
void FC_GLOBAL3(drotg,DROTG)(double *a, double *b, double *c, double *s)
{
    FC_GLOBAL(drotg,DROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_drotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    *(void **) &fn = current_backend->blas.drotg;
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_drotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_real_drotg_")));
#else
void flexiblas_real_drotg(void *a, void *b, void *c, void *s)
{
    flexiblas_real_drotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_drotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    hook_pos_drotg++;
    if ( hook_pos_drotg < __flexiblas_hooks->drotg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->drotg.hook_function[hook_pos_drotg];
    } else {
        hook_pos_drotg = 0;
        *(void **) &fn = current_backend->blas.drotg;
    }
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_drotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_chain_drotg_")));
#else
void flexiblas_chain_drotg(void *a, void *b, void *c, void *s)
{
    flexiblas_chain_drotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_drotm = 0;
#endif

/* Wrapper for drotm */
void FC_GLOBAL(drotm,DROTM)
    (blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *dparam)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *dparam);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *dx, void *incx, void *dy, void *incy, void *dparam);
    hook_pos_drotm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.drotm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->drotm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    } else {
        fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    }
    #else
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(drotm,DROTM)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *dparam) __attribute__((alias(MTS(FC_GLOBAL(drotm,DROTM)))));
void FC_GLOBAL3(drotm,DROTM)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *dparam) __attribute__((alias(MTS(FC_GLOBAL(drotm,DROTM)))));
#else
void FC_GLOBAL2(drotm,DROTM)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *dparam)
{
    FC_GLOBAL(drotm,DROTM)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
}
void FC_GLOBAL3(drotm,DROTM)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy, double *dparam)
{
    FC_GLOBAL(drotm,DROTM)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_drotm_(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *dparam);
    *(void **) &fn = current_backend->blas.drotm;
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    return;
}
#ifndef __APPLE__
void flexiblas_real_drotm(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam) __attribute__((alias("flexiblas_real_drotm_")));
#else
void flexiblas_real_drotm(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam)
{
    flexiblas_real_drotm_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    return;
}
#endif

void flexiblas_chain_drotm_(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy, void *dparam);
    hook_pos_drotm++;
    if ( hook_pos_drotm < __flexiblas_hooks->drotm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->drotm.hook_function[hook_pos_drotm];
    } else {
        hook_pos_drotm = 0;
        *(void **) &fn = current_backend->blas.drotm;
    }
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_drotm(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam) __attribute__((alias("flexiblas_chain_drotm_")));
#else
void flexiblas_chain_drotm(void *n, void *dx, void *incx, void *dy, void *incy, void *dparam)
{
    flexiblas_chain_drotm_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy, (void *) dparam);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_drotmg = 0;
#endif

/* Wrapper for drotmg */
void FC_GLOBAL(drotmg,DROTMG)
    (double *dd1, double *dd2, double *dx1, double *dy1, double *dparam)
{
    void (*fn) (void *dd1, void *dd2, void *dx1, void *dy1, void *dparam);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *dd1, void *dd2, void *dx1, void *dy1, void *dparam);
    hook_pos_drotmg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.drotmg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->drotmg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    } else {
        fn((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    }
    #else
    fn((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(drotmg,DROTMG)(double *dd1, double *dd2, double *dx1, double *dy1, double *dparam) __attribute__((alias(MTS(FC_GLOBAL(drotmg,DROTMG)))));
void FC_GLOBAL3(drotmg,DROTMG)(double *dd1, double *dd2, double *dx1, double *dy1, double *dparam) __attribute__((alias(MTS(FC_GLOBAL(drotmg,DROTMG)))));
#else
void FC_GLOBAL2(drotmg,DROTMG)(double *dd1, double *dd2, double *dx1, double *dy1, double *dparam)
{
    FC_GLOBAL(drotmg,DROTMG)((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
}
void FC_GLOBAL3(drotmg,DROTMG)(double *dd1, double *dd2, double *dx1, double *dy1, double *dparam)
{
    FC_GLOBAL(drotmg,DROTMG)((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_drotmg_(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam)
{
    void (*fn) (void *dd1, void *dd2, void *dx1, void *dy1, void *dparam);
    *(void **) &fn = current_backend->blas.drotmg;
    fn((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    return;
}
#ifndef __APPLE__
void flexiblas_real_drotmg(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam) __attribute__((alias("flexiblas_real_drotmg_")));
#else
void flexiblas_real_drotmg(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam)
{
    flexiblas_real_drotmg_((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    return;
}
#endif

void flexiblas_chain_drotmg_(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam)
{
    void (*fn) (void *dd1, void *dd2, void *dx1, void *dy1, void *dparam);
    hook_pos_drotmg++;
    if ( hook_pos_drotmg < __flexiblas_hooks->drotmg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->drotmg.hook_function[hook_pos_drotmg];
    } else {
        hook_pos_drotmg = 0;
        *(void **) &fn = current_backend->blas.drotmg;
    }
    fn((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_drotmg(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam) __attribute__((alias("flexiblas_chain_drotmg_")));
#else
void flexiblas_chain_drotmg(void *dd1, void *dd2, void *dx1, void *dy1, void *dparam)
{
    flexiblas_chain_drotmg_((void *) dd1, (void *) dd2, (void *) dx1, (void *) dy1, (void *) dparam);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsbmv = 0;
#endif

/* Wrapper for dsbmv */
void FC_GLOBAL(dsbmv,DSBMV)
    (char *uplo, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsbmv,DSBMV)(char *uplo, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsbmv,DSBMV)))));
void FC_GLOBAL3(dsbmv,DSBMV)(char *uplo, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsbmv,DSBMV)))));
#else
void FC_GLOBAL2(dsbmv,DSBMV)(char *uplo, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsbmv,DSBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(dsbmv,DSBMV)(char *uplo, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsbmv,DSBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dsbmv;
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dsbmv_")));
#else
void flexiblas_real_dsbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dsbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_dsbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsbmv++;
    if ( hook_pos_dsbmv < __flexiblas_hooks->dsbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsbmv.hook_function[hook_pos_dsbmv];
    } else {
        hook_pos_dsbmv = 0;
        *(void **) &fn = current_backend->blas.dsbmv;
    }
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dsbmv_")));
#else
void flexiblas_chain_dsbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dsbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dscal = 0;
#endif

/* Wrapper for dscal */
void FC_GLOBAL(dscal,DSCAL)
    (blasint *n, double *da, double *dx, blasint *incx)
{
    void (*fn) (void *n, void *da, void *dx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *da, void *dx, void *incx);
    hook_pos_dscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) da, (void *) dx, (void *) incx);
    } else {
        fn((void *) n, (void *) da, (void *) dx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) da, (void *) dx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dscal,DSCAL)(blasint *n, double *da, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dscal,DSCAL)))));
void FC_GLOBAL3(dscal,DSCAL)(blasint *n, double *da, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dscal,DSCAL)))));
#else
void FC_GLOBAL2(dscal,DSCAL)(blasint *n, double *da, double *dx, blasint *incx)
{
    FC_GLOBAL(dscal,DSCAL)((void *) n, (void *) da, (void *) dx, (void *) incx);
}
void FC_GLOBAL3(dscal,DSCAL)(blasint *n, double *da, double *dx, blasint *incx)
{
    FC_GLOBAL(dscal,DSCAL)((void *) n, (void *) da, (void *) dx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dscal_(void *n, void *da, void *dx, void *incx)
{
    void (*fn) (void *n, void *da, void *dx, void *incx);
    *(void **) &fn = current_backend->blas.dscal;
    fn((void *) n, (void *) da, (void *) dx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dscal(void *n, void *da, void *dx, void *incx) __attribute__((alias("flexiblas_real_dscal_")));
#else
void flexiblas_real_dscal(void *n, void *da, void *dx, void *incx)
{
    flexiblas_real_dscal_((void *) n, (void *) da, (void *) dx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_dscal_(void *n, void *da, void *dx, void *incx)
{
    void (*fn) (void *n, void *da, void *dx, void *incx);
    hook_pos_dscal++;
    if ( hook_pos_dscal < __flexiblas_hooks->dscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dscal.hook_function[hook_pos_dscal];
    } else {
        hook_pos_dscal = 0;
        *(void **) &fn = current_backend->blas.dscal;
    }
    fn((void *) n, (void *) da, (void *) dx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dscal(void *n, void *da, void *dx, void *incx) __attribute__((alias("flexiblas_chain_dscal_")));
#else
void flexiblas_chain_dscal(void *n, void *da, void *dx, void *incx)
{
    flexiblas_chain_dscal_((void *) n, (void *) da, (void *) dx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsdot = 0;
#endif

/* Wrapper for dsdot */
double FC_GLOBAL(dsdot,DSDOT)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    double (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_dsdot = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsdot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsdot.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    }
    #else
    ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(dsdot,DSDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dsdot,DSDOT)))));
double FC_GLOBAL3(dsdot,DSDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dsdot,DSDOT)))));
#else
double FC_GLOBAL2(dsdot,DSDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(dsdot,DSDOT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
double FC_GLOBAL3(dsdot,DSDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(dsdot,DSDOT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_dsdot_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    double (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    double ret;
    *(void **) &fn = current_backend->blas.dsdot;
    ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_dsdot(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_dsdot_")));
#else
double flexiblas_real_dsdot(void *n, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_real_dsdot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif

double flexiblas_chain_dsdot_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    double (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    double ret;
    hook_pos_dsdot++;
    if ( hook_pos_dsdot < __flexiblas_hooks->dsdot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsdot.hook_function[hook_pos_dsdot];
    } else {
        hook_pos_dsdot = 0;
        *(void **) &fn = current_backend->blas.dsdot;
    }
    ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_dsdot(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_dsdot_")));
#else
double flexiblas_chain_dsdot(void *n, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_chain_dsdot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dspmv = 0;
#endif

/* Wrapper for dspmv */
void FC_GLOBAL(dspmv,DSPMV)
    (char *uplo, blasint *n, double *alpha, double *ap, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dspmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dspmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dspmv,DSPMV)(char *uplo, blasint *n, double *alpha, double *ap, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspmv,DSPMV)))));
void FC_GLOBAL3(dspmv,DSPMV)(char *uplo, blasint *n, double *alpha, double *ap, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspmv,DSPMV)))));
#else
void FC_GLOBAL2(dspmv,DSPMV)(char *uplo, blasint *n, double *alpha, double *ap, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspmv,DSPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(dspmv,DSPMV)(char *uplo, blasint *n, double *alpha, double *ap, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspmv,DSPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dspmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dspmv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dspmv_")));
#else
void flexiblas_real_dspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dspmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_dspmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspmv++;
    if ( hook_pos_dspmv < __flexiblas_hooks->dspmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dspmv.hook_function[hook_pos_dspmv];
    } else {
        hook_pos_dspmv = 0;
        *(void **) &fn = current_backend->blas.dspmv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dspmv_")));
#else
void flexiblas_chain_dspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dspmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dspr = 0;
#endif

/* Wrapper for dspr */
void FC_GLOBAL(dspr,DSPR)
    (char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dspr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dspr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dspr,DSPR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspr,DSPR)))));
void FC_GLOBAL3(dspr,DSPR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspr,DSPR)))));
#else
void FC_GLOBAL2(dspr,DSPR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspr,DSPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
void FC_GLOBAL3(dspr,DSPR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspr,DSPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dspr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dspr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dspr_")));
#else
void flexiblas_real_dspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dspr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_dspr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspr++;
    if ( hook_pos_dspr < __flexiblas_hooks->dspr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dspr.hook_function[hook_pos_dspr];
    } else {
        hook_pos_dspr = 0;
        *(void **) &fn = current_backend->blas.dspr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dspr_")));
#else
void flexiblas_chain_dspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dspr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dspr2 = 0;
#endif

/* Wrapper for dspr2 */
void FC_GLOBAL(dspr2,DSPR2)
    (char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dspr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dspr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dspr2,DSPR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspr2,DSPR2)))));
void FC_GLOBAL3(dspr2,DSPR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dspr2,DSPR2)))));
#else
void FC_GLOBAL2(dspr2,DSPR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspr2,DSPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
void FC_GLOBAL3(dspr2,DSPR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dspr2,DSPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dspr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dspr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dspr2_")));
#else
void flexiblas_real_dspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dspr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_dspr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dspr2++;
    if ( hook_pos_dspr2 < __flexiblas_hooks->dspr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dspr2.hook_function[hook_pos_dspr2];
    } else {
        hook_pos_dspr2 = 0;
        *(void **) &fn = current_backend->blas.dspr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dspr2_")));
#else
void flexiblas_chain_dspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dspr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dswap = 0;
#endif

/* Wrapper for dswap */
void FC_GLOBAL(dswap,DSWAP)
    (blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *dx, void *incx, void *dy, void *incy);
    hook_pos_dswap = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dswap;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dswap.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    } else {
        fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dswap,DSWAP)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dswap,DSWAP)))));
void FC_GLOBAL3(dswap,DSWAP)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(dswap,DSWAP)))));
#else
void FC_GLOBAL2(dswap,DSWAP)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(dswap,DSWAP)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
void FC_GLOBAL3(dswap,DSWAP)(blasint *n, double *dx, blasint *incx, double *dy, blasint *incy)
{
    FC_GLOBAL(dswap,DSWAP)((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dswap_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    *(void **) &fn = current_backend->blas.dswap;
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dswap(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_real_dswap_")));
#else
void flexiblas_real_dswap(void *n, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_real_dswap_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_dswap_(void *n, void *dx, void *incx, void *dy, void *incy)
{
    void (*fn) (void *n, void *dx, void *incx, void *dy, void *incy);
    hook_pos_dswap++;
    if ( hook_pos_dswap < __flexiblas_hooks->dswap.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dswap.hook_function[hook_pos_dswap];
    } else {
        hook_pos_dswap = 0;
        *(void **) &fn = current_backend->blas.dswap;
    }
    fn((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dswap(void *n, void *dx, void *incx, void *dy, void *incy) __attribute__((alias("flexiblas_chain_dswap_")));
#else
void flexiblas_chain_dswap(void *n, void *dx, void *incx, void *dy, void *incy)
{
    flexiblas_chain_dswap_((void *) n, (void *) dx, (void *) incx, (void *) dy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsymm = 0;
#endif

/* Wrapper for dsymm */
void FC_GLOBAL(dsymm,DSYMM)
    (char *side, char *uplo, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsymm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsymm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsymm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsymm,DSYMM)(char *side, char *uplo, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsymm,DSYMM)))));
void FC_GLOBAL3(dsymm,DSYMM)(char *side, char *uplo, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsymm,DSYMM)))));
#else
void FC_GLOBAL2(dsymm,DSYMM)(char *side, char *uplo, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsymm,DSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(dsymm,DSYMM)(char *side, char *uplo, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsymm,DSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dsymm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dsymm_")));
#else
void flexiblas_real_dsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dsymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_dsymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsymm++;
    if ( hook_pos_dsymm < __flexiblas_hooks->dsymm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsymm.hook_function[hook_pos_dsymm];
    } else {
        hook_pos_dsymm = 0;
        *(void **) &fn = current_backend->blas.dsymm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dsymm_")));
#else
void flexiblas_chain_dsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dsymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsymv = 0;
#endif

/* Wrapper for dsymv */
void FC_GLOBAL(dsymv,DSYMV)
    (char *uplo, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsymv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsymv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsymv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsymv,DSYMV)(char *uplo, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsymv,DSYMV)))));
void FC_GLOBAL3(dsymv,DSYMV)(char *uplo, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsymv,DSYMV)))));
#else
void FC_GLOBAL2(dsymv,DSYMV)(char *uplo, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsymv,DSYMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(dsymv,DSYMV)(char *uplo, blasint *n, double *alpha, double *a, blasint *lda, double *x, blasint *incx, double *beta, double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsymv,DSYMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsymv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dsymv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dsymv_")));
#else
void flexiblas_real_dsymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dsymv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_dsymv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsymv++;
    if ( hook_pos_dsymv < __flexiblas_hooks->dsymv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsymv.hook_function[hook_pos_dsymv];
    } else {
        hook_pos_dsymv = 0;
        *(void **) &fn = current_backend->blas.dsymv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dsymv_")));
#else
void flexiblas_chain_dsymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dsymv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsyr = 0;
#endif

/* Wrapper for dsyr */
void FC_GLOBAL(dsyr,DSYR)
    (char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsyr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsyr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsyr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsyr,DSYR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr,DSYR)))));
void FC_GLOBAL3(dsyr,DSYR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr,DSYR)))));
#else
void FC_GLOBAL2(dsyr,DSYR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsyr,DSYR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(dsyr,DSYR)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsyr,DSYR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsyr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dsyr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dsyr_")));
#else
void flexiblas_real_dsyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dsyr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_dsyr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsyr++;
    if ( hook_pos_dsyr < __flexiblas_hooks->dsyr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsyr.hook_function[hook_pos_dsyr];
    } else {
        hook_pos_dsyr = 0;
        *(void **) &fn = current_backend->blas.dsyr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dsyr_")));
#else
void flexiblas_chain_dsyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dsyr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsyr2 = 0;
#endif

/* Wrapper for dsyr2 */
void FC_GLOBAL(dsyr2,DSYR2)
    (char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsyr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsyr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsyr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsyr2,DSYR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr2,DSYR2)))));
void FC_GLOBAL3(dsyr2,DSYR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr2,DSYR2)))));
#else
void FC_GLOBAL2(dsyr2,DSYR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsyr2,DSYR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(dsyr2,DSYR2)(char *uplo, blasint *n, double *alpha, double *x, blasint *incx, double *y, blasint *incy, double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(dsyr2,DSYR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsyr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.dsyr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_dsyr2_")));
#else
void flexiblas_real_dsyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_dsyr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_dsyr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_dsyr2++;
    if ( hook_pos_dsyr2 < __flexiblas_hooks->dsyr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsyr2.hook_function[hook_pos_dsyr2];
    } else {
        hook_pos_dsyr2 = 0;
        *(void **) &fn = current_backend->blas.dsyr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_dsyr2_")));
#else
void flexiblas_chain_dsyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_dsyr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsyr2k = 0;
#endif

/* Wrapper for dsyr2k */
void FC_GLOBAL(dsyr2k,DSYR2K)
    (char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dsyr2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsyr2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsyr2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsyr2k,DSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr2k,DSYR2K)))));
void FC_GLOBAL3(dsyr2k,DSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dsyr2k,DSYR2K)))));
#else
void FC_GLOBAL2(dsyr2k,DSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dsyr2k,DSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(dsyr2k,DSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dsyr2k,DSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.dsyr2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_dsyr2k_")));
#else
void flexiblas_real_dsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_dsyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_dsyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dsyr2k++;
    if ( hook_pos_dsyr2k < __flexiblas_hooks->dsyr2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsyr2k.hook_function[hook_pos_dsyr2k];
    } else {
        hook_pos_dsyr2k = 0;
        *(void **) &fn = current_backend->blas.dsyr2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_dsyr2k_")));
#else
void flexiblas_chain_dsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_dsyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dsyrk = 0;
#endif

/* Wrapper for dsyrk */
void FC_GLOBAL(dsyrk,DSYRK)
    (char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dsyrk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dsyrk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dsyrk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dsyrk,DSYRK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dsyrk,DSYRK)))));
void FC_GLOBAL3(dsyrk,DSYRK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(dsyrk,DSYRK)))));
#else
void FC_GLOBAL2(dsyrk,DSYRK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dsyrk,DSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(dsyrk,DSYRK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, double *a, blasint *lda, double *beta, double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(dsyrk,DSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dsyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.dsyrk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_dsyrk_")));
#else
void flexiblas_real_dsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_dsyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_dsyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_dsyrk++;
    if ( hook_pos_dsyrk < __flexiblas_hooks->dsyrk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dsyrk.hook_function[hook_pos_dsyrk];
    } else {
        hook_pos_dsyrk = 0;
        *(void **) &fn = current_backend->blas.dsyrk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_dsyrk_")));
#else
void flexiblas_chain_dsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_dsyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtbmv = 0;
#endif

/* Wrapper for dtbmv */
void FC_GLOBAL(dtbmv,DTBMV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtbmv,DTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtbmv,DTBMV)))));
void FC_GLOBAL3(dtbmv,DTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtbmv,DTBMV)))));
#else
void FC_GLOBAL2(dtbmv,DTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtbmv,DTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtbmv,DTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtbmv,DTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtbmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtbmv_")));
#else
void flexiblas_real_dtbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtbmv++;
    if ( hook_pos_dtbmv < __flexiblas_hooks->dtbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtbmv.hook_function[hook_pos_dtbmv];
    } else {
        hook_pos_dtbmv = 0;
        *(void **) &fn = current_backend->blas.dtbmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtbmv_")));
#else
void flexiblas_chain_dtbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtbsv = 0;
#endif

/* Wrapper for dtbsv */
void FC_GLOBAL(dtbsv,DTBSV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtbsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtbsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtbsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtbsv,DTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtbsv,DTBSV)))));
void FC_GLOBAL3(dtbsv,DTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtbsv,DTBSV)))));
#else
void FC_GLOBAL2(dtbsv,DTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtbsv,DTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtbsv,DTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtbsv,DTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtbsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtbsv_")));
#else
void flexiblas_real_dtbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtbsv++;
    if ( hook_pos_dtbsv < __flexiblas_hooks->dtbsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtbsv.hook_function[hook_pos_dtbsv];
    } else {
        hook_pos_dtbsv = 0;
        *(void **) &fn = current_backend->blas.dtbsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtbsv_")));
#else
void flexiblas_chain_dtbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtpmv = 0;
#endif

/* Wrapper for dtpmv */
void FC_GLOBAL(dtpmv,DTPMV)
    (char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtpmv,DTPMV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtpmv,DTPMV)))));
void FC_GLOBAL3(dtpmv,DTPMV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtpmv,DTPMV)))));
#else
void FC_GLOBAL2(dtpmv,DTPMV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtpmv,DTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtpmv,DTPMV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtpmv,DTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtpmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtpmv_")));
#else
void flexiblas_real_dtpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtpmv++;
    if ( hook_pos_dtpmv < __flexiblas_hooks->dtpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtpmv.hook_function[hook_pos_dtpmv];
    } else {
        hook_pos_dtpmv = 0;
        *(void **) &fn = current_backend->blas.dtpmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtpmv_")));
#else
void flexiblas_chain_dtpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtpsv = 0;
#endif

/* Wrapper for dtpsv */
void FC_GLOBAL(dtpsv,DTPSV)
    (char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtpsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtpsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtpsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtpsv,DTPSV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtpsv,DTPSV)))));
void FC_GLOBAL3(dtpsv,DTPSV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtpsv,DTPSV)))));
#else
void FC_GLOBAL2(dtpsv,DTPSV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtpsv,DTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtpsv,DTPSV)(char *uplo, char *trans, char *diag, blasint *n, double *ap, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtpsv,DTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtpsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtpsv_")));
#else
void flexiblas_real_dtpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtpsv++;
    if ( hook_pos_dtpsv < __flexiblas_hooks->dtpsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtpsv.hook_function[hook_pos_dtpsv];
    } else {
        hook_pos_dtpsv = 0;
        *(void **) &fn = current_backend->blas.dtpsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtpsv_")));
#else
void flexiblas_chain_dtpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtrmm = 0;
#endif

/* Wrapper for dtrmm */
void FC_GLOBAL(dtrmm,DTRMM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrmm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtrmm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtrmm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtrmm,DTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrmm,DTRMM)))));
void FC_GLOBAL3(dtrmm,DTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrmm,DTRMM)))));
#else
void FC_GLOBAL2(dtrmm,DTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrmm,DTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(dtrmm,DTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrmm,DTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtrmm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtrmm_")));
#else
void flexiblas_real_dtrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrmm++;
    if ( hook_pos_dtrmm < __flexiblas_hooks->dtrmm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtrmm.hook_function[hook_pos_dtrmm];
    } else {
        hook_pos_dtrmm = 0;
        *(void **) &fn = current_backend->blas.dtrmm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtrmm_")));
#else
void flexiblas_chain_dtrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtrmv = 0;
#endif

/* Wrapper for dtrmv */
void FC_GLOBAL(dtrmv,DTRMV)
    (char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtrmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtrmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtrmv,DTRMV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrmv,DTRMV)))));
void FC_GLOBAL3(dtrmv,DTRMV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrmv,DTRMV)))));
#else
void FC_GLOBAL2(dtrmv,DTRMV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrmv,DTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtrmv,DTRMV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrmv,DTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtrmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtrmv_")));
#else
void flexiblas_real_dtrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrmv++;
    if ( hook_pos_dtrmv < __flexiblas_hooks->dtrmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtrmv.hook_function[hook_pos_dtrmv];
    } else {
        hook_pos_dtrmv = 0;
        *(void **) &fn = current_backend->blas.dtrmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtrmv_")));
#else
void flexiblas_chain_dtrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtrsm = 0;
#endif

/* Wrapper for dtrsm */
void FC_GLOBAL(dtrsm,DTRSM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrsm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtrsm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtrsm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtrsm,DTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrsm,DTRSM)))));
void FC_GLOBAL3(dtrsm,DTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrsm,DTRSM)))));
#else
void FC_GLOBAL2(dtrsm,DTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrsm,DTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(dtrsm,DTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, double *alpha, double *a, blasint *lda, double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrsm,DTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtrsm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtrsm_")));
#else
void flexiblas_real_dtrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrsm++;
    if ( hook_pos_dtrsm < __flexiblas_hooks->dtrsm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtrsm.hook_function[hook_pos_dtrsm];
    } else {
        hook_pos_dtrsm = 0;
        *(void **) &fn = current_backend->blas.dtrsm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtrsm_")));
#else
void flexiblas_chain_dtrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dtrsv = 0;
#endif

/* Wrapper for dtrsv */
void FC_GLOBAL(dtrsv,DTRSV)
    (char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dtrsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dtrsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(dtrsv,DTRSV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrsv,DTRSV)))));
void FC_GLOBAL3(dtrsv,DTRSV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(dtrsv,DTRSV)))));
#else
void FC_GLOBAL2(dtrsv,DTRSV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrsv,DTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(dtrsv,DTRSV)(char *uplo, char *trans, char *diag, blasint *n, double *a, blasint *lda, double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(dtrsv,DTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_dtrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.dtrsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_dtrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_dtrsv_")));
#else
void flexiblas_real_dtrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_dtrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_dtrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_dtrsv++;
    if ( hook_pos_dtrsv < __flexiblas_hooks->dtrsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dtrsv.hook_function[hook_pos_dtrsv];
    } else {
        hook_pos_dtrsv = 0;
        *(void **) &fn = current_backend->blas.dtrsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_dtrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_dtrsv_")));
#else
void flexiblas_chain_dtrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_dtrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dzasum = 0;
#endif

/* Wrapper for dzasum */
double FC_GLOBAL(dzasum,DZASUM)
    (blasint *n, blas_complex_double *zx, blasint *incx)
{
    double (*fn) (void *n, void *zx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *zx, void *incx);
    hook_pos_dzasum = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dzasum;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dzasum.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) zx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) zx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) zx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(dzasum,DZASUM)(blasint *n, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dzasum,DZASUM)))));
double FC_GLOBAL3(dzasum,DZASUM)(blasint *n, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dzasum,DZASUM)))));
#else
double FC_GLOBAL2(dzasum,DZASUM)(blasint *n, blas_complex_double *zx, blasint *incx)
{
    return FC_GLOBAL(dzasum,DZASUM)((void *) n, (void *) zx, (void *) incx);
}
double FC_GLOBAL3(dzasum,DZASUM)(blasint *n, blas_complex_double *zx, blasint *incx)
{
    return FC_GLOBAL(dzasum,DZASUM)((void *) n, (void *) zx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_dzasum_(void *n, void *zx, void *incx)
{
    double (*fn) (void *n, void *zx, void *incx);
    double ret;
    *(void **) &fn = current_backend->blas.dzasum;
    ret = fn((void *) n, (void *) zx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_dzasum(void *n, void *zx, void *incx) __attribute__((alias("flexiblas_real_dzasum_")));
#else
double flexiblas_real_dzasum(void *n, void *zx, void *incx)
{
    return flexiblas_real_dzasum_((void *) n, (void *) zx, (void *) incx);
}
#endif

double flexiblas_chain_dzasum_(void *n, void *zx, void *incx)
{
    double (*fn) (void *n, void *zx, void *incx);
    double ret;
    hook_pos_dzasum++;
    if ( hook_pos_dzasum < __flexiblas_hooks->dzasum.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dzasum.hook_function[hook_pos_dzasum];
    } else {
        hook_pos_dzasum = 0;
        *(void **) &fn = current_backend->blas.dzasum;
    }
    ret = fn((void *) n, (void *) zx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_dzasum(void *n, void *zx, void *incx) __attribute__((alias("flexiblas_chain_dzasum_")));
#else
double flexiblas_chain_dzasum(void *n, void *zx, void *incx)
{
    return flexiblas_chain_dzasum_((void *) n, (void *) zx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_dznrm2 = 0;
#endif

/* Wrapper for dznrm2 */
double FC_GLOBAL(dznrm2,DZNRM2)
    (blasint *n, blas_complex_double *x, blasint *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    double (*fn_hook) (void *n, void *x, void *incx);
    hook_pos_dznrm2 = 0;
    #endif
    double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.dznrm2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->dznrm2.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) x, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) x, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) x, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
double FC_GLOBAL2(dznrm2,DZNRM2)(blasint *n, blas_complex_double *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dznrm2,DZNRM2)))));
double FC_GLOBAL3(dznrm2,DZNRM2)(blasint *n, blas_complex_double *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(dznrm2,DZNRM2)))));
#else
double FC_GLOBAL2(dznrm2,DZNRM2)(blasint *n, blas_complex_double *x, blasint *incx)
{
    return FC_GLOBAL(dznrm2,DZNRM2)((void *) n, (void *) x, (void *) incx);
}
double FC_GLOBAL3(dznrm2,DZNRM2)(blasint *n, blas_complex_double *x, blasint *incx)
{
    return FC_GLOBAL(dznrm2,DZNRM2)((void *) n, (void *) x, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
double flexiblas_real_dznrm2_(void *n, void *x, void *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    double ret;
    *(void **) &fn = current_backend->blas.dznrm2;
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_real_dznrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_real_dznrm2_")));
#else
double flexiblas_real_dznrm2(void *n, void *x, void *incx)
{
    return flexiblas_real_dznrm2_((void *) n, (void *) x, (void *) incx);
}
#endif

double flexiblas_chain_dznrm2_(void *n, void *x, void *incx)
{
    double (*fn) (void *n, void *x, void *incx);
    double ret;
    hook_pos_dznrm2++;
    if ( hook_pos_dznrm2 < __flexiblas_hooks->dznrm2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->dznrm2.hook_function[hook_pos_dznrm2];
    } else {
        hook_pos_dznrm2 = 0;
        *(void **) &fn = current_backend->blas.dznrm2;
    }
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
double flexiblas_chain_dznrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_chain_dznrm2_")));
#else
double flexiblas_chain_dznrm2(void *n, void *x, void *incx)
{
    return flexiblas_chain_dznrm2_((void *) n, (void *) x, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_icamax = 0;
#endif

/* Wrapper for icamax */
blasint FC_GLOBAL(icamax,ICAMAX)
    (blasint *n, blas_complex_float *cx, blasint *incx)
{
    blasint (*fn) (void *n, void *cx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    blasint (*fn_hook) (void *n, void *cx, void *incx);
    hook_pos_icamax = 0;
    #endif
    blasint ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.icamax;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->icamax.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) cx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) cx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) cx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
blasint FC_GLOBAL2(icamax,ICAMAX)(blasint *n, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(icamax,ICAMAX)))));
blasint FC_GLOBAL3(icamax,ICAMAX)(blasint *n, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(icamax,ICAMAX)))));
#else
blasint FC_GLOBAL2(icamax,ICAMAX)(blasint *n, blas_complex_float *cx, blasint *incx)
{
    return FC_GLOBAL(icamax,ICAMAX)((void *) n, (void *) cx, (void *) incx);
}
blasint FC_GLOBAL3(icamax,ICAMAX)(blasint *n, blas_complex_float *cx, blasint *incx)
{
    return FC_GLOBAL(icamax,ICAMAX)((void *) n, (void *) cx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
blasint flexiblas_real_icamax_(void *n, void *cx, void *incx)
{
    blasint (*fn) (void *n, void *cx, void *incx);
    blasint ret;
    *(void **) &fn = current_backend->blas.icamax;
    ret = fn((void *) n, (void *) cx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_real_icamax(void *n, void *cx, void *incx) __attribute__((alias("flexiblas_real_icamax_")));
#else
blasint flexiblas_real_icamax(void *n, void *cx, void *incx)
{
    return flexiblas_real_icamax_((void *) n, (void *) cx, (void *) incx);
}
#endif

blasint flexiblas_chain_icamax_(void *n, void *cx, void *incx)
{
    blasint (*fn) (void *n, void *cx, void *incx);
    blasint ret;
    hook_pos_icamax++;
    if ( hook_pos_icamax < __flexiblas_hooks->icamax.nhook ) {
        *(void **) &fn = __flexiblas_hooks->icamax.hook_function[hook_pos_icamax];
    } else {
        hook_pos_icamax = 0;
        *(void **) &fn = current_backend->blas.icamax;
    }
    ret = fn((void *) n, (void *) cx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_chain_icamax(void *n, void *cx, void *incx) __attribute__((alias("flexiblas_chain_icamax_")));
#else
blasint flexiblas_chain_icamax(void *n, void *cx, void *incx)
{
    return flexiblas_chain_icamax_((void *) n, (void *) cx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_idamax = 0;
#endif

/* Wrapper for idamax */
blasint FC_GLOBAL(idamax,IDAMAX)
    (blasint *n, double *dx, blasint *incx)
{
    blasint (*fn) (void *n, void *dx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    blasint (*fn_hook) (void *n, void *dx, void *incx);
    hook_pos_idamax = 0;
    #endif
    blasint ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.idamax;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->idamax.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) dx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) dx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) dx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
blasint FC_GLOBAL2(idamax,IDAMAX)(blasint *n, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(idamax,IDAMAX)))));
blasint FC_GLOBAL3(idamax,IDAMAX)(blasint *n, double *dx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(idamax,IDAMAX)))));
#else
blasint FC_GLOBAL2(idamax,IDAMAX)(blasint *n, double *dx, blasint *incx)
{
    return FC_GLOBAL(idamax,IDAMAX)((void *) n, (void *) dx, (void *) incx);
}
blasint FC_GLOBAL3(idamax,IDAMAX)(blasint *n, double *dx, blasint *incx)
{
    return FC_GLOBAL(idamax,IDAMAX)((void *) n, (void *) dx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
blasint flexiblas_real_idamax_(void *n, void *dx, void *incx)
{
    blasint (*fn) (void *n, void *dx, void *incx);
    blasint ret;
    *(void **) &fn = current_backend->blas.idamax;
    ret = fn((void *) n, (void *) dx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_real_idamax(void *n, void *dx, void *incx) __attribute__((alias("flexiblas_real_idamax_")));
#else
blasint flexiblas_real_idamax(void *n, void *dx, void *incx)
{
    return flexiblas_real_idamax_((void *) n, (void *) dx, (void *) incx);
}
#endif

blasint flexiblas_chain_idamax_(void *n, void *dx, void *incx)
{
    blasint (*fn) (void *n, void *dx, void *incx);
    blasint ret;
    hook_pos_idamax++;
    if ( hook_pos_idamax < __flexiblas_hooks->idamax.nhook ) {
        *(void **) &fn = __flexiblas_hooks->idamax.hook_function[hook_pos_idamax];
    } else {
        hook_pos_idamax = 0;
        *(void **) &fn = current_backend->blas.idamax;
    }
    ret = fn((void *) n, (void *) dx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_chain_idamax(void *n, void *dx, void *incx) __attribute__((alias("flexiblas_chain_idamax_")));
#else
blasint flexiblas_chain_idamax(void *n, void *dx, void *incx)
{
    return flexiblas_chain_idamax_((void *) n, (void *) dx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_isamax = 0;
#endif

/* Wrapper for isamax */
blasint FC_GLOBAL(isamax,ISAMAX)
    (blasint *n, float *sx, blasint *incx)
{
    blasint (*fn) (void *n, void *sx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    blasint (*fn_hook) (void *n, void *sx, void *incx);
    hook_pos_isamax = 0;
    #endif
    blasint ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.isamax;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->isamax.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) sx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) sx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) sx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
blasint FC_GLOBAL2(isamax,ISAMAX)(blasint *n, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(isamax,ISAMAX)))));
blasint FC_GLOBAL3(isamax,ISAMAX)(blasint *n, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(isamax,ISAMAX)))));
#else
blasint FC_GLOBAL2(isamax,ISAMAX)(blasint *n, float *sx, blasint *incx)
{
    return FC_GLOBAL(isamax,ISAMAX)((void *) n, (void *) sx, (void *) incx);
}
blasint FC_GLOBAL3(isamax,ISAMAX)(blasint *n, float *sx, blasint *incx)
{
    return FC_GLOBAL(isamax,ISAMAX)((void *) n, (void *) sx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
blasint flexiblas_real_isamax_(void *n, void *sx, void *incx)
{
    blasint (*fn) (void *n, void *sx, void *incx);
    blasint ret;
    *(void **) &fn = current_backend->blas.isamax;
    ret = fn((void *) n, (void *) sx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_real_isamax(void *n, void *sx, void *incx) __attribute__((alias("flexiblas_real_isamax_")));
#else
blasint flexiblas_real_isamax(void *n, void *sx, void *incx)
{
    return flexiblas_real_isamax_((void *) n, (void *) sx, (void *) incx);
}
#endif

blasint flexiblas_chain_isamax_(void *n, void *sx, void *incx)
{
    blasint (*fn) (void *n, void *sx, void *incx);
    blasint ret;
    hook_pos_isamax++;
    if ( hook_pos_isamax < __flexiblas_hooks->isamax.nhook ) {
        *(void **) &fn = __flexiblas_hooks->isamax.hook_function[hook_pos_isamax];
    } else {
        hook_pos_isamax = 0;
        *(void **) &fn = current_backend->blas.isamax;
    }
    ret = fn((void *) n, (void *) sx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_chain_isamax(void *n, void *sx, void *incx) __attribute__((alias("flexiblas_chain_isamax_")));
#else
blasint flexiblas_chain_isamax(void *n, void *sx, void *incx)
{
    return flexiblas_chain_isamax_((void *) n, (void *) sx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_izamax = 0;
#endif

/* Wrapper for izamax */
blasint FC_GLOBAL(izamax,IZAMAX)
    (blasint *n, blas_complex_double *zx, blasint *incx)
{
    blasint (*fn) (void *n, void *zx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    blasint (*fn_hook) (void *n, void *zx, void *incx);
    hook_pos_izamax = 0;
    #endif
    blasint ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.izamax;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->izamax.hook_function[0];

    if ( fn_hook != NULL ) {
        ret = fn_hook((void *) n, (void *) zx, (void *) incx);
    } else {
        ret = fn((void *) n, (void *) zx, (void *) incx);
    }
    #else
    ret = fn((void *) n, (void *) zx, (void *) incx);
    #endif
    return ret;
}

#ifndef __APPLE__
blasint FC_GLOBAL2(izamax,IZAMAX)(blasint *n, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(izamax,IZAMAX)))));
blasint FC_GLOBAL3(izamax,IZAMAX)(blasint *n, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(izamax,IZAMAX)))));
#else
blasint FC_GLOBAL2(izamax,IZAMAX)(blasint *n, blas_complex_double *zx, blasint *incx)
{
    return FC_GLOBAL(izamax,IZAMAX)((void *) n, (void *) zx, (void *) incx);
}
blasint FC_GLOBAL3(izamax,IZAMAX)(blasint *n, blas_complex_double *zx, blasint *incx)
{
    return FC_GLOBAL(izamax,IZAMAX)((void *) n, (void *) zx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
blasint flexiblas_real_izamax_(void *n, void *zx, void *incx)
{
    blasint (*fn) (void *n, void *zx, void *incx);
    blasint ret;
    *(void **) &fn = current_backend->blas.izamax;
    ret = fn((void *) n, (void *) zx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_real_izamax(void *n, void *zx, void *incx) __attribute__((alias("flexiblas_real_izamax_")));
#else
blasint flexiblas_real_izamax(void *n, void *zx, void *incx)
{
    return flexiblas_real_izamax_((void *) n, (void *) zx, (void *) incx);
}
#endif

blasint flexiblas_chain_izamax_(void *n, void *zx, void *incx)
{
    blasint (*fn) (void *n, void *zx, void *incx);
    blasint ret;
    hook_pos_izamax++;
    if ( hook_pos_izamax < __flexiblas_hooks->izamax.nhook ) {
        *(void **) &fn = __flexiblas_hooks->izamax.hook_function[hook_pos_izamax];
    } else {
        hook_pos_izamax = 0;
        *(void **) &fn = current_backend->blas.izamax;
    }
    ret = fn((void *) n, (void *) zx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
blasint flexiblas_chain_izamax(void *n, void *zx, void *incx) __attribute__((alias("flexiblas_chain_izamax_")));
#else
blasint flexiblas_chain_izamax(void *n, void *zx, void *incx)
{
    return flexiblas_chain_izamax_((void *) n, (void *) zx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sasum = 0;
#endif

/* Wrapper for sasum */
float FC_GLOBAL(sasum,SASUM)
    (blasint *n, float *sx, blasint *incx)
{
    float (*fn) (void *n, void *sx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *sx, void *incx);
    hook_pos_sasum = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *sx, void *incx);
         *(void **) &fn_double = current_backend->blas.sasum;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sasum.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sx, (void *) incx);
         } else {
             ret = (float) fn_double((void *) n, (void *) sx, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) sx, (void *) incx);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.sasum;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sasum.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sx, (void *) incx);
         } else {
             ret = fn((void *) n, (void *) sx, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) sx, (void *) incx);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(sasum,SASUM)(blasint *n, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(sasum,SASUM)))));
float FC_GLOBAL3(sasum,SASUM)(blasint *n, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(sasum,SASUM)))));
#else
float FC_GLOBAL2(sasum,SASUM)(blasint *n, float *sx, blasint *incx)
{
    return FC_GLOBAL(sasum,SASUM)((void *) n, (void *) sx, (void *) incx);
}
float FC_GLOBAL3(sasum,SASUM)(blasint *n, float *sx, blasint *incx)
{
    return FC_GLOBAL(sasum,SASUM)((void *) n, (void *) sx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_sasum_(void *n, void *sx, void *incx)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *sx, void *incx);
        float ret;
        *(void **) &fn_double = current_backend->blas.sasum;
        ret = (float) fn_double((void *) n, (void *) sx, (void *) incx);
        return ret;
    }
    float (*fn) (void *n, void *sx, void *incx);
    float ret;
    *(void **) &fn = current_backend->blas.sasum;
    ret = fn((void *) n, (void *) sx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_sasum(void *n, void *sx, void *incx) __attribute__((alias("flexiblas_real_sasum_")));
#else
float flexiblas_real_sasum(void *n, void *sx, void *incx)
{
    return flexiblas_real_sasum_((void *) n, (void *) sx, (void *) incx);
}
#endif

float flexiblas_chain_sasum_(void *n, void *sx, void *incx)
{
    float (*fn) (void *n, void *sx, void *incx);
    float ret;
    hook_pos_sasum++;
    if ( hook_pos_sasum < __flexiblas_hooks->sasum.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sasum.hook_function[hook_pos_sasum];
        ret = fn((void *) n, (void *) sx, (void *) incx);
    } else {
        hook_pos_sasum = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.sasum;
            ret = fn((void *) n, (void *) sx, (void *) incx);
        } else {
            double (*fn_double) (void *n, void *sx, void *incx);
            *(void **) &fn_double = current_backend->blas.sasum;
            ret = (float) fn_double((void *) n, (void *) sx, (void *) incx);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_sasum(void *n, void *sx, void *incx) __attribute__((alias("flexiblas_chain_sasum_")));
#else
float flexiblas_chain_sasum(void *n, void *sx, void *incx)
{
    return flexiblas_chain_sasum_((void *) n, (void *) sx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_saxpby = 0;
#endif

/* Wrapper for saxpby */
void FC_GLOBAL(saxpby,SAXPBY)
    (blasint *n, float *sa, float *sx, blasint *incx, float *sb, float *sy, blasint *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy);
    hook_pos_saxpby = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.saxpby;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->saxpby.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    } else {
        fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(saxpby,SAXPBY)(blasint *n, float *sa, float *sx, blasint *incx, float *sb, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(saxpby,SAXPBY)))));
void FC_GLOBAL3(saxpby,SAXPBY)(blasint *n, float *sa, float *sx, blasint *incx, float *sb, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(saxpby,SAXPBY)))));
#else
void FC_GLOBAL2(saxpby,SAXPBY)(blasint *n, float *sa, float *sx, blasint *incx, float *sb, float *sy, blasint *incy)
{
    FC_GLOBAL(saxpby,SAXPBY)((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
}
void FC_GLOBAL3(saxpby,SAXPBY)(blasint *n, float *sa, float *sx, blasint *incx, float *sb, float *sy, blasint *incy)
{
    FC_GLOBAL(saxpby,SAXPBY)((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_saxpby_(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy);
    *(void **) &fn = current_backend->blas.saxpby;
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_saxpby(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy) __attribute__((alias("flexiblas_real_saxpby_")));
#else
void flexiblas_real_saxpby(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy)
{
    flexiblas_real_saxpby_((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_saxpby_(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy);
    hook_pos_saxpby++;
    if ( hook_pos_saxpby < __flexiblas_hooks->saxpby.nhook ) {
        *(void **) &fn = __flexiblas_hooks->saxpby.hook_function[hook_pos_saxpby];
    } else {
        hook_pos_saxpby = 0;
        *(void **) &fn = current_backend->blas.saxpby;
    }
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_saxpby(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy) __attribute__((alias("flexiblas_chain_saxpby_")));
#else
void flexiblas_chain_saxpby(void *n, void *sa, void *sx, void *incx, void *sb, void *sy, void *incy)
{
    flexiblas_chain_saxpby_((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sb, (void *) sy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_saxpy = 0;
#endif

/* Wrapper for saxpy */
void FC_GLOBAL(saxpy,SAXPY)
    (blasint *n, float *sa, float *sx, blasint *incx, float *sy, blasint *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sa, void *sx, void *incx, void *sy, void *incy);
    hook_pos_saxpy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.saxpy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->saxpy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(saxpy,SAXPY)(blasint *n, float *sa, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(saxpy,SAXPY)))));
void FC_GLOBAL3(saxpy,SAXPY)(blasint *n, float *sa, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(saxpy,SAXPY)))));
#else
void FC_GLOBAL2(saxpy,SAXPY)(blasint *n, float *sa, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(saxpy,SAXPY)((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
void FC_GLOBAL3(saxpy,SAXPY)(blasint *n, float *sa, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(saxpy,SAXPY)((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_saxpy_(void *n, void *sa, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sy, void *incy);
    *(void **) &fn = current_backend->blas.saxpy;
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_saxpy(void *n, void *sa, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_saxpy_")));
#else
void flexiblas_real_saxpy(void *n, void *sa, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_real_saxpy_((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_saxpy_(void *n, void *sa, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx, void *sy, void *incy);
    hook_pos_saxpy++;
    if ( hook_pos_saxpy < __flexiblas_hooks->saxpy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->saxpy.hook_function[hook_pos_saxpy];
    } else {
        hook_pos_saxpy = 0;
        *(void **) &fn = current_backend->blas.saxpy;
    }
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_saxpy(void *n, void *sa, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_saxpy_")));
#else
void flexiblas_chain_saxpy(void *n, void *sa, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_chain_saxpy_((void *) n, (void *) sa, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_scasum = 0;
#endif

/* Wrapper for scasum */
float FC_GLOBAL(scasum,SCASUM)
    (blasint *n, blas_complex_float *cx, blasint *incx)
{
    float (*fn) (void *n, void *cx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *cx, void *incx);
    hook_pos_scasum = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *cx, void *incx);
         *(void **) &fn_double = current_backend->blas.scasum;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->scasum.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) cx, (void *) incx);
         } else {
             ret = (float) fn_double((void *) n, (void *) cx, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) cx, (void *) incx);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.scasum;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->scasum.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) cx, (void *) incx);
         } else {
             ret = fn((void *) n, (void *) cx, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) cx, (void *) incx);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(scasum,SCASUM)(blasint *n, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(scasum,SCASUM)))));
float FC_GLOBAL3(scasum,SCASUM)(blasint *n, blas_complex_float *cx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(scasum,SCASUM)))));
#else
float FC_GLOBAL2(scasum,SCASUM)(blasint *n, blas_complex_float *cx, blasint *incx)
{
    return FC_GLOBAL(scasum,SCASUM)((void *) n, (void *) cx, (void *) incx);
}
float FC_GLOBAL3(scasum,SCASUM)(blasint *n, blas_complex_float *cx, blasint *incx)
{
    return FC_GLOBAL(scasum,SCASUM)((void *) n, (void *) cx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_scasum_(void *n, void *cx, void *incx)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *cx, void *incx);
        float ret;
        *(void **) &fn_double = current_backend->blas.scasum;
        ret = (float) fn_double((void *) n, (void *) cx, (void *) incx);
        return ret;
    }
    float (*fn) (void *n, void *cx, void *incx);
    float ret;
    *(void **) &fn = current_backend->blas.scasum;
    ret = fn((void *) n, (void *) cx, (void *) incx);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_scasum(void *n, void *cx, void *incx) __attribute__((alias("flexiblas_real_scasum_")));
#else
float flexiblas_real_scasum(void *n, void *cx, void *incx)
{
    return flexiblas_real_scasum_((void *) n, (void *) cx, (void *) incx);
}
#endif

float flexiblas_chain_scasum_(void *n, void *cx, void *incx)
{
    float (*fn) (void *n, void *cx, void *incx);
    float ret;
    hook_pos_scasum++;
    if ( hook_pos_scasum < __flexiblas_hooks->scasum.nhook ) {
        *(void **) &fn = __flexiblas_hooks->scasum.hook_function[hook_pos_scasum];
        ret = fn((void *) n, (void *) cx, (void *) incx);
    } else {
        hook_pos_scasum = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.scasum;
            ret = fn((void *) n, (void *) cx, (void *) incx);
        } else {
            double (*fn_double) (void *n, void *cx, void *incx);
            *(void **) &fn_double = current_backend->blas.scasum;
            ret = (float) fn_double((void *) n, (void *) cx, (void *) incx);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_scasum(void *n, void *cx, void *incx) __attribute__((alias("flexiblas_chain_scasum_")));
#else
float flexiblas_chain_scasum(void *n, void *cx, void *incx)
{
    return flexiblas_chain_scasum_((void *) n, (void *) cx, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_scnrm2 = 0;
#endif

/* Wrapper for scnrm2 */
float FC_GLOBAL(scnrm2,SCNRM2)
    (blasint *n, blas_complex_float *x, blasint *incx)
{
    float (*fn) (void *n, void *x, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *x, void *incx);
    hook_pos_scnrm2 = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *x, void *incx);
         *(void **) &fn_double = current_backend->blas.scnrm2;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->scnrm2.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) x, (void *) incx);
         } else {
             ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) x, (void *) incx);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.scnrm2;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->scnrm2.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) x, (void *) incx);
         } else {
             ret = fn((void *) n, (void *) x, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) x, (void *) incx);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(scnrm2,SCNRM2)(blasint *n, blas_complex_float *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(scnrm2,SCNRM2)))));
float FC_GLOBAL3(scnrm2,SCNRM2)(blasint *n, blas_complex_float *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(scnrm2,SCNRM2)))));
#else
float FC_GLOBAL2(scnrm2,SCNRM2)(blasint *n, blas_complex_float *x, blasint *incx)
{
    return FC_GLOBAL(scnrm2,SCNRM2)((void *) n, (void *) x, (void *) incx);
}
float FC_GLOBAL3(scnrm2,SCNRM2)(blasint *n, blas_complex_float *x, blasint *incx)
{
    return FC_GLOBAL(scnrm2,SCNRM2)((void *) n, (void *) x, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_scnrm2_(void *n, void *x, void *incx)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *x, void *incx);
        float ret;
        *(void **) &fn_double = current_backend->blas.scnrm2;
        ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
        return ret;
    }
    float (*fn) (void *n, void *x, void *incx);
    float ret;
    *(void **) &fn = current_backend->blas.scnrm2;
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_scnrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_real_scnrm2_")));
#else
float flexiblas_real_scnrm2(void *n, void *x, void *incx)
{
    return flexiblas_real_scnrm2_((void *) n, (void *) x, (void *) incx);
}
#endif

float flexiblas_chain_scnrm2_(void *n, void *x, void *incx)
{
    float (*fn) (void *n, void *x, void *incx);
    float ret;
    hook_pos_scnrm2++;
    if ( hook_pos_scnrm2 < __flexiblas_hooks->scnrm2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->scnrm2.hook_function[hook_pos_scnrm2];
        ret = fn((void *) n, (void *) x, (void *) incx);
    } else {
        hook_pos_scnrm2 = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.scnrm2;
            ret = fn((void *) n, (void *) x, (void *) incx);
        } else {
            double (*fn_double) (void *n, void *x, void *incx);
            *(void **) &fn_double = current_backend->blas.scnrm2;
            ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_scnrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_chain_scnrm2_")));
#else
float flexiblas_chain_scnrm2(void *n, void *x, void *incx)
{
    return flexiblas_chain_scnrm2_((void *) n, (void *) x, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_scopy = 0;
#endif

/* Wrapper for scopy */
void FC_GLOBAL(scopy,SCOPY)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_scopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.scopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->scopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(scopy,SCOPY)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(scopy,SCOPY)))));
void FC_GLOBAL3(scopy,SCOPY)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(scopy,SCOPY)))));
#else
void FC_GLOBAL2(scopy,SCOPY)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(scopy,SCOPY)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
void FC_GLOBAL3(scopy,SCOPY)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(scopy,SCOPY)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_scopy_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    *(void **) &fn = current_backend->blas.scopy;
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_scopy(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_scopy_")));
#else
void flexiblas_real_scopy(void *n, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_real_scopy_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_scopy_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_scopy++;
    if ( hook_pos_scopy < __flexiblas_hooks->scopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->scopy.hook_function[hook_pos_scopy];
    } else {
        hook_pos_scopy = 0;
        *(void **) &fn = current_backend->blas.scopy;
    }
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_scopy(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_scopy_")));
#else
void flexiblas_chain_scopy(void *n, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_chain_scopy_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sdot = 0;
#endif

/* Wrapper for sdot */
float FC_GLOBAL(sdot,SDOT)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    float (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_sdot = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *sx, void *incx, void *sy, void *incy);
         *(void **) &fn_double = current_backend->blas.sdot;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sdot.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         } else {
             ret = (float) fn_double((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         }
         #else
         ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.sdot;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sdot.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         } else {
             ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         }
         #else
         ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(sdot,SDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sdot,SDOT)))));
float FC_GLOBAL3(sdot,SDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sdot,SDOT)))));
#else
float FC_GLOBAL2(sdot,SDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(sdot,SDOT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
float FC_GLOBAL3(sdot,SDOT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(sdot,SDOT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_sdot_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *sx, void *incx, void *sy, void *incy);
        float ret;
        *(void **) &fn_double = current_backend->blas.sdot;
        ret = (float) fn_double((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        return ret;
    }
    float (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    float ret;
    *(void **) &fn = current_backend->blas.sdot;
    ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_sdot(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_sdot_")));
#else
float flexiblas_real_sdot(void *n, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_real_sdot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif

float flexiblas_chain_sdot_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    float (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    float ret;
    hook_pos_sdot++;
    if ( hook_pos_sdot < __flexiblas_hooks->sdot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sdot.hook_function[hook_pos_sdot];
        ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        hook_pos_sdot = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.sdot;
            ret = fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        } else {
            double (*fn_double) (void *n, void *sx, void *incx, void *sy, void *incy);
            *(void **) &fn_double = current_backend->blas.sdot;
            ret = (float) fn_double((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_sdot(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_sdot_")));
#else
float flexiblas_chain_sdot(void *n, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_chain_sdot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sdsdot = 0;
#endif

/* Wrapper for sdsdot */
float FC_GLOBAL(sdsdot,SDSDOT)
    (blasint *n, float *sb, float *sx, blasint *incx, float *sy, blasint *incy)
{
    float (*fn) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
    hook_pos_sdsdot = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
         *(void **) &fn_double = current_backend->blas.sdsdot;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sdsdot.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         } else {
             ret = (float) fn_double((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         }
         #else
         ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.sdsdot;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->sdsdot.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         } else {
             ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         }
         #else
         ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(sdsdot,SDSDOT)(blasint *n, float *sb, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sdsdot,SDSDOT)))));
float FC_GLOBAL3(sdsdot,SDSDOT)(blasint *n, float *sb, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sdsdot,SDSDOT)))));
#else
float FC_GLOBAL2(sdsdot,SDSDOT)(blasint *n, float *sb, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(sdsdot,SDSDOT)((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
float FC_GLOBAL3(sdsdot,SDSDOT)(blasint *n, float *sb, float *sx, blasint *incx, float *sy, blasint *incy)
{
    return FC_GLOBAL(sdsdot,SDSDOT)((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_sdsdot_(void *n, void *sb, void *sx, void *incx, void *sy, void *incy)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
        float ret;
        *(void **) &fn_double = current_backend->blas.sdsdot;
        ret = (float) fn_double((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        return ret;
    }
    float (*fn) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
    float ret;
    *(void **) &fn = current_backend->blas.sdsdot;
    ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_sdsdot(void *n, void *sb, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_sdsdot_")));
#else
float flexiblas_real_sdsdot(void *n, void *sb, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_real_sdsdot_((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif

float flexiblas_chain_sdsdot_(void *n, void *sb, void *sx, void *incx, void *sy, void *incy)
{
    float (*fn) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
    float ret;
    hook_pos_sdsdot++;
    if ( hook_pos_sdsdot < __flexiblas_hooks->sdsdot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sdsdot.hook_function[hook_pos_sdsdot];
        ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        hook_pos_sdsdot = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.sdsdot;
            ret = fn((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        } else {
            double (*fn_double) (void *n, void *sb, void *sx, void *incx, void *sy, void *incy);
            *(void **) &fn_double = current_backend->blas.sdsdot;
            ret = (float) fn_double((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_sdsdot(void *n, void *sb, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_sdsdot_")));
#else
float flexiblas_chain_sdsdot(void *n, void *sb, void *sx, void *incx, void *sy, void *incy)
{
    return flexiblas_chain_sdsdot_((void *) n, (void *) sb, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sgbmv = 0;
#endif

/* Wrapper for sgbmv */
void FC_GLOBAL(sgbmv,SGBMV)
    (char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_sgbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sgbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sgbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sgbmv,SGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(sgbmv,SGBMV)))));
void FC_GLOBAL3(sgbmv,SGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(sgbmv,SGBMV)))));
#else
void FC_GLOBAL2(sgbmv,SGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(sgbmv,SGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(sgbmv,SGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(sgbmv,SGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.sgbmv;
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_sgbmv_")));
#else
void flexiblas_real_sgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_sgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_sgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_sgbmv++;
    if ( hook_pos_sgbmv < __flexiblas_hooks->sgbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sgbmv.hook_function[hook_pos_sgbmv];
    } else {
        hook_pos_sgbmv = 0;
        *(void **) &fn = current_backend->blas.sgbmv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_sgbmv_")));
#else
void flexiblas_chain_sgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_sgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sgeadd = 0;
#endif

/* Wrapper for sgeadd */
void FC_GLOBAL(sgeadd,SGEADD)
    (blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *beta, float *b, blasint *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_sgeadd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sgeadd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sgeadd.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sgeadd,SGEADD)(blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *beta, float *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(sgeadd,SGEADD)))));
void FC_GLOBAL3(sgeadd,SGEADD)(blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *beta, float *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(sgeadd,SGEADD)))));
#else
void FC_GLOBAL2(sgeadd,SGEADD)(blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *beta, float *b, blasint *ldb)
{
    FC_GLOBAL(sgeadd,SGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
void FC_GLOBAL3(sgeadd,SGEADD)(blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *beta, float *b, blasint *ldb)
{
    FC_GLOBAL(sgeadd,SGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    *(void **) &fn = current_backend->blas.sgeadd;
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_real_sgeadd_")));
#else
void flexiblas_real_sgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_real_sgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif

void flexiblas_chain_sgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_sgeadd++;
    if ( hook_pos_sgeadd < __flexiblas_hooks->sgeadd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sgeadd.hook_function[hook_pos_sgeadd];
    } else {
        hook_pos_sgeadd = 0;
        *(void **) &fn = current_backend->blas.sgeadd;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_chain_sgeadd_")));
#else
void flexiblas_chain_sgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_chain_sgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sgemm = 0;
#endif

/* Wrapper for sgemm */
void FC_GLOBAL(sgemm,SGEMM)
    (char *transa, char *transb, blasint *m, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_sgemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sgemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sgemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    } else {
        fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    }
    #else
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sgemm,SGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemm,SGEMM)))));
void FC_GLOBAL3(sgemm,SGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemm,SGEMM)))));
#else
void FC_GLOBAL2(sgemm,SGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemm,SGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
void FC_GLOBAL3(sgemm,SGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemm,SGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.sgemm;
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_sgemm_")));
#else
void flexiblas_real_sgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_sgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_sgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_sgemm++;
    if ( hook_pos_sgemm < __flexiblas_hooks->sgemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sgemm.hook_function[hook_pos_sgemm];
    } else {
        hook_pos_sgemm = 0;
        *(void **) &fn = current_backend->blas.sgemm;
    }
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_sgemm_")));
#else
void flexiblas_chain_sgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_sgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sgemmtr = 0;
#endif

/* Wrapper for sgemmtr */
void FC_GLOBAL(sgemmtr,SGEMMTR)
    (char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_sgemmtr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sgemmtr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sgemmtr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    } else {
        fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    }
    #else
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sgemmtr,SGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemmtr,SGEMMTR)))));
void FC_GLOBAL3(sgemmtr,SGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemmtr,SGEMMTR)))));
#else
void FC_GLOBAL2(sgemmtr,SGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemmtr,SGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(sgemmtr,SGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemmtr,SGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.sgemmtr;
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_sgemmtr_")));
#else
void flexiblas_real_sgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_sgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_sgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_sgemmtr++;
    if ( hook_pos_sgemmtr < __flexiblas_hooks->sgemmtr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sgemmtr.hook_function[hook_pos_sgemmtr];
    } else {
        hook_pos_sgemmtr = 0;
        *(void **) &fn = current_backend->blas.sgemmtr;
    }
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_sgemmtr_")));
#else
void flexiblas_chain_sgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_sgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif
#endif
/* Alternative function name for sgemmtr -- sgemmt */
#ifndef __APPLE__
void FC_GLOBAL(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemmtr,SGEMMTR)))));
void FC_GLOBAL2(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemmtr,SGEMMTR)))));
void FC_GLOBAL3(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(sgemmtr,SGEMMTR)))));
#else
void FC_GLOBAL(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemmtr,SGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL2(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemmtr,SGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(sgemmt,SGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(sgemmtr,SGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif

#ifdef FLEXIBLAS_HOOK_API
#ifndef __APPLE__
void flexiblas_real_sgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_sgemmtr")));
void flexiblas_real_sgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_sgemmtr")));
void flexiblas_chain_sgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_sgemmtr")));
void flexiblas_chain_sgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_sgemmtr")));

#else
void flexiblas_real_sgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_sgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
void flexiblas_real_sgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_sgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_sgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_sgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_sgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_sgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sgemv = 0;
#endif

/* Wrapper for sgemv */
void FC_GLOBAL(sgemv,SGEMV)
    (char *trans, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_sgemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sgemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sgemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sgemv,SGEMV)(char *trans, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(sgemv,SGEMV)))));
void FC_GLOBAL3(sgemv,SGEMV)(char *trans, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(sgemv,SGEMV)))));
#else
void FC_GLOBAL2(sgemv,SGEMV)(char *trans, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(sgemv,SGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(sgemv,SGEMV)(char *trans, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(sgemv,SGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.sgemv;
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_sgemv_")));
#else
void flexiblas_real_sgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_sgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_sgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_sgemv++;
    if ( hook_pos_sgemv < __flexiblas_hooks->sgemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sgemv.hook_function[hook_pos_sgemv];
    } else {
        hook_pos_sgemv = 0;
        *(void **) &fn = current_backend->blas.sgemv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_sgemv_")));
#else
void flexiblas_chain_sgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_sgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sger = 0;
#endif

/* Wrapper for sger */
void FC_GLOBAL(sger,SGER)
    (blasint *m, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_sger = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sger;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sger.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sger,SGER)(blasint *m, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(sger,SGER)))));
void FC_GLOBAL3(sger,SGER)(blasint *m, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(sger,SGER)))));
#else
void FC_GLOBAL2(sger,SGER)(blasint *m, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda)
{
    FC_GLOBAL(sger,SGER)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(sger,SGER)(blasint *m, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda)
{
    FC_GLOBAL(sger,SGER)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sger_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.sger;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_sger_")));
#else
void flexiblas_real_sger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_sger_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_sger_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_sger++;
    if ( hook_pos_sger < __flexiblas_hooks->sger.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sger.hook_function[hook_pos_sger];
    } else {
        hook_pos_sger = 0;
        *(void **) &fn = current_backend->blas.sger;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_sger_")));
#else
void flexiblas_chain_sger(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_sger_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_simatcopy = 0;
#endif

/* Wrapper for simatcopy */
void FC_GLOBAL(simatcopy,SIMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_simatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.simatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->simatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(simatcopy,SIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(simatcopy,SIMATCOPY)))));
void FC_GLOBAL3(simatcopy,SIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(simatcopy,SIMATCOPY)))));
#else
void FC_GLOBAL2(simatcopy,SIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(simatcopy,SIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
void FC_GLOBAL3(simatcopy,SIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(simatcopy,SIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_simatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.simatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_simatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_simatcopy_")));
#else
void flexiblas_real_simatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_simatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif

void flexiblas_chain_simatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_simatcopy++;
    if ( hook_pos_simatcopy < __flexiblas_hooks->simatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->simatcopy.hook_function[hook_pos_simatcopy];
    } else {
        hook_pos_simatcopy = 0;
        *(void **) &fn = current_backend->blas.simatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_simatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_simatcopy_")));
#else
void flexiblas_chain_simatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_simatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_snrm2 = 0;
#endif

/* Wrapper for snrm2 */
float FC_GLOBAL(snrm2,SNRM2)
    (blasint *n, float *x, blasint *incx)
{
    float (*fn) (void *n, void *x, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    float (*fn_hook) (void *n, void *x, void *incx);
    hook_pos_snrm2 = 0;
    #endif
    float ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    if ( current_backend->info.f2c_float_return == 1) {
         double (*fn_double) (void *n, void *x, void *incx);
         *(void **) &fn_double = current_backend->blas.snrm2;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->snrm2.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) x, (void *) incx);
         } else {
             ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) x, (void *) incx);
         #endif

    } else {

         *(void **) &fn = current_backend->blas.snrm2;
         #ifdef FLEXIBLAS_HOOK_API
         *(void **) &fn_hook = __flexiblas_hooks->snrm2.hook_function[0];

         if ( fn_hook != NULL ) {
             ret = fn_hook((void *) n, (void *) x, (void *) incx);
         } else {
             ret = fn((void *) n, (void *) x, (void *) incx);
         }
         #else
         ret = fn((void *) n, (void *) x, (void *) incx);
         #endif
    }
    return ret;
}

#ifndef __APPLE__
float FC_GLOBAL2(snrm2,SNRM2)(blasint *n, float *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(snrm2,SNRM2)))));
float FC_GLOBAL3(snrm2,SNRM2)(blasint *n, float *x, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(snrm2,SNRM2)))));
#else
float FC_GLOBAL2(snrm2,SNRM2)(blasint *n, float *x, blasint *incx)
{
    return FC_GLOBAL(snrm2,SNRM2)((void *) n, (void *) x, (void *) incx);
}
float FC_GLOBAL3(snrm2,SNRM2)(blasint *n, float *x, blasint *incx)
{
    return FC_GLOBAL(snrm2,SNRM2)((void *) n, (void *) x, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
float flexiblas_real_snrm2_(void *n, void *x, void *incx)
{
    if ( current_backend->info.f2c_float_return == 1 ) {
        double (*fn_double) (void *n, void *x, void *incx);
        float ret;
        *(void **) &fn_double = current_backend->blas.snrm2;
        ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
        return ret;
    }
    float (*fn) (void *n, void *x, void *incx);
    float ret;
    *(void **) &fn = current_backend->blas.snrm2;
    ret = fn((void *) n, (void *) x, (void *) incx);
    return ret;
}
#ifndef __APPLE__
float flexiblas_real_snrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_real_snrm2_")));
#else
float flexiblas_real_snrm2(void *n, void *x, void *incx)
{
    return flexiblas_real_snrm2_((void *) n, (void *) x, (void *) incx);
}
#endif

float flexiblas_chain_snrm2_(void *n, void *x, void *incx)
{
    float (*fn) (void *n, void *x, void *incx);
    float ret;
    hook_pos_snrm2++;
    if ( hook_pos_snrm2 < __flexiblas_hooks->snrm2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->snrm2.hook_function[hook_pos_snrm2];
        ret = fn((void *) n, (void *) x, (void *) incx);
    } else {
        hook_pos_snrm2 = 0;
        if ( current_backend->info.f2c_float_return == 0 ) {
            *(void **) &fn = current_backend->blas.snrm2;
            ret = fn((void *) n, (void *) x, (void *) incx);
        } else {
            double (*fn_double) (void *n, void *x, void *incx);
            *(void **) &fn_double = current_backend->blas.snrm2;
            ret = (float) fn_double((void *) n, (void *) x, (void *) incx);
        }
       
    }
    
    return ret;
}
#ifndef __APPLE__
float flexiblas_chain_snrm2(void *n, void *x, void *incx) __attribute__((alias("flexiblas_chain_snrm2_")));
#else
float flexiblas_chain_snrm2(void *n, void *x, void *incx)
{
    return flexiblas_chain_snrm2_((void *) n, (void *) x, (void *) incx);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_somatcopy = 0;
#endif

/* Wrapper for somatcopy */
void FC_GLOBAL(somatcopy,SOMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_somatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.somatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->somatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(somatcopy,SOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(somatcopy,SOMATCOPY)))));
void FC_GLOBAL3(somatcopy,SOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(somatcopy,SOMATCOPY)))));
#else
void FC_GLOBAL2(somatcopy,SOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(somatcopy,SOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
void FC_GLOBAL3(somatcopy,SOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(somatcopy,SOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_somatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    *(void **) &fn = current_backend->blas.somatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_somatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_real_somatcopy_")));
#else
void flexiblas_real_somatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_real_somatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif

void flexiblas_chain_somatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_somatcopy++;
    if ( hook_pos_somatcopy < __flexiblas_hooks->somatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->somatcopy.hook_function[hook_pos_somatcopy];
    } else {
        hook_pos_somatcopy = 0;
        *(void **) &fn = current_backend->blas.somatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_somatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_chain_somatcopy_")));
#else
void flexiblas_chain_somatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_chain_somatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_srot = 0;
#endif

/* Wrapper for srot */
void FC_GLOBAL(srot,SROT)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *c, float *s)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s);
    hook_pos_srot = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.srot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->srot.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    } else {
        fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    }
    #else
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(srot,SROT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(srot,SROT)))));
void FC_GLOBAL3(srot,SROT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(srot,SROT)))));
#else
void FC_GLOBAL2(srot,SROT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *c, float *s)
{
    FC_GLOBAL(srot,SROT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
}
void FC_GLOBAL3(srot,SROT)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *c, float *s)
{
    FC_GLOBAL(srot,SROT)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_srot_(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s);
    *(void **) &fn = current_backend->blas.srot;
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_srot(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_real_srot_")));
#else
void flexiblas_real_srot(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s)
{
    flexiblas_real_srot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_srot_(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s);
    hook_pos_srot++;
    if ( hook_pos_srot < __flexiblas_hooks->srot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->srot.hook_function[hook_pos_srot];
    } else {
        hook_pos_srot = 0;
        *(void **) &fn = current_backend->blas.srot;
    }
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_srot(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_chain_srot_")));
#else
void flexiblas_chain_srot(void *n, void *sx, void *incx, void *sy, void *incy, void *c, void *s)
{
    flexiblas_chain_srot_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_srotg = 0;
#endif

/* Wrapper for srotg */
void FC_GLOBAL(srotg,SROTG)
    (float *a, float *b, float *c, float *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *a, void *b, void *c, void *s);
    hook_pos_srotg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.srotg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->srotg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) a, (void *) b, (void *) c, (void *) s);
    } else {
        fn((void *) a, (void *) b, (void *) c, (void *) s);
    }
    #else
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(srotg,SROTG)(float *a, float *b, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(srotg,SROTG)))));
void FC_GLOBAL3(srotg,SROTG)(float *a, float *b, float *c, float *s) __attribute__((alias(MTS(FC_GLOBAL(srotg,SROTG)))));
#else
void FC_GLOBAL2(srotg,SROTG)(float *a, float *b, float *c, float *s)
{
    FC_GLOBAL(srotg,SROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
void FC_GLOBAL3(srotg,SROTG)(float *a, float *b, float *c, float *s)
{
    FC_GLOBAL(srotg,SROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_srotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    *(void **) &fn = current_backend->blas.srotg;
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_srotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_real_srotg_")));
#else
void flexiblas_real_srotg(void *a, void *b, void *c, void *s)
{
    flexiblas_real_srotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_srotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    hook_pos_srotg++;
    if ( hook_pos_srotg < __flexiblas_hooks->srotg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->srotg.hook_function[hook_pos_srotg];
    } else {
        hook_pos_srotg = 0;
        *(void **) &fn = current_backend->blas.srotg;
    }
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_srotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_chain_srotg_")));
#else
void flexiblas_chain_srotg(void *a, void *b, void *c, void *s)
{
    flexiblas_chain_srotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_srotm = 0;
#endif

/* Wrapper for srotm */
void FC_GLOBAL(srotm,SROTM)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *sparam)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *sparam);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy, void *sparam);
    hook_pos_srotm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.srotm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->srotm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    } else {
        fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    }
    #else
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(srotm,SROTM)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *sparam) __attribute__((alias(MTS(FC_GLOBAL(srotm,SROTM)))));
void FC_GLOBAL3(srotm,SROTM)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *sparam) __attribute__((alias(MTS(FC_GLOBAL(srotm,SROTM)))));
#else
void FC_GLOBAL2(srotm,SROTM)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *sparam)
{
    FC_GLOBAL(srotm,SROTM)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
}
void FC_GLOBAL3(srotm,SROTM)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy, float *sparam)
{
    FC_GLOBAL(srotm,SROTM)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_srotm_(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *sparam);
    *(void **) &fn = current_backend->blas.srotm;
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    return;
}
#ifndef __APPLE__
void flexiblas_real_srotm(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam) __attribute__((alias("flexiblas_real_srotm_")));
#else
void flexiblas_real_srotm(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam)
{
    flexiblas_real_srotm_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    return;
}
#endif

void flexiblas_chain_srotm_(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy, void *sparam);
    hook_pos_srotm++;
    if ( hook_pos_srotm < __flexiblas_hooks->srotm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->srotm.hook_function[hook_pos_srotm];
    } else {
        hook_pos_srotm = 0;
        *(void **) &fn = current_backend->blas.srotm;
    }
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_srotm(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam) __attribute__((alias("flexiblas_chain_srotm_")));
#else
void flexiblas_chain_srotm(void *n, void *sx, void *incx, void *sy, void *incy, void *sparam)
{
    flexiblas_chain_srotm_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy, (void *) sparam);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_srotmg = 0;
#endif

/* Wrapper for srotmg */
void FC_GLOBAL(srotmg,SROTMG)
    (float *sd1, float *sd2, float *sx1, float *sy1, float *sparam)
{
    void (*fn) (void *sd1, void *sd2, void *sx1, void *sy1, void *sparam);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *sd1, void *sd2, void *sx1, void *sy1, void *sparam);
    hook_pos_srotmg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.srotmg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->srotmg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    } else {
        fn((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    }
    #else
    fn((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(srotmg,SROTMG)(float *sd1, float *sd2, float *sx1, float *sy1, float *sparam) __attribute__((alias(MTS(FC_GLOBAL(srotmg,SROTMG)))));
void FC_GLOBAL3(srotmg,SROTMG)(float *sd1, float *sd2, float *sx1, float *sy1, float *sparam) __attribute__((alias(MTS(FC_GLOBAL(srotmg,SROTMG)))));
#else
void FC_GLOBAL2(srotmg,SROTMG)(float *sd1, float *sd2, float *sx1, float *sy1, float *sparam)
{
    FC_GLOBAL(srotmg,SROTMG)((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
}
void FC_GLOBAL3(srotmg,SROTMG)(float *sd1, float *sd2, float *sx1, float *sy1, float *sparam)
{
    FC_GLOBAL(srotmg,SROTMG)((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_srotmg_(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam)
{
    void (*fn) (void *sd1, void *sd2, void *sx1, void *sy1, void *sparam);
    *(void **) &fn = current_backend->blas.srotmg;
    fn((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    return;
}
#ifndef __APPLE__
void flexiblas_real_srotmg(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam) __attribute__((alias("flexiblas_real_srotmg_")));
#else
void flexiblas_real_srotmg(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam)
{
    flexiblas_real_srotmg_((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    return;
}
#endif

void flexiblas_chain_srotmg_(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam)
{
    void (*fn) (void *sd1, void *sd2, void *sx1, void *sy1, void *sparam);
    hook_pos_srotmg++;
    if ( hook_pos_srotmg < __flexiblas_hooks->srotmg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->srotmg.hook_function[hook_pos_srotmg];
    } else {
        hook_pos_srotmg = 0;
        *(void **) &fn = current_backend->blas.srotmg;
    }
    fn((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_srotmg(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam) __attribute__((alias("flexiblas_chain_srotmg_")));
#else
void flexiblas_chain_srotmg(void *sd1, void *sd2, void *sx1, void *sy1, void *sparam)
{
    flexiblas_chain_srotmg_((void *) sd1, (void *) sd2, (void *) sx1, (void *) sy1, (void *) sparam);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssbmv = 0;
#endif

/* Wrapper for ssbmv */
void FC_GLOBAL(ssbmv,SSBMV)
    (char *uplo, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssbmv,SSBMV)(char *uplo, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssbmv,SSBMV)))));
void FC_GLOBAL3(ssbmv,SSBMV)(char *uplo, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssbmv,SSBMV)))));
#else
void FC_GLOBAL2(ssbmv,SSBMV)(char *uplo, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssbmv,SSBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(ssbmv,SSBMV)(char *uplo, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssbmv,SSBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.ssbmv;
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_ssbmv_")));
#else
void flexiblas_real_ssbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_ssbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_ssbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssbmv++;
    if ( hook_pos_ssbmv < __flexiblas_hooks->ssbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssbmv.hook_function[hook_pos_ssbmv];
    } else {
        hook_pos_ssbmv = 0;
        *(void **) &fn = current_backend->blas.ssbmv;
    }
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_ssbmv_")));
#else
void flexiblas_chain_ssbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_ssbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sscal = 0;
#endif

/* Wrapper for sscal */
void FC_GLOBAL(sscal,SSCAL)
    (blasint *n, float *sa, float *sx, blasint *incx)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sa, void *sx, void *incx);
    hook_pos_sscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sa, (void *) sx, (void *) incx);
    } else {
        fn((void *) n, (void *) sa, (void *) sx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sscal,SSCAL)(blasint *n, float *sa, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(sscal,SSCAL)))));
void FC_GLOBAL3(sscal,SSCAL)(blasint *n, float *sa, float *sx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(sscal,SSCAL)))));
#else
void FC_GLOBAL2(sscal,SSCAL)(blasint *n, float *sa, float *sx, blasint *incx)
{
    FC_GLOBAL(sscal,SSCAL)((void *) n, (void *) sa, (void *) sx, (void *) incx);
}
void FC_GLOBAL3(sscal,SSCAL)(blasint *n, float *sa, float *sx, blasint *incx)
{
    FC_GLOBAL(sscal,SSCAL)((void *) n, (void *) sa, (void *) sx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sscal_(void *n, void *sa, void *sx, void *incx)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx);
    *(void **) &fn = current_backend->blas.sscal;
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sscal(void *n, void *sa, void *sx, void *incx) __attribute__((alias("flexiblas_real_sscal_")));
#else
void flexiblas_real_sscal(void *n, void *sa, void *sx, void *incx)
{
    flexiblas_real_sscal_((void *) n, (void *) sa, (void *) sx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_sscal_(void *n, void *sa, void *sx, void *incx)
{
    void (*fn) (void *n, void *sa, void *sx, void *incx);
    hook_pos_sscal++;
    if ( hook_pos_sscal < __flexiblas_hooks->sscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sscal.hook_function[hook_pos_sscal];
    } else {
        hook_pos_sscal = 0;
        *(void **) &fn = current_backend->blas.sscal;
    }
    fn((void *) n, (void *) sa, (void *) sx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sscal(void *n, void *sa, void *sx, void *incx) __attribute__((alias("flexiblas_chain_sscal_")));
#else
void flexiblas_chain_sscal(void *n, void *sa, void *sx, void *incx)
{
    flexiblas_chain_sscal_((void *) n, (void *) sa, (void *) sx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sspmv = 0;
#endif

/* Wrapper for sspmv */
void FC_GLOBAL(sspmv,SSPMV)
    (char *uplo, blasint *n, float *alpha, float *ap, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sspmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sspmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sspmv,SSPMV)(char *uplo, blasint *n, float *alpha, float *ap, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspmv,SSPMV)))));
void FC_GLOBAL3(sspmv,SSPMV)(char *uplo, blasint *n, float *alpha, float *ap, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspmv,SSPMV)))));
#else
void FC_GLOBAL2(sspmv,SSPMV)(char *uplo, blasint *n, float *alpha, float *ap, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspmv,SSPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(sspmv,SSPMV)(char *uplo, blasint *n, float *alpha, float *ap, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspmv,SSPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sspmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.sspmv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_sspmv_")));
#else
void flexiblas_real_sspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_sspmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_sspmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspmv++;
    if ( hook_pos_sspmv < __flexiblas_hooks->sspmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sspmv.hook_function[hook_pos_sspmv];
    } else {
        hook_pos_sspmv = 0;
        *(void **) &fn = current_backend->blas.sspmv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_sspmv_")));
#else
void flexiblas_chain_sspmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_sspmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sspr = 0;
#endif

/* Wrapper for sspr */
void FC_GLOBAL(sspr,SSPR)
    (char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sspr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sspr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sspr,SSPR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspr,SSPR)))));
void FC_GLOBAL3(sspr,SSPR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspr,SSPR)))));
#else
void FC_GLOBAL2(sspr,SSPR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspr,SSPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
void FC_GLOBAL3(sspr,SSPR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspr,SSPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sspr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.sspr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_sspr_")));
#else
void flexiblas_real_sspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_sspr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_sspr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspr++;
    if ( hook_pos_sspr < __flexiblas_hooks->sspr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sspr.hook_function[hook_pos_sspr];
    } else {
        hook_pos_sspr = 0;
        *(void **) &fn = current_backend->blas.sspr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_sspr_")));
#else
void flexiblas_chain_sspr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_sspr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sspr2 = 0;
#endif

/* Wrapper for sspr2 */
void FC_GLOBAL(sspr2,SSPR2)
    (char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sspr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sspr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sspr2,SSPR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspr2,SSPR2)))));
void FC_GLOBAL3(sspr2,SSPR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(sspr2,SSPR2)))));
#else
void FC_GLOBAL2(sspr2,SSPR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspr2,SSPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
void FC_GLOBAL3(sspr2,SSPR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(sspr2,SSPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sspr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.sspr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_sspr2_")));
#else
void flexiblas_real_sspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_sspr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_sspr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_sspr2++;
    if ( hook_pos_sspr2 < __flexiblas_hooks->sspr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sspr2.hook_function[hook_pos_sspr2];
    } else {
        hook_pos_sspr2 = 0;
        *(void **) &fn = current_backend->blas.sspr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_sspr2_")));
#else
void flexiblas_chain_sspr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_sspr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_sswap = 0;
#endif

/* Wrapper for sswap */
void FC_GLOBAL(sswap,SSWAP)
    (blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_sswap = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.sswap;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->sswap.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    } else {
        fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(sswap,SSWAP)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sswap,SSWAP)))));
void FC_GLOBAL3(sswap,SSWAP)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(sswap,SSWAP)))));
#else
void FC_GLOBAL2(sswap,SSWAP)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(sswap,SSWAP)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
void FC_GLOBAL3(sswap,SSWAP)(blasint *n, float *sx, blasint *incx, float *sy, blasint *incy)
{
    FC_GLOBAL(sswap,SSWAP)((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_sswap_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    *(void **) &fn = current_backend->blas.sswap;
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_sswap(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_real_sswap_")));
#else
void flexiblas_real_sswap(void *n, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_real_sswap_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_sswap_(void *n, void *sx, void *incx, void *sy, void *incy)
{
    void (*fn) (void *n, void *sx, void *incx, void *sy, void *incy);
    hook_pos_sswap++;
    if ( hook_pos_sswap < __flexiblas_hooks->sswap.nhook ) {
        *(void **) &fn = __flexiblas_hooks->sswap.hook_function[hook_pos_sswap];
    } else {
        hook_pos_sswap = 0;
        *(void **) &fn = current_backend->blas.sswap;
    }
    fn((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_sswap(void *n, void *sx, void *incx, void *sy, void *incy) __attribute__((alias("flexiblas_chain_sswap_")));
#else
void flexiblas_chain_sswap(void *n, void *sx, void *incx, void *sy, void *incy)
{
    flexiblas_chain_sswap_((void *) n, (void *) sx, (void *) incx, (void *) sy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssymm = 0;
#endif

/* Wrapper for ssymm */
void FC_GLOBAL(ssymm,SSYMM)
    (char *side, char *uplo, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssymm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssymm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssymm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssymm,SSYMM)(char *side, char *uplo, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssymm,SSYMM)))));
void FC_GLOBAL3(ssymm,SSYMM)(char *side, char *uplo, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssymm,SSYMM)))));
#else
void FC_GLOBAL2(ssymm,SSYMM)(char *side, char *uplo, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssymm,SSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(ssymm,SSYMM)(char *side, char *uplo, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssymm,SSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.ssymm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_ssymm_")));
#else
void flexiblas_real_ssymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_ssymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_ssymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssymm++;
    if ( hook_pos_ssymm < __flexiblas_hooks->ssymm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssymm.hook_function[hook_pos_ssymm];
    } else {
        hook_pos_ssymm = 0;
        *(void **) &fn = current_backend->blas.ssymm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_ssymm_")));
#else
void flexiblas_chain_ssymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_ssymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssymv = 0;
#endif

/* Wrapper for ssymv */
void FC_GLOBAL(ssymv,SSYMV)
    (char *uplo, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssymv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssymv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssymv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssymv,SSYMV)(char *uplo, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssymv,SSYMV)))));
void FC_GLOBAL3(ssymv,SSYMV)(char *uplo, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssymv,SSYMV)))));
#else
void FC_GLOBAL2(ssymv,SSYMV)(char *uplo, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssymv,SSYMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(ssymv,SSYMV)(char *uplo, blasint *n, float *alpha, float *a, blasint *lda, float *x, blasint *incx, float *beta, float *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssymv,SSYMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssymv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.ssymv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_ssymv_")));
#else
void flexiblas_real_ssymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_ssymv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_ssymv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssymv++;
    if ( hook_pos_ssymv < __flexiblas_hooks->ssymv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssymv.hook_function[hook_pos_ssymv];
    } else {
        hook_pos_ssymv = 0;
        *(void **) &fn = current_backend->blas.ssymv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_ssymv_")));
#else
void flexiblas_chain_ssymv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_ssymv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssyr = 0;
#endif

/* Wrapper for ssyr */
void FC_GLOBAL(ssyr,SSYR)
    (char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssyr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssyr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssyr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssyr,SSYR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr,SSYR)))));
void FC_GLOBAL3(ssyr,SSYR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr,SSYR)))));
#else
void FC_GLOBAL2(ssyr,SSYR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssyr,SSYR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(ssyr,SSYR)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssyr,SSYR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssyr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.ssyr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_ssyr_")));
#else
void flexiblas_real_ssyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_ssyr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_ssyr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssyr++;
    if ( hook_pos_ssyr < __flexiblas_hooks->ssyr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssyr.hook_function[hook_pos_ssyr];
    } else {
        hook_pos_ssyr = 0;
        *(void **) &fn = current_backend->blas.ssyr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_ssyr_")));
#else
void flexiblas_chain_ssyr(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_ssyr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssyr2 = 0;
#endif

/* Wrapper for ssyr2 */
void FC_GLOBAL(ssyr2,SSYR2)
    (char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssyr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssyr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssyr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssyr2,SSYR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr2,SSYR2)))));
void FC_GLOBAL3(ssyr2,SSYR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr2,SSYR2)))));
#else
void FC_GLOBAL2(ssyr2,SSYR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssyr2,SSYR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(ssyr2,SSYR2)(char *uplo, blasint *n, float *alpha, float *x, blasint *incx, float *y, blasint *incy, float *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(ssyr2,SSYR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssyr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.ssyr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_ssyr2_")));
#else
void flexiblas_real_ssyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_ssyr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_ssyr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_ssyr2++;
    if ( hook_pos_ssyr2 < __flexiblas_hooks->ssyr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssyr2.hook_function[hook_pos_ssyr2];
    } else {
        hook_pos_ssyr2 = 0;
        *(void **) &fn = current_backend->blas.ssyr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_ssyr2_")));
#else
void flexiblas_chain_ssyr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_ssyr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssyr2k = 0;
#endif

/* Wrapper for ssyr2k */
void FC_GLOBAL(ssyr2k,SSYR2K)
    (char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_ssyr2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssyr2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssyr2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssyr2k,SSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr2k,SSYR2K)))));
void FC_GLOBAL3(ssyr2k,SSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(ssyr2k,SSYR2K)))));
#else
void FC_GLOBAL2(ssyr2k,SSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(ssyr2k,SSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(ssyr2k,SSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(ssyr2k,SSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.ssyr2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_ssyr2k_")));
#else
void flexiblas_real_ssyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_ssyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_ssyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_ssyr2k++;
    if ( hook_pos_ssyr2k < __flexiblas_hooks->ssyr2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssyr2k.hook_function[hook_pos_ssyr2k];
    } else {
        hook_pos_ssyr2k = 0;
        *(void **) &fn = current_backend->blas.ssyr2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_ssyr2k_")));
#else
void flexiblas_chain_ssyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_ssyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ssyrk = 0;
#endif

/* Wrapper for ssyrk */
void FC_GLOBAL(ssyrk,SSYRK)
    (char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_ssyrk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ssyrk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ssyrk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ssyrk,SSYRK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(ssyrk,SSYRK)))));
void FC_GLOBAL3(ssyrk,SSYRK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(ssyrk,SSYRK)))));
#else
void FC_GLOBAL2(ssyrk,SSYRK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(ssyrk,SSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(ssyrk,SSYRK)(char *uplo, char *trans, blasint *n, blasint *k, float *alpha, float *a, blasint *lda, float *beta, float *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(ssyrk,SSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ssyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.ssyrk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ssyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_ssyrk_")));
#else
void flexiblas_real_ssyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_ssyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_ssyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_ssyrk++;
    if ( hook_pos_ssyrk < __flexiblas_hooks->ssyrk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ssyrk.hook_function[hook_pos_ssyrk];
    } else {
        hook_pos_ssyrk = 0;
        *(void **) &fn = current_backend->blas.ssyrk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ssyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_ssyrk_")));
#else
void flexiblas_chain_ssyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_ssyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_stbmv = 0;
#endif

/* Wrapper for stbmv */
void FC_GLOBAL(stbmv,STBMV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.stbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->stbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(stbmv,STBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stbmv,STBMV)))));
void FC_GLOBAL3(stbmv,STBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stbmv,STBMV)))));
#else
void FC_GLOBAL2(stbmv,STBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stbmv,STBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(stbmv,STBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stbmv,STBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_stbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.stbmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_stbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_stbmv_")));
#else
void flexiblas_real_stbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_stbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_stbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stbmv++;
    if ( hook_pos_stbmv < __flexiblas_hooks->stbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->stbmv.hook_function[hook_pos_stbmv];
    } else {
        hook_pos_stbmv = 0;
        *(void **) &fn = current_backend->blas.stbmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_stbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_stbmv_")));
#else
void flexiblas_chain_stbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_stbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_stbsv = 0;
#endif

/* Wrapper for stbsv */
void FC_GLOBAL(stbsv,STBSV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stbsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.stbsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->stbsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(stbsv,STBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stbsv,STBSV)))));
void FC_GLOBAL3(stbsv,STBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stbsv,STBSV)))));
#else
void FC_GLOBAL2(stbsv,STBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stbsv,STBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(stbsv,STBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stbsv,STBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_stbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.stbsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_stbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_stbsv_")));
#else
void flexiblas_real_stbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_stbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_stbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stbsv++;
    if ( hook_pos_stbsv < __flexiblas_hooks->stbsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->stbsv.hook_function[hook_pos_stbsv];
    } else {
        hook_pos_stbsv = 0;
        *(void **) &fn = current_backend->blas.stbsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_stbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_stbsv_")));
#else
void flexiblas_chain_stbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_stbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_stpmv = 0;
#endif

/* Wrapper for stpmv */
void FC_GLOBAL(stpmv,STPMV)
    (char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.stpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->stpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(stpmv,STPMV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stpmv,STPMV)))));
void FC_GLOBAL3(stpmv,STPMV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stpmv,STPMV)))));
#else
void FC_GLOBAL2(stpmv,STPMV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stpmv,STPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(stpmv,STPMV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stpmv,STPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_stpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.stpmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_stpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_stpmv_")));
#else
void flexiblas_real_stpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_stpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_stpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stpmv++;
    if ( hook_pos_stpmv < __flexiblas_hooks->stpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->stpmv.hook_function[hook_pos_stpmv];
    } else {
        hook_pos_stpmv = 0;
        *(void **) &fn = current_backend->blas.stpmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_stpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_stpmv_")));
#else
void flexiblas_chain_stpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_stpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_stpsv = 0;
#endif

/* Wrapper for stpsv */
void FC_GLOBAL(stpsv,STPSV)
    (char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stpsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.stpsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->stpsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(stpsv,STPSV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stpsv,STPSV)))));
void FC_GLOBAL3(stpsv,STPSV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(stpsv,STPSV)))));
#else
void FC_GLOBAL2(stpsv,STPSV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stpsv,STPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(stpsv,STPSV)(char *uplo, char *trans, char *diag, blasint *n, float *ap, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(stpsv,STPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_stpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.stpsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_stpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_stpsv_")));
#else
void flexiblas_real_stpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_stpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_stpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_stpsv++;
    if ( hook_pos_stpsv < __flexiblas_hooks->stpsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->stpsv.hook_function[hook_pos_stpsv];
    } else {
        hook_pos_stpsv = 0;
        *(void **) &fn = current_backend->blas.stpsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_stpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_stpsv_")));
#else
void flexiblas_chain_stpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_stpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_strmm = 0;
#endif

/* Wrapper for strmm */
void FC_GLOBAL(strmm,STRMM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strmm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.strmm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->strmm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(strmm,STRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strmm,STRMM)))));
void FC_GLOBAL3(strmm,STRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strmm,STRMM)))));
#else
void FC_GLOBAL2(strmm,STRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strmm,STRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(strmm,STRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strmm,STRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_strmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.strmm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_strmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_strmm_")));
#else
void flexiblas_real_strmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_strmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_strmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strmm++;
    if ( hook_pos_strmm < __flexiblas_hooks->strmm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->strmm.hook_function[hook_pos_strmm];
    } else {
        hook_pos_strmm = 0;
        *(void **) &fn = current_backend->blas.strmm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_strmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_strmm_")));
#else
void flexiblas_chain_strmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_strmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_strmv = 0;
#endif

/* Wrapper for strmv */
void FC_GLOBAL(strmv,STRMV)
    (char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.strmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->strmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(strmv,STRMV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strmv,STRMV)))));
void FC_GLOBAL3(strmv,STRMV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strmv,STRMV)))));
#else
void FC_GLOBAL2(strmv,STRMV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strmv,STRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(strmv,STRMV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strmv,STRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_strmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.strmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_strmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_strmv_")));
#else
void flexiblas_real_strmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_strmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_strmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strmv++;
    if ( hook_pos_strmv < __flexiblas_hooks->strmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->strmv.hook_function[hook_pos_strmv];
    } else {
        hook_pos_strmv = 0;
        *(void **) &fn = current_backend->blas.strmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_strmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_strmv_")));
#else
void flexiblas_chain_strmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_strmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_strsm = 0;
#endif

/* Wrapper for strsm */
void FC_GLOBAL(strsm,STRSM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strsm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.strsm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->strsm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(strsm,STRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strsm,STRSM)))));
void FC_GLOBAL3(strsm,STRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strsm,STRSM)))));
#else
void FC_GLOBAL2(strsm,STRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strsm,STRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(strsm,STRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, float *alpha, float *a, blasint *lda, float *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strsm,STRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_strsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.strsm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_strsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_strsm_")));
#else
void flexiblas_real_strsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_strsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_strsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strsm++;
    if ( hook_pos_strsm < __flexiblas_hooks->strsm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->strsm.hook_function[hook_pos_strsm];
    } else {
        hook_pos_strsm = 0;
        *(void **) &fn = current_backend->blas.strsm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_strsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_strsm_")));
#else
void flexiblas_chain_strsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_strsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_strsv = 0;
#endif

/* Wrapper for strsv */
void FC_GLOBAL(strsv,STRSV)
    (char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.strsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->strsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(strsv,STRSV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strsv,STRSV)))));
void FC_GLOBAL3(strsv,STRSV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(strsv,STRSV)))));
#else
void FC_GLOBAL2(strsv,STRSV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strsv,STRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(strsv,STRSV)(char *uplo, char *trans, char *diag, blasint *n, float *a, blasint *lda, float *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(strsv,STRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_strsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.strsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_strsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_strsv_")));
#else
void flexiblas_real_strsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_strsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_strsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_strsv++;
    if ( hook_pos_strsv < __flexiblas_hooks->strsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->strsv.hook_function[hook_pos_strsv];
    } else {
        hook_pos_strsv = 0;
        *(void **) &fn = current_backend->blas.strsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_strsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_strsv_")));
#else
void flexiblas_chain_strsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_strsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zaxpby = 0;
#endif

/* Wrapper for zaxpby */
void FC_GLOBAL(zaxpby,ZAXPBY)
    (blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zb, blas_complex_double *zy, blasint *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy);
    hook_pos_zaxpby = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zaxpby;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zaxpby.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    } else {
        fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zaxpby,ZAXPBY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zb, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zaxpby,ZAXPBY)))));
void FC_GLOBAL3(zaxpby,ZAXPBY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zb, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zaxpby,ZAXPBY)))));
#else
void FC_GLOBAL2(zaxpby,ZAXPBY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zb, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zaxpby,ZAXPBY)((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
}
void FC_GLOBAL3(zaxpby,ZAXPBY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zb, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zaxpby,ZAXPBY)((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zaxpby_(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy);
    *(void **) &fn = current_backend->blas.zaxpby;
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zaxpby(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy) __attribute__((alias("flexiblas_real_zaxpby_")));
#else
void flexiblas_real_zaxpby(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy)
{
    flexiblas_real_zaxpby_((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_zaxpby_(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy);
    hook_pos_zaxpby++;
    if ( hook_pos_zaxpby < __flexiblas_hooks->zaxpby.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zaxpby.hook_function[hook_pos_zaxpby];
    } else {
        hook_pos_zaxpby = 0;
        *(void **) &fn = current_backend->blas.zaxpby;
    }
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zaxpby(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zaxpby_")));
#else
void flexiblas_chain_zaxpby(void *n, void *za, void *zx, void *incx, void *zb, void *zy, void *incy)
{
    flexiblas_chain_zaxpby_((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zb, (void *) zy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zaxpy = 0;
#endif

/* Wrapper for zaxpy */
void FC_GLOBAL(zaxpy,ZAXPY)
    (blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *za, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zaxpy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zaxpy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zaxpy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zaxpy,ZAXPY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zaxpy,ZAXPY)))));
void FC_GLOBAL3(zaxpy,ZAXPY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zaxpy,ZAXPY)))));
#else
void FC_GLOBAL2(zaxpy,ZAXPY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zaxpy,ZAXPY)((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
void FC_GLOBAL3(zaxpy,ZAXPY)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zaxpy,ZAXPY)((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zaxpy_(void *n, void *za, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zy, void *incy);
    *(void **) &fn = current_backend->blas.zaxpy;
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zaxpy(void *n, void *za, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_real_zaxpy_")));
#else
void flexiblas_real_zaxpy(void *n, void *za, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_real_zaxpy_((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_zaxpy_(void *n, void *za, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *za, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zaxpy++;
    if ( hook_pos_zaxpy < __flexiblas_hooks->zaxpy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zaxpy.hook_function[hook_pos_zaxpy];
    } else {
        hook_pos_zaxpy = 0;
        *(void **) &fn = current_backend->blas.zaxpy;
    }
    fn((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zaxpy(void *n, void *za, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zaxpy_")));
#else
void flexiblas_chain_zaxpy(void *n, void *za, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_chain_zaxpy_((void *) n, (void *) za, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zcopy = 0;
#endif

/* Wrapper for zcopy */
void FC_GLOBAL(zcopy,ZCOPY)
    (blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zcopy,ZCOPY)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zcopy,ZCOPY)))));
void FC_GLOBAL3(zcopy,ZCOPY)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zcopy,ZCOPY)))));
#else
void FC_GLOBAL2(zcopy,ZCOPY)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zcopy,ZCOPY)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
void FC_GLOBAL3(zcopy,ZCOPY)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zcopy,ZCOPY)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zcopy_(void *n, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    *(void **) &fn = current_backend->blas.zcopy;
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zcopy(void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_real_zcopy_")));
#else
void flexiblas_real_zcopy(void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_real_zcopy_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_zcopy_(void *n, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zcopy++;
    if ( hook_pos_zcopy < __flexiblas_hooks->zcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zcopy.hook_function[hook_pos_zcopy];
    } else {
        hook_pos_zcopy = 0;
        *(void **) &fn = current_backend->blas.zcopy;
    }
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zcopy(void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zcopy_")));
#else
void flexiblas_chain_zcopy(void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_chain_zcopy_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zdotc = 0;
#endif

/* Wrapper for zdotc */
blas_complex_double FC_GLOBAL(zdotc,ZDOTC)
    (blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);

    #ifdef FLEXIBLAS_HOOK_API
    blas_complex_double (*fn_hook) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel_hook) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zdotc = 0;
    #endif
    blas_complex_double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zdotc;
    *(void **) &fn_intel = current_backend->blas.zdotc;

    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zdotc.hook_function[0];
    *(void **) &fn_intel_hook = __flexiblas_hooks->zdotc.hook_function[0];

    if ( fn_hook != NULL) {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn_hook((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        } else {
            fn_intel_hook( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        }
    } else {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        } else {
            fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        }
    }
    #else
    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    #endif
    return ret;
}

#ifndef __APPLE__
blas_complex_double FC_GLOBAL2(zdotc,ZDOTC)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zdotc,ZDOTC)))));
blas_complex_double FC_GLOBAL3(zdotc,ZDOTC)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zdotc,ZDOTC)))));
#else
blas_complex_double FC_GLOBAL2(zdotc,ZDOTC)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    return FC_GLOBAL(zdotc,ZDOTC)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
blas_complex_double FC_GLOBAL3(zdotc,ZDOTC)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    return FC_GLOBAL(zdotc,ZDOTC)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zdotc_(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    blas_complex_double ret;
    *(void **) &fn = current_backend->blas.zdotc;
    *(void **) &fn_intel = current_backend->blas.zdotc;

    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    *((blas_complex_double *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_real_zdotc(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_real_zdotc_")));
#else
void flexiblas_real_zdotc(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_real_zdotc_(returnvalue, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif

void flexiblas_chain_zdotc_(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    blas_complex_double ret;
    hook_pos_zdotc++;
    if ( hook_pos_zdotc < __flexiblas_hooks->zdotc.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zdotc.hook_function[hook_pos_zdotc];
        *(void **) &fn_intel = __flexiblas_hooks->zdotc.hook_function[hook_pos_zdotc];
    } else {
        hook_pos_zdotc = 0;
        *(void **) &fn = current_backend->blas.zdotc;
        *(void **) &fn_intel = current_backend->blas.zdotc;
    }
    if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
            fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    *((blas_complex_double *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zdotc(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zdotc_")));
#else
void flexiblas_chain_zdotc(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_chain_zdotc_(returnvalue, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zdotu = 0;
#endif

/* Wrapper for zdotu */
blas_complex_double FC_GLOBAL(zdotu,ZDOTU)
    (blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);

    #ifdef FLEXIBLAS_HOOK_API
    blas_complex_double (*fn_hook) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel_hook) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zdotu = 0;
    #endif
    blas_complex_double ret;

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zdotu;
    *(void **) &fn_intel = current_backend->blas.zdotu;

    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zdotu.hook_function[0];
    *(void **) &fn_intel_hook = __flexiblas_hooks->zdotu.hook_function[0];

    if ( fn_hook != NULL) {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn_hook((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        } else {
            fn_intel_hook( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        }
    } else {
        if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        } else {
            fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
        }
    }
    #else
    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    #endif
    return ret;
}

#ifndef __APPLE__
blas_complex_double FC_GLOBAL2(zdotu,ZDOTU)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zdotu,ZDOTU)))));
blas_complex_double FC_GLOBAL3(zdotu,ZDOTU)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zdotu,ZDOTU)))));
#else
blas_complex_double FC_GLOBAL2(zdotu,ZDOTU)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    return FC_GLOBAL(zdotu,ZDOTU)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
blas_complex_double FC_GLOBAL3(zdotu,ZDOTU)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    return FC_GLOBAL(zdotu,ZDOTU)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zdotu_(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    blas_complex_double ret;
    *(void **) &fn = current_backend->blas.zdotu;
    *(void **) &fn_intel = current_backend->blas.zdotu;

    if(current_backend->info.intel_interface == 0 ) {
        ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    *((blas_complex_double *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_real_zdotu(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_real_zdotu_")));
#else
void flexiblas_real_zdotu(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_real_zdotu_(returnvalue, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif

void flexiblas_chain_zdotu_(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    blas_complex_double (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    void (*fn_intel) (blas_complex_double *ret, void *n, void *zx, void *incx, void *zy, void *incy);
    blas_complex_double ret;
    hook_pos_zdotu++;
    if ( hook_pos_zdotu < __flexiblas_hooks->zdotu.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zdotu.hook_function[hook_pos_zdotu];
        *(void **) &fn_intel = __flexiblas_hooks->zdotu.hook_function[hook_pos_zdotu];
    } else {
        hook_pos_zdotu = 0;
        *(void **) &fn = current_backend->blas.zdotu;
        *(void **) &fn_intel = current_backend->blas.zdotu;
    }
    if(current_backend->info.intel_interface == 0 ) {
            ret = fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
            fn_intel( &ret, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    *((blas_complex_double *)returnvalue) = ret;
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zdotu(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zdotu_")));
#else
void flexiblas_chain_zdotu(blas_complex_double *returnvalue, void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_chain_zdotu_(returnvalue, (void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zdrot = 0;
#endif

/* Wrapper for zdrot */
void FC_GLOBAL(zdrot,ZDROT)
    (blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy, double *c, double *s)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s);
    hook_pos_zdrot = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zdrot;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zdrot.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    } else {
        fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    }
    #else
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zdrot,ZDROT)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(zdrot,ZDROT)))));
void FC_GLOBAL3(zdrot,ZDROT)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy, double *c, double *s) __attribute__((alias(MTS(FC_GLOBAL(zdrot,ZDROT)))));
#else
void FC_GLOBAL2(zdrot,ZDROT)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy, double *c, double *s)
{
    FC_GLOBAL(zdrot,ZDROT)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
}
void FC_GLOBAL3(zdrot,ZDROT)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy, double *c, double *s)
{
    FC_GLOBAL(zdrot,ZDROT)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zdrot_(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s);
    *(void **) &fn = current_backend->blas.zdrot;
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zdrot(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_real_zdrot_")));
#else
void flexiblas_real_zdrot(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s)
{
    flexiblas_real_zdrot_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_zdrot_(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s);
    hook_pos_zdrot++;
    if ( hook_pos_zdrot < __flexiblas_hooks->zdrot.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zdrot.hook_function[hook_pos_zdrot];
    } else {
        hook_pos_zdrot = 0;
        *(void **) &fn = current_backend->blas.zdrot;
    }
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zdrot(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s) __attribute__((alias("flexiblas_chain_zdrot_")));
#else
void flexiblas_chain_zdrot(void *n, void *zx, void *incx, void *zy, void *incy, void *c, void *s)
{
    flexiblas_chain_zdrot_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zdscal = 0;
#endif

/* Wrapper for zdscal */
void FC_GLOBAL(zdscal,ZDSCAL)
    (blasint *n, double *da, blas_complex_double *zx, blasint *incx)
{
    void (*fn) (void *n, void *da, void *zx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *da, void *zx, void *incx);
    hook_pos_zdscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zdscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zdscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) da, (void *) zx, (void *) incx);
    } else {
        fn((void *) n, (void *) da, (void *) zx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) da, (void *) zx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zdscal,ZDSCAL)(blasint *n, double *da, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(zdscal,ZDSCAL)))));
void FC_GLOBAL3(zdscal,ZDSCAL)(blasint *n, double *da, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(zdscal,ZDSCAL)))));
#else
void FC_GLOBAL2(zdscal,ZDSCAL)(blasint *n, double *da, blas_complex_double *zx, blasint *incx)
{
    FC_GLOBAL(zdscal,ZDSCAL)((void *) n, (void *) da, (void *) zx, (void *) incx);
}
void FC_GLOBAL3(zdscal,ZDSCAL)(blasint *n, double *da, blas_complex_double *zx, blasint *incx)
{
    FC_GLOBAL(zdscal,ZDSCAL)((void *) n, (void *) da, (void *) zx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zdscal_(void *n, void *da, void *zx, void *incx)
{
    void (*fn) (void *n, void *da, void *zx, void *incx);
    *(void **) &fn = current_backend->blas.zdscal;
    fn((void *) n, (void *) da, (void *) zx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zdscal(void *n, void *da, void *zx, void *incx) __attribute__((alias("flexiblas_real_zdscal_")));
#else
void flexiblas_real_zdscal(void *n, void *da, void *zx, void *incx)
{
    flexiblas_real_zdscal_((void *) n, (void *) da, (void *) zx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_zdscal_(void *n, void *da, void *zx, void *incx)
{
    void (*fn) (void *n, void *da, void *zx, void *incx);
    hook_pos_zdscal++;
    if ( hook_pos_zdscal < __flexiblas_hooks->zdscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zdscal.hook_function[hook_pos_zdscal];
    } else {
        hook_pos_zdscal = 0;
        *(void **) &fn = current_backend->blas.zdscal;
    }
    fn((void *) n, (void *) da, (void *) zx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zdscal(void *n, void *da, void *zx, void *incx) __attribute__((alias("flexiblas_chain_zdscal_")));
#else
void flexiblas_chain_zdscal(void *n, void *da, void *zx, void *incx)
{
    flexiblas_chain_zdscal_((void *) n, (void *) da, (void *) zx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgbmv = 0;
#endif

/* Wrapper for zgbmv */
void FC_GLOBAL(zgbmv,ZGBMV)
    (char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zgbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgbmv,ZGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zgbmv,ZGBMV)))));
void FC_GLOBAL3(zgbmv,ZGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zgbmv,ZGBMV)))));
#else
void FC_GLOBAL2(zgbmv,ZGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zgbmv,ZGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(zgbmv,ZGBMV)(char *trans, blasint *m, blasint *n, blasint *kl, blasint *ku, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zgbmv,ZGBMV)((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zgbmv;
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zgbmv_")));
#else
void flexiblas_real_zgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_zgbmv_(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zgbmv++;
    if ( hook_pos_zgbmv < __flexiblas_hooks->zgbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgbmv.hook_function[hook_pos_zgbmv];
    } else {
        hook_pos_zgbmv = 0;
        *(void **) &fn = current_backend->blas.zgbmv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zgbmv_")));
#else
void flexiblas_chain_zgbmv(void *trans, void *m, void *n, void *kl, void *ku, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zgbmv_((void *) trans, (void *) m, (void *) n, (void *) kl, (void *) ku, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgeadd = 0;
#endif

/* Wrapper for zgeadd */
void FC_GLOBAL(zgeadd,ZGEADD)
    (blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *b, blasint *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_zgeadd = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgeadd;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgeadd.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgeadd,ZGEADD)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(zgeadd,ZGEADD)))));
void FC_GLOBAL3(zgeadd,ZGEADD)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *b, blasint *ldb) __attribute__((alias(MTS(FC_GLOBAL(zgeadd,ZGEADD)))));
#else
void FC_GLOBAL2(zgeadd,ZGEADD)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *b, blasint *ldb)
{
    FC_GLOBAL(zgeadd,ZGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
void FC_GLOBAL3(zgeadd,ZGEADD)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *b, blasint *ldb)
{
    FC_GLOBAL(zgeadd,ZGEADD)((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    *(void **) &fn = current_backend->blas.zgeadd;
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_real_zgeadd_")));
#else
void flexiblas_real_zgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_real_zgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif

void flexiblas_chain_zgeadd_(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    void (*fn) (void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb);
    hook_pos_zgeadd++;
    if ( hook_pos_zgeadd < __flexiblas_hooks->zgeadd.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgeadd.hook_function[hook_pos_zgeadd];
    } else {
        hook_pos_zgeadd = 0;
        *(void **) &fn = current_backend->blas.zgeadd;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb) __attribute__((alias("flexiblas_chain_zgeadd_")));
#else
void flexiblas_chain_zgeadd(void *m, void *n, void *alpha, void *a, void *lda, void *beta, void *b, void *ldb)
{
    flexiblas_chain_zgeadd_((void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) b, (void *) ldb);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgemm = 0;
#endif

/* Wrapper for zgemm */
void FC_GLOBAL(zgemm,ZGEMM)
    (char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_zgemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    } else {
        fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    }
    #else
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgemm,ZGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemm,ZGEMM)))));
void FC_GLOBAL3(zgemm,ZGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemm,ZGEMM)))));
#else
void FC_GLOBAL2(zgemm,ZGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemm,ZGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
void FC_GLOBAL3(zgemm,ZGEMM)(char *transa, char *transb, blasint *m, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemm,ZGEMM)((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.zgemm;
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_zgemm_")));
#else
void flexiblas_real_zgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_zgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_zgemm_(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_zgemm++;
    if ( hook_pos_zgemm < __flexiblas_hooks->zgemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgemm.hook_function[hook_pos_zgemm];
    } else {
        hook_pos_zgemm = 0;
        *(void **) &fn = current_backend->blas.zgemm;
    }
    fn((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_zgemm_")));
#else
void flexiblas_chain_zgemm(void *transa, void *transb, void *m, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_zgemm_((void *) transa, (void *) transb, (void *) m, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, transa_len, transb_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgemmtr = 0;
#endif

/* Wrapper for zgemmtr */
void FC_GLOBAL(zgemmtr,ZGEMMTR)
    (char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_zgemmtr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgemmtr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgemmtr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    } else {
        fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    }
    #else
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgemmtr,ZGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemmtr,ZGEMMTR)))));
void FC_GLOBAL3(zgemmtr,ZGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemmtr,ZGEMMTR)))));
#else
void FC_GLOBAL2(zgemmtr,ZGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemmtr,ZGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(zgemmtr,ZGEMMTR)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemmtr,ZGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    *(void **) &fn = current_backend->blas.zgemmtr;
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_zgemmtr_")));
#else
void flexiblas_real_zgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_zgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

void flexiblas_chain_zgemmtr_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    void (*fn) (void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len);
    hook_pos_zgemmtr++;
    if ( hook_pos_zgemmtr < __flexiblas_hooks->zgemmtr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgemmtr.hook_function[hook_pos_zgemmtr];
    } else {
        hook_pos_zgemmtr = 0;
        *(void **) &fn = current_backend->blas.zgemmtr;
    }
    fn((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_zgemmtr_")));
#else
void flexiblas_chain_zgemmtr(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_zgemmtr_((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif
#endif
/* Alternative function name for zgemmtr -- zgemmt */
#ifndef __APPLE__
void FC_GLOBAL(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemmtr,ZGEMMTR)))));
void FC_GLOBAL2(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemmtr,ZGEMMTR)))));
void FC_GLOBAL3(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias(MTS(FC_GLOBAL(zgemmtr,ZGEMMTR)))));
#else
void FC_GLOBAL(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemmtr,ZGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL2(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemmtr,ZGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
void FC_GLOBAL3(zgemmt,ZGEMMT)(char *uplo, char *transa, char *transb, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    FC_GLOBAL(zgemmtr,ZGEMMTR)((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
}
#endif

#ifdef FLEXIBLAS_HOOK_API
#ifndef __APPLE__
void flexiblas_real_zgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_zgemmtr")));
void flexiblas_real_zgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_real_zgemmtr")));
void flexiblas_chain_zgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_zgemmtr")));
void flexiblas_chain_zgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len) __attribute__((alias("flexiblas_chain_zgemmtr")));

#else
void flexiblas_real_zgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_zgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
void flexiblas_real_zgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_real_zgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_zgemmt(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_zgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}

void flexiblas_chain_zgemmt_(void *uplo, void *transa, void *transb, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t transb_len)
{
    flexiblas_chain_zgemmtr((void *) uplo, (void *) transa, (void *) transb, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, transa_len, transb_len);
    return;
}
#endif

#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgemv = 0;
#endif

/* Wrapper for zgemv */
void FC_GLOBAL(zgemv,ZGEMV)
    (char *trans, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zgemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    } else {
        fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    }
    #else
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgemv,ZGEMV)(char *trans, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zgemv,ZGEMV)))));
void FC_GLOBAL3(zgemv,ZGEMV)(char *trans, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zgemv,ZGEMV)))));
#else
void FC_GLOBAL2(zgemv,ZGEMV)(char *trans, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zgemv,ZGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
void FC_GLOBAL3(zgemv,ZGEMV)(char *trans, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zgemv,ZGEMV)((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zgemv;
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zgemv_")));
#else
void flexiblas_real_zgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif

void flexiblas_chain_zgemv_(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zgemv++;
    if ( hook_pos_zgemv < __flexiblas_hooks->zgemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgemv.hook_function[hook_pos_zgemv];
    } else {
        hook_pos_zgemv = 0;
        *(void **) &fn = current_backend->blas.zgemv;
    }
    fn((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zgemv_")));
#else
void flexiblas_chain_zgemv(void *trans, void *m, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zgemv_((void *) trans, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgerc = 0;
#endif

/* Wrapper for zgerc */
void FC_GLOBAL(zgerc,ZGERC)
    (blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_zgerc = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgerc;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgerc.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgerc,ZGERC)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(zgerc,ZGERC)))));
void FC_GLOBAL3(zgerc,ZGERC)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(zgerc,ZGERC)))));
#else
void FC_GLOBAL2(zgerc,ZGERC)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    FC_GLOBAL(zgerc,ZGERC)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(zgerc,ZGERC)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    FC_GLOBAL(zgerc,ZGERC)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgerc_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.zgerc;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_zgerc_")));
#else
void flexiblas_real_zgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_zgerc_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_zgerc_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_zgerc++;
    if ( hook_pos_zgerc < __flexiblas_hooks->zgerc.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgerc.hook_function[hook_pos_zgerc];
    } else {
        hook_pos_zgerc = 0;
        *(void **) &fn = current_backend->blas.zgerc;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_zgerc_")));
#else
void flexiblas_chain_zgerc(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_zgerc_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zgeru = 0;
#endif

/* Wrapper for zgeru */
void FC_GLOBAL(zgeru,ZGERU)
    (blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_zgeru = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zgeru;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zgeru.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    } else {
        fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    }
    #else
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zgeru,ZGERU)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(zgeru,ZGERU)))));
void FC_GLOBAL3(zgeru,ZGERU)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda) __attribute__((alias(MTS(FC_GLOBAL(zgeru,ZGERU)))));
#else
void FC_GLOBAL2(zgeru,ZGERU)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    FC_GLOBAL(zgeru,ZGERU)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
void FC_GLOBAL3(zgeru,ZGERU)(blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda)
{
    FC_GLOBAL(zgeru,ZGERU)((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zgeru_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    *(void **) &fn = current_backend->blas.zgeru;
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_real_zgeru_")));
#else
void flexiblas_real_zgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_real_zgeru_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif

void flexiblas_chain_zgeru_(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    void (*fn) (void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda);
    hook_pos_zgeru++;
    if ( hook_pos_zgeru < __flexiblas_hooks->zgeru.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zgeru.hook_function[hook_pos_zgeru];
    } else {
        hook_pos_zgeru = 0;
        *(void **) &fn = current_backend->blas.zgeru;
    }
    fn((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda) __attribute__((alias("flexiblas_chain_zgeru_")));
#else
void flexiblas_chain_zgeru(void *m, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda)
{
    flexiblas_chain_zgeru_((void *) m, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhbmv = 0;
#endif

/* Wrapper for zhbmv */
void FC_GLOBAL(zhbmv,ZHBMV)
    (char *uplo, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhbmv,ZHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhbmv,ZHBMV)))));
void FC_GLOBAL3(zhbmv,ZHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhbmv,ZHBMV)))));
#else
void FC_GLOBAL2(zhbmv,ZHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhbmv,ZHBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(zhbmv,ZHBMV)(char *uplo, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhbmv,ZHBMV)((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhbmv;
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhbmv_")));
#else
void flexiblas_real_zhbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhbmv_(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhbmv++;
    if ( hook_pos_zhbmv < __flexiblas_hooks->zhbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhbmv.hook_function[hook_pos_zhbmv];
    } else {
        hook_pos_zhbmv = 0;
        *(void **) &fn = current_backend->blas.zhbmv;
    }
    fn((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhbmv_")));
#else
void flexiblas_chain_zhbmv(void *uplo, void *n, void *k, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhbmv_((void *) uplo, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhemm = 0;
#endif

/* Wrapper for zhemm */
void FC_GLOBAL(zhemm,ZHEMM)
    (char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhemm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhemm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhemm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhemm,ZHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhemm,ZHEMM)))));
void FC_GLOBAL3(zhemm,ZHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhemm,ZHEMM)))));
#else
void FC_GLOBAL2(zhemm,ZHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhemm,ZHEMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(zhemm,ZHEMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhemm,ZHEMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhemm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhemm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhemm_")));
#else
void flexiblas_real_zhemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhemm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhemm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhemm++;
    if ( hook_pos_zhemm < __flexiblas_hooks->zhemm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhemm.hook_function[hook_pos_zhemm];
    } else {
        hook_pos_zhemm = 0;
        *(void **) &fn = current_backend->blas.zhemm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhemm_")));
#else
void flexiblas_chain_zhemm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhemm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhemv = 0;
#endif

/* Wrapper for zhemv */
void FC_GLOBAL(zhemv,ZHEMV)
    (char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhemv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhemv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhemv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhemv,ZHEMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhemv,ZHEMV)))));
void FC_GLOBAL3(zhemv,ZHEMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhemv,ZHEMV)))));
#else
void FC_GLOBAL2(zhemv,ZHEMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhemv,ZHEMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(zhemv,ZHEMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhemv,ZHEMV)((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhemv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhemv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhemv_")));
#else
void flexiblas_real_zhemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhemv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhemv_(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhemv++;
    if ( hook_pos_zhemv < __flexiblas_hooks->zhemv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhemv.hook_function[hook_pos_zhemv];
    } else {
        hook_pos_zhemv = 0;
        *(void **) &fn = current_backend->blas.zhemv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhemv_")));
#else
void flexiblas_chain_zhemv(void *uplo, void *n, void *alpha, void *a, void *lda, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhemv_((void *) uplo, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zher = 0;
#endif

/* Wrapper for zher */
void FC_GLOBAL(zher,ZHER)
    (char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zher = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zher;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zher.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zher,ZHER)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zher,ZHER)))));
void FC_GLOBAL3(zher,ZHER)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zher,ZHER)))));
#else
void FC_GLOBAL2(zher,ZHER)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zher,ZHER)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(zher,ZHER)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zher,ZHER)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zher_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zher;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zher_")));
#else
void flexiblas_real_zher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zher_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_zher_(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zher++;
    if ( hook_pos_zher < __flexiblas_hooks->zher.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zher.hook_function[hook_pos_zher];
    } else {
        hook_pos_zher = 0;
        *(void **) &fn = current_backend->blas.zher;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zher_")));
#else
void flexiblas_chain_zher(void *uplo, void *n, void *alpha, void *x, void *incx, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zher_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zher2 = 0;
#endif

/* Wrapper for zher2 */
void FC_GLOBAL(zher2,ZHER2)
    (char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zher2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zher2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zher2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zher2,ZHER2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zher2,ZHER2)))));
void FC_GLOBAL3(zher2,ZHER2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zher2,ZHER2)))));
#else
void FC_GLOBAL2(zher2,ZHER2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zher2,ZHER2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
void FC_GLOBAL3(zher2,ZHER2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *a, blasint *lda, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zher2,ZHER2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zher2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zher2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zher2_")));
#else
void flexiblas_real_zher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zher2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif

void flexiblas_chain_zher2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zher2++;
    if ( hook_pos_zher2 < __flexiblas_hooks->zher2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zher2.hook_function[hook_pos_zher2];
    } else {
        hook_pos_zher2 = 0;
        *(void **) &fn = current_backend->blas.zher2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zher2_")));
#else
void flexiblas_chain_zher2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *a, void *lda, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zher2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) a, (void *) lda, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zher2k = 0;
#endif

/* Wrapper for zher2k */
void FC_GLOBAL(zher2k,ZHER2K)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zher2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zher2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zher2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zher2k,ZHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zher2k,ZHER2K)))));
void FC_GLOBAL3(zher2k,ZHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zher2k,ZHER2K)))));
#else
void FC_GLOBAL2(zher2k,ZHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zher2k,ZHER2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(zher2k,ZHER2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zher2k,ZHER2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zher2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zher2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zher2k_")));
#else
void flexiblas_real_zher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zher2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_zher2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zher2k++;
    if ( hook_pos_zher2k < __flexiblas_hooks->zher2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zher2k.hook_function[hook_pos_zher2k];
    } else {
        hook_pos_zher2k = 0;
        *(void **) &fn = current_backend->blas.zher2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zher2k_")));
#else
void flexiblas_chain_zher2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zher2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zherk = 0;
#endif

/* Wrapper for zherk */
void FC_GLOBAL(zherk,ZHERK)
    (char *uplo, char *trans, blasint *n, blasint *k, double *alpha, blas_complex_double *a, blasint *lda, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zherk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zherk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zherk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zherk,ZHERK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, blas_complex_double *a, blasint *lda, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zherk,ZHERK)))));
void FC_GLOBAL3(zherk,ZHERK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, blas_complex_double *a, blasint *lda, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zherk,ZHERK)))));
#else
void FC_GLOBAL2(zherk,ZHERK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, blas_complex_double *a, blasint *lda, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zherk,ZHERK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(zherk,ZHERK)(char *uplo, char *trans, blasint *n, blasint *k, double *alpha, blas_complex_double *a, blasint *lda, double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zherk,ZHERK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zherk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zherk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zherk_")));
#else
void flexiblas_real_zherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zherk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_zherk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zherk++;
    if ( hook_pos_zherk < __flexiblas_hooks->zherk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zherk.hook_function[hook_pos_zherk];
    } else {
        hook_pos_zherk = 0;
        *(void **) &fn = current_backend->blas.zherk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zherk_")));
#else
void flexiblas_chain_zherk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zherk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhpmv = 0;
#endif

/* Wrapper for zhpmv */
void FC_GLOBAL(zhpmv,ZHPMV)
    (char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *ap, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhpmv,ZHPMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *ap, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpmv,ZHPMV)))));
void FC_GLOBAL3(zhpmv,ZHPMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *ap, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpmv,ZHPMV)))));
#else
void FC_GLOBAL2(zhpmv,ZHPMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *ap, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpmv,ZHPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
void FC_GLOBAL3(zhpmv,ZHPMV)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *ap, blas_complex_double *x, blasint *incx, blas_complex_double *beta, blas_complex_double *y, blasint *incy, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpmv,ZHPMV)((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhpmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhpmv;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhpmv_")));
#else
void flexiblas_real_zhpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhpmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhpmv_(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpmv++;
    if ( hook_pos_zhpmv < __flexiblas_hooks->zhpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhpmv.hook_function[hook_pos_zhpmv];
    } else {
        hook_pos_zhpmv = 0;
        *(void **) &fn = current_backend->blas.zhpmv;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhpmv_")));
#else
void flexiblas_chain_zhpmv(void *uplo, void *n, void *alpha, void *ap, void *x, void *incx, void *beta, void *y, void *incy, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhpmv_((void *) uplo, (void *) n, (void *) alpha, (void *) ap, (void *) x, (void *) incx, (void *) beta, (void *) y, (void *) incy, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhpr = 0;
#endif

/* Wrapper for zhpr */
void FC_GLOBAL(zhpr,ZHPR)
    (char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpr = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhpr;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhpr.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhpr,ZHPR)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpr,ZHPR)))));
void FC_GLOBAL3(zhpr,ZHPR)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpr,ZHPR)))));
#else
void FC_GLOBAL2(zhpr,ZHPR)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpr,ZHPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
void FC_GLOBAL3(zhpr,ZHPR)(char *uplo, blasint *n, double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpr,ZHPR)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhpr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhpr;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhpr_")));
#else
void flexiblas_real_zhpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhpr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhpr_(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpr++;
    if ( hook_pos_zhpr < __flexiblas_hooks->zhpr.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhpr.hook_function[hook_pos_zhpr];
    } else {
        hook_pos_zhpr = 0;
        *(void **) &fn = current_backend->blas.zhpr;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhpr_")));
#else
void flexiblas_chain_zhpr(void *uplo, void *n, void *alpha, void *x, void *incx, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhpr_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zhpr2 = 0;
#endif

/* Wrapper for zhpr2 */
void FC_GLOBAL(zhpr2,ZHPR2)
    (char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpr2 = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zhpr2;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zhpr2.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    } else {
        fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    }
    #else
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zhpr2,ZHPR2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpr2,ZHPR2)))));
void FC_GLOBAL3(zhpr2,ZHPR2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zhpr2,ZHPR2)))));
#else
void FC_GLOBAL2(zhpr2,ZHPR2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpr2,ZHPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
void FC_GLOBAL3(zhpr2,ZHPR2)(char *uplo, blasint *n, blas_complex_double *alpha, blas_complex_double *x, blasint *incx, blas_complex_double *y, blasint *incy, blas_complex_double *ap, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zhpr2,ZHPR2)((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zhpr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zhpr2;
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zhpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zhpr2_")));
#else
void flexiblas_real_zhpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zhpr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif

void flexiblas_chain_zhpr2_(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zhpr2++;
    if ( hook_pos_zhpr2 < __flexiblas_hooks->zhpr2.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zhpr2.hook_function[hook_pos_zhpr2];
    } else {
        hook_pos_zhpr2 = 0;
        *(void **) &fn = current_backend->blas.zhpr2;
    }
    fn((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zhpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zhpr2_")));
#else
void flexiblas_chain_zhpr2(void *uplo, void *n, void *alpha, void *x, void *incx, void *y, void *incy, void *ap, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zhpr2_((void *) uplo, (void *) n, (void *) alpha, (void *) x, (void *) incx, (void *) y, (void *) incy, (void *) ap, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zimatcopy = 0;
#endif

/* Wrapper for zimatcopy */
void FC_GLOBAL(zimatcopy,ZIMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zimatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zimatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zimatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zimatcopy,ZIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zimatcopy,ZIMATCOPY)))));
void FC_GLOBAL3(zimatcopy,ZIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zimatcopy,ZIMATCOPY)))));
#else
void FC_GLOBAL2(zimatcopy,ZIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zimatcopy,ZIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
void FC_GLOBAL3(zimatcopy,ZIMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blasint *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zimatcopy,ZIMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zimatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zimatcopy_")));
#else
void flexiblas_real_zimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif

void flexiblas_chain_zimatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zimatcopy++;
    if ( hook_pos_zimatcopy < __flexiblas_hooks->zimatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zimatcopy.hook_function[hook_pos_zimatcopy];
    } else {
        hook_pos_zimatcopy = 0;
        *(void **) &fn = current_backend->blas.zimatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zimatcopy_")));
#else
void flexiblas_chain_zimatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *ldb, flexiblas_fortran_charlen_t order_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zimatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) ldb, order_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zomatcopy = 0;
#endif

/* Wrapper for zomatcopy */
void FC_GLOBAL(zomatcopy,ZOMATCOPY)
    (char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_zomatcopy = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zomatcopy;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zomatcopy.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    } else {
        fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    }
    #else
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zomatcopy,ZOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(zomatcopy,ZOMATCOPY)))));
void FC_GLOBAL3(zomatcopy,ZOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias(MTS(FC_GLOBAL(zomatcopy,ZOMATCOPY)))));
#else
void FC_GLOBAL2(zomatcopy,ZOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(zomatcopy,ZOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
void FC_GLOBAL3(zomatcopy,ZOMATCOPY)(char *ORDER, char *TRANS, blasint *rows, blasint *cols, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    FC_GLOBAL(zomatcopy,ZOMATCOPY)((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zomatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    *(void **) &fn = current_backend->blas.zomatcopy;
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zomatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_real_zomatcopy_")));
#else
void flexiblas_real_zomatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_real_zomatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif

void flexiblas_chain_zomatcopy_(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    void (*fn) (void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len);
    hook_pos_zomatcopy++;
    if ( hook_pos_zomatcopy < __flexiblas_hooks->zomatcopy.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zomatcopy.hook_function[hook_pos_zomatcopy];
    } else {
        hook_pos_zomatcopy = 0;
        *(void **) &fn = current_backend->blas.zomatcopy;
    }
    fn((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zomatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len) __attribute__((alias("flexiblas_chain_zomatcopy_")));
#else
void flexiblas_chain_zomatcopy(void *ORDER, void *TRANS, void *rows, void *cols, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t ORDER_len, flexiblas_fortran_charlen_t TRANS_len)
{
    flexiblas_chain_zomatcopy_((void *) ORDER, (void *) TRANS, (void *) rows, (void *) cols, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, ORDER_len, TRANS_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zrotg = 0;
#endif

/* Wrapper for zrotg */
void FC_GLOBAL(zrotg,ZROTG)
    (blas_complex_double *a, blas_complex_double *b, double *c, blas_complex_double *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *a, void *b, void *c, void *s);
    hook_pos_zrotg = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zrotg;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zrotg.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) a, (void *) b, (void *) c, (void *) s);
    } else {
        fn((void *) a, (void *) b, (void *) c, (void *) s);
    }
    #else
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zrotg,ZROTG)(blas_complex_double *a, blas_complex_double *b, double *c, blas_complex_double *s) __attribute__((alias(MTS(FC_GLOBAL(zrotg,ZROTG)))));
void FC_GLOBAL3(zrotg,ZROTG)(blas_complex_double *a, blas_complex_double *b, double *c, blas_complex_double *s) __attribute__((alias(MTS(FC_GLOBAL(zrotg,ZROTG)))));
#else
void FC_GLOBAL2(zrotg,ZROTG)(blas_complex_double *a, blas_complex_double *b, double *c, blas_complex_double *s)
{
    FC_GLOBAL(zrotg,ZROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
void FC_GLOBAL3(zrotg,ZROTG)(blas_complex_double *a, blas_complex_double *b, double *c, blas_complex_double *s)
{
    FC_GLOBAL(zrotg,ZROTG)((void *) a, (void *) b, (void *) c, (void *) s);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zrotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    *(void **) &fn = current_backend->blas.zrotg;
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zrotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_real_zrotg_")));
#else
void flexiblas_real_zrotg(void *a, void *b, void *c, void *s)
{
    flexiblas_real_zrotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif

void flexiblas_chain_zrotg_(void *a, void *b, void *c, void *s)
{
    void (*fn) (void *a, void *b, void *c, void *s);
    hook_pos_zrotg++;
    if ( hook_pos_zrotg < __flexiblas_hooks->zrotg.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zrotg.hook_function[hook_pos_zrotg];
    } else {
        hook_pos_zrotg = 0;
        *(void **) &fn = current_backend->blas.zrotg;
    }
    fn((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zrotg(void *a, void *b, void *c, void *s) __attribute__((alias("flexiblas_chain_zrotg_")));
#else
void flexiblas_chain_zrotg(void *a, void *b, void *c, void *s)
{
    flexiblas_chain_zrotg_((void *) a, (void *) b, (void *) c, (void *) s);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zscal = 0;
#endif

/* Wrapper for zscal */
void FC_GLOBAL(zscal,ZSCAL)
    (blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx)
{
    void (*fn) (void *n, void *za, void *zx, void *incx);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *za, void *zx, void *incx);
    hook_pos_zscal = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zscal;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zscal.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) za, (void *) zx, (void *) incx);
    } else {
        fn((void *) n, (void *) za, (void *) zx, (void *) incx);
    }
    #else
    fn((void *) n, (void *) za, (void *) zx, (void *) incx);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zscal,ZSCAL)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(zscal,ZSCAL)))));
void FC_GLOBAL3(zscal,ZSCAL)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx) __attribute__((alias(MTS(FC_GLOBAL(zscal,ZSCAL)))));
#else
void FC_GLOBAL2(zscal,ZSCAL)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx)
{
    FC_GLOBAL(zscal,ZSCAL)((void *) n, (void *) za, (void *) zx, (void *) incx);
}
void FC_GLOBAL3(zscal,ZSCAL)(blasint *n, blas_complex_double *za, blas_complex_double *zx, blasint *incx)
{
    FC_GLOBAL(zscal,ZSCAL)((void *) n, (void *) za, (void *) zx, (void *) incx);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zscal_(void *n, void *za, void *zx, void *incx)
{
    void (*fn) (void *n, void *za, void *zx, void *incx);
    *(void **) &fn = current_backend->blas.zscal;
    fn((void *) n, (void *) za, (void *) zx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zscal(void *n, void *za, void *zx, void *incx) __attribute__((alias("flexiblas_real_zscal_")));
#else
void flexiblas_real_zscal(void *n, void *za, void *zx, void *incx)
{
    flexiblas_real_zscal_((void *) n, (void *) za, (void *) zx, (void *) incx);
    return;
}
#endif

void flexiblas_chain_zscal_(void *n, void *za, void *zx, void *incx)
{
    void (*fn) (void *n, void *za, void *zx, void *incx);
    hook_pos_zscal++;
    if ( hook_pos_zscal < __flexiblas_hooks->zscal.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zscal.hook_function[hook_pos_zscal];
    } else {
        hook_pos_zscal = 0;
        *(void **) &fn = current_backend->blas.zscal;
    }
    fn((void *) n, (void *) za, (void *) zx, (void *) incx);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zscal(void *n, void *za, void *zx, void *incx) __attribute__((alias("flexiblas_chain_zscal_")));
#else
void flexiblas_chain_zscal(void *n, void *za, void *zx, void *incx)
{
    flexiblas_chain_zscal_((void *) n, (void *) za, (void *) zx, (void *) incx);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zswap = 0;
#endif

/* Wrapper for zswap */
void FC_GLOBAL(zswap,ZSWAP)
    (blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zswap = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zswap;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zswap.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    } else {
        fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    }
    #else
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zswap,ZSWAP)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zswap,ZSWAP)))));
void FC_GLOBAL3(zswap,ZSWAP)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy) __attribute__((alias(MTS(FC_GLOBAL(zswap,ZSWAP)))));
#else
void FC_GLOBAL2(zswap,ZSWAP)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zswap,ZSWAP)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
void FC_GLOBAL3(zswap,ZSWAP)(blasint *n, blas_complex_double *zx, blasint *incx, blas_complex_double *zy, blasint *incy)
{
    FC_GLOBAL(zswap,ZSWAP)((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zswap_(void *n, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    *(void **) &fn = current_backend->blas.zswap;
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zswap(void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_real_zswap_")));
#else
void flexiblas_real_zswap(void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_real_zswap_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif

void flexiblas_chain_zswap_(void *n, void *zx, void *incx, void *zy, void *incy)
{
    void (*fn) (void *n, void *zx, void *incx, void *zy, void *incy);
    hook_pos_zswap++;
    if ( hook_pos_zswap < __flexiblas_hooks->zswap.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zswap.hook_function[hook_pos_zswap];
    } else {
        hook_pos_zswap = 0;
        *(void **) &fn = current_backend->blas.zswap;
    }
    fn((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zswap(void *n, void *zx, void *incx, void *zy, void *incy) __attribute__((alias("flexiblas_chain_zswap_")));
#else
void flexiblas_chain_zswap(void *n, void *zx, void *incx, void *zy, void *incy)
{
    flexiblas_chain_zswap_((void *) n, (void *) zx, (void *) incx, (void *) zy, (void *) incy);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zsymm = 0;
#endif

/* Wrapper for zsymm */
void FC_GLOBAL(zsymm,ZSYMM)
    (char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zsymm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zsymm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zsymm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zsymm,ZSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zsymm,ZSYMM)))));
void FC_GLOBAL3(zsymm,ZSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias(MTS(FC_GLOBAL(zsymm,ZSYMM)))));
#else
void FC_GLOBAL2(zsymm,ZSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zsymm,ZSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
void FC_GLOBAL3(zsymm,ZSYMM)(char *side, char *uplo, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    FC_GLOBAL(zsymm,ZSYMM)((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zsymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    *(void **) &fn = current_backend->blas.zsymm;
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_real_zsymm_")));
#else
void flexiblas_real_zsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_real_zsymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif

void flexiblas_chain_zsymm_(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    void (*fn) (void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len);
    hook_pos_zsymm++;
    if ( hook_pos_zsymm < __flexiblas_hooks->zsymm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zsymm.hook_function[hook_pos_zsymm];
    } else {
        hook_pos_zsymm = 0;
        *(void **) &fn = current_backend->blas.zsymm;
    }
    fn((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len) __attribute__((alias("flexiblas_chain_zsymm_")));
#else
void flexiblas_chain_zsymm(void *side, void *uplo, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len)
{
    flexiblas_chain_zsymm_((void *) side, (void *) uplo, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, side_len, uplo_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zsyr2k = 0;
#endif

/* Wrapper for zsyr2k */
void FC_GLOBAL(zsyr2k,ZSYR2K)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zsyr2k = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zsyr2k;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zsyr2k.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zsyr2k,ZSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zsyr2k,ZSYR2K)))));
void FC_GLOBAL3(zsyr2k,ZSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zsyr2k,ZSYR2K)))));
#else
void FC_GLOBAL2(zsyr2k,ZSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zsyr2k,ZSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(zsyr2k,ZSYR2K)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zsyr2k,ZSYR2K)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zsyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zsyr2k;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zsyr2k_")));
#else
void flexiblas_real_zsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zsyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_zsyr2k_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zsyr2k++;
    if ( hook_pos_zsyr2k < __flexiblas_hooks->zsyr2k.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zsyr2k.hook_function[hook_pos_zsyr2k];
    } else {
        hook_pos_zsyr2k = 0;
        *(void **) &fn = current_backend->blas.zsyr2k;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zsyr2k_")));
#else
void flexiblas_chain_zsyr2k(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *b, void *ldb, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zsyr2k_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_zsyrk = 0;
#endif

/* Wrapper for zsyrk */
void FC_GLOBAL(zsyrk,ZSYRK)
    (char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zsyrk = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.zsyrk;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->zsyrk.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(zsyrk,ZSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zsyrk,ZSYRK)))));
void FC_GLOBAL3(zsyrk,ZSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias(MTS(FC_GLOBAL(zsyrk,ZSYRK)))));
#else
void FC_GLOBAL2(zsyrk,ZSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zsyrk,ZSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
void FC_GLOBAL3(zsyrk,ZSYRK)(char *uplo, char *trans, blasint *n, blasint *k, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *beta, blas_complex_double *c, blasint *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    FC_GLOBAL(zsyrk,ZSYRK)((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_zsyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    *(void **) &fn = current_backend->blas.zsyrk;
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_zsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_real_zsyrk_")));
#else
void flexiblas_real_zsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_real_zsyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif

void flexiblas_chain_zsyrk_(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    void (*fn) (void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len);
    hook_pos_zsyrk++;
    if ( hook_pos_zsyrk < __flexiblas_hooks->zsyrk.nhook ) {
        *(void **) &fn = __flexiblas_hooks->zsyrk.hook_function[hook_pos_zsyrk];
    } else {
        hook_pos_zsyrk = 0;
        *(void **) &fn = current_backend->blas.zsyrk;
    }
    fn((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_zsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len) __attribute__((alias("flexiblas_chain_zsyrk_")));
#else
void flexiblas_chain_zsyrk(void *uplo, void *trans, void *n, void *k, void *alpha, void *a, void *lda, void *beta, void *c, void *ldc, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len)
{
    flexiblas_chain_zsyrk_((void *) uplo, (void *) trans, (void *) n, (void *) k, (void *) alpha, (void *) a, (void *) lda, (void *) beta, (void *) c, (void *) ldc, uplo_len, trans_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztbmv = 0;
#endif

/* Wrapper for ztbmv */
void FC_GLOBAL(ztbmv,ZTBMV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztbmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztbmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztbmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztbmv,ZTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztbmv,ZTBMV)))));
void FC_GLOBAL3(ztbmv,ZTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztbmv,ZTBMV)))));
#else
void FC_GLOBAL2(ztbmv,ZTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztbmv,ZTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztbmv,ZTBMV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztbmv,ZTBMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztbmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztbmv_")));
#else
void flexiblas_real_ztbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztbmv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztbmv++;
    if ( hook_pos_ztbmv < __flexiblas_hooks->ztbmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztbmv.hook_function[hook_pos_ztbmv];
    } else {
        hook_pos_ztbmv = 0;
        *(void **) &fn = current_backend->blas.ztbmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztbmv_")));
#else
void flexiblas_chain_ztbmv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztbmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztbsv = 0;
#endif

/* Wrapper for ztbsv */
void FC_GLOBAL(ztbsv,ZTBSV)
    (char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztbsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztbsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztbsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztbsv,ZTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztbsv,ZTBSV)))));
void FC_GLOBAL3(ztbsv,ZTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztbsv,ZTBSV)))));
#else
void FC_GLOBAL2(ztbsv,ZTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztbsv,ZTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztbsv,ZTBSV)(char *uplo, char *trans, char *diag, blasint *n, blasint *k, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztbsv,ZTBSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztbsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztbsv_")));
#else
void flexiblas_real_ztbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztbsv_(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztbsv++;
    if ( hook_pos_ztbsv < __flexiblas_hooks->ztbsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztbsv.hook_function[hook_pos_ztbsv];
    } else {
        hook_pos_ztbsv = 0;
        *(void **) &fn = current_backend->blas.ztbsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztbsv_")));
#else
void flexiblas_chain_ztbsv(void *uplo, void *trans, void *diag, void *n, void *k, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztbsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) k, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztpmv = 0;
#endif

/* Wrapper for ztpmv */
void FC_GLOBAL(ztpmv,ZTPMV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztpmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztpmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztpmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztpmv,ZTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztpmv,ZTPMV)))));
void FC_GLOBAL3(ztpmv,ZTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztpmv,ZTPMV)))));
#else
void FC_GLOBAL2(ztpmv,ZTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztpmv,ZTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztpmv,ZTPMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztpmv,ZTPMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztpmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztpmv_")));
#else
void flexiblas_real_ztpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztpmv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztpmv++;
    if ( hook_pos_ztpmv < __flexiblas_hooks->ztpmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztpmv.hook_function[hook_pos_ztpmv];
    } else {
        hook_pos_ztpmv = 0;
        *(void **) &fn = current_backend->blas.ztpmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztpmv_")));
#else
void flexiblas_chain_ztpmv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztpmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztpsv = 0;
#endif

/* Wrapper for ztpsv */
void FC_GLOBAL(ztpsv,ZTPSV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztpsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztpsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztpsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztpsv,ZTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztpsv,ZTPSV)))));
void FC_GLOBAL3(ztpsv,ZTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztpsv,ZTPSV)))));
#else
void FC_GLOBAL2(ztpsv,ZTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztpsv,ZTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztpsv,ZTPSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *ap, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztpsv,ZTPSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztpsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztpsv_")));
#else
void flexiblas_real_ztpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztpsv_(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztpsv++;
    if ( hook_pos_ztpsv < __flexiblas_hooks->ztpsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztpsv.hook_function[hook_pos_ztpsv];
    } else {
        hook_pos_ztpsv = 0;
        *(void **) &fn = current_backend->blas.ztpsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztpsv_")));
#else
void flexiblas_chain_ztpsv(void *uplo, void *trans, void *diag, void *n, void *ap, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztpsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) ap, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztrmm = 0;
#endif

/* Wrapper for ztrmm */
void FC_GLOBAL(ztrmm,ZTRMM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrmm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztrmm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztrmm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztrmm,ZTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrmm,ZTRMM)))));
void FC_GLOBAL3(ztrmm,ZTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrmm,ZTRMM)))));
#else
void FC_GLOBAL2(ztrmm,ZTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrmm,ZTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(ztrmm,ZTRMM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrmm,ZTRMM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztrmm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztrmm_")));
#else
void flexiblas_real_ztrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztrmm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrmm++;
    if ( hook_pos_ztrmm < __flexiblas_hooks->ztrmm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztrmm.hook_function[hook_pos_ztrmm];
    } else {
        hook_pos_ztrmm = 0;
        *(void **) &fn = current_backend->blas.ztrmm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztrmm_")));
#else
void flexiblas_chain_ztrmm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztrmm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztrmv = 0;
#endif

/* Wrapper for ztrmv */
void FC_GLOBAL(ztrmv,ZTRMV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrmv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztrmv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztrmv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztrmv,ZTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrmv,ZTRMV)))));
void FC_GLOBAL3(ztrmv,ZTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrmv,ZTRMV)))));
#else
void FC_GLOBAL2(ztrmv,ZTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrmv,ZTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztrmv,ZTRMV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrmv,ZTRMV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztrmv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztrmv_")));
#else
void flexiblas_real_ztrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztrmv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrmv++;
    if ( hook_pos_ztrmv < __flexiblas_hooks->ztrmv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztrmv.hook_function[hook_pos_ztrmv];
    } else {
        hook_pos_ztrmv = 0;
        *(void **) &fn = current_backend->blas.ztrmv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztrmv_")));
#else
void flexiblas_chain_ztrmv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztrmv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztrsm = 0;
#endif

/* Wrapper for ztrsm */
void FC_GLOBAL(ztrsm,ZTRSM)
    (char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrsm = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztrsm;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztrsm.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    } else {
        fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    }
    #else
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztrsm,ZTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrsm,ZTRSM)))));
void FC_GLOBAL3(ztrsm,ZTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrsm,ZTRSM)))));
#else
void FC_GLOBAL2(ztrsm,ZTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrsm,ZTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
void FC_GLOBAL3(ztrsm,ZTRSM)(char *side, char *uplo, char *transa, char *diag, blasint *m, blasint *n, blas_complex_double *alpha, blas_complex_double *a, blasint *lda, blas_complex_double *b, blasint *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrsm,ZTRSM)((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztrsm;
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztrsm_")));
#else
void flexiblas_real_ztrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztrsm_(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrsm++;
    if ( hook_pos_ztrsm < __flexiblas_hooks->ztrsm.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztrsm.hook_function[hook_pos_ztrsm];
    } else {
        hook_pos_ztrsm = 0;
        *(void **) &fn = current_backend->blas.ztrsm;
    }
    fn((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztrsm_")));
#else
void flexiblas_chain_ztrsm(void *side, void *uplo, void *transa, void *diag, void *m, void *n, void *alpha, void *a, void *lda, void *b, void *ldb, flexiblas_fortran_charlen_t side_len, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t transa_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztrsm_((void *) side, (void *) uplo, (void *) transa, (void *) diag, (void *) m, (void *) n, (void *) alpha, (void *) a, (void *) lda, (void *) b, (void *) ldb, side_len, uplo_len, transa_len, diag_len);
    return;
}
#endif
#endif
#ifdef FLEXIBLAS_HOOK_API
static TLS_STORE uint8_t hook_pos_ztrsv = 0;
#endif

/* Wrapper for ztrsv */
void FC_GLOBAL(ztrsv,ZTRSV)
    (char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    #ifdef FLEXIBLAS_HOOK_API
    void (*fn_hook) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrsv = 0;
    #endif

    /* Post init if necessary */
    if ( unlikely(current_backend->post_init != 0) ) {
        __flexiblas_backend_init(current_backend);
        current_backend->post_init = 0;
    }

    *(void **) &fn = current_backend->blas.ztrsv;
    #ifdef FLEXIBLAS_HOOK_API
    *(void **) &fn_hook = __flexiblas_hooks->ztrsv.hook_function[0];

    if ( fn_hook != NULL ) {
        fn_hook((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    } else {
        fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    }
    #else
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    #endif
    return;
}

#ifndef __APPLE__
void FC_GLOBAL2(ztrsv,ZTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrsv,ZTRSV)))));
void FC_GLOBAL3(ztrsv,ZTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias(MTS(FC_GLOBAL(ztrsv,ZTRSV)))));
#else
void FC_GLOBAL2(ztrsv,ZTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrsv,ZTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
void FC_GLOBAL3(ztrsv,ZTRSV)(char *uplo, char *trans, char *diag, blasint *n, blas_complex_double *a, blasint *lda, blas_complex_double *x, blasint *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    FC_GLOBAL(ztrsv,ZTRSV)((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
}
#endif


#ifdef FLEXIBLAS_HOOK_API
void flexiblas_real_ztrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    *(void **) &fn = current_backend->blas.ztrsv;
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_real_ztrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_real_ztrsv_")));
#else
void flexiblas_real_ztrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_real_ztrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif

void flexiblas_chain_ztrsv_(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    void (*fn) (void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len);
    hook_pos_ztrsv++;
    if ( hook_pos_ztrsv < __flexiblas_hooks->ztrsv.nhook ) {
        *(void **) &fn = __flexiblas_hooks->ztrsv.hook_function[hook_pos_ztrsv];
    } else {
        hook_pos_ztrsv = 0;
        *(void **) &fn = current_backend->blas.ztrsv;
    }
    fn((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#ifndef __APPLE__
void flexiblas_chain_ztrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len) __attribute__((alias("flexiblas_chain_ztrsv_")));
#else
void flexiblas_chain_ztrsv(void *uplo, void *trans, void *diag, void *n, void *a, void *lda, void *x, void *incx, flexiblas_fortran_charlen_t uplo_len, flexiblas_fortran_charlen_t trans_len, flexiblas_fortran_charlen_t diag_len)
{
    flexiblas_chain_ztrsv_((void *) uplo, (void *) trans, (void *) diag, (void *) n, (void *) a, (void *) lda, (void *) x, (void *) incx, uplo_len, trans_len, diag_len);
    return;
}
#endif
#endif

HIDDEN int __flexiblas_load_fblas ( flexiblas_backend_t *handle, int *loaded, int *failed )  {
	int _ifailed = *failed;

    #ifndef FLEXIBLAS_INTEGER8
    if (handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64 ) {
        DPRINTF(0, "The selected BLAS library contains an ILP64 version of BLAS. Loading fallback instead."); 
    }
    #endif
  
    
    // LOAD_FBLAS(handle,blas.caxpby,caxpby);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "caxpby", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "caxpby", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.caxpby = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load caxpby from internal fallback BLAS.\n");
            handle->blas.caxpby = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load caxpby.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.caxpby = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load caxpby from internal fallback BLAS.\n");
            handle->blas.caxpby = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load caxpby.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.caxpy,caxpy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "caxpy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "caxpy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.caxpy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load caxpy from internal fallback BLAS.\n");
            handle->blas.caxpy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load caxpy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.caxpy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load caxpy from internal fallback BLAS.\n");
            handle->blas.caxpy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load caxpy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ccopy,ccopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ccopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ccopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ccopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ccopy from internal fallback BLAS.\n");
            handle->blas.ccopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ccopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ccopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ccopy from internal fallback BLAS.\n");
            handle->blas.ccopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ccopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cdotc,cdotc);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cdotc", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cdotc", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cdotc = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cdotc from internal fallback BLAS.\n");
            handle->blas.cdotc = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cdotc.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cdotc = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cdotc from internal fallback BLAS.\n");
            handle->blas.cdotc = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cdotc.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cdotu,cdotu);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cdotu", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cdotu", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cdotu = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cdotu from internal fallback BLAS.\n");
            handle->blas.cdotu = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cdotu.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cdotu = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cdotu from internal fallback BLAS.\n");
            handle->blas.cdotu = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cdotu.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgbmv,cgbmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgbmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgbmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgbmv from internal fallback BLAS.\n");
            handle->blas.cgbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgbmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgbmv from internal fallback BLAS.\n");
            handle->blas.cgbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgbmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgeadd,cgeadd);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgeadd", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgeadd", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgeadd = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgeadd from internal fallback BLAS.\n");
            handle->blas.cgeadd = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgeadd.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgeadd = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgeadd from internal fallback BLAS.\n");
            handle->blas.cgeadd = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgeadd.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgemm,cgemm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgemm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgemm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemm from internal fallback BLAS.\n");
            handle->blas.cgemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemm from internal fallback BLAS.\n");
            handle->blas.cgemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgemmtr,cgemmtr);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgemmtr", "cgemmt", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgemmtr", "cgemmt", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgemmtr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemmtr from internal fallback BLAS.\n");
            handle->blas.cgemmtr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemmtr.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgemmtr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemmtr from internal fallback BLAS.\n");
            handle->blas.cgemmtr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemmtr.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgemv,cgemv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgemv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgemv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemv from internal fallback BLAS.\n");
            handle->blas.cgemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgemv from internal fallback BLAS.\n");
            handle->blas.cgemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgemv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgerc,cgerc);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgerc", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgerc", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgerc = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgerc from internal fallback BLAS.\n");
            handle->blas.cgerc = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgerc.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgerc = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgerc from internal fallback BLAS.\n");
            handle->blas.cgerc = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgerc.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cgeru,cgeru);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cgeru", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cgeru", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cgeru = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgeru from internal fallback BLAS.\n");
            handle->blas.cgeru = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgeru.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cgeru = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cgeru from internal fallback BLAS.\n");
            handle->blas.cgeru = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cgeru.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chbmv,chbmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chbmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chbmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chbmv from internal fallback BLAS.\n");
            handle->blas.chbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chbmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chbmv from internal fallback BLAS.\n");
            handle->blas.chbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chbmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chemm,chemm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chemm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chemm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chemm from internal fallback BLAS.\n");
            handle->blas.chemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chemm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chemm from internal fallback BLAS.\n");
            handle->blas.chemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chemm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chemv,chemv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chemv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chemv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chemv from internal fallback BLAS.\n");
            handle->blas.chemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chemv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chemv from internal fallback BLAS.\n");
            handle->blas.chemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chemv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cher,cher);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cher", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cher", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cher = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher from internal fallback BLAS.\n");
            handle->blas.cher = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cher = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher from internal fallback BLAS.\n");
            handle->blas.cher = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cher2,cher2);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cher2", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cher2", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cher2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher2 from internal fallback BLAS.\n");
            handle->blas.cher2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher2.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cher2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher2 from internal fallback BLAS.\n");
            handle->blas.cher2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher2.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cher2k,cher2k);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cher2k", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cher2k", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cher2k = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher2k from internal fallback BLAS.\n");
            handle->blas.cher2k = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher2k.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cher2k = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cher2k from internal fallback BLAS.\n");
            handle->blas.cher2k = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cher2k.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cherk,cherk);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cherk", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cherk", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cherk = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cherk from internal fallback BLAS.\n");
            handle->blas.cherk = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cherk.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cherk = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cherk from internal fallback BLAS.\n");
            handle->blas.cherk = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cherk.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chpmv,chpmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chpmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chpmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chpmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpmv from internal fallback BLAS.\n");
            handle->blas.chpmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chpmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpmv from internal fallback BLAS.\n");
            handle->blas.chpmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chpr,chpr);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chpr", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chpr", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chpr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpr from internal fallback BLAS.\n");
            handle->blas.chpr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpr.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chpr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpr from internal fallback BLAS.\n");
            handle->blas.chpr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpr.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.chpr2,chpr2);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "chpr2", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "chpr2", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.chpr2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpr2 from internal fallback BLAS.\n");
            handle->blas.chpr2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpr2.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.chpr2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load chpr2 from internal fallback BLAS.\n");
            handle->blas.chpr2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load chpr2.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cimatcopy,cimatcopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cimatcopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cimatcopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cimatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cimatcopy from internal fallback BLAS.\n");
            handle->blas.cimatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cimatcopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cimatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cimatcopy from internal fallback BLAS.\n");
            handle->blas.cimatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cimatcopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.comatcopy,comatcopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "comatcopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "comatcopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.comatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load comatcopy from internal fallback BLAS.\n");
            handle->blas.comatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load comatcopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.comatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load comatcopy from internal fallback BLAS.\n");
            handle->blas.comatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load comatcopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.crotg,crotg);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "crotg", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "crotg", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.crotg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load crotg from internal fallback BLAS.\n");
            handle->blas.crotg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load crotg.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.crotg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load crotg from internal fallback BLAS.\n");
            handle->blas.crotg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load crotg.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cscal,cscal);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cscal", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cscal", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cscal from internal fallback BLAS.\n");
            handle->blas.cscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cscal.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cscal from internal fallback BLAS.\n");
            handle->blas.cscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cscal.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.csrot,csrot);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "csrot", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "csrot", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.csrot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csrot from internal fallback BLAS.\n");
            handle->blas.csrot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csrot.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.csrot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csrot from internal fallback BLAS.\n");
            handle->blas.csrot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csrot.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.csscal,csscal);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "csscal", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "csscal", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.csscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csscal from internal fallback BLAS.\n");
            handle->blas.csscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csscal.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.csscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csscal from internal fallback BLAS.\n");
            handle->blas.csscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csscal.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.cswap,cswap);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "cswap", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "cswap", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.cswap = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cswap from internal fallback BLAS.\n");
            handle->blas.cswap = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cswap.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.cswap = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load cswap from internal fallback BLAS.\n");
            handle->blas.cswap = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load cswap.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.csymm,csymm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "csymm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "csymm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.csymm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csymm from internal fallback BLAS.\n");
            handle->blas.csymm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csymm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.csymm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csymm from internal fallback BLAS.\n");
            handle->blas.csymm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csymm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.csyr2k,csyr2k);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "csyr2k", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "csyr2k", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.csyr2k = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csyr2k from internal fallback BLAS.\n");
            handle->blas.csyr2k = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csyr2k.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.csyr2k = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csyr2k from internal fallback BLAS.\n");
            handle->blas.csyr2k = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csyr2k.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.csyrk,csyrk);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "csyrk", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "csyrk", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.csyrk = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csyrk from internal fallback BLAS.\n");
            handle->blas.csyrk = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csyrk.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.csyrk = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load csyrk from internal fallback BLAS.\n");
            handle->blas.csyrk = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load csyrk.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctbmv,ctbmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctbmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctbmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctbmv from internal fallback BLAS.\n");
            handle->blas.ctbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctbmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctbmv from internal fallback BLAS.\n");
            handle->blas.ctbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctbmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctbsv,ctbsv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctbsv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctbsv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctbsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctbsv from internal fallback BLAS.\n");
            handle->blas.ctbsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctbsv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctbsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctbsv from internal fallback BLAS.\n");
            handle->blas.ctbsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctbsv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctpmv,ctpmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctpmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctpmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctpmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctpmv from internal fallback BLAS.\n");
            handle->blas.ctpmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctpmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctpmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctpmv from internal fallback BLAS.\n");
            handle->blas.ctpmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctpmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctpsv,ctpsv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctpsv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctpsv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctpsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctpsv from internal fallback BLAS.\n");
            handle->blas.ctpsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctpsv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctpsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctpsv from internal fallback BLAS.\n");
            handle->blas.ctpsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctpsv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctrmm,ctrmm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctrmm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctrmm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctrmm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrmm from internal fallback BLAS.\n");
            handle->blas.ctrmm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrmm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctrmm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrmm from internal fallback BLAS.\n");
            handle->blas.ctrmm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrmm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctrmv,ctrmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctrmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctrmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctrmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrmv from internal fallback BLAS.\n");
            handle->blas.ctrmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctrmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrmv from internal fallback BLAS.\n");
            handle->blas.ctrmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctrsm,ctrsm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctrsm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctrsm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctrsm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrsm from internal fallback BLAS.\n");
            handle->blas.ctrsm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrsm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctrsm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrsm from internal fallback BLAS.\n");
            handle->blas.ctrsm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrsm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ctrsv,ctrsv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ctrsv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ctrsv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ctrsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrsv from internal fallback BLAS.\n");
            handle->blas.ctrsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrsv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ctrsv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ctrsv from internal fallback BLAS.\n");
            handle->blas.ctrsv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ctrsv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dasum,dasum);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dasum", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dasum", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dasum = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dasum from internal fallback BLAS.\n");
            handle->blas.dasum = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dasum.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dasum = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dasum from internal fallback BLAS.\n");
            handle->blas.dasum = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dasum.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.daxpby,daxpby);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "daxpby", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "daxpby", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.daxpby = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load daxpby from internal fallback BLAS.\n");
            handle->blas.daxpby = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load daxpby.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.daxpby = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load daxpby from internal fallback BLAS.\n");
            handle->blas.daxpby = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load daxpby.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.daxpy,daxpy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "daxpy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "daxpy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.daxpy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load daxpy from internal fallback BLAS.\n");
            handle->blas.daxpy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load daxpy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.daxpy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load daxpy from internal fallback BLAS.\n");
            handle->blas.daxpy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load daxpy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dcopy,dcopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dcopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dcopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dcopy from internal fallback BLAS.\n");
            handle->blas.dcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dcopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dcopy from internal fallback BLAS.\n");
            handle->blas.dcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dcopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.ddot,ddot);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "ddot", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "ddot", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.ddot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ddot from internal fallback BLAS.\n");
            handle->blas.ddot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ddot.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.ddot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load ddot from internal fallback BLAS.\n");
            handle->blas.ddot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load ddot.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dgbmv,dgbmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dgbmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dgbmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dgbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgbmv from internal fallback BLAS.\n");
            handle->blas.dgbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgbmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dgbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgbmv from internal fallback BLAS.\n");
            handle->blas.dgbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgbmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dgeadd,dgeadd);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dgeadd", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dgeadd", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dgeadd = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgeadd from internal fallback BLAS.\n");
            handle->blas.dgeadd = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgeadd.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dgeadd = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgeadd from internal fallback BLAS.\n");
            handle->blas.dgeadd = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgeadd.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dgemm,dgemm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dgemm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dgemm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dgemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemm from internal fallback BLAS.\n");
            handle->blas.dgemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dgemm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemm from internal fallback BLAS.\n");
            handle->blas.dgemm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dgemmtr,dgemmtr);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dgemmtr", "dgemmt", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dgemmtr", "dgemmt", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dgemmtr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemmtr from internal fallback BLAS.\n");
            handle->blas.dgemmtr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemmtr.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dgemmtr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemmtr from internal fallback BLAS.\n");
            handle->blas.dgemmtr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemmtr.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dgemv,dgemv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dgemv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dgemv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dgemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemv from internal fallback BLAS.\n");
            handle->blas.dgemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dgemv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dgemv from internal fallback BLAS.\n");
            handle->blas.dgemv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dgemv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dger,dger);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dger", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dger", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dger = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dger from internal fallback BLAS.\n");
            handle->blas.dger = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dger.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dger = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dger from internal fallback BLAS.\n");
            handle->blas.dger = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dger.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dimatcopy,dimatcopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dimatcopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dimatcopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dimatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dimatcopy from internal fallback BLAS.\n");
            handle->blas.dimatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dimatcopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dimatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dimatcopy from internal fallback BLAS.\n");
            handle->blas.dimatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dimatcopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dnrm2,dnrm2);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dnrm2", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dnrm2", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dnrm2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dnrm2 from internal fallback BLAS.\n");
            handle->blas.dnrm2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dnrm2.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dnrm2 = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dnrm2 from internal fallback BLAS.\n");
            handle->blas.dnrm2 = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dnrm2.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.domatcopy,domatcopy);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "domatcopy", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "domatcopy", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.domatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load domatcopy from internal fallback BLAS.\n");
            handle->blas.domatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load domatcopy.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.domatcopy = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load domatcopy from internal fallback BLAS.\n");
            handle->blas.domatcopy = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load domatcopy.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.drot,drot);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "drot", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "drot", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.drot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drot from internal fallback BLAS.\n");
            handle->blas.drot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drot.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.drot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drot from internal fallback BLAS.\n");
            handle->blas.drot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drot.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.drotg,drotg);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "drotg", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "drotg", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.drotg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotg from internal fallback BLAS.\n");
            handle->blas.drotg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotg.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.drotg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotg from internal fallback BLAS.\n");
            handle->blas.drotg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotg.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.drotm,drotm);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "drotm", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "drotm", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.drotm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotm from internal fallback BLAS.\n");
            handle->blas.drotm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotm.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.drotm = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotm from internal fallback BLAS.\n");
            handle->blas.drotm = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotm.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.drotmg,drotmg);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "drotmg", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "drotmg", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.drotmg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotmg from internal fallback BLAS.\n");
            handle->blas.drotmg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotmg.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.drotmg = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load drotmg from internal fallback BLAS.\n");
            handle->blas.drotmg = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load drotmg.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dsbmv,dsbmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dsbmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dsbmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dsbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dsbmv from internal fallback BLAS.\n");
            handle->blas.dsbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dsbmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dsbmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dsbmv from internal fallback BLAS.\n");
            handle->blas.dsbmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dsbmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dscal,dscal);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dscal", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dscal", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dscal from internal fallback BLAS.\n");
            handle->blas.dscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dscal.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dscal = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dscal from internal fallback BLAS.\n");
            handle->blas.dscal = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dscal.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dsdot,dsdot);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dsdot", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dsdot", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dsdot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dsdot from internal fallback BLAS.\n");
            handle->blas.dsdot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dsdot.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dsdot = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dsdot from internal fallback BLAS.\n");
            handle->blas.dsdot = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dsdot.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dspmv,dspmv);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dspmv", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dspmv", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dspmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dspmv from internal fallback BLAS.\n");
            handle->blas.dspmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dspmv.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dspmv = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dspmv from internal fallback BLAS.\n");
            handle->blas.dspmv = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dspmv.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dspr,dspr);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle, "dspr", NULL);
        void *ptr_fallback = __flexiblas_lookup_fortran_function(__flexiblas_blas_fallback, "dspr", NULL);

        #ifndef FLEXIBLAS_INTEGER8
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_LP64) {
            handle->blas.dspr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dspr from internal fallback BLAS.\n");
            handle->blas.dspr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dspr.\n");
        }
        #else 
        if ( ptr_library != NULL && handle->integer_interface == FLEXIBLAS_INTERFACE_ILP64) {
            handle->blas.dspr = ptr_library;
            *loaded = *loaded +1; 
        } else if (ptr_fallback != NULL ) {
            DPRINTF(2, "Load dspr from internal fallback BLAS.\n");
            handle->blas.dspr = ptr_fallback;
            *loaded = *loaded +1; 
        } else {
            *failed = *failed + 1; 
            flexiblas_print_error("flexiblas",__FILE__, __LINE__, "Can not load dspr.\n");
        }
        #endif
    } while(0);



    // LOAD_FBLAS(handle,blas.dspr2,dspr2);
    do {
        void *ptr_library = __flexiblas_lookup_fortran_function(handle->library_handle,