; AleoBFT Library
;
; Copyright (C) 2024 Provable Inc.
;
; License: See the LICENSE file distributed with this library.
;
; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (bendyarm on GitHub)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ALEOBFT-DYNAMIC")

(include-book "unequivocal-accepted-certificates-def-and-init")

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ unequivocal-dags
  :parents (correctness)
  :short "Invariant that DAGs of correct validators are unequivocal."
  :long
  (xdoc::topstring
   (xdoc::p
    "Here we state and prove that the DAGs of two validators are unequivocal.
     This is a simple consequence of the invariant that
     the sets of accepted certificates of two validators are unequivocal.
     The proof of that invariant needs
     the whole set of accepted certificates (namely, DAG and buffer)
     for a sufficiently strong induction hypothesis;
     however, for the purpose of proving further properties of the protocol,
     we are only interested in the non-equivocation of DAGs.
     So here we state and prove the non-equivocation of DAGs.")
   (xdoc::p
    "The two validators could be the same validator,
     in which case we obtain the special case that
     the DAG of a validator is unequivocal.
     This is similar to the situation for accepted certificates:
     see @('unequivocal-accepted-certificates-p-necc-single')
     in @(tsee unequivocal-accepted-certificates-p)."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-sk unequivocal-dags-p ((systate system-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant:
          the DAGs of every pair of correct validator are unequivocal."
  :long
  (xdoc::topstring
   (xdoc::p
    "We prove that this invariant,
     which is defined as non-equivocation for two validators,
     implies non-equivocation for a single validator.
     This is similar to the @('-necc') rule generated by the definition,
     but it involves a single validator;
     this similarity is the reason for the name of the rule."))
  (forall (val1 val2)
          (implies (and (set::in val1 (correct-addresses systate))
                        (set::in val2 (correct-addresses systate)))
                   (certificate-sets-unequivocalp
                    (validator-state->dag
                     (get-validator-state val1 systate))
                    (validator-state->dag
                     (get-validator-state val2 systate)))))

  ///

  (defruled unequivocal-dags-necc-single
    (implies (and (unequivocal-dags-p systate)
                  (set::in val (correct-addresses systate)))
             (certificate-set-unequivocalp
              (validator-state->dag
               (get-validator-state val systate))))
    :enable certificate-sets-unequivocalp-of-same-set-converse))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled unequivocal-dags-p-when-unequivocal-accepted-certificates-p
  :short "Proof of the specialized invariant from the general invariant."
  :long
  (xdoc::topstring
   (xdoc::p
    "We just show that this system-level invariant
     is implied by the more general one.
     There is no need to prove explicitly
     the establishment and preservation of the specialized invariant,
     as that has already been done for the more general one,
     and the specialized one is just implied by that."))
  (implies (unequivocal-accepted-certificates-p systate)
           (unequivocal-dags-p systate))
  :enable (unequivocal-dags-p
           accepted-certificates)
  :use ((:instance unequivocal-accepted-certificates-p-necc
                   (val1 (mv-nth 0 (unequivocal-dags-p-witness systate)))
                   (val2 (mv-nth 1 (unequivocal-dags-p-witness systate))))
        (:instance certificate-sets-unequivocalp-when-subsets
                   (certs01
                    (validator-state->dag
                     (get-validator-state
                      (mv-nth 0 (unequivocal-dags-p-witness systate))
                      systate)))
                   (certs02
                    (validator-state->dag
                     (get-validator-state
                      (mv-nth 1 (unequivocal-dags-p-witness systate))
                      systate)))
                   (certs1 (accepted-certificates
                            (mv-nth 0 (unequivocal-dags-p-witness systate))
                            systate))
                   (certs2 (accepted-certificates
                            (mv-nth 1 (unequivocal-dags-p-witness systate))
                            systate)))))
