//! Implementation code for testing a channel manager.

// TODO: this should probably move into a crate of its own, along with
// some of our testing code from tor-proto, with which it shares some
// logic.

use futures::io::{AsyncWrite, AsyncWriteExt};
use std::io::Result as IoResult;

/// Respond to a newly opened channel using a canned set of certificates.
///
/// Note that this isn't a real responder-side handshake, and doesn't check
/// whether the client is behaving correctly at all.
pub(crate) async fn answer_channel_req<S>(conn: &mut S) -> IoResult<()>
where
    S: AsyncWrite + Send + Unpin,
{
    conn.write_all(msgs::VERSIONS_CELL).await?;
    conn.write_all(msgs::CERTS_CELL).await?;
    let mut netinfo_cell: Vec<u8> = msgs::NETINFO_PREFIX.into();
    netinfo_cell.resize(514, 0);
    conn.write_all(&netinfo_cell[..]).await?;

    Ok(())
}

/// Canned messages to send in order to complete a simple relay
/// handshake.
///
/// Taken from the a client on the Tor network around April 29, 2020.
pub(crate) mod msgs {
    use hex_literal::hex;

    // Address of the target relay.
    pub(crate) const ADDR: &str = "75.176.45.87:9001";

    // Identities of the target relay
    pub(crate) const ED_ID: [u8; 32] =
        hex!("b9f1a5e4f6c3106d7a8bfd622b6bb2b532f999309cd598ab3e527f35db88547a");
    pub(crate) const RSA_ID: [u8; 20] = hex!("5492e760648aabb7bec08bc87e8f72f55fbba90d");

    // A time (in seconds since unix epoch) at which the test is expected to succeed.
    pub(crate) const NOW: u64 = 1_619_710_465;

    /// contents of a versions cell.
    pub(crate) const VERSIONS_CELL: &[u8] = &hex!("0000 07 0006 0003 0004 0005");

    /// contents of the relay's x.509 certificate
    pub(crate) const X509_CERT: &[u8] = &hex!(
        "30820242308201aba00302010202081376e7ffb6100370300d06092a864886f70d01010b05003020311e301c06035504030c157777772e623563343577676f696166786d2e636f6d301e170d3230313230373030303030305a170d3231303630393233353935395a30233121301f06035504030c187777772e6a72336c6974353735696a7a6f6767692e6e657430820122300d06092a864886f70d01010105000382010f003082010a0282010100ce630e2570ada1b2347d705235cd8b63eee8b4b0dbc10729019c0fd22c9d074a33bb57d4e30bc952de9b7f8be31852b1e9bb0d8be9df028be0321f2786e1fb496698e1c1b19a6f66e9edddb05f9d723abe3cfdc8210c13f6f1670dca4d1c19b07675469bfd71cfedec070e1d53b14f877dab3a7b2f2574fed00e8a2f9faf3bec34757ef2e12d457814ffc4f26690ef29608acac7bf44d0a52e71a41d69193d958680c1b198ff339d34a5fb935d35be3ef70a855d44612b8ed71a611c6a5b102fc9a0a8bc98a4e06baa705ba12617b59843d5286e66baf5b5bbc758cc7b9cd7cf034b51f13a2778e858e62878b6812deac178d9b3fc3a39c4b3babf9d9a0d67010203010001300d06092a864886f70d01010b05000381810091c52d227b226d3056538ed5d6b31499f4476b384cad4d8b9bfde657e48150b28fbad7c635ad9e57fa085195c8f097d1fe8431deb3c6582c2f7cd99ba5aa0f471b40f235bf1f20184e38c666b8e342a1d9ef0822d8959673d2ed697f0e34edad0b1d07116174b544a6b8bc3ee1a7abe6db87b2fe3b854a8ea3cabf5a6a6f72f3"
    );

    /// contents of the relay's CERTS cell
    pub(crate) const CERTS_CELL: &[u8] = &hex!(
        "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"
    );

    /// Start of the relay's netinfo cell. (Pad up to 514 bytes.)
    pub(crate) const NETINFO_PREFIX: &[u8] =
        &hex!("0000000008608acef00404485d55f30104044bb02d570000");
}
