'use strict';

const fixtures = require('../../common/fixtures');

function getKeyFileName(type, suffix) {
  return `${type.replaceAll('-', '_')}_${suffix}.pem`;
}

module.exports = function() {
  const pkcs8 = {
    'ML-DSA-44': fixtures.readKey(getKeyFileName('ml-dsa-44', 'private_seed_only'), 'ascii'),
    'ML-DSA-65': fixtures.readKey(getKeyFileName('ml-dsa-65', 'private_seed_only'), 'ascii'),
    'ML-DSA-87': fixtures.readKey(getKeyFileName('ml-dsa-87', 'private_seed_only'), 'ascii'),
  }

  const spki = {
    'ML-DSA-44': fixtures.readKey(getKeyFileName('ml-dsa-44', 'public'), 'ascii'),
    'ML-DSA-65': fixtures.readKey(getKeyFileName('ml-dsa-65', 'public'), 'ascii'),
    'ML-DSA-87': fixtures.readKey(getKeyFileName('ml-dsa-87', 'public'), 'ascii'),
  }

  const data = Buffer.from(
    '2b7ed0bc7795694ab4acd35903fe8cd7d80f6a1c8688a6c3414409457514a1457855bb' +
    'b219e30a1beea8fe869082d99fc8282f9050d024e59eaf0730ba9db70a', 'hex');

  // For verification tests.
  const signatures = {
    // eslint-disable-next-line @stylistic/js/max-len
    'ML-DSA-44': Buffer.from('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', 'hex'),
    // eslint-disable-next-line @stylistic/js/max-len
    'ML-DSA-65': Buffer.from('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', 'hex'),
    // eslint-disable-next-line @stylistic/js/max-len
    'ML-DSA-87': Buffer.from('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', 'hex'),
  }

  const algorithms = ['ML-DSA-44', 'ML-DSA-65', 'ML-DSA-87'];

  const vectors = algorithms.map((algorithm) => ({
    publicKeyPem: spki[algorithm],
    privateKeyPem: pkcs8[algorithm],
    name: algorithm,
    data,
    signature: signatures[algorithm],
  }));

  return vectors;
}
