/* Example use of std::error implicit conversion
(C) 2024-2025 Niall Douglas <http://www.nedproductions.biz/> (5 commits)
File Created: Sept 2018


Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "../../../include/boost/outcome/experimental/status-code/quick_status_code_from_enum.hpp"
#include "../../../include/boost/outcome/experimental/status_result.hpp"
#include "outcome/experimental/status-code/config.hpp"


//! [preamble]
// My custom enum type
enum class custom_failure
{
  not_found,
  bad_argument
};

// Tell `status_code` to stamp out a custom code domain for this enum type
BOOST_OUTCOME_SYSTEM_ERROR2_NAMESPACE_BEGIN
template <> struct quick_status_code_from_enum<custom_failure> : quick_status_code_from_enum_defaults<custom_failure>
{
  // Text name of the enum
  static constexpr const auto domain_name = "My custom failure";
  // Unique UUID for the enum. PLEASE use https://www.random.org/cgi-bin/randbyte?nbytes=16&format=h
  static constexpr const auto domain_uuid = "{be201f65-3962-dd0e-1266-a72e63776a42}";
  // Map of each enum value to its text string, and list of semantically equivalent errc's
  static const std::initializer_list<mapping> &value_mappings()
  {
    static const std::initializer_list<mapping> v = {
    // Format is: { enum value, "string representation", { list of errc mappings ... } }
    {custom_failure::not_found, "item not found", {errc::no_such_file_or_directory}},  //
    {custom_failure::bad_argument, "invoked wrong", {errc::invalid_argument}},         //
    };
    return v;
  }
  // Completely optional definition of mixin for the status code synthesised from `Enum`. It can be omitted.
  template <class Base> struct mixin : Base
  {
    using Base::Base;
    constexpr int custom_method() const { return 42; }
  };
};
BOOST_OUTCOME_SYSTEM_ERROR2_NAMESPACE_END
//! [preamble]

//! [implicit_construction]
// This is the status code generated for your custom enum type. It will implicitly construct from
// values of enum custom_failure.
using custom_failure_code = BOOST_OUTCOME_SYSTEM_ERROR2_NAMESPACE::quick_status_code_from_enum_code<custom_failure>;

namespace outcome_e = BOOST_OUTCOME_V2_NAMESPACE::experimental;

// You don't usually need to use the status code type explicitly, because this "just works":
outcome_e::status_result<int> positive_only(int x)
{
  if(x < 0)
  {
    // Outcome's result sees that status_code will implicitly construct from this enum,
    // and it returns an errored result
    return custom_failure::bad_argument;
  }
  return x;
}

// Semantic comparisons work
bool test(int x)
{
  if(auto r = positive_only(x); !r)
  {
    if(r.error() == outcome_e::errc::invalid_argument)
    {
      std::cerr << "Positive numbers only!" << std::endl;
      return false;
    }
  }
  return true;
}
//! [implicit_construction]

int main(void)
{
  if(!test(0))
    abort();
  if(test(-1))
    abort();
  return 0;
}
