/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.entitlement.runtime.policy;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.stream.Stream;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.entitlement.runtime.policy.FileUtils;
import org.elasticsearch.entitlement.runtime.policy.PathLookup;
import org.elasticsearch.entitlement.runtime.policy.Platform;
import org.elasticsearch.entitlement.runtime.policy.entitlements.FilesEntitlement;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;

public final class FileAccessTree {
    private static final Logger logger = LogManager.getLogger(FileAccessTree.class);
    private static final String FILE_SEPARATOR = PathUtils.getDefaultFileSystem().getSeparator();
    private final String[] exclusivePaths;
    private final String[] readPaths;
    private final String[] writePaths;

    static List<ExclusivePath> buildExclusivePathList(List<ExclusiveFileEntitlement> exclusiveFileEntitlements, PathLookup pathLookup) {
        HashMap<String, ExclusivePath> exclusivePaths = new HashMap<String, ExclusivePath>();
        for (ExclusiveFileEntitlement efe : exclusiveFileEntitlements) {
            for (FilesEntitlement.FileData fd : efe.filesEntitlement().filesData()) {
                if (!fd.exclusive()) continue;
                List<Path> paths = fd.resolvePaths(pathLookup).toList();
                for (Path path : paths) {
                    String normalizedPath = FileAccessTree.normalizePath(path);
                    ExclusivePath exclusivePath = exclusivePaths.computeIfAbsent(normalizedPath, k -> new ExclusivePath(efe.componentName(), new HashSet<String>(), normalizedPath));
                    if (!exclusivePath.componentName().equals(efe.componentName())) {
                        throw new IllegalArgumentException("Path [" + normalizedPath + "] is already exclusive to [" + exclusivePath.componentName() + "]" + String.valueOf(exclusivePath.moduleNames) + ", cannot add exclusive access for [" + efe.componentName() + "][" + efe.moduleName + "]");
                    }
                    exclusivePath.moduleNames.add(efe.moduleName());
                }
            }
        }
        return exclusivePaths.values().stream().sorted(Comparator.comparing(ExclusivePath::path, FileUtils.PATH_ORDER)).distinct().toList();
    }

    static void validateExclusivePaths(List<ExclusivePath> exclusivePaths) {
        if (!exclusivePaths.isEmpty()) {
            ExclusivePath currentExclusivePath = exclusivePaths.get(0);
            for (int i = 1; i < exclusivePaths.size(); ++i) {
                ExclusivePath nextPath = exclusivePaths.get(i);
                if (currentExclusivePath.path().equals(nextPath.path) || FileAccessTree.isParent(currentExclusivePath.path(), nextPath.path())) {
                    throw new IllegalArgumentException("duplicate/overlapping exclusive paths found in files entitlements: " + String.valueOf(currentExclusivePath) + " and " + String.valueOf(nextPath));
                }
                currentExclusivePath = nextPath;
            }
        }
    }

    private FileAccessTree(String componentName, String moduleName, FilesEntitlement filesEntitlement, PathLookup pathLookup, Path componentPath, List<ExclusivePath> exclusivePaths) {
        ArrayList<String> updatedExclusivePaths = new ArrayList<String>();
        for (ExclusivePath exclusivePath : exclusivePaths) {
            if (exclusivePath.componentName().equals(componentName) && exclusivePath.moduleNames().contains(moduleName)) continue;
            updatedExclusivePaths.add(exclusivePath.path());
        }
        ArrayList<String> readPaths = new ArrayList<String>();
        ArrayList<String> writePaths = new ArrayList<String>();
        BiConsumer<Path, FilesEntitlement.Mode> addPath = (path, mode) -> {
            String normalized = FileAccessTree.normalizePath(path);
            if (mode == FilesEntitlement.Mode.READ_WRITE) {
                writePaths.add(normalized);
            }
            readPaths.add(normalized);
        };
        BiConsumer<Path, FilesEntitlement.Mode> addPathAndMaybeLink = (path, mode) -> {
            addPath.accept((Path)path, (FilesEntitlement.Mode)((Object)mode));
            if (Files.exists(path, new LinkOption[0])) {
                try {
                    Path realPath = path.toRealPath(new LinkOption[0]);
                    if (!realPath.equals(path)) {
                        addPath.accept(realPath, (FilesEntitlement.Mode)((Object)mode));
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
        };
        for (FilesEntitlement.FileData fileData : filesEntitlement.filesData()) {
            Platform platform = fileData.platform();
            if (platform != null && !platform.isCurrent()) continue;
            FilesEntitlement.Mode mode2 = fileData.mode();
            Stream<Path> paths = fileData.resolvePaths(pathLookup);
            paths.forEach(path -> {
                if (path == null) {
                    return;
                }
                addPathAndMaybeLink.accept((Path)path, mode2);
            });
        }
        addPathAndMaybeLink.accept(pathLookup.tempDir(), FilesEntitlement.Mode.READ_WRITE);
        addPathAndMaybeLink.accept(pathLookup.configDir(), FilesEntitlement.Mode.READ);
        if (componentPath != null) {
            addPathAndMaybeLink.accept(componentPath, FilesEntitlement.Mode.READ);
        }
        Path jdk = Paths.get(System.getProperty("java.home"), new String[0]);
        addPathAndMaybeLink.accept(jdk.resolve("conf"), FilesEntitlement.Mode.READ);
        updatedExclusivePaths.sort(FileUtils.PATH_ORDER);
        readPaths.sort(FileUtils.PATH_ORDER);
        writePaths.sort(FileUtils.PATH_ORDER);
        this.exclusivePaths = updatedExclusivePaths.toArray(new String[0]);
        this.readPaths = FileAccessTree.pruneSortedPaths(readPaths).toArray(new String[0]);
        this.writePaths = FileAccessTree.pruneSortedPaths(writePaths).toArray(new String[0]);
    }

    static List<String> pruneSortedPaths(List<String> paths) {
        ArrayList<String> prunedReadPaths = new ArrayList<String>();
        if (!paths.isEmpty()) {
            String currentPath = paths.get(0);
            prunedReadPaths.add(currentPath);
            for (int i = 1; i < paths.size(); ++i) {
                String nextPath = paths.get(i);
                if (currentPath.equals(nextPath) || FileAccessTree.isParent(currentPath, nextPath)) continue;
                prunedReadPaths.add(nextPath);
                currentPath = nextPath;
            }
        }
        return prunedReadPaths;
    }

    public static FileAccessTree of(String componentName, String moduleName, FilesEntitlement filesEntitlement, PathLookup pathLookup, @Nullable Path componentPath, List<ExclusivePath> exclusivePaths) {
        return new FileAccessTree(componentName, moduleName, filesEntitlement, pathLookup, componentPath, exclusivePaths);
    }

    boolean canRead(Path path) {
        return this.checkPath(FileAccessTree.normalizePath(path), this.readPaths);
    }

    boolean canWrite(Path path) {
        return this.checkPath(FileAccessTree.normalizePath(path), this.writePaths);
    }

    static String normalizePath(Path path) {
        String result = path.toAbsolutePath().normalize().toString();
        while (result.endsWith(FILE_SEPARATOR)) {
            result = result.substring(0, result.length() - FILE_SEPARATOR.length());
        }
        return result;
    }

    private boolean checkPath(String path, String[] paths) {
        logger.trace(() -> Strings.format((String)"checking [%s] against [%s]", (Object[])new Object[]{path, String.join((CharSequence)",", paths)}));
        if (paths.length == 0) {
            return false;
        }
        int endx = Arrays.binarySearch(this.exclusivePaths, path, FileUtils.PATH_ORDER);
        if (endx < -1 && FileAccessTree.isParent(this.exclusivePaths[-endx - 2], path) || endx >= 0) {
            return false;
        }
        int ndx = Arrays.binarySearch(paths, path, FileUtils.PATH_ORDER);
        if (ndx < -1) {
            return FileAccessTree.isParent(paths[-ndx - 2], path);
        }
        return ndx >= 0;
    }

    private static boolean isParent(String maybeParent, String path) {
        logger.trace(() -> Strings.format((String)"checking isParent [%s] for [%s]", (Object[])new Object[]{maybeParent, path}));
        return path.startsWith(maybeParent) && path.startsWith(FILE_SEPARATOR, maybeParent.length());
    }

    public boolean equals(Object o) {
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        FileAccessTree that = (FileAccessTree)o;
        return Objects.deepEquals(this.readPaths, that.readPaths) && Objects.deepEquals(this.writePaths, that.writePaths);
    }

    public int hashCode() {
        return Objects.hash(Arrays.hashCode(this.readPaths), Arrays.hashCode(this.writePaths));
    }

    record ExclusiveFileEntitlement(String componentName, String moduleName, FilesEntitlement filesEntitlement) {
    }

    record ExclusivePath(String componentName, Set<String> moduleNames, String path) {
        @Override
        public String toString() {
            return "[[" + this.componentName + "] " + String.valueOf(this.moduleNames) + " [" + this.path + "]]";
        }
    }
}

