import { provideHttpClientTesting } from '@angular/common/http/testing'
import { ComponentFixture, TestBed, waitForAsync } from '@angular/core/testing'
import { provideRouter } from '@angular/router'
import { MessageService, ConfirmationService, Confirmation } from 'primeng/api'

import { EventsService } from '../backend'
import { EventsPanelComponent } from './events-panel.component'
import { ServerSentEventsService, ServerSentEventsTestingService } from '../server-sent-events.service'
import { of } from 'rxjs'
import { provideHttpClient, withInterceptorsFromDi } from '@angular/common/http'
import { SelectChangeEvent } from 'primeng/select'

/**
 * Fake event value.
 */
class TestEventValue {
    public id: any
    public value: any
}

/**
 * Fake event object.
 */
class TestEvent implements SelectChangeEvent {
    public originalEvent: Event
    public value: TestEventValue
    constructor() {
        this.value = new TestEventValue()
    }
}

describe('EventsPanelComponent', () => {
    let component: EventsPanelComponent
    let fixture: ComponentFixture<EventsPanelComponent>
    let sseService: ServerSentEventsService
    let eventsApi: EventsService
    let confirmationService: ConfirmationService

    beforeEach(waitForAsync(() => {
        TestBed.configureTestingModule({
            providers: [
                MessageService,
                ConfirmationService,
                { provide: ServerSentEventsService, useClass: ServerSentEventsTestingService },
                provideHttpClient(withInterceptorsFromDi()),
                provideHttpClientTesting(),
                provideRouter([]),
            ],
        }).compileComponents()
    }))

    beforeEach(() => {
        fixture = TestBed.createComponent(EventsPanelComponent)
        component = fixture.componentInstance
        fixture.detectChanges()
        sseService = fixture.debugElement.injector.get(ServerSentEventsService)
        eventsApi = fixture.debugElement.injector.get(EventsService)
        confirmationService = fixture.debugElement.injector.get(ConfirmationService)
    })

    it('should create', () => {
        expect(component).toBeTruthy()
    })

    it('should establish SSE connection with correct filtering rules', () => {
        component.filter.level = 1
        component.filter.machineId = 2
        component.filter.daemonName = 'dhcp4'
        component.filter.userId = 3

        spyOn(sseService, 'receivePriorityAndMessageEvents').and.returnValue(
            of({
                stream: 'foo',
                originalEvent: {},
            })
        )

        component.ngOnInit()
        fixture.detectChanges()

        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledOnceWith(component.filter)
    })

    it('should renew subscription upon filter changes', () => {
        component.filter.level = 1

        spyOn(sseService, 'receivePriorityAndMessageEvents').and.returnValue(
            of({
                stream: 'foo',
                originalEvent: {},
            })
        )

        component.ngOnInit()
        fixture.detectChanges()

        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledTimes(1)
        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledWith(component.filter)

        component.filter.level = 2
        component.ngOnChanges()

        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledTimes(2)
        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledWith(component.filter)
    })

    it('should re-establish SSE connection on events', () => {
        spyOn(sseService, 'receivePriorityAndMessageEvents').and.returnValue(
            of({
                stream: 'foo',
                originalEvent: {},
            })
        )

        component.ngOnInit()
        fixture.detectChanges()

        // Select specific machine, daemon name and user. In each
        // case, the SSE connection should be re-established with appropriate
        // filtering parameters.

        const event = new TestEvent()

        event.value.id = 1
        component.onMachineSelect(event)
        expect(component.filter.machineId).toBe(1)
        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledWith(component.filter)

        event.value.value = 'dhcp4'
        component.onDaemonNameSelect(event)
        expect(component.filter.daemonName).toBe('dhcp4')
        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledWith(component.filter)

        event.value.id = 5
        component.onUserSelect(event)
        expect(component.filter.userId).toBe(5)
        expect(sseService.receivePriorityAndMessageEvents).toHaveBeenCalledWith(component.filter)
    })

    it('should unsubscribe from events on destroy', () => {
        spyOn(sseService, 'receivePriorityAndMessageEvents').and.returnValue(
            of({
                stream: 'foo',
                originalEvent: {},
            })
        )
        component.ngOnInit()
        fixture.detectChanges()

        spyOn(component.eventSubscription, 'unsubscribe')
        component.ngOnDestroy()
        expect(component.eventSubscription.unsubscribe).toHaveBeenCalled()
    })

    it('should recognize the layout type', () => {
        component.ui = 'table'
        expect(component.isBare).toBeFalse()
        expect(component.isTable).toBeTrue()
        component.ui = 'bare'
        expect(component.isBare).toBeTrue()
        expect(component.isTable).toBeFalse()
    })

    it('should toggle event details expansion', () => {
        expect(component.expandedEvents.size).toBe(0)

        component.onToggleExpandEventDetails(42)
        expect(component.expandedEvents.has(42)).toBeTrue()

        component.onToggleExpandEventDetails(42)
        expect(component.expandedEvents.has(42)).toBeFalse()
    })

    it('should remove the events on onClear', () => {
        spyOn(eventsApi, 'deleteEvents')
        const confirmSpy = spyOn(confirmationService, 'confirm')
        // 1. Confirm that the onClear function clears events when the user clicks
        // the clear button and accepts the confirmation dialog.
        confirmSpy.and.callFake((confirmation: Confirmation) => {
            return confirmation.accept()
        })
        component.onClear()
        expect(eventsApi.deleteEvents).toHaveBeenCalledTimes(1)

        // 2. Confirm that the onClear function doesn't clear events when the user
        // clicks the clear button and then *rejects* the confirmation dialog.
        confirmSpy.and.callFake((confirmation: Confirmation) => {
            if (confirmation.reject) {
                return confirmation.reject()
            }
        })
        component.onClear()
        // deleteEvents should still have been called one time (i.e. *not* called
        // a second time).
        expect(eventsApi.deleteEvents).toHaveBeenCalledTimes(1)
    })
})
