import { Component, OnInit, ViewChild } from '@angular/core'

import { Popover } from 'primeng/popover'

import { SearchService } from '../backend/api/api'
import { FormsModule } from '@angular/forms'
import { InputText } from 'primeng/inputtext'
import { NgIf, NgFor } from '@angular/common'
import { RouterLink } from '@angular/router'
import { EntityLinkComponent } from '../entity-link/entity-link.component'
import { SearchResult } from '../backend'

const recordTypes = ['subnets', 'sharedNetworks', 'hosts', 'machines', 'daemons', 'users', 'groups']

/**
 * Component for handling global search. It provides box
 * for entering search text and a panel with results.
 */
@Component({
    selector: 'app-global-search',
    templateUrl: './global-search.component.html',
    styleUrls: ['./global-search.component.sass'],
    imports: [FormsModule, InputText, Popover, NgIf, NgFor, RouterLink, EntityLinkComponent],
})
export class GlobalSearchComponent implements OnInit {
    @ViewChild('searchResultsBox') searchResultsBox: Popover

    searchText: string
    searchResults: Required<SearchResult>

    constructor(protected searchApi: SearchService) {}

    ngOnInit(): void {
        this.resetResults()
    }

    /**
     * Reset results to be empty.
     */
    resetResults() {
        const searchResults: SearchResult = {}
        for (const rt of recordTypes) {
            searchResults[rt] = { items: [] }
        }
        this.searchResults = searchResults as Required<SearchResult>
    }

    /**
     * Search for records server-side, in the database.
     */
    searchRecords(event) {
        if (event.key === 'Escape') {
            this.resetResults()
            this.searchText = ''
            this.searchResultsBox.hide()
        } else if (this.searchText?.length >= 2 || event.key === 'Enter') {
            this.searchApi.searchRecords(this.searchText).subscribe((data) => {
                this.resetResults()
                for (const k of recordTypes) {
                    this.searchResults[k] = data[k]?.items ? data[k] : { items: [] }
                }
                this.searchResultsBox.show(event)
                // this is a workaround to fix position when content of overlay panel changes
                setTimeout(() => {
                    this.searchResultsBox.align()
                }, 1000)
            })
        } else {
            this.resetResults()
        }
    }

    /**
     * Return true if there are no results.
     */
    noResults() {
        let count = 0
        for (const rt of recordTypes) {
            count += this.searchResults[rt].items.length
        }
        return count === 0
    }
}
