import { ComponentFixture, TestBed, fakeAsync, flush, tick } from '@angular/core/testing'
import { provideHttpClientTesting } from '@angular/common/http/testing'
import { By } from '@angular/platform-browser'
import { BehaviorSubject, Observable, of, throwError } from 'rxjs'

import { ConfirmationService, ToastMessageOptions, MessageService } from 'primeng/api'

import { MachinesPageComponent } from './machines-page.component'
import { AppsVersions, GetMachinesServerToken200Response, Machine, ServicesService, SettingsService } from '../backend'
import { BreadcrumbsComponent } from '../breadcrumbs/breadcrumbs.component'
import { provideNoopAnimations } from '@angular/platform-browser/animations'
import { provideHttpClient, withInterceptorsFromDi } from '@angular/common/http'
import {
    ActivatedRoute,
    ActivatedRouteSnapshot,
    convertToParamMap,
    NavigationEnd,
    Router,
    provideRouter,
} from '@angular/router'
import { Severity, VersionService } from '../version.service'
import createSpyObj = jasmine.createSpyObj
import objectContaining = jasmine.objectContaining
import { AuthService } from '../auth.service'

describe('MachinesPageComponent', () => {
    let component: MachinesPageComponent
    let fixture: ComponentFixture<MachinesPageComponent>
    let servicesApi: any
    let msgService: MessageService
    let router: Router
    let route: ActivatedRoute
    let versionServiceStub: Partial<VersionService>
    let routerEventSubject: BehaviorSubject<NavigationEnd>
    let unauthorizedMachinesCountBadge: HTMLElement
    let getSettingsSpy: jasmine.Spy<() => Observable<any>>
    let getMachinesSpy: jasmine.Spy<
        (
            start?: number,
            limit?: number,
            text?: string,
            authorized?: boolean,
            sortField?: string,
            sortDir?: number
        ) => Observable<{ items?: Array<Partial<Machine>>; total?: number }>
    >
    let getMachinesServerTokenSpy: jasmine.Spy<() => Observable<GetMachinesServerToken200Response>>
    let msgSrvAddSpy: jasmine.Spy<(message: ToastMessageOptions) => void>
    let authService: AuthService

    // prepare responses for api calls
    const getUnauthorizedMachinesResp = {
        items: [
            { hostname: 'aaa', id: 1, address: 'addr1', authorized: false },
            { hostname: 'bbb', id: 2, address: 'addr2', authorized: false },
            { hostname: 'ccc', id: 3, address: 'addr3', authorized: false },
        ],
        total: 3,
    }
    const getAuthorizedMachinesResp = {
        items: [
            { hostname: 'zzz', id: 4, authorized: true },
            { hostname: 'xxx', id: 5, authorized: true },
        ],
        total: 2,
    }
    const getAllMachinesResp = {
        items: [...getUnauthorizedMachinesResp.items, ...getAuthorizedMachinesResp.items],
        total: 5,
    }
    const serverTokenResp = { token: 'ABC' }

    beforeEach(async () => {
        versionServiceStub = {
            sanitizeSemver: () => '1.2.3',
            getCurrentData: () => of({} as AppsVersions),
            getSoftwareVersionFeedback: () => ({ severity: Severity.success, messages: ['test feedback'] }),
        }

        // fake SettingsService
        const registrationEnabled = {
            enableMachineRegistration: true,
        }
        const settingsService = createSpyObj('SettingsService', ['getSettings'])
        getSettingsSpy = settingsService.getSettings.and.returnValue(of(registrationEnabled))

        // fake ServicesService
        servicesApi = createSpyObj('ServicesService', [
            'getMachines',
            'getMachinesServerToken',
            'regenerateMachinesServerToken',
            'getUnauthorizedMachinesCount',
            'updateMachine',
        ])

        getMachinesSpy = servicesApi.getMachines.and.returnValue(of(getAllMachinesResp))
        getMachinesSpy.withArgs(0, 10, null, true, null, null).and.returnValue(of(getAuthorizedMachinesResp))
        getMachinesSpy.withArgs(0, 10, null, false, null, null).and.returnValue(of(getUnauthorizedMachinesResp))

        getMachinesServerTokenSpy = servicesApi.getMachinesServerToken.and.returnValue(of(serverTokenResp))
        servicesApi.getUnauthorizedMachinesCount.and.returnValue(of(3))

        await TestBed.configureTestingModule({
            providers: [
                MessageService,
                ConfirmationService,
                { provide: ServicesService, useValue: servicesApi },
                { provide: VersionService, useValue: versionServiceStub },
                { provide: SettingsService, useValue: settingsService },
                provideHttpClient(withInterceptorsFromDi()),
                provideHttpClientTesting(),
                provideNoopAnimations(),
                provideRouter([
                    {
                        path: 'machines',
                        pathMatch: 'full',
                        redirectTo: 'machines/all',
                    },
                    {
                        path: 'machines/:id',
                        component: MachinesPageComponent,
                    },
                ]),
            ],
        }).compileComponents()

        fixture = TestBed.createComponent(MachinesPageComponent)
        component = fixture.componentInstance
        msgService = fixture.debugElement.injector.get(MessageService)
        fixture.debugElement.injector.get(VersionService)
        route = fixture.debugElement.injector.get(ActivatedRoute)
        route.snapshot = {
            paramMap: convertToParamMap({}),
            queryParamMap: convertToParamMap({}),
        } as ActivatedRouteSnapshot
        router = fixture.debugElement.injector.get(Router)
        routerEventSubject = new BehaviorSubject(new NavigationEnd(1, 'machines', 'machines/all'))
        spyOnProperty(router, 'events').and.returnValue(routerEventSubject)
        msgSrvAddSpy = spyOn(msgService, 'add')
        authService = fixture.debugElement.injector.get(AuthService)
        spyOn(authService, 'superAdmin').and.returnValue(true)

        fixture.detectChanges()
        unauthorizedMachinesCountBadge = fixture.nativeElement.querySelector('.p-togglebutton .p-badge')

        // Wait until table's data loading is finished.
        await fixture.whenStable()
        fixture.detectChanges()
    })

    /**
     * Triggers the component handler called when the route changes.
     * @param params The parameters to pass to the route.
     * @param queryParams The queryParameters to pass to the route.
     */
    function navigate(
        params: { id?: number | string },
        queryParams?: { authorized?: 'true' | 'false'; text?: string }
    ) {
        route.snapshot = {
            paramMap: convertToParamMap(params),
            queryParamMap: convertToParamMap(queryParams || {}),
        } as ActivatedRouteSnapshot

        const queryParamsList = []
        for (const k in queryParams) {
            if (queryParams[k]) {
                queryParamsList.push(`${encodeURIComponent(k)}=${encodeURIComponent(queryParams[k])}`)
            }
        }

        const eid = routerEventSubject.getValue().id + 1
        routerEventSubject.next(
            new NavigationEnd(
                eid,
                `machines/${params.id}?${queryParamsList.join('&')}`,
                `machines/${params.id}?${queryParamsList.join('&')}`
            )
        )
    }

    it('should create', () => {
        expect(component).toBeTruthy()
    })

    it('should not display agent installation instruction if there is an error in getMachinesServerToken', fakeAsync(() => {
        // dialog should be hidden
        expect(component.displayAgentInstallationInstruction).toBeFalse()

        // prepare error response for call to getMachinesServerToken
        const serverTokenRespErr: any = { statusText: 'some error' }
        getMachinesServerTokenSpy.and.returnValue(throwError(() => serverTokenRespErr))

        const showBtnEl = fixture.debugElement.query(By.css('#show-agent-installation-instruction-button'))
        expect(showBtnEl).toBeTruthy()

        // show instruction but error should appear, so it should be handled
        showBtnEl.nativeElement.click()

        tick() // async message service add()

        // check if it is NOT displayed and server token is still empty
        expect(component.displayAgentInstallationInstruction).toBeFalse()
        expect(getMachinesServerTokenSpy).toHaveBeenCalledTimes(1)
        expect(component.serverToken).toBe('')

        // error message should be issued
        expect(msgSrvAddSpy).toHaveBeenCalledOnceWith(
            objectContaining({ severity: 'error', summary: 'Cannot get server token' })
        )
    }))

    it('should display agent installation instruction if all is ok', async () => {
        // dialog should be hidden
        expect(component.displayAgentInstallationInstruction).toBeFalse()

        const showBtnEl = fixture.debugElement.query(By.css('#show-agent-installation-instruction-button'))
        expect(showBtnEl).toBeTruthy()

        // show instruction
        showBtnEl.triggerEventHandler('click', null)
        await fixture.whenStable()
        fixture.detectChanges()

        // check if it is displayed and server token retrieved
        expect(component.displayAgentInstallationInstruction).toBeTrue()
        expect(getMachinesServerTokenSpy).toHaveBeenCalled()
        expect(component.serverToken).toBe('ABC')

        // regenerate server token
        const regenerateMachinesServerTokenResp: any = { token: 'DEF' }
        servicesApi.regenerateMachinesServerToken.and.returnValue(of(regenerateMachinesServerTokenResp))
        component.regenerateServerToken()
        await fixture.whenStable()
        fixture.detectChanges()

        // check if server token has changed
        expect(component.serverToken).toBe('DEF')

        // close instruction
        const closeBtnEl = fixture.debugElement.query(By.css('#close-agent-installation-instruction-button'))
        expect(closeBtnEl).toBeTruthy()
        closeBtnEl.triggerEventHandler('click', null)

        // now dialog should be hidden
        expect(component.displayAgentInstallationInstruction).toBeFalse()
    })

    it('should error msg if regenerateServerToken fails', async () => {
        // dialog should be hidden
        expect(component.displayAgentInstallationInstruction).toBeFalse()

        const showBtnEl = fixture.debugElement.query(By.css('#show-agent-installation-instruction-button'))
        expect(showBtnEl).toBeTruthy()

        // show instruction but error should appear, so it should be handled
        showBtnEl.nativeElement.click()
        await fixture.whenStable()
        fixture.detectChanges()

        // check if it is displayed and server token retrieved
        expect(component.displayAgentInstallationInstruction).toBeTrue()
        expect(getMachinesServerTokenSpy).toHaveBeenCalledTimes(1)
        expect(component.serverToken).toBe('ABC')

        // regenerate server token but it returns error, so in UI token should not change
        const regenerateMachinesServerTokenRespErr: any = { statusText: 'some error' }
        servicesApi.regenerateMachinesServerToken.and.returnValue(
            throwError(() => regenerateMachinesServerTokenRespErr)
        )

        const regenerateBtnDe = fixture.debugElement.query(By.css('#regenerate-server-token-button'))
        expect(regenerateBtnDe).toBeTruthy()
        regenerateBtnDe.nativeElement.click()
        await fixture.whenStable()
        fixture.detectChanges()

        // check if server token has NOT changed
        expect(component.serverToken).toBe('ABC')

        // error message should be issued
        expect(msgSrvAddSpy).toHaveBeenCalledOnceWith(
            objectContaining({ severity: 'error', summary: 'Cannot regenerate server token' })
        )

        // close instruction
        const closeBtnEl = fixture.debugElement.query(By.css('#close-agent-installation-instruction-button'))
        expect(closeBtnEl).toBeTruthy()
        closeBtnEl.nativeElement.click()
        await fixture.whenStable()
        fixture.detectChanges()

        // now dialog should be hidden
        expect(component.displayAgentInstallationInstruction).toBeFalse()
    })

    it('should refresh unauthorized machines count', fakeAsync(() => {
        component.machinesTable().unauthorizedMachinesCountChange.emit(4)
        tick()
        fixture.detectChanges()

        expect(component.unauthorizedMachinesCount).toBe(4)
        expect(unauthorizedMachinesCountBadge.textContent).toBe('4')

        flush()
    }))

    it('should list unauthorized machines requested via URL', fakeAsync(() => {
        // Navigate to Unauthorized machines only view.
        navigate({ id: 'all' }, { authorized: 'false' })
        tick()

        expect(component.showAuthorized()).toBeFalse()
    }))

    it('should button menu click trigger the download handler', fakeAsync(() => {
        // Navigate to Authorized machines only view.
        navigate({ id: 'all' }, { authorized: 'true' })
        tick()
        fixture.detectChanges()

        expect(component.showAuthorized()).toBeTrue()
        expect(component.machinesTable().totalRecords).toBe(2)

        // Show the menu for the machine with ID=4.
        const menuButton = fixture.debugElement.query(By.css('#show-machines-menu-4'))
        expect(menuButton).not.toBeNull()

        menuButton.nativeElement.click()
        flush()
        fixture.detectChanges()

        // Check the dump button.
        // The menu items don't render the IDs in PrimeNG >= 16.
        const dumpButton = fixture.debugElement.query(By.css('#dump-single-machine a'))
        expect(dumpButton).not.toBeNull()

        const downloadSpy = spyOn(component, 'downloadDump').and.returnValue()

        const dumpButtonElement = dumpButton.nativeElement as HTMLButtonElement
        dumpButtonElement.click()
        flush()
        fixture.detectChanges()

        expect(downloadSpy).toHaveBeenCalledOnceWith(objectContaining({ id: 4 }))
    }))

    it('should have breadcrumbs', () => {
        const breadcrumbsElement = fixture.debugElement.query(By.directive(BreadcrumbsComponent))
        expect(breadcrumbsElement).not.toBeNull()
        const breadcrumbsComponent = breadcrumbsElement.componentInstance as BreadcrumbsComponent
        expect(breadcrumbsComponent).not.toBeNull()
        expect(breadcrumbsComponent.items).toHaveSize(2)
        expect(breadcrumbsComponent.items[0].label).toEqual('Services')
        expect(breadcrumbsComponent.items[1].label).toEqual('Machines')
    })

    it('should display a warning about disabled registration', fakeAsync(() => {
        // Prepare response for the call to getMachines().
        const getMachinesResp: any = {
            items: [],
            total: 0,
        }
        getMachinesSpy.withArgs(0, 10, null, true, null, null).and.returnValue(of(getMachinesResp))
        getMachinesSpy.withArgs(0, 10, null, false, null, null).and.returnValue(of(getMachinesResp))

        // Simulate disabled machine registration.
        const getSettingsResp: any = {
            enableMachineRegistration: false,
        }
        getSettingsSpy.and.returnValue(of(getSettingsResp))

        component.ngOnInit()
        tick()
        fixture.detectChanges()

        // Navigate to Authorized machines only view.
        navigate({ id: 'all' }, { authorized: 'true' })
        tick()
        fixture.detectChanges()

        // Initially, we show authorized machines. In that case we don't show a warning.
        expect(component.showAuthorized()).toBeTrue()
        let messages = fixture.debugElement.query(By.css('p-message'))
        expect(messages).toBeFalsy()

        // Show unauthorized machines.
        navigate({ id: 'all' }, { authorized: 'false' })
        tick()
        fixture.detectChanges()

        expect(component.showAuthorized()).toBeFalse()

        // This time we should show the warning that the machines registration is disabled.
        messages = fixture.debugElement.query(By.css('p-message'))
        expect(messages).toBeTruthy()
        expect(messages.nativeElement.innerText).toContain('Registration of new machines is disabled')
    }))

    it('should not display a warning about disabled registration', fakeAsync(() => {
        // Prepare response for the call to getMachines().
        const getMachinesResp: any = {
            items: [],
            total: 0,
        }
        getMachinesSpy.withArgs(0, 10, null, true, null, null).and.returnValue(of(getMachinesResp))
        getMachinesSpy.withArgs(0, 10, null, false, null, null).and.returnValue(of(getMachinesResp))

        // Navigate to Authorized machines only view.
        navigate({ id: 'all' }, { authorized: 'true' })
        tick()
        fixture.detectChanges()

        // Showing authorized machines. The warning is never displayed in such a case.
        expect(component.showAuthorized()).toBeTrue()
        let messages = fixture.debugElement.query(By.css('p-message'))
        expect(messages).toBeFalsy()

        // Show unauthorized machines.
        navigate({ id: 'all' }, { authorized: 'false' })
        tick()
        fixture.detectChanges()

        expect(component.showAuthorized()).toBeFalse()

        // The warning should not be displayed because the registration is enabled.
        messages = fixture.debugElement.query(By.css('p-message'))
        expect(messages).toBeFalsy()
    }))
})
