"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultComponentOptions = void 0;
const core_1 = require("@angular-devkit/core");
const schema_1 = require("@schematics/angular/component/schema");
/**
 * Returns the default options for the `@schematics/angular:component` schematic which would
 * have been specified at project initialization (ng new or ng init).
 *
 * This is necessary because the Angular CLI only exposes the default values for the "--style",
 * "--inlineStyle", "--skipTests" and "--inlineTemplate" options to the "component" schematic.
 */
function getDefaultComponentOptions(project) {
    // Note: Not all options which are available when running "ng new" will be stored in the
    // workspace config. List of options which will be available in the configuration:
    // angular/angular-cli/blob/main/packages/schematics/angular/application/index.ts#L109-L131
    let skipTests = getDefaultComponentOption(project, ['skipTests'], null);
    // In case "skipTests" is not set explicitly, also look for the "spec" option. The "spec"
    // option has been deprecated but can be still used in older Angular CLI projects.
    // See: https://github.com/angular/angular-cli/commit/a12a4e02a4689b5bdbc6e740c0d9865afb55671a
    if (skipTests === null) {
        skipTests = !getDefaultComponentOption(project, ['spec'], true);
    }
    return {
        style: getDefaultComponentOption(project, ['style', 'styleext'], schema_1.Style.Css),
        inlineStyle: getDefaultComponentOption(project, ['inlineStyle'], false),
        inlineTemplate: getDefaultComponentOption(project, ['inlineTemplate'], false),
        skipTests: skipTests,
    };
}
exports.getDefaultComponentOptions = getDefaultComponentOptions;
/**
 * Gets the default value for the specified option. The default options will be determined
 * by looking at the stored schematic options for `@schematics/angular:component` in the
 * CLI workspace configuration.
 */
function getDefaultComponentOption(project, optionNames, fallbackValue) {
    const schematicOptions = (0, core_1.isJsonObject)(project.extensions.schematics || null)
        ? project.extensions.schematics
        : null;
    const defaultSchematic = schematicOptions
        ? schematicOptions['@schematics/angular:component']
        : null;
    for (const optionName of optionNames) {
        if (defaultSchematic && defaultSchematic[optionName] != null) {
            return defaultSchematic[optionName];
        }
    }
    return fallbackValue;
}
//# sourceMappingURL=data:application/json;base64,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