#include "ref_import.h"
#include "tr_local.h"
#include "tr_model.h"
#include "tr_shader.h"

#define LL(x) x = LittleLong(x)

/*
=============================================================

UNCOMPRESSING BONES

=============================================================
*/

#define MC_BITS_X (16)
#define MC_BITS_Y (16)
#define MC_BITS_Z (16)
#define MC_BITS_VECT (16)

#define MC_SCALE_X (1.0f / 64)
#define MC_SCALE_Y (1.0f / 64)
#define MC_SCALE_Z (1.0f / 64)

#define MC_MASK_X ((1 << (MC_BITS_X)) - 1)
#define MC_MASK_Y ((1 << (MC_BITS_Y)) - 1)
#define MC_MASK_Z ((1 << (MC_BITS_Z)) - 1)
#define MC_MASK_VECT ((1 << (MC_BITS_VECT)) - 1)

#define MC_SCALE_VECT (1.0f / (float)((1 << (MC_BITS_VECT - 1)) - 2))

#define MC_POS_X (0)
#define MC_SHIFT_X (0)

#define MC_POS_Y ((((MC_BITS_X)) / 8))
#define MC_SHIFT_Y ((((MC_BITS_X) % 8)))

#define MC_POS_Z ((((MC_BITS_X + MC_BITS_Y)) / 8))
#define MC_SHIFT_Z ((((MC_BITS_X + MC_BITS_Y) % 8)))

#define MC_POS_V11 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z)) / 8))
#define MC_SHIFT_V11 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z) % 8)))

#define MC_POS_V12 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT)) / 8))
#define MC_SHIFT_V12 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT) % 8)))

#define MC_POS_V13 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 2)) / 8))
#define MC_SHIFT_V13 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 2) % 8)))

#define MC_POS_V21 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 3)) / 8))
#define MC_SHIFT_V21 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 3) % 8)))

#define MC_POS_V22 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 4)) / 8))
#define MC_SHIFT_V22 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 4) % 8)))

#define MC_POS_V23 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 5)) / 8))
#define MC_SHIFT_V23 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 5) % 8)))

#define MC_POS_V31 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 6)) / 8))
#define MC_SHIFT_V31 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 6) % 8)))

#define MC_POS_V32 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 7)) / 8))
#define MC_SHIFT_V32 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 7) % 8)))

#define MC_POS_V33 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 8)) / 8))
#define MC_SHIFT_V33 ((((MC_BITS_X + MC_BITS_Y + MC_BITS_Z + MC_BITS_VECT * 8) % 8)))

static void MC_UnCompress(float mat[3][4], const unsigned char *comp) {
	int val;

	val = (int)((const unsigned short *)(comp))[0];
	val -= 1 << (MC_BITS_X - 1);
	mat[0][3] = ((float)(val)) * MC_SCALE_X;

	val = (int)((const unsigned short *)(comp))[1];
	val -= 1 << (MC_BITS_Y - 1);
	mat[1][3] = ((float)(val)) * MC_SCALE_Y;

	val = (int)((const unsigned short *)(comp))[2];
	val -= 1 << (MC_BITS_Z - 1);
	mat[2][3] = ((float)(val)) * MC_SCALE_Z;

	val = (int)((const unsigned short *)(comp))[3];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[0][0] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[4];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[0][1] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[5];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[0][2] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[6];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[1][0] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[7];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[1][1] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[8];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[1][2] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[9];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[2][0] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[10];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[2][1] = ((float)(val)) * MC_SCALE_VECT;

	val = (int)((const unsigned short *)(comp))[11];
	val -= 1 << (MC_BITS_VECT - 1);
	mat[2][2] = ((float)(val)) * MC_SCALE_VECT;
}

qboolean R_LoadMDR(model_t *mod, void *buffer, int filesize, const char *mod_name) {
	int i, j, k, l;
	mdrHeader_t *pinmodel, *mdr;
	mdrFrame_t *frame;
	mdrLOD_t *lod, *curlod;
	mdrSurface_t *surf, *cursurf;
	mdrTriangle_t *tri, *curtri;
	mdrVertex_t *v, *curv;
	mdrWeight_t *weight, *curweight;
	mdrTag_t *tag, *curtag;
	int size;
	shader_t *sh;

	pinmodel = (mdrHeader_t *)buffer;

	pinmodel->version = LittleLong(pinmodel->version);
	if (pinmodel->version != MDR_VERSION) {
		ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has wrong version (%i should be %i)\n", mod_name, pinmodel->version,
				  MDR_VERSION);
		return qfalse;
	}

	size = LittleLong(pinmodel->ofsEnd);

	if (size > filesize) {
		ri.Printf(PRINT_WARNING, "R_LoadMDR: Header of %s is broken. Wrong filesize declared!\n", mod_name);
		return qfalse;
	}

	mod->type = MOD_MDR;

	LL(pinmodel->numFrames);
	LL(pinmodel->numBones);
	LL(pinmodel->ofsFrames);

	// This is a model that uses some type of compressed Bones. We don't want to uncompress every bone for each rendered
	// frame over and over again, we'll uncompress it in this function already, so we must adjust the size of the target
	// mdr.
	if (pinmodel->ofsFrames < 0) {
		// mdrFrame_t is larger than mdrCompFrame_t:
		size += pinmodel->numFrames * sizeof(frame->name);
		// now add enough space for the uncompressed bones.
		size += pinmodel->numFrames * pinmodel->numBones * ((sizeof(mdrBone_t) - sizeof(mdrCompBone_t)));
	}

	// simple bounds check
	if (pinmodel->numBones < 0 ||
		sizeof(*mdr) + pinmodel->numFrames * (sizeof(*frame) + (pinmodel->numBones - 1) * sizeof(*frame->bones)) >
			size) {
		ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
		return qfalse;
	}

	mod->dataSize += size;
	mod->modelData = mdr = ri.Hunk_Alloc(size, h_low);

	// Copy all the values over from the file and fix endian issues in the process, if necessary.

	mdr->ident = LittleLong(pinmodel->ident);
	mdr->version = pinmodel->version; // Don't need to swap byte order on this one, we already did above.
	Q_strncpyz(mdr->name, pinmodel->name, sizeof(mdr->name));
	mdr->numFrames = pinmodel->numFrames;
	mdr->numBones = pinmodel->numBones;
	mdr->numLODs = LittleLong(pinmodel->numLODs);
	mdr->numTags = LittleLong(pinmodel->numTags);
	// We don't care about the other offset values, we'll generate them ourselves while loading.

	mod->numLods = mdr->numLODs;

	if (mdr->numFrames < 1) {
		ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has no frames\n", mod_name);
		return qfalse;
	}

	/* The first frame will be put into the first free space after the header */
	frame = (mdrFrame_t *)(mdr + 1);
	mdr->ofsFrames = (int)((byte *)frame - (byte *)mdr);

	if (pinmodel->ofsFrames < 0) {
		mdrCompFrame_t *cframe;

		// compressed model...
		cframe = (mdrCompFrame_t *)((byte *)pinmodel - pinmodel->ofsFrames);

		for (i = 0; i < mdr->numFrames; i++) {
			for (j = 0; j < 3; j++) {
				frame->bounds[0][j] = LittleFloat(cframe->bounds[0][j]);
				frame->bounds[1][j] = LittleFloat(cframe->bounds[1][j]);
				frame->localOrigin[j] = LittleFloat(cframe->localOrigin[j]);
			}

			frame->radius = LittleFloat(cframe->radius);
			frame->name[0] = '\0'; // No name supplied in the compressed version.

			for (j = 0; j < mdr->numBones; j++) {
#if defined(Q3_BIG_ENDIAN)
				for (k = 0; k < (sizeof(cframe->bones[j].Comp) / 2); k++) {
					// Do swapping for the uncompressing functions. They seem to use shorts
					// values only, so I assume this will work. Never tested it on other
					// platforms, though.

					((unsigned short *)(cframe->bones[j].Comp))[k] =
						LittleShort(((unsigned short *)(cframe->bones[j].Comp))[k]);
				}
#endif
				/* Now do the actual uncompressing */
				MC_UnCompress(frame->bones[j].matrix, cframe->bones[j].Comp);
			}

			// Next Frame...
			// cframe = (mdrCompFrame_t *) &cframe->bones[j];
			// frame = (mdrFrame_t *) &frame->bones[j];
			// this suppress GCC strict-aliasing warning
			{
				// Next Frame...
				mdrCompBone_t *p = &(cframe->bones[j]);
				cframe = (mdrCompFrame_t *)p;
			}

			{
				mdrBone_t *p = &frame->bones[j];
				frame = (mdrFrame_t *)p;
			}
		}
	} else {
		mdrFrame_t *curframe;

		// uncompressed model...
		//

		curframe = (mdrFrame_t *)((byte *)pinmodel + pinmodel->ofsFrames);

		// swap all the frames
		for (i = 0; i < mdr->numFrames; i++) {
			for (j = 0; j < 3; j++) {
				frame->bounds[0][j] = LittleFloat(curframe->bounds[0][j]);
				frame->bounds[1][j] = LittleFloat(curframe->bounds[1][j]);
				frame->localOrigin[j] = LittleFloat(curframe->localOrigin[j]);
			}

			frame->radius = LittleFloat(curframe->radius);
			Q_strncpyz(frame->name, curframe->name, sizeof(frame->name));

			// suppress GCC strict-aliasing warning
#if defined(Q3_BIG_ENDIAN)
			for (j = 0; j < (int)(mdr->numBones * sizeof(mdrBone_t) / 4); j++) {
				((float *)frame->bones)[j] = FloatSwap(&((float *)curframe->bones)[j]);
			}
#else
			for (j = 0; j < mdr->numBones; j++) {
				frame->bones[j] = curframe->bones[j];
			}
#endif

			// curframe = (mdrFrame_t *) &curframe->bones[mdr->numBones];
			// frame = (mdrFrame_t *) &frame->bones[mdr->numBones];
			// suppress GCC strict-aliasing warning
			{
				mdrBone_t *p = &curframe->bones[mdr->numBones];
				curframe = (mdrFrame_t *)p;
			}

			{
				mdrBone_t *p = &frame->bones[mdr->numBones];
				frame = (mdrFrame_t *)p;
			}
		}
	}

	// frame should now point to the first free address after all frames.
	lod = (mdrLOD_t *)frame;
	mdr->ofsLODs = (int)((byte *)lod - (byte *)mdr);

	curlod = (mdrLOD_t *)((byte *)pinmodel + LittleLong(pinmodel->ofsLODs));

	// swap all the LOD's
	for (l = 0; l < mdr->numLODs; l++) {
		// simple bounds check
		if ((byte *)(lod + 1) > (byte *)mdr + size) {
			ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
			return qfalse;
		}

		lod->numSurfaces = LittleLong(curlod->numSurfaces);

		// swap all the surfaces
		surf = (mdrSurface_t *)(lod + 1);
		lod->ofsSurfaces = (int)((byte *)surf - (byte *)lod);
		cursurf = (mdrSurface_t *)((byte *)curlod + LittleLong(curlod->ofsSurfaces));

		for (i = 0; i < lod->numSurfaces; i++) {
			// simple bounds check
			if ((byte *)(surf + 1) > (byte *)mdr + size) {
				ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
				return qfalse;
			}

			// first do some copying stuff

			surf->ident = SF_MDR;
			Q_strncpyz(surf->name, cursurf->name, sizeof(surf->name));
			Q_strncpyz(surf->shader, cursurf->shader, sizeof(surf->shader));

			surf->ofsHeader = (const byte *)mdr - (const byte *)surf;

			surf->numVerts = LittleLong(cursurf->numVerts);
			surf->numTriangles = LittleLong(cursurf->numTriangles);
			// numBoneReferences and BoneReferences generally seem to be unused

			// now do the checks that may fail.
			if (surf->numVerts >= SHADER_MAX_VERTEXES) {
				ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has more than %i verts on %s (%i).\n", mod_name,
						  SHADER_MAX_VERTEXES - 1, surf->name[0] ? surf->name : "a surface", surf->numVerts);
				return qfalse;
			}
			if (surf->numTriangles * 3 >= SHADER_MAX_INDEXES) {
				ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has more than %i triangles on %s (%i).\n", mod_name,
						  (SHADER_MAX_INDEXES / 3) - 1, surf->name[0] ? surf->name : "a surface", surf->numTriangles);
				return qfalse;
			}
			// lowercase the surface name so skin compares are faster
			Q_strlwr(surf->name);

			// register the shaders
			sh = R_FindShader(surf->shader, LIGHTMAP_NONE, qtrue);
			if (sh->defaultShader) {
				surf->shaderIndex = 0;
			} else {
				surf->shaderIndex = sh->index;
			}

			// now copy the vertexes.
			v = (mdrVertex_t *)(surf + 1);
			surf->ofsVerts = (int)((byte *)v - (byte *)surf);
			curv = (mdrVertex_t *)((byte *)cursurf + LittleLong(cursurf->ofsVerts));

			for (j = 0; j < surf->numVerts; j++) {
				LL(curv->numWeights);

				// simple bounds check
				if (curv->numWeights < 0 ||
					(byte *)(v + 1) + (curv->numWeights - 1) * sizeof(*weight) > (byte *)mdr + size) {
					ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
					return qfalse;
				}

				v->normal[0] = LittleFloat(curv->normal[0]);
				v->normal[1] = LittleFloat(curv->normal[1]);
				v->normal[2] = LittleFloat(curv->normal[2]);

				v->texCoords[0] = LittleFloat(curv->texCoords[0]);
				v->texCoords[1] = LittleFloat(curv->texCoords[1]);

				v->numWeights = curv->numWeights;
				weight = &v->weights[0];
				curweight = &curv->weights[0];

				// Now copy all the weights
				for (k = 0; k < v->numWeights; k++) {
					weight->boneIndex = LittleLong(curweight->boneIndex);
					weight->boneWeight = LittleFloat(curweight->boneWeight);

					weight->offset[0] = LittleFloat(curweight->offset[0]);
					weight->offset[1] = LittleFloat(curweight->offset[1]);
					weight->offset[2] = LittleFloat(curweight->offset[2]);

					weight++;
					curweight++;
				}

				v = (mdrVertex_t *)weight;
				curv = (mdrVertex_t *)curweight;
			}

			// we know the offset to the triangles now:
			tri = (mdrTriangle_t *)v;
			surf->ofsTriangles = (int)((byte *)tri - (byte *)surf);
			curtri = (mdrTriangle_t *)((byte *)cursurf + LittleLong(cursurf->ofsTriangles));

			// simple bounds check
			if (surf->numTriangles < 0 || (byte *)(tri + surf->numTriangles) > (byte *)mdr + size) {
				ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
				return qfalse;
			}

			for (j = 0; j < surf->numTriangles; j++) {
				tri->indexes[0] = LittleLong(curtri->indexes[0]);
				tri->indexes[1] = LittleLong(curtri->indexes[1]);
				tri->indexes[2] = LittleLong(curtri->indexes[2]);

				tri++;
				curtri++;
			}

			// tri now points to the end of the surface.
			surf->ofsEnd = (const byte *)tri - (const byte *)surf;
			surf = (mdrSurface_t *)tri;

			// find the next surface.
			cursurf = (mdrSurface_t *)((byte *)cursurf + LittleLong(cursurf->ofsEnd));
		}

		// surf points to the next lod now.
		lod->ofsEnd = (int)((byte *)surf - (byte *)lod);
		lod = (mdrLOD_t *)surf;

		// find the next LOD.
		curlod = (mdrLOD_t *)((byte *)curlod + LittleLong(curlod->ofsEnd));
	}

	// lod points to the first tag now, so update the offset too.
	tag = (mdrTag_t *)lod;
	mdr->ofsTags = (int)((byte *)tag - (byte *)mdr);
	curtag = (mdrTag_t *)((byte *)pinmodel + LittleLong(pinmodel->ofsTags));

	// simple bounds check
	if (mdr->numTags < 0 || (byte *)(tag + mdr->numTags) > (byte *)mdr + size) {
		ri.Printf(PRINT_WARNING, "R_LoadMDR: %s has broken structure.\n", mod_name);
		return qfalse;
	}

	for (i = 0; i < mdr->numTags; i++) {
		tag->boneIndex = LittleLong(curtag->boneIndex);
		Q_strncpyz(tag->name, curtag->name, sizeof(tag->name));

		tag++;
		curtag++;
	}

	// And finally we know the real offset to the end.
	mdr->ofsEnd = (int)((byte *)tag - (byte *)mdr);

	// phew! we're done.

	return qtrue;
}

qhandle_t R_RegisterMDR(const char *name, model_t *mod) {
	char *buf;
	qboolean loaded = qfalse;
	int filesize = ri.FS_ReadFile(name, (void**)&buf);
	int ident;

	if (!buf) {
		mod->type = MOD_BAD;
		return 0;
	}

	ident = LittleLong(*(int *)buf);
	if (ident == MDR_IDENT)
		loaded = R_LoadMDR(mod, buf, filesize, name);

	ri.FS_FreeFile(buf);

	if (!loaded) {
		ri.Printf(PRINT_WARNING, "R_RegisterMDR: couldn't load mdr file %s\n", name);
		mod->type = MOD_BAD;
		return 0;
	}

	return mod->index;
}
