/** @file
  Provides library services to get and set Platform Configuration Database entries.

  PCD Library Class provides a PCD usage macro interface for all PCD types.
  It should be included in any module that uses PCD. If a module uses dynamic/dynamicex
  PCD, module should be linked to a PEIM/DXE library instance to access that PCD.
  If a module uses PatchableInModule type PCD, it also needs the library instance to produce
  LibPatchPcdSetPtr() interface. For FeatureFlag/Fixed PCD, the macro interface is
  translated to a variable or macro that is auto-generated by build tool in
  module's autogen.h/autogen.c.
  The PcdGetXX(), PcdSetXX(), PcdToken(), and PcdGetNextTokenSpace() operations are
  only available prior to ExitBootServices().  If access to PCD values are required
  at runtime, then their values must be collected prior to ExitBootServices().
  There are no restrictions on the use of FeaturePcd(), FixedPcdGetXX(),
  PatchPcdGetXX(), and PatchPcdSetXX().

Copyright (c) 2006 - 2018, Intel Corporation. All rights reserved.<BR>
SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#ifndef __PCD_LIB_H__
#define __PCD_LIB_H__


/**
  Retrieves a token number based on a token name.

  Returns the token number associated with the PCD token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve the token number for.

  @return  The token number associated with the PCD.

**/
#define PcdToken(TokenName)                 _PCD_TOKEN_##TokenName


/**
  Retrieves a Boolean PCD feature flag based on a token name.

  Returns the Boolean value for the PCD feature flag specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a feature flag PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  Boolean value for the PCD feature flag.

**/
#define FeaturePcdGet(TokenName)            _PCD_GET_MODE_BOOL_##TokenName


/**
  Retrieves an 8-bit fixed PCD token value based on a token name.

  Returns the 8-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  8-bit value for the token specified by TokenName.

**/
#define FixedPcdGet8(TokenName)             _PCD_VALUE_##TokenName


/**
  Retrieves a 16-bit fixed PCD token value based on a token name.

  Returns the 16-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  16-bit value for the token specified by TokenName.

**/
#define FixedPcdGet16(TokenName)            _PCD_VALUE_##TokenName


/**
  Retrieves a 32-bit fixed PCD token value based on a token name.

  Returns the 32-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  32-bit value for the token specified by TokenName.

**/
#define FixedPcdGet32(TokenName)            _PCD_VALUE_##TokenName


/**
  Retrieves a 64-bit fixed PCD token value based on a token name.

  Returns the 64-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  64-bit value for the token specified by TokenName.

**/
#define FixedPcdGet64(TokenName)            _PCD_VALUE_##TokenName


/**
  Retrieves a Boolean fixed PCD token value based on a token name.

  Returns the Boolean value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  The Boolean value for the token.

**/
#define FixedPcdGetBool(TokenName)          _PCD_VALUE_##TokenName


/**
  Retrieves a pointer to a fixed PCD token buffer based on a token name.

  Returns a pointer to the buffer for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a fixed at build PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A pointer to the buffer.

**/
#define FixedPcdGetPtr(TokenName)           ((VOID *)_PCD_VALUE_##TokenName)


/**
  Retrieves an 8-bit binary patchable PCD token value based on a token name.

  Returns the 8-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  An 8-bit binary patchable PCD token value.

**/
#define PatchPcdGet8(TokenName)             _gPcd_BinaryPatch_##TokenName

/**
  Retrieves a 16-bit binary patchable PCD token value based on a token name.

  Returns the 16-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A 16-bit binary patchable PCD token value.

**/
#define PatchPcdGet16(TokenName)            _gPcd_BinaryPatch_##TokenName


/**
  Retrieves a 32-bit binary patchable PCD token value based on a token name.

  Returns the 32-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A 32-bit binary patchable PCD token value.

**/
#define PatchPcdGet32(TokenName)            _gPcd_BinaryPatch_##TokenName


/**
  Retrieves a 64-bit binary patchable PCD token value based on a token name.

  Returns the 64-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A 64-bit binary patchable PCD token value.

**/
#define PatchPcdGet64(TokenName)            _gPcd_BinaryPatch_##TokenName


/**
  Retrieves a Boolean binary patchable PCD token value based on a token name.

  Returns the Boolean value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  The Boolean value for the token.

**/
#define PatchPcdGetBool(TokenName)          _gPcd_BinaryPatch_##TokenName


/**
  Retrieves a pointer to a binary patchable PCD token buffer based on a token name.

  Returns a pointer to the buffer for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A pointer to the buffer for the token.

**/
#define PatchPcdGetPtr(TokenName)           ((VOID *)_gPcd_BinaryPatch_##TokenName)


/**
  Sets an 8-bit binary patchable PCD token value based on a token name.

  Sets the 8-bit value for the token specified by TokenName. Value is returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the binary patchable PCD token to set the current value for.
  @param   Value      The 8-bit value to set.

  @return Return the Value that was set.

**/
#define PatchPcdSet8(TokenName, Value)      (_gPcd_BinaryPatch_##TokenName = (Value))


/**
  Sets a 16-bit binary patchable PCD token value based on a token name.

  Sets the 16-bit value for the token specified by TokenName. Value is returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the binary patchable PCD token to set the current value for.
  @param   Value      The 16-bit value to set.

  @return Return the Value that was set.

**/
#define PatchPcdSet16(TokenName, Value)     (_gPcd_BinaryPatch_##TokenName = (Value))


/**
  Sets a 32-bit binary patchable PCD token value based on a token name.

  Sets the 32-bit value for the token specified by TokenName. Value is returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the binary patchable PCD token to set the current value for.
  @param   Value      The 32-bit value to set.

  @return Return the Value that was set.

**/
#define PatchPcdSet32(TokenName, Value)     (_gPcd_BinaryPatch_##TokenName = (Value))


/**
  Sets a 64-bit binary patchable PCD token value based on a token name.

  Sets the 64-bit value for the token specified by TokenName. Value is returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the binary patchable PCD token to set the current value for.
  @param   Value      The 64-bit value to set.

  @return Return the Value that was set.

**/
#define PatchPcdSet64(TokenName, Value)     (_gPcd_BinaryPatch_##TokenName = (Value))


/**
  Sets a Boolean binary patchable PCD token value based on a token name.

  Sets the Boolean value for the token specified by TokenName. Value is returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  @param   TokenName  The name of the binary patchable PCD token to set the current value for.
  @param   Value      The boolean value to set.

  @return Return the Value that was set.

**/
#define PatchPcdSetBool(TokenName, Value)   (_gPcd_BinaryPatch_##TokenName = (Value))


/**
  Sets a pointer to a binary patchable PCD token buffer based on a token name.

  Sets the buffer for the token specified by TokenName.  Buffer is returned.
  If SizeOfBuffer is greater than the maximum size supported by TokenName, then set SizeOfBuffer
  to the maximum size supported by TokenName and return NULL to indicate that the set operation
  was not actually performed.  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be
  set to the maximum size supported by TokenName and NULL must be returned.
  If TokenName is not a valid token in the token space, then the module will not build.
  If TokenName is not a patchable in module PCD, then the module will not build.

  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param   TokenName      The name of the binary patchable PCD token to set the current value for.
  @param   SizeOfBuffer   A pointer to the size, in bytes, of Buffer.
  @param   Buffer         Pointer to the value to set.

  @return Return the pointer to the Buffer that was set.

**/
#define PatchPcdSetPtr(TokenName, Size, Buffer) \
                                            LibPatchPcdSetPtrAndSize (                 \
                                              (VOID *)_gPcd_BinaryPatch_##TokenName,   \
                                              &_gPcd_BinaryPatch_Size_##TokenName,     \
                                              (UINTN)_PCD_PATCHABLE_##TokenName##_SIZE, \
                                              (Size),                                  \
                                              (Buffer)                                 \
                                              )
/**
  Retrieves an 8-bit PCD token value based on a token name.

  Returns the 8-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  8-bit value for the token specified by TokenName.

**/
#define PcdGet8(TokenName)                  _PCD_GET_MODE_8_##TokenName


/**
  Retrieves a 16-bit PCD token value based on a token name.

  Returns the 16-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  16-bit value for the token specified by TokenName.

**/
#define PcdGet16(TokenName)                 _PCD_GET_MODE_16_##TokenName


/**
  Retrieves a 32-bit PCD token value based on a token name.

  Returns the 32-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  32-bit value for the token specified by TokenName.

**/
#define PcdGet32(TokenName)                 _PCD_GET_MODE_32_##TokenName


/**
  Retrieves a 64-bit PCD token value based on a token name.

  Returns the 64-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  64-bit value for the token specified by TokenName.

**/
#define PcdGet64(TokenName)                 _PCD_GET_MODE_64_##TokenName


/**
  Retrieves a pointer to a PCD token buffer based on a token name.

  Returns a pointer to the buffer for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A pointer to the buffer.

**/
#define PcdGetPtr(TokenName)                _PCD_GET_MODE_PTR_##TokenName


/**
  Retrieves a Boolean PCD token value based on a token name.

  Returns the Boolean value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   TokenName  The name of the PCD token to retrieve a current value for.

  @return  A Boolean PCD token value.

**/
#define PcdGetBool(TokenName)               _PCD_GET_MODE_BOOL_##TokenName


/**
  Retrieves the size of a fixed PCD token based on a token name.

  Returns the size of the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param[in]  TokenName  The name of the PCD token to retrieve a current value size for.

  @return     Return the size

**/
#define FixedPcdGetSize(TokenName)    _PCD_SIZE_##TokenName


/**
  Retrieves the size of a binary patchable PCD token based on a token name.

  Returns the size of the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param[in]  TokenName  The name of the PCD token to retrieve a current value size for.

  @return     Return the size

**/
#define PatchPcdGetSize(TokenName)    _gPcd_BinaryPatch_Size_##TokenName


/**
  Retrieves the size of the PCD token based on a token name.

  Returns the size of the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param[in]   TokenName  The name of the PCD token to retrieve a current value size for.

  @return      Return the size

**/
#define PcdGetSize(TokenName)         _PCD_GET_MODE_SIZE_##TokenName


/**
  Retrieve the size of a given PCD token.

  Returns the size of the token specified by TokenNumber and Guid.
  If Guid is NULL, then ASSERT().

  @param[in]  Guid          Pointer to a 128-bit unique value that designates
                            which namespace to retrieve a value from.
  @param[in]  TokenNumber   The PCD token number to retrieve a current value size for.

  @return     Return the size.

**/
#define PcdGetExSize(Guid, TokenName) LibPcdGetExSize ((Guid), PcdTokenEx(Guid,TokenName))

/**
  Sets a 8-bit PCD token value based on a token name.

  Sets the 8-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param TokenName  The name of the PCD token to retrieve a current value for.
  @param Value      The 8-bit value to set.

  @return The status of the set operation.

**/
#define PcdSet8S(TokenName, Value)          _PCD_SET_MODE_8_S_##TokenName    ((Value))

/**
  Sets a 16-bit PCD token value based on a token name.

  Sets the 16-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param TokenName  The name of the PCD token to retrieve a current value for.
  @param Value      The 16-bit value to set.

  @return The status of the set operation.

**/
#define PcdSet16S(TokenName, Value)         _PCD_SET_MODE_16_S_##TokenName    ((Value))

/**
  Sets a 32-bit PCD token value based on a token name.

  Sets the 32-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param TokenName  The name of the PCD token to retrieve a current value for.
  @param Value      The 32-bit value to set.

  @return The status of the set operation.

**/
#define PcdSet32S(TokenName, Value)         _PCD_SET_MODE_32_S_##TokenName    ((Value))

/**
  Sets a 64-bit PCD token value based on a token name.

  Sets the 64-bit value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param TokenName  The name of the PCD token to retrieve a current value for.
  @param Value      The 64-bit value to set.

  @return The status of the set operation.

**/
#define PcdSet64S(TokenName, Value)         _PCD_SET_MODE_64_S_##TokenName    ((Value))

/**
  Sets a pointer to a PCD token buffer based on a token name.

  Sets the buffer for the token specified by TokenName.
  If SizeOfBuffer is greater than the maximum size supported by TokenName,
  then set SizeOfBuffer to the maximum size supported by TokenName and return
  RETURN_INVALID_PARAMETER to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to the maximum size
  supported by TokenName and RETURN_INVALID_PARAMETER must be returned.
  If TokenName is not a valid token in the token space, then the module will not build.

  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param TokenName      The name of the PCD token to set the current value for.
  @param SizeOfBuffer   A pointer to the size, in bytes, of Buffer.
  @param Buffer         A pointer to the buffer to set.

  @return The status of the set operation.

**/
#define PcdSetPtrS(TokenName, SizeOfBuffer, Buffer) \
                                            _PCD_SET_MODE_PTR_S_##TokenName   ((SizeOfBuffer), (Buffer))



/**
  Sets a boolean PCD token value based on a token name.

  Sets the boolean value for the token specified by TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param TokenName  The name of the PCD token to retrieve a current value for.
  @param Value      The boolean value to set.

  @return The status of the set operation.

**/
#define PcdSetBoolS(TokenName, Value)       _PCD_SET_MODE_BOOL_S_##TokenName    ((Value))

/**
  Retrieves a token number based on a GUID and a token name.

  Returns the token number for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space, then the module will not build.

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  Return the token number.

**/
#define PcdTokenEx(Guid,TokenName)  _PCD_TOKEN_EX_##TokenName(Guid)

/**
  Retrieves an 8-bit PCD token value based on a GUID and a token name.

  Returns the 8-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  An 8-bit PCD token value.

**/
#define PcdGetEx8(Guid, TokenName)          LibPcdGetEx8 ((Guid), PcdTokenEx(Guid,TokenName))

/**
  Retrieves a 16-bit PCD token value based on a GUID and a token name.

  Returns the 16-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  A 16-bit PCD token value.

**/
#define PcdGetEx16(Guid, TokenName)         LibPcdGetEx16 ((Guid), PcdTokenEx(Guid,TokenName))


/**
  Retrieves a 32-bit PCD token value based on a GUID and a token name.

  Returns the 32-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  A 32-bit PCD token value.

**/
#define PcdGetEx32(Guid, TokenName)         LibPcdGetEx32 ((Guid), PcdTokenEx(Guid,TokenName))


/**
  Retrieves a 64-bit PCD token value based on a GUID and a token name.

  Returns the 64-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  A 64-bit PCD token value.

**/
#define PcdGetEx64(Guid, TokenName)         LibPcdGetEx64 ((Guid), PcdTokenEx(Guid,TokenName))


/**
  Retrieves a pointer to a PCD token buffer based on a GUID and a token name.

  Returns a pointer to the buffer for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  A pointer to a PCD token buffer.

**/
#define PcdGetExPtr(Guid, TokenName)        LibPcdGetExPtr ((Guid), PcdTokenEx(Guid,TokenName))


/**
  Retrieves a Boolean PCD token value based on a GUID and a token name.

  Returns the Boolean value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param   Guid        Pointer to a 128-bit unique value that designates
                       which namespace to retrieve a value from.
  @param   TokenName   The name of the PCD token to retrieve a current value for.

  @return  A Boolean PCD token value.

**/
#define PcdGetExBool(Guid, TokenName)       LibPcdGetExBool  ((Guid), PcdTokenEx(Guid,TokenName))



/**
  Sets an 8-bit PCD token value based on a GUID and a token name.

  Sets the 8-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param Guid       Pointer to a 128-bit unique value that designates
                    which namespace to retrieve a value from.
  @param TokenName  The name of the PCD token to set the current value for.
  @param Value      The 8-bit value to set.

  @return The status of the set operation.

**/
#define PcdSetEx8S(Guid, TokenName, Value)     LibPcdSetEx8S ((Guid), PcdTokenEx(Guid,TokenName), (Value))

/**
  Sets an 16-bit PCD token value based on a GUID and a token name.

  Sets the 16-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param Guid       Pointer to a 128-bit unique value that designates
                    which namespace to retrieve a value from.
  @param TokenName  The name of the PCD token to set the current value for.
  @param Value      The 16-bit value to set.

  @return The status of the set operation.

**/
#define PcdSetEx16S(Guid, TokenName, Value)    LibPcdSetEx16S ((Guid), PcdTokenEx(Guid,TokenName), (Value))

/**
  Sets an 32-bit PCD token value based on a GUID and a token name.

  Sets the 32-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param Guid       Pointer to a 128-bit unique value that designates
                    which namespace to retrieve a value from.
  @param TokenName  The name of the PCD token to set the current value for.
  @param Value      The 32-bit value to set.

  @return The status of the set operation.

**/
#define PcdSetEx32S(Guid, TokenName, Value)    LibPcdSetEx32S ((Guid), PcdTokenEx(Guid,TokenName), (Value))

/**
  Sets an 64-bit PCD token value based on a GUID and a token name.

  Sets the 64-bit value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param Guid       Pointer to a 128-bit unique value that designates
                    which namespace to retrieve a value from.
  @param TokenName  The name of the PCD token to set the current value for.
  @param Value      The 64-bit value to set.

  @return The status of the set operation.

**/
#define PcdSetEx64S(Guid, TokenName, Value)    LibPcdSetEx64S ((Guid), PcdTokenEx(Guid,TokenName), (Value))

/**
  Sets a pointer to a PCD token buffer based on a GUID and a token name.

  Sets the buffer for the token specified by Guid and TokenName.
  If SizeOfBuffer is greater than the maximum size supported by Guid and TokenName,
  then set SizeOfBuffer to the maximum size supported by Guid and TokenName and return
  RETURN_INVALID_PARAMETER to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to the maximum size
  supported by Guid and TokenName and RETURN_INVALID_PARAMETER must be returned.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param Guid           Pointer to a 128-bit unique value that designates
                        which namespace to retrieve a value from.
  @param TokenName      The name of the PCD token to set the current value for.
  @param SizeOfBuffer   A pointer to the size, in bytes, of Buffer.
  @param Buffer         Pointer to the buffer to set.

  @return The status of the set operation.

**/
#define PcdSetExPtrS(Guid, TokenName, SizeOfBuffer, Buffer) \
                                            LibPcdSetExPtrS ((Guid), PcdTokenEx(Guid,TokenName), (SizeOfBuffer), (Buffer))


/**
  Sets an boolean PCD token value based on a GUID and a token name.

  Sets the boolean value for the token specified by Guid and TokenName.
  If TokenName is not a valid token in the token space specified by Guid,
  then the module will not build.

  If Guid is NULL, then ASSERT().

  @param Guid       Pointer to a 128-bit unique value that designates
                    which namespace to retrieve a value from.
  @param TokenName  The name of the PCD token to set the current value for.
  @param Value      The boolean value to set.

  @return The status of the set operation.

**/
#define PcdSetExBoolS(Guid, TokenName, Value) \
                                            LibPcdSetExBoolS ((Guid), PcdTokenEx(Guid,TokenName), (Value))

/**
  This function provides a means by which SKU support can be established in the PCD infrastructure.

  Sets the current SKU in the PCD database to the value specified by SkuId.  SkuId is returned.

  @param  SkuId   The SKU value that will be used when the PCD service retrieves and sets values
                  associated with a PCD token.

  @return  Return the SKU ID that was set.

**/
UINTN
EFIAPI
LibPcdSetSku (
  IN UINTN   SkuId
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 8-bit value for the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the 8-bit value for the token specified by TokenNumber.

**/
UINT8
EFIAPI
LibPcdGet8 (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 16-bit value for the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the 16-bit value for the token specified by TokenNumber.

**/
UINT16
EFIAPI
LibPcdGet16 (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 32-bit value for the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the 32-bit value for the token specified by TokenNumber.

**/
UINT32
EFIAPI
LibPcdGet32 (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 64-bit value for the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the 64-bit value for the token specified by TokenNumber.

**/
UINT64
EFIAPI
LibPcdGet64 (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the pointer to the buffer of the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the pointer to the token specified by TokenNumber.

**/
VOID *
EFIAPI
LibPcdGetPtr (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the Boolean value of the token specified by TokenNumber.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the Boolean value of the token specified by TokenNumber.

**/
BOOLEAN
EFIAPI
LibPcdGetBool (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve the size of a given PCD token.

  @param[in]  TokenNumber The PCD token number to retrieve a current value for.

  @return Returns the size of the token specified by TokenNumber.

**/
UINTN
EFIAPI
LibPcdGetSize (
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 8-bit value for the token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid         Pointer to a 128-bit unique value that designates
                           which namespace to retrieve a value from.
  @param[in]  TokenNumber  The PCD token number to retrieve a current value for.

  @return Return the UINT8.

**/
UINT8
EFIAPI
LibPcdGetEx8 (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 16-bit value for the token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid         Pointer to a 128-bit unique value that designates
                           which namespace to retrieve a value from.
  @param[in]  TokenNumber  The PCD token number to retrieve a current value for.

  @return Return the UINT16.

**/
UINT16
EFIAPI
LibPcdGetEx16 (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  Returns the 32-bit value for the token specified by TokenNumber and Guid.
  If Guid is NULL, then ASSERT().

  @param[in]  Guid         Pointer to a 128-bit unique value that designates
                           which namespace to retrieve a value from.
  @param[in]  TokenNumber  The PCD token number to retrieve a current value for.

  @return Return the UINT32.

**/
UINT32
EFIAPI
LibPcdGetEx32 (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the 64-bit value for the token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid          Pointer to a 128-bit unique value that designates
                            which namespace to retrieve a value from.
  @param[in]  TokenNumber   The PCD token number to retrieve a current value for.

  @return Return the UINT64.

**/
UINT64
EFIAPI
LibPcdGetEx64 (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the pointer to the buffer of token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid          Pointer to a 128-bit unique value that designates
                            which namespace to retrieve a value from.
  @param[in]  TokenNumber   The PCD token number to retrieve a current value for.

  @return Return the VOID* pointer.

**/
VOID *
EFIAPI
LibPcdGetExPtr (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve a value for a given PCD token.

  Returns the Boolean value of the token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid          Pointer to a 128-bit unique value that designates
                            which namespace to retrieve a value from.
  @param[in]  TokenNumber   The PCD token number to retrieve a current value for.

  @return Return the BOOLEAN.

**/
BOOLEAN
EFIAPI
LibPcdGetExBool (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to retrieve the size of a given PCD token.

  Returns the size of the token specified by TokenNumber and Guid.

  If Guid is NULL, then ASSERT().

  @param[in]  Guid          Pointer to a 128-bit unique value that designates
                            which namespace to retrieve a value from.
  @param[in]  TokenNumber   The PCD token number to retrieve a current value for.

  @return Return the size.

**/
UINTN
EFIAPI
LibPcdGetExSize (
  IN CONST GUID        *Guid,
  IN UINTN             TokenNumber
  );


/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 8-bit value for the token specified by TokenNumber
  to the value specified by Value.

  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 8-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSet8S (
  IN UINTN          TokenNumber,
  IN UINT8          Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 16-bit value for the token specified by TokenNumber
  to the value specified by Value.

  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 16-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSet16S (
  IN UINTN          TokenNumber,
  IN UINT16         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 32-bit value for the token specified by TokenNumber
  to the value specified by Value.

  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 32-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSet32S (
  IN UINTN          TokenNumber,
  IN UINT32         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 64-bit value for the token specified by TokenNumber
  to the value specified by Value.

  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 64-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSet64S (
  IN UINTN          TokenNumber,
  IN UINT64         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets a buffer for the token specified by TokenNumber to the value specified
  by Buffer and SizeOfBuffer. If SizeOfBuffer is greater than the maximum size
  support by TokenNumber, then set SizeOfBuffer to the maximum size supported by
  TokenNumber and return RETURN_INVALID_PARAMETER to indicate that the set operation
  was not actually performed.

  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to the
  maximum size supported by TokenName and RETURN_INVALID_PARAMETER must be returned.

  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[in]      TokenNumber   The PCD token number to set a current value for.
  @param[in, out] SizeOfBuffer  The size, in bytes, of Buffer.
  @param[in]      Buffer        A pointer to the buffer to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetPtrS (
  IN       UINTN    TokenNumber,
  IN OUT   UINTN    *SizeOfBuffer,
  IN CONST VOID     *Buffer
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the boolean value for the token specified by TokenNumber
  to the value specified by Value.

  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The boolean value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetBoolS (
  IN UINTN          TokenNumber,
  IN BOOLEAN        Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 8-bit value for the token specified by TokenNumber
  to the value specified by Value.

  If Guid is NULL, then ASSERT().

  @param[in] Guid           The pointer to a 128-bit unique value that
                            designates which namespace to set a value from.
  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 8-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetEx8S (
  IN CONST GUID     *Guid,
  IN UINTN          TokenNumber,
  IN UINT8          Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 16-bit value for the token specified by TokenNumber
  to the value specified by Value.

  If Guid is NULL, then ASSERT().

  @param[in] Guid           The pointer to a 128-bit unique value that
                            designates which namespace to set a value from.
  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 16-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetEx16S (
  IN CONST GUID     *Guid,
  IN UINTN          TokenNumber,
  IN UINT16         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 32-bit value for the token specified by TokenNumber
  to the value specified by Value.

  If Guid is NULL, then ASSERT().

  @param[in] Guid           The pointer to a 128-bit unique value that
                            designates which namespace to set a value from.
  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 32-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetEx32S (
  IN CONST GUID     *Guid,
  IN UINTN          TokenNumber,
  IN UINT32         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the 64-bit value for the token specified by TokenNumber
  to the value specified by Value.

  If Guid is NULL, then ASSERT().

  @param[in] Guid           The pointer to a 128-bit unique value that
                            designates which namespace to set a value from.
  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The 64-bit value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetEx64S (
  IN CONST GUID     *Guid,
  IN UINTN          TokenNumber,
  IN UINT64         Value
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets a buffer for the token specified by TokenNumber to the value specified by
  Buffer and SizeOfBuffer. If SizeOfBuffer is greater than the maximum size
  support by TokenNumber, then set SizeOfBuffer to the maximum size supported by
  TokenNumber and return RETURN_INVALID_PARAMETER to indicate that the set operation
  was not actually performed.

  If Guid is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[in]      Guid          Pointer to a 128-bit unique value that
                                designates which namespace to set a value from.
  @param[in]      TokenNumber   The PCD token number to set a current value for.
  @param[in, out] SizeOfBuffer  The size, in bytes, of Buffer.
  @param[in]      Buffer        A pointer to the buffer to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetExPtrS (
  IN CONST GUID     *Guid,
  IN       UINTN    TokenNumber,
  IN OUT   UINTN    *SizeOfBuffer,
  IN       VOID     *Buffer
  );

/**
  This function provides a means by which to set a value for a given PCD token.

  Sets the boolean value for the token specified by TokenNumber
  to the value specified by Value.

  If Guid is NULL, then ASSERT().

  @param[in] Guid           The pointer to a 128-bit unique value that
                            designates which namespace to set a value from.
  @param[in] TokenNumber    The PCD token number to set a current value for.
  @param[in] Value          The boolean value to set.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPcdSetExBoolS (
  IN CONST GUID     *Guid,
  IN UINTN          TokenNumber,
  IN BOOLEAN        Value
  );

/**
  This notification function serves two purposes.

  Firstly, it notifies the module that did the registration that the value of this
  PCD token has been set.
  Secondly, it provides a mechanism for the module that did the registration to intercept
  the set operation and override the value been set if necessary. After the invocation of
  the callback function, TokenData will be used by PCD service PEIM or driver to modify th
  internal data in PCD database.

  @param[in]      CallBackGuid    The PCD token GUID being set.
  @param[in]      CallBackToken   The PCD token number being set.
  @param[in, out] TokenData       A pointer to the token data being set.
  @param[in]      TokenDataSize   The size, in bytes, of the data being set.

**/
typedef
VOID
(EFIAPI *PCD_CALLBACK)(
  IN        CONST GUID        *CallBackGuid, OPTIONAL
  IN        UINTN             CallBackToken,
  IN  OUT   VOID              *TokenData,
  IN        UINTN             TokenDataSize
  );


/**
  Set up a notification function that is called when a specified token is set.

  When the token specified by TokenNumber and Guid is set,
  then notification function specified by NotificationFunction is called.
  If Guid is NULL, then the default token space is used.
  If NotificationFunction is NULL, then ASSERT().

  @param[in]  Guid                  Pointer to a 128-bit unique value that designates which
                                    namespace to set a value from.  If NULL, then the default
                                    token space is used.
  @param[in]  TokenNumber           The PCD token number to monitor.
  @param[in]  NotificationFunction  The function to call when the token
                                    specified by Guid and TokenNumber is set.

**/
VOID
EFIAPI
LibPcdCallbackOnSet (
  IN CONST GUID               *Guid,       OPTIONAL
  IN UINTN                    TokenNumber,
  IN PCD_CALLBACK             NotificationFunction
  );


/**
  Disable a notification function that was established with LibPcdCallbackonSet().

  Disable a notification function that was previously established with LibPcdCallbackOnSet().
  If NotificationFunction is NULL, then ASSERT().
  If LibPcdCallbackOnSet() was not previously called with Guid, TokenNumber,
  and NotificationFunction, then ASSERT().

  @param[in]  Guid                 Specify the GUID token space.
  @param[in]  TokenNumber          Specify the token number.
  @param[in]  NotificationFunction The callback function to be unregistered.

**/
VOID
EFIAPI
LibPcdCancelCallback (
  IN CONST GUID               *Guid,       OPTIONAL
  IN UINTN                    TokenNumber,
  IN PCD_CALLBACK             NotificationFunction
  );


/**
  Retrieves the next token in a token space.

  Retrieves the next PCD token number from the token space specified by Guid.
  If Guid is NULL, then the default token space is used.  If TokenNumber is 0,
  then the first token number is returned.  Otherwise, the token number that
  follows TokenNumber in the token space is returned.  If TokenNumber is the last
  token number in the token space, then 0 is returned.

  If TokenNumber is not 0 and is not in the token space specified by Guid, then ASSERT().

  @param[in]  Guid        Pointer to a 128-bit unique value that designates which namespace
                          to set a value from.  If NULL, then the default token space is used.
  @param[in]  TokenNumber The previous PCD token number.  If 0, then retrieves the first PCD
                          token number.

  @return The next valid token number.

**/
UINTN
EFIAPI
LibPcdGetNextToken (
  IN CONST GUID               *Guid,       OPTIONAL
  IN UINTN                    TokenNumber
  );



/**
  Used to retrieve the list of available PCD token space GUIDs.

  Returns the PCD token space GUID that follows TokenSpaceGuid in the list of token spaces
  in the platform.
  If TokenSpaceGuid is NULL, then a pointer to the first PCD token spaces returned.
  If TokenSpaceGuid is the last PCD token space GUID in the list, then NULL is returned.

  @param  TokenSpaceGuid  Pointer to the a PCD token space GUID

  @return The next valid token namespace.

**/
GUID *
EFIAPI
LibPcdGetNextTokenSpace (
  IN CONST GUID  *TokenSpaceGuid
  );


/**
  Sets a value of a patchable PCD entry that is type pointer.

  Sets the PCD entry specified by PatchVariable to the value specified by Buffer
  and SizeOfBuffer.  Buffer is returned.  If SizeOfBuffer is greater than
  MaximumDatumSize, then set SizeOfBuffer to MaximumDatumSize and return
  NULL to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to
  MaximumDatumSize and NULL must be returned.

  If PatchVariable is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[out] PatchVariable     A pointer to the global variable in a module that is
                                the target of the set operation.
  @param[in] MaximumDatumSize   The maximum size allowed for the PCD entry specified by PatchVariable.
  @param[in, out] SizeOfBuffer  A pointer to the size, in bytes, of Buffer.
  @param[in] Buffer             A pointer to the buffer to used to set the target variable.

  @return Return the pointer to the Buffer that was set.

**/
VOID *
EFIAPI
LibPatchPcdSetPtr (
  OUT        VOID       *PatchVariable,
  IN        UINTN       MaximumDatumSize,
  IN OUT    UINTN       *SizeOfBuffer,
  IN CONST  VOID        *Buffer
  );

/**
  Sets a value of a patchable PCD entry that is type pointer.

  Sets the PCD entry specified by PatchVariable to the value specified
  by Buffer and SizeOfBuffer. If SizeOfBuffer is greater than MaximumDatumSize,
  then set SizeOfBuffer to MaximumDatumSize and return RETURN_INVALID_PARAMETER
  to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to
  MaximumDatumSize and RETURN_INVALID_PARAMETER must be returned.

  If PatchVariable is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[out] PatchVariable     A pointer to the global variable in a module that is
                                the target of the set operation.
  @param[in] MaximumDatumSize   The maximum size allowed for the PCD entry specified by PatchVariable.
  @param[in, out] SizeOfBuffer  A pointer to the size, in bytes, of Buffer.
  @param[in] Buffer             A pointer to the buffer to used to set the target variable.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPatchPcdSetPtrS (
  OUT      VOID     *PatchVariable,
  IN       UINTN    MaximumDatumSize,
  IN OUT   UINTN    *SizeOfBuffer,
  IN CONST VOID     *Buffer
  );

/**
  Sets a value and size of a patchable PCD entry that is type pointer.

  Sets the PCD entry specified by PatchVariable to the value specified by Buffer
  and SizeOfBuffer. Buffer is returned.  If SizeOfBuffer is greater than
  MaximumDatumSize, then set SizeOfBuffer to MaximumDatumSize and return
  NULL to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to
  MaximumDatumSize and NULL must be returned.

  If PatchVariable is NULL, then ASSERT().
  If SizeOfPatchVariable is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[out] PatchVariable     A pointer to the global variable in a module that is
                                the target of the set operation.
  @param[out] SizeOfPatchVariable A pointer to the size, in bytes, of PatchVariable.
  @param[in] MaximumDatumSize   The maximum size allowed for the PCD entry specified by PatchVariable.
  @param[in, out] SizeOfBuffer  A pointer to the size, in bytes, of Buffer.
  @param[in] Buffer             A pointer to the buffer to used to set the target variable.

  @return Return the pointer to the Buffer that was set.

**/
VOID *
EFIAPI
LibPatchPcdSetPtrAndSize (
  OUT       VOID        *PatchVariable,
  OUT       UINTN       *SizeOfPatchVariable,
  IN        UINTN       MaximumDatumSize,
  IN OUT    UINTN       *SizeOfBuffer,
  IN CONST  VOID        *Buffer
  );

/**
  Sets a value and size of a patchable PCD entry that is type pointer.

  Sets the PCD entry specified by PatchVariable to the value specified
  by Buffer and SizeOfBuffer. If SizeOfBuffer is greater than MaximumDatumSize,
  then set SizeOfBuffer to MaximumDatumSize and return RETURN_INVALID_PARAMETER
  to indicate that the set operation was not actually performed.
  If SizeOfBuffer is set to MAX_ADDRESS, then SizeOfBuffer must be set to
  MaximumDatumSize and RETURN_INVALID_PARAMETER must be returned.

  If PatchVariable is NULL, then ASSERT().
  If SizeOfPatchVariable is NULL, then ASSERT().
  If SizeOfBuffer is NULL, then ASSERT().
  If SizeOfBuffer > 0 and Buffer is NULL, then ASSERT().

  @param[out] PatchVariable     A pointer to the global variable in a module that is
                                the target of the set operation.
  @param[out] SizeOfPatchVariable A pointer to the size, in bytes, of PatchVariable.
  @param[in] MaximumDatumSize   The maximum size allowed for the PCD entry specified by PatchVariable.
  @param[in, out] SizeOfBuffer  A pointer to the size, in bytes, of Buffer.
  @param[in] Buffer             A pointer to the buffer to used to set the target variable.

  @return The status of the set operation.

**/
RETURN_STATUS
EFIAPI
LibPatchPcdSetPtrAndSizeS (
  OUT      VOID     *PatchVariable,
  OUT      UINTN    *SizeOfPatchVariable,
  IN       UINTN    MaximumDatumSize,
  IN OUT   UINTN    *SizeOfBuffer,
  IN CONST VOID     *Buffer
  );

typedef enum {
  PCD_TYPE_8,
  PCD_TYPE_16,
  PCD_TYPE_32,
  PCD_TYPE_64,
  PCD_TYPE_BOOL,
  PCD_TYPE_PTR
} PCD_TYPE;

typedef struct {
  ///
  /// The returned information associated with the requested TokenNumber. If
  /// TokenNumber is 0, then PcdType is set to PCD_TYPE_8.
  ///
  PCD_TYPE          PcdType;
  ///
  /// The size of the data in bytes associated with the TokenNumber specified. If
  /// TokenNumber is 0, then PcdSize is set 0.
  ///
  UINTN             PcdSize;
  ///
  /// The null-terminated ASCII string associated with a given token. If the
  /// TokenNumber specified was 0, then this field corresponds to the null-terminated
  /// ASCII string associated with the token's namespace Guid. If NULL, there is no
  /// name associated with this request.
  ///
  CHAR8             *PcdName;
} PCD_INFO;


/**
  Retrieve additional information associated with a PCD token.

  This includes information such as the type of value the TokenNumber is associated with as well as possible
  human readable name that is associated with the token.

  If TokenNumber is not in the default token space specified, then ASSERT().

  @param[in]    TokenNumber The PCD token number.
  @param[out]   PcdInfo     The returned information associated with the requested TokenNumber.
                            The caller is responsible for freeing the buffer that is allocated by callee for PcdInfo->PcdName.
**/
VOID
EFIAPI
LibPcdGetInfo (
  IN        UINTN           TokenNumber,
  OUT       PCD_INFO        *PcdInfo
  );

/**
  Retrieve additional information associated with a PCD token.

  This includes information such as the type of value the TokenNumber is associated with as well as possible
  human readable name that is associated with the token.

  If TokenNumber is not in the token space specified by Guid, then ASSERT().

  @param[in]    Guid        The 128-bit unique value that designates the namespace from which to extract the value.
  @param[in]    TokenNumber The PCD token number.
  @param[out]   PcdInfo     The returned information associated with the requested TokenNumber.
                            The caller is responsible for freeing the buffer that is allocated by callee for PcdInfo->PcdName.
**/
VOID
EFIAPI
LibPcdGetInfoEx (
  IN CONST  GUID            *Guid,
  IN        UINTN           TokenNumber,
  OUT       PCD_INFO        *PcdInfo
  );

/**
  Retrieve the currently set SKU Id.

  @return   The currently set SKU Id. If the platform has not set at a SKU Id, then the
            default SKU Id value of 0 is returned. If the platform has set a SKU Id, then the currently set SKU
            Id is returned.
**/
UINTN
EFIAPI
LibPcdGetSku (
  VOID
  );

#endif
