﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/cloudformation/model/RegistrationStatus.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace CloudFormation {
namespace Model {
namespace RegistrationStatusMapper {

static const int COMPLETE_HASH = HashingUtils::HashString("COMPLETE");
static const int IN_PROGRESS_HASH = HashingUtils::HashString("IN_PROGRESS");
static const int FAILED_HASH = HashingUtils::HashString("FAILED");

RegistrationStatus GetRegistrationStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == COMPLETE_HASH) {
    return RegistrationStatus::COMPLETE;
  } else if (hashCode == IN_PROGRESS_HASH) {
    return RegistrationStatus::IN_PROGRESS;
  } else if (hashCode == FAILED_HASH) {
    return RegistrationStatus::FAILED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<RegistrationStatus>(hashCode);
  }

  return RegistrationStatus::NOT_SET;
}

Aws::String GetNameForRegistrationStatus(RegistrationStatus enumValue) {
  switch (enumValue) {
    case RegistrationStatus::NOT_SET:
      return {};
    case RegistrationStatus::COMPLETE:
      return "COMPLETE";
    case RegistrationStatus::IN_PROGRESS:
      return "IN_PROGRESS";
    case RegistrationStatus::FAILED:
      return "FAILED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace RegistrationStatusMapper
}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
