/******************************************************************************
 * dpms.c - Functions to get/set DPMS modes for the Rage 128                  *
 *                                                                            *
 * This module contains the following functions:                              *
 *     void R128_SetDPMS (WORD dpms_state)                                    *
 *     WORD R128_GetDPMS (void)                                               *
 *                                                                            *
 * Copyright (c) 1999 ATI Technologies Inc.  All rights reserved.             *
 ******************************************************************************/
#include <stdio.h>
#include "regdef.h"
#include "main.h"

/******************************************************************************
 * Function:   void R128_SetDPMS (WORD dpms_state)                            *
 *  Purpose:   set DPMS (Display Power Mgmt Signaling) mode                   *
 *    Input:   dpms_state: 0 = active                                         *
 *                         1 = standby                                        *
 *                         2 = suspend                                        *
 *                         3 = off                                            *
 *                         4 = blank the display                              *
 *                             (this is not a DPMS state;                     *
 *                             the DPMS state is left alone)                  *
 *  Returns:    none.                                                         *
 ******************************************************************************/
void R128_SetDPMS (WORD dpms_state)
{
    if (dpms_state > 4) return;

    // map the passed dpms_state values to hsync disable, vsync disable,
    // and blank bits in byte 1 of the CRTC_EXT_CNTL reg
    // bits<2:0> = Blank, VSyncDis, HSyncDis
    // (set other bits of this byte=0)

    // set blanking if dpms_state <> 0
    if (dpms_state == 4)
    {
        dpms_state = (regr8 (CRTC_EXT_CNTL + 1) & 3 | 4);
    }
    else if (dpms_state != 0)
    {
        dpms_state |= 4;
    }

    regw8 (CRTC_EXT_CNTL + 1, dpms_state);    // write the reg byte
    return;

} // R128_SetDPMS ()



/******************************************************************************
 * Function:   WORD R128_GetDPMS (void)                                       *
 *  Purpose:   Get the current DPMS mode                                      *
 *    Input:   none                                                           *
 *  Returns:   Current DPMS mode                                              *
 *                 0 = active                                                 *
 *                 1 = standby                                                *
 *                 2 = suspend                                                *
 *                 3 = off                                                    *
 ******************************************************************************/
WORD R128_GetDPMS (void)
{
    // return value = CRTC_EXT_CNTL reg bits<9:8>
    return ((WORD)(regr8 (CRTC_EXT_CNTL+1) & 3));

} // R128_GetDPMS ()


