/******************************************************************************
 * REGRDWRT.C - Chapter 3 sample code                                         *
 *                                                                            *
 * This module contains functions that read and write the various registers   *
 * of the Rage 128.  Functions in this module are:                            *
 *                                                                            *
 *              void regw (WORD regindex, DWORD data)                         *
 *              void regw16 (WORD regindex, WORD data)                        *
 *              void regw8 (WORD regindex, BYTE data)                         *
 *              void PLL_regw (BYTE index, DWORD data)                        *
 *              void PLL_regw16 (BYTE index, WORD data)                       *
 *              void PLL_regw8 (BYTE index, BYTE data)                        *
 *              DWORD regr (WORD regindex)                                    *
 *              WORD regr16 (WORD regindex)                                   *
 *              BYTE regr8 (WORD regindex)                                    *
 *              DWORD PLL_regr (BYTE regindex)                                *
 *              WORD PLL_regr16 (BYTE regindex)                               *
 *              BYTE PLL_regr8 (BYTE regindex)                                *
 *              void iow8 (DWORD regindex, BYTE data)                         *
 *              BYTE ior8 (DWORD regindex)                                    *
 *                                                                            *
 *  Copyright (c) 1999 ATI Technologies Inc.  All rights reserved.            *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include "defines.h"
#include "main.h"
#include "regdef.h"

/******************************************************************************
 *  regw (WORD regindex, DWORD data)                                          *
 *                                                                            *
 *  Function: This function will provide write access to the memory           *
 *            mapped registers. Each register is 32 bits wide.                *
 *    Inputs: regindex - Register byte offset from start of register block.   *
 *            data - 32bit data value to fill register.                       *
 *   Outputs: 0 - unsuccessful                                                *
 *            1 - successful                                                  *
 ******************************************************************************/
void regw (WORD regindex, DWORD data)
{
    DWORD *appaddr;             // Pointer to actual register.

    // Calculate linear aperture address.
    appaddr = (DWORD *) (R128_AdapterInfo.virtual_REG_BASE + regindex);

    // Write to appaddr.
    *appaddr = data;
    return;

} // regw

/******************************************************************************
 *   void regw16 (WORD regindex, WORD data)                                   *
 *                                                                            *
 *      16 bit version of regw (..)                                           *
 ******************************************************************************/
void regw16 (WORD regindex, WORD data)
{
    WORD *appaddr;
    appaddr = (WORD *) (R128_AdapterInfo.virtual_REG_BASE + regindex);
    *appaddr = data;
    return;
}


/******************************************************************************
 *   void regw8 (WORD regindex, BYTE data)                                    *
 *                                                                            *
 *      8 bit version of regw (..)                                           *
 ******************************************************************************/
void regw8 (WORD regindex, BYTE data)
{
    BYTE *appaddr;

    appaddr = (BYTE *) (R128_AdapterInfo.virtual_REG_BASE + regindex);
    *appaddr = data;
    return;
}

/******************************************************************************
 *  PLL_regw (BYTE index, DWORD data)                                         *
 *                                                                            *
 *  Function: This function will provide write access to the memory           *
 *            mapped PLL registers. Each register is 32 bits wide.            *
 *    Inputs: index - Register byte index into PLL registers                  *
 *            data - 32 bit data value to fill register.                      *
 *   Outputs: NONE                                                            *
 ******************************************************************************/
void PLL_regw (BYTE index, DWORD data)
{
    regw8 (CLOCK_CNTL_INDEX, (index & 0x1F) | PLL_WR_EN);
    regw (CLOCK_CNTL_DATA, data);
    return;
}

/******************************************************************************
 *   void PLL_regw16 (BYTE index, WORD data)                                  *
 *                                                                            *
 *      16 bit version of PLL_regw (..)                                       *
 ******************************************************************************/
void PLL_regw16 (BYTE index, WORD data)
{
    regw8 (CLOCK_CNTL_INDEX, (index & 0x1F) | PLL_WR_EN);
    regw16  (CLOCK_CNTL_DATA, data);
    return;
}

/******************************************************************************
 *   void PLL_regw8 (BYTE index, BYTE data                                    *
 *                                                                            *
 *      16 bit version of PLL_regw (..)                                       *
 ******************************************************************************/
void PLL_regw8 (BYTE index, BYTE data)
{
    regw8 (CLOCK_CNTL_INDEX, (index & 0x1F) | PLL_WR_EN);
    regw8  (CLOCK_CNTL_DATA, data);
    return;
}


/******************************************************************************
 *   regr (WORD regindex)                                                     *
 *                                                                            *
 *  Function: This function will provide read access to the memory            *
 *            mapped registers. Each register is 32 bits wide.                *
 *    Inputs: regindex - Register byte offset from start of register block.   *
 *   Outputs: data - 32 bit contents of register.                             *
 ******************************************************************************/
DWORD regr (WORD regindex)
{
    DWORD *appaddr;             // Pointer to actual register.

    // Determine offset address.

    // Calculate linear aperture address.
    appaddr = (DWORD *) (R128_AdapterInfo.virtual_REG_BASE + regindex);

    // Read appaddr.
    return (*appaddr);

} // regr


/******************************************************************************
 *   WORD regr16 (WORD regindex)                                              *
 *                                                                            *
 *      16 bit version of regr (..)                                           *
 ******************************************************************************/
WORD regr16 (WORD regindex)
{
    WORD *appaddr;             // Pointer to actual register.

    // Determine offset address.

    // Calculate linear aperture address.
    appaddr = (WORD *) (R128_AdapterInfo.virtual_REG_BASE + regindex);

    // Read appaddr.
    return (*appaddr);
}

/******************************************************************************
 *   BYTE regr8 (WORD regindex)                                               *
 *                                                                            *
 *      8 bit version of regr (..)                                            *
 ******************************************************************************/
BYTE regr8 (WORD regindex)
{
    BYTE *appaddr;             // Pointer to actual register.

    // Determine offset address.

    // Calculate linear aperture address.
    appaddr = (BYTE *) (R128_AdapterInfo.virtual_REG_BASE + regindex);

    // Read appaddr.
    return (*appaddr);
}

/***********************************************************************************
 *  PLL_regr (BYTE regindex)                                                       *
 *                                                                                 *
 *  Function: This function will provide read access to the PLL memory             *
 *            mapped registers. Each register is 32 bits wide.                     *
 *    Inputs: regindex - Register byte offset into PLL register block.             *
 *   Outputs: data - 32 bit contents of register.                                  *
 ***********************************************************************************/
DWORD PLL_regr (BYTE regindex)
{
    regw8 (CLOCK_CNTL_INDEX, regindex & 0x1F);
    return (regr (CLOCK_CNTL_DATA));
}


/******************************************************************************
 *   WORD PLL_regr16 (BYTE regindex)                                          *
 *                                                                            *
 *      16 bit version of PLL_regr (..)                                       *
 ******************************************************************************/
WORD PLL_regr16 (BYTE regindex)
{
    regw8 (CLOCK_CNTL_INDEX, regindex & 0x1F);
    return (regr16 (CLOCK_CNTL_DATA));
}

/******************************************************************************
 *   BYTE PLL_regr8 (BYTE regindex)                                           *
 *                                                                            *
 *      8 bit version of PLL_regr8 (..)                                       *
 ******************************************************************************/
BYTE PLL_regr8 (BYTE regindex)
{
    regw8 (CLOCK_CNTL_INDEX, regindex & 0x1F);
    return (regr8 (CLOCK_CNTL_DATA));
}

/******************************************************************************
 *  ior8 (DWORD regindex)                                                     *
 *                                                                            *
 *  Function: reads a BYTE from an I/O port                                   *
 *                                                                            *
 *    Inputs: regindex - byte offset from start of I/O register block.        *
 *   Outputs: data - BYTE contents of I/O port .                              *
 ******************************************************************************/
BYTE ior8 (DWORD regindex)
{
    BYTE data;
    data = inp ((int)(R128_AdapterInfo.IO_BASE + regindex));
    return (data);
} // ior8

/******************************************************************************
 *  iow8 (DWORD regindex, BYTE data)                                          *
 *                                                                            *
 *  Function: writes a BYTE to an I/O mapped register                         *
 *                                                                            *
 *    Inputs: regindex - byte offset from start of I/O register block.        *
 *            data - BYTE contents to be written to register.                 *
 *   Outputs: NONE.                                                           *
 ******************************************************************************/
void iow8 (DWORD regindex, BYTE data)
{
    int address;
    address = (int)(R128_AdapterInfo.IO_BASE + regindex);
    outp (address, (int)data);

    return;
} // iow8

