/****************************************************************************
 * viewimg.c - Functions to load IMG files                                  *
 *                                                                          *
 * This module contains the following functions:                            *
 *                                                                          *
 *          img_handle *get_img_header(char *filename)                      *
 *          WORD load_img(img_handle *himage, int x, int y)                 *
 *                                                                          *
 * Copyright (c) 1999 ATI Technologies Inc.  All rights reserved.           *
 ****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "defines.h"
#include "main.h"


/****************************************************************************
 * get_img_header (char *filename)                                          *
 *  Function: Retrives information from the file header such as:            *
 *            bpp, height, width, format_id.                                *
 *    Inputs: char *filename - name of file to load                         *
 *   Outputs: img_handle * - handle of IMG file                             *
 ****************************************************************************/
img_handle *get_img_header(char *filename)
{
   FILE *handle;
   img_handle *himage;

   // open image file
   handle = fopen (filename, "rb");

   if (handle == NULL)
   {
      return (0);
   }

   // allocate memory for new pointer
   himage = calloc (1,sizeof(img_handle));

   // store filename into himage
   himage->filespec = filename;

   // read header from file
   fread (himage->image, sizeof(img_header), 1, handle);

   // check for IMG id code
   if (strcmpi (himage->image->img_code, "_IMG") != 0)
   {
      free (himage);
      fclose (handle);
      return (0);
   }

   // check format id
   switch(himage->image->format_id)
   {
      case FORMAT_TYPE_8BPP:
      case FORMAT_TYPE_15BPP:
      case FORMAT_TYPE_16BPP:
      case FORMAT_TYPE_32BPP:
      case FORMAT_TYPE_YUV9:
      case FORMAT_TYPE_YUV12:
      case FORMAT_TYPE_VYUY:
      case FORMAT_TYPE_YVYU:
         break;

      default:
         fclose (handle);
         free (himage);
         return (0);
   }

   fclose(handle);
   return(himage);
} // get_img_header ()..


/****************************************************************************
 * load_img (img_handle *himage, WORD x, WORD y)                            *
 *  Function: Transfers the image to the screen.                            *
 *    Inputs: img_handle *himage - handle of image to be loaded             *
 *            WORD x - where to place the file horizontally                 *
 *            WORD y - where to place the file vertically                   *
 *   Outputs: WORD value returning a LOAD_SUCCESSFUL if completed           *
 ****************************************************************************/
WORD load_img(img_handle *himage, int x, int y)
{
    int byte_pitch, i;
    char *dstaddr;
    FILE *handle;

    // ensure engine is idle before writing to video memory
    R128_WaitForIdle ();

    // open image file
    handle = fopen(himage->filespec, "rb");
    if (fseek(handle, 32, SEEK_SET))
    {
        printf ("\nCannot get image header!!");
        exit(1);
    }

    switch(R128_AdapterInfo.bpp)
    {
        case 8:
                byte_pitch = R128_AdapterInfo.pitch;
                break;
        case 15:
        case 16:
                byte_pitch = R128_AdapterInfo.pitch * 16;
                break;
        case 24:
                byte_pitch = R128_AdapterInfo.pitch * 24;
                break;
        case 32:
                byte_pitch = R128_AdapterInfo.pitch * 32;
                break;
        default:
                free(himage);
                fclose(handle);
                return (0);
    }

    dstaddr = (char *) (R128_AdapterInfo.virtual_MEM_BASE +(byte_pitch*y) +
                (x * R128_AdapterInfo.bpp/8));

    for (i = 0; i < himage->image->height; i++)
    {
        fread(dstaddr, himage->image->bytes_per_pixel,
                         himage->image->width, handle);
        dstaddr += byte_pitch;
    }

    // free allocated memory to pointer "himage"
    free(himage);

    // close file
    fclose(handle);

    return (1);
} // load_img












