/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.shard;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.opensearch.action.LatchedActionListener;
import org.opensearch.action.bulk.BackoffPolicy;
import org.opensearch.action.support.GroupedActionListener;
import org.opensearch.common.CheckedFunction;
import org.opensearch.common.concurrent.GatedCloseable;
import org.opensearch.common.logging.Loggers;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.util.UploadListener;
import org.opensearch.core.action.ActionListener;
import org.opensearch.index.engine.EngineException;
import org.opensearch.index.engine.InternalEngine;
import org.opensearch.index.remote.RemoteSegmentTransferTracker;
import org.opensearch.index.shard.CloseableRetryableRefreshListener;
import org.opensearch.index.shard.IndexShard;
import org.opensearch.index.shard.IndexShardState;
import org.opensearch.index.store.RemoteSegmentStoreDirectory;
import org.opensearch.index.store.remote.metadata.RemoteSegmentMetadata;
import org.opensearch.index.translog.Translog;
import org.opensearch.indices.replication.checkpoint.ReplicationCheckpoint;
import org.opensearch.indices.replication.checkpoint.SegmentReplicationCheckpointPublisher;

public final class RemoteStoreRefreshListener
extends CloseableRetryableRefreshListener {
    private final Logger logger;
    private static final int REMOTE_REFRESH_RETRY_BASE_INTERVAL_MILLIS = 1000;
    private static final int REMOTE_REFRESH_RETRY_MAX_INTERVAL_MILLIS = 10000;
    private static final int INVALID_PRIMARY_TERM = -1;
    private static final BackoffPolicy EXPONENTIAL_BACKOFF_POLICY = BackoffPolicy.exponentialEqualJitterBackoff(1000L, 10000L);
    public static final Set<String> EXCLUDE_FILES = Set.of("write.lock");
    public static final int LAST_N_METADATA_FILES_TO_KEEP = 10;
    private final IndexShard indexShard;
    private final Directory storeDirectory;
    private final RemoteSegmentStoreDirectory remoteDirectory;
    private final RemoteSegmentTransferTracker segmentTracker;
    private final Map<String, String> localSegmentChecksumMap;
    private long primaryTerm;
    private volatile Iterator<TimeValue> backoffDelayIterator;
    private final SegmentReplicationCheckpointPublisher checkpointPublisher;

    public RemoteStoreRefreshListener(IndexShard indexShard, SegmentReplicationCheckpointPublisher checkpointPublisher, RemoteSegmentTransferTracker segmentTracker) {
        super(indexShard.getThreadPool());
        this.logger = Loggers.getLogger(this.getClass(), indexShard.shardId(), new String[0]);
        this.indexShard = indexShard;
        this.storeDirectory = indexShard.store().directory();
        this.remoteDirectory = (RemoteSegmentStoreDirectory)((FilterDirectory)((FilterDirectory)indexShard.remoteStore().directory()).getDelegate()).getDelegate();
        this.localSegmentChecksumMap = new HashMap<String, String>();
        RemoteSegmentMetadata remoteSegmentMetadata = null;
        if (indexShard.routingEntry().primary()) {
            try {
                remoteSegmentMetadata = this.remoteDirectory.init();
            }
            catch (IOException e) {
                this.logger.error("Exception while initialising RemoteSegmentStoreDirectory", (Throwable)e);
            }
        }
        this.primaryTerm = remoteSegmentMetadata != null ? remoteSegmentMetadata.getPrimaryTerm() : -1L;
        this.segmentTracker = segmentTracker;
        this.resetBackOffDelayIterator();
        this.checkpointPublisher = checkpointPublisher;
    }

    public void beforeRefresh() throws IOException {
    }

    @Override
    protected void runAfterRefreshExactlyOnce(boolean didRefresh) {
        if (this.shouldSync(didRefresh)) {
            this.segmentTracker.updateLocalRefreshTimeAndSeqNo();
            try {
                if (this.primaryTerm != this.indexShard.getOperationPrimaryTerm()) {
                    this.primaryTerm = this.indexShard.getOperationPrimaryTerm();
                    this.remoteDirectory.init();
                }
                try (GatedCloseable<SegmentInfos> segmentInfosGatedCloseable = this.indexShard.getSegmentInfosSnapshot();){
                    Collection localSegmentsPostRefresh = segmentInfosGatedCloseable.get().files(true);
                    this.updateLocalSizeMapAndTracker(localSegmentsPostRefresh);
                }
            }
            catch (Throwable t) {
                this.logger.error("Exception in runAfterRefreshExactlyOnce() method", t);
            }
        }
    }

    @Override
    protected boolean performAfterRefreshWithPermit(boolean didRefresh) {
        boolean successful = this.shouldSync(didRefresh) ? this.syncSegments() : true;
        return successful;
    }

    private boolean shouldSync(boolean didRefresh) {
        return this.primaryTerm != this.indexShard.getOperationPrimaryTerm() || didRefresh || this.remoteDirectory.getSegmentsUploadedToRemoteStore().isEmpty();
    }

    private boolean syncSegments() {
        if (!this.indexShard.getReplicationTracker().isPrimaryMode() || this.indexShard.state() == IndexShardState.CLOSED) {
            this.logger.trace("Skipped syncing segments with primaryMode={} indexShardState={}", (Object)this.indexShard.getReplicationTracker().isPrimaryMode(), (Object)this.indexShard.state());
            return this.indexShard.state() != IndexShardState.STARTED || !(this.indexShard.getEngine() instanceof InternalEngine);
        }
        final ReplicationCheckpoint checkpoint = this.indexShard.getLatestReplicationCheckpoint();
        this.beforeSegmentsSync();
        final long refreshTimeMs = this.segmentTracker.getLocalRefreshTimeMs();
        final long refreshClockTimeMs = this.segmentTracker.getLocalRefreshClockTimeMs();
        final long refreshSeqNo = this.segmentTracker.getLocalRefreshSeqNo();
        long bytesBeforeUpload = this.segmentTracker.getUploadBytesSucceeded();
        long startTimeInNS = System.nanoTime();
        final AtomicBoolean successful = new AtomicBoolean(false);
        try {
            try {
                if (this.isRefreshAfterCommit()) {
                    this.remoteDirectory.deleteStaleSegmentsAsync(10);
                }
                try (GatedCloseable<SegmentInfos> segmentInfosGatedCloseable = this.indexShard.getSegmentInfosSnapshot();){
                    final SegmentInfos segmentInfos = segmentInfosGatedCloseable.get();
                    assert (segmentInfos.getGeneration() == checkpoint.getSegmentsGen()) : "SegmentInfos generation: " + segmentInfos.getGeneration() + " does not match metadata generation: " + checkpoint.getSegmentsGen();
                    final long lastRefreshedCheckpoint = ((InternalEngine)this.indexShard.getEngine()).lastRefreshedCheckpoint();
                    final Collection localSegmentsPostRefresh = segmentInfos.files(true);
                    this.updateLocalSizeMapAndTracker(localSegmentsPostRefresh);
                    CountDownLatch latch = new CountDownLatch(1);
                    LatchedActionListener<Void> segmentUploadsCompletedListener = new LatchedActionListener<Void>(new ActionListener<Void>(){

                        public void onResponse(Void unused) {
                            try {
                                RemoteStoreRefreshListener.this.uploadMetadata(localSegmentsPostRefresh, segmentInfos, checkpoint);
                                RemoteStoreRefreshListener.this.clearStaleFilesFromLocalSegmentChecksumMap(localSegmentsPostRefresh);
                                RemoteStoreRefreshListener.this.onSuccessfulSegmentsSync(refreshTimeMs, refreshClockTimeMs, refreshSeqNo, lastRefreshedCheckpoint, checkpoint);
                                successful.set(true);
                            }
                            catch (Exception e) {
                                RemoteStoreRefreshListener.this.logger.warn("Exception in post new segment upload actions", (Throwable)e);
                            }
                        }

                        public void onFailure(Exception e) {
                            RemoteStoreRefreshListener.this.logger.warn("Exception while uploading new segments to the remote segment store", (Throwable)e);
                        }
                    }, latch);
                    this.uploadNewSegments(localSegmentsPostRefresh, segmentUploadsCompletedListener);
                    latch.await();
                }
                catch (EngineException e) {
                    this.logger.warn("Exception while reading SegmentInfosSnapshot", (Throwable)((Object)e));
                }
            }
            catch (IOException e) {
                this.logger.warn("Exception while uploading new segments to the remote segment store", (Throwable)e);
            }
        }
        catch (Throwable t) {
            this.logger.error("Exception in RemoteStoreRefreshListener.afterRefresh()", t);
        }
        this.updateFinalStatusInSegmentTracker(successful.get(), bytesBeforeUpload, startTimeInNS);
        return successful.get();
    }

    private void clearStaleFilesFromLocalSegmentChecksumMap(Collection<String> localSegmentsPostRefresh) {
        this.localSegmentChecksumMap.keySet().stream().filter(file -> !localSegmentsPostRefresh.contains(file)).collect(Collectors.toSet()).forEach(this.localSegmentChecksumMap::remove);
    }

    private void beforeSegmentsSync() {
        this.segmentTracker.incrementTotalUploadsStarted();
    }

    private void onSuccessfulSegmentsSync(long refreshTimeMs, long refreshClockTimeMs, long refreshSeqNo, long lastRefreshedCheckpoint, ReplicationCheckpoint checkpoint) {
        this.segmentTracker.setLatestUploadedFiles(this.segmentTracker.getLatestLocalFileNameLengthMap().keySet());
        this.updateRemoteRefreshTimeAndSeqNo(refreshTimeMs, refreshClockTimeMs, refreshSeqNo);
        this.resetBackOffDelayIterator();
        ((InternalEngine)this.indexShard.getEngine()).translogManager().setMinSeqNoToKeep(lastRefreshedCheckpoint + 1L);
        this.checkpointPublisher.publish(this.indexShard, checkpoint);
    }

    private void resetBackOffDelayIterator() {
        this.backoffDelayIterator = EXPONENTIAL_BACKOFF_POLICY.iterator();
    }

    @Override
    protected TimeValue getNextRetryInterval() {
        return this.backoffDelayIterator.next();
    }

    @Override
    protected String getRetryThreadPoolName() {
        return "remote_refresh_retry";
    }

    private boolean isRefreshAfterCommit() throws IOException {
        String lastCommittedLocalSegmentFileName = SegmentInfos.getLastCommitSegmentsFileName((Directory)this.storeDirectory);
        return lastCommittedLocalSegmentFileName != null && !this.remoteDirectory.containsFile(lastCommittedLocalSegmentFileName, this.getChecksumOfLocalFile(lastCommittedLocalSegmentFileName));
    }

    void uploadMetadata(Collection<String> localSegmentsPostRefresh, SegmentInfos segmentInfos, ReplicationCheckpoint replicationCheckpoint) throws IOException {
        long maxSeqNo = ((InternalEngine)this.indexShard.getEngine()).currentOngoingRefreshCheckpoint();
        SegmentInfos segmentInfosSnapshot = segmentInfos.clone();
        Map userData = segmentInfosSnapshot.getUserData();
        userData.put("local_checkpoint", String.valueOf(maxSeqNo));
        userData.put("max_seq_no", Long.toString(maxSeqNo));
        segmentInfosSnapshot.setUserData(userData, false);
        Translog.TranslogGeneration translogGeneration = ((InternalEngine)this.indexShard.getEngine()).translogManager().getTranslogGeneration();
        if (translogGeneration == null) {
            throw new UnsupportedOperationException("Encountered null TranslogGeneration while uploading metadata to remote segment store");
        }
        long translogFileGeneration = translogGeneration.translogFileGeneration;
        this.remoteDirectory.uploadMetadata(localSegmentsPostRefresh, segmentInfosSnapshot, this.storeDirectory, translogFileGeneration, replicationCheckpoint);
    }

    private void uploadNewSegments(Collection<String> localSegmentsPostRefresh, ActionListener<Void> listener) {
        Collection filteredFiles = localSegmentsPostRefresh.stream().filter(file -> !this.skipUpload((String)file)).collect(Collectors.toList());
        if (filteredFiles.size() == 0) {
            listener.onResponse(null);
            return;
        }
        ActionListener mappedListener = ActionListener.map(listener, resp -> null);
        GroupedActionListener batchUploadListener = new GroupedActionListener(mappedListener, filteredFiles.size());
        for (String src : filteredFiles) {
            UploadListener statsListener = this.createUploadListener();
            ActionListener aggregatedListener = ActionListener.wrap(resp -> {
                statsListener.onSuccess(src);
                batchUploadListener.onResponse(resp);
            }, ex -> {
                this.logger.warn(() -> new ParameterizedMessage("Exception: [{}] while uploading segment files", ex), (Throwable)ex);
                if (ex instanceof CorruptIndexException) {
                    this.indexShard.failShard(ex.getMessage(), (Exception)ex);
                }
                statsListener.onFailure(src);
                batchUploadListener.onFailure((Exception)ex);
            });
            statsListener.beforeUpload(src);
            this.remoteDirectory.copyFrom(this.storeDirectory, src, IOContext.DEFAULT, (ActionListener<Void>)aggregatedListener);
        }
    }

    private boolean skipUpload(String file) {
        try {
            return EXCLUDE_FILES.contains(file) || this.remoteDirectory.containsFile(file, this.getChecksumOfLocalFile(file));
        }
        catch (IOException e) {
            this.logger.error("Exception while reading checksum of local segment file: {}, ignoring the exception and re-uploading the file", (Object)file);
            return false;
        }
    }

    private String getChecksumOfLocalFile(String file) throws IOException {
        if (!this.localSegmentChecksumMap.containsKey(file)) {
            try (IndexInput indexInput = this.storeDirectory.openInput(file, IOContext.DEFAULT);){
                String checksum = Long.toString(CodecUtil.retrieveChecksum((IndexInput)indexInput));
                this.localSegmentChecksumMap.put(file, checksum);
            }
        }
        return this.localSegmentChecksumMap.get(file);
    }

    private void updateRemoteRefreshTimeAndSeqNo(long refreshTimeMs, long refreshClockTimeMs, long refreshSeqNo) {
        this.segmentTracker.updateRemoteRefreshClockTimeMs(refreshClockTimeMs);
        this.segmentTracker.updateRemoteRefreshTimeMs(refreshTimeMs);
        this.segmentTracker.updateRemoteRefreshSeqNo(refreshSeqNo);
    }

    private void updateLocalSizeMapAndTracker(Collection<String> segmentFiles) {
        this.segmentTracker.updateLatestLocalFileNameLengthMap(segmentFiles, (CheckedFunction<String, Long, IOException>)((CheckedFunction)arg_0 -> ((Directory)this.storeDirectory).fileLength(arg_0)));
    }

    private void updateFinalStatusInSegmentTracker(boolean uploadStatus, long bytesBeforeUpload, long startTimeInNS) {
        if (uploadStatus) {
            long bytesUploaded = this.segmentTracker.getUploadBytesSucceeded() - bytesBeforeUpload;
            long timeTakenInMS = TimeValue.nsecToMSec((long)(System.nanoTime() - startTimeInNS));
            this.segmentTracker.incrementTotalUploadsSucceeded();
            this.segmentTracker.updateUploadBytesMovingAverage(bytesUploaded);
            this.segmentTracker.updateUploadBytesPerSecMovingAverage(bytesUploaded * 1000L / Math.max(1L, timeTakenInMS));
            this.segmentTracker.updateUploadTimeMovingAverage(timeTakenInMS);
        } else {
            this.segmentTracker.incrementTotalUploadsFailed();
        }
    }

    private UploadListener createUploadListener() {
        return new UploadListener(){
            private long uploadStartTime = 0L;

            @Override
            public void beforeUpload(String file) {
                RemoteStoreRefreshListener.this.segmentTracker.addUploadBytesStarted(RemoteStoreRefreshListener.this.segmentTracker.getLatestLocalFileNameLengthMap().get(file));
                this.uploadStartTime = System.currentTimeMillis();
            }

            @Override
            public void onSuccess(String file) {
                RemoteStoreRefreshListener.this.segmentTracker.addUploadBytesSucceeded(RemoteStoreRefreshListener.this.segmentTracker.getLatestLocalFileNameLengthMap().get(file));
                RemoteStoreRefreshListener.this.segmentTracker.addToLatestUploadedFiles(file);
                RemoteStoreRefreshListener.this.segmentTracker.addUploadTimeInMillis(Math.max(1L, System.currentTimeMillis() - this.uploadStartTime));
            }

            @Override
            public void onFailure(String file) {
                RemoteStoreRefreshListener.this.segmentTracker.addUploadBytesFailed(RemoteStoreRefreshListener.this.segmentTracker.getLatestLocalFileNameLengthMap().get(file));
                RemoteStoreRefreshListener.this.segmentTracker.addUploadTimeInMillis(Math.max(1L, System.currentTimeMillis() - this.uploadStartTime));
            }
        };
    }

    @Override
    protected Logger getLogger() {
        return this.logger;
    }

    @Override
    protected boolean isRetryEnabled() {
        return true;
    }
}

