"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.kdfCommitKeyInfo = exports.kdfInfo = void 0;
/*
 * This public interface for constructing info for the extract step of the KDF
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * The Key Derivation Algorithm section
 */
const material_management_1 = require("@aws-crypto/material-management");
const concat_buffers_1 = require("./concat_buffers");
const uint_util_1 = require("./uint_util");
function kdfInfo(suiteId, messageId) {
    /* Precondition: Info for non-committing suites *only*. */
    (0, material_management_1.needs)(material_management_1.NonCommittingAlgorithmSuiteIdentifier[suiteId], 'Committing algorithm suite not supported.');
    return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(suiteId), messageId);
}
exports.kdfInfo = kdfInfo;
/* Since these values are static
 * there is no need to import
 * a fromUtf8 function to convert them.
 *
 * [...Buffer.from('DERIVEKEY')]
 * 1. KeyLabel := DERIVEKEY as UTF-8 encoded bytes
 * [...Buffer.from('COMMITKEY')]
 * 2. CommitLabel := COMMITKEY as UTF-8 encoded bytes
 */
const KEY_LABEL = new Uint8Array([68, 69, 82, 73, 86, 69, 75, 69, 89]);
const COMMIT_LABEL = new Uint8Array([67, 79, 77, 77, 73, 84, 75, 69, 89]);
function kdfCommitKeyInfo(suite) {
    /* Precondition: Info for committing algorithm suites only. */
    (0, material_management_1.needs)(suite.commitment === 'KEY', 'Non committing algorithm suite not supported.');
    return {
        keyLabel: (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(suite.id), KEY_LABEL),
        commitLabel: COMMIT_LABEL.slice(),
    };
}
exports.kdfCommitKeyInfo = kdfCommitKeyInfo;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2RmX2luZm8uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMva2RmX2luZm8udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUV0Qzs7Ozs7Ozs7R0FRRztBQUVILHlFQUt3QztBQUV4QyxxREFBZ0Q7QUFDaEQsMkNBQXNDO0FBRXRDLFNBQWdCLE9BQU8sQ0FDckIsT0FBaUMsRUFDakMsU0FBcUI7SUFFckIsMERBQTBEO0lBQzFELElBQUEsMkJBQUssRUFDSCwyREFBcUMsQ0FBQyxPQUFPLENBQUMsRUFDOUMsMkNBQTJDLENBQzVDLENBQUE7SUFDRCxPQUFPLElBQUEsOEJBQWEsRUFBQyxJQUFBLG9CQUFRLEVBQUMsT0FBTyxDQUFDLEVBQUUsU0FBUyxDQUFDLENBQUE7QUFDcEQsQ0FBQztBQVZELDBCQVVDO0FBRUQ7Ozs7Ozs7O0dBUUc7QUFDSCxNQUFNLFNBQVMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQTtBQUN0RSxNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQTtBQUV6RSxTQUFnQixnQkFBZ0IsQ0FBQyxLQUFxQjtJQUNwRCw4REFBOEQ7SUFDOUQsSUFBQSwyQkFBSyxFQUNILEtBQUssQ0FBQyxVQUFVLEtBQUssS0FBSyxFQUMxQiwrQ0FBK0MsQ0FDaEQsQ0FBQTtJQUNELE9BQU87UUFDTCxRQUFRLEVBQUUsSUFBQSw4QkFBYSxFQUFDLElBQUEsb0JBQVEsRUFBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEVBQUUsU0FBUyxDQUFDO1FBQ3RELFdBQVcsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFO0tBQ2xDLENBQUE7QUFDSCxDQUFDO0FBVkQsNENBVUMifQ==