"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeBacktick = exports.preprocessQuery = exports.getIndexPatternFromRawQuery = exports.composeFinalQuery = exports.buildRawQuery = exports.buildQuery = exports.buildPatternsQuery = void 0;

var _datemath = _interopRequireDefault(require("@elastic/datemath"));

var _lodash = require("lodash");

var _explorer = require("../../common/constants/explorer");

var _shared = require("../../common/constants/shared");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @param literal - string literal that will be put inside single quotes in PPL command
 * @returns string with inner single quotes escaped
 */
const escapeQuotes = literal => {
  return literal.replaceAll("'", "''");
};

const getIndexPatternFromRawQuery = query => {
  const matches = query.match(_shared.PPL_INDEX_REGEX);

  if (matches) {
    return matches[2];
  }

  return '';
}; // insert time filter command and additional commands based on raw query


exports.getIndexPatternFromRawQuery = getIndexPatternFromRawQuery;

const preprocessQuery = ({
  rawQuery,
  startTime,
  endTime,
  timeField,
  isLiveQuery,
  selectedPatternField,
  patternRegex,
  filteredPattern,
  whereClause
}) => {
  var _datemath$parse, _datemath$parse2;

  let finalQuery = '';
  if ((0, _lodash.isEmpty)(rawQuery)) return finalQuery; // convert to moment

  const start = (_datemath$parse = _datemath.default.parse(startTime)) === null || _datemath$parse === void 0 ? void 0 : _datemath$parse.utc().format(_explorer.DATE_PICKER_FORMAT);
  const end = (_datemath$parse2 = _datemath.default.parse(endTime, {
    roundUp: true
  })) === null || _datemath$parse2 === void 0 ? void 0 : _datemath$parse2.utc().format(_explorer.DATE_PICKER_FORMAT);
  const tokens = rawQuery.replaceAll(_shared.PPL_NEWLINE_REGEX, '').match(_shared.PPL_INDEX_INSERT_POINT_REGEX);
  if ((0, _lodash.isEmpty)(tokens)) return finalQuery;
  finalQuery = `${tokens[1]}=${tokens[2]} | where ${timeField} >= '${start}' and ${timeField} <= '${end}'`;

  if (whereClause) {
    finalQuery += ` AND ${whereClause}`;
  }

  finalQuery += tokens[3];

  if (isLiveQuery) {
    finalQuery = finalQuery + ` | sort - ${timeField}`;
  } // if a pattern is selected as filter, build it into finalQuery


  if (selectedPatternField && filteredPattern) finalQuery = buildPatternsQuery(finalQuery, selectedPatternField, patternRegex, filteredPattern);
  return finalQuery;
};

exports.preprocessQuery = preprocessQuery;

const buildPatternsQuery = (baseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  let finalQuery = baseQuery;

  if (selectedPatternField) {
    finalQuery += ` | patterns `;

    if (patternRegex && patternRegex !== _explorer.PPL_DEFAULT_PATTERN_REGEX_FILETER) {
      finalQuery += `pattern='${escapeQuotes(patternRegex)}' `;
    }

    finalQuery += `\`${selectedPatternField}\` `;

    if (filteredPattern) {
      finalQuery += `| where patterns_field='${escapeQuotes(filteredPattern)}'`;
    }
  }

  return finalQuery;
};

exports.buildPatternsQuery = buildPatternsQuery;

const buildQuery = (baseQuery, currQuery) => {
  let fullQuery;

  if (baseQuery) {
    fullQuery = baseQuery;

    if (currQuery) {
      fullQuery += '| ' + currQuery;
    }
  } else {
    fullQuery = currQuery;
  }

  return fullQuery;
};

exports.buildQuery = buildQuery;

const buildRawQuery = (query, appBaseQuery) => {
  const rawQueryStr = query.rawQuery.includes(appBaseQuery) ? query.rawQuery : buildQuery(appBaseQuery, query.rawQuery);
  return rawQueryStr;
};

exports.buildRawQuery = buildRawQuery;

const composeFinalQuery = (curQuery, startingTime, endingTime, timeField, isLiveQuery, appBaseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  const fullQuery = curQuery.includes(appBaseQuery) ? curQuery : buildQuery(appBaseQuery, curQuery);
  if ((0, _lodash.isEmpty)(fullQuery)) return '';
  return preprocessQuery({
    rawQuery: fullQuery,
    startTime: startingTime,
    endTime: endingTime,
    timeField,
    isLiveQuery,
    selectedPatternField,
    patternRegex,
    filteredPattern
  });
};

exports.composeFinalQuery = composeFinalQuery;

const removeBacktick = stringContainsBacktick => {
  if (!stringContainsBacktick) return '';
  return stringContainsBacktick.replace(/`/g, '');
};

exports.removeBacktick = removeBacktick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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