"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LegacyLoggingServer = void 0;
exports.attachMetaData = attachMetaData;
exports.metadataSymbol = void 0;

var _podium = _interopRequireDefault(require("@hapi/podium"));

var _config = require("../../../../legacy/server/config");

var _logging = require("../../../../legacy/server/logging");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const metadataSymbol = Symbol('log message with metadata');
exports.metadataSymbol = metadataSymbol;

function attachMetaData(message, metadata = {}) {
  return {
    [metadataSymbol]: {
      message,
      metadata
    }
  };
}

const isEmptyObject = obj => Object.keys(obj).length === 0;

function getDataToLog(error, metadata, message) {
  if (error) return error;
  if (!isEmptyObject(metadata)) return attachMetaData(message, metadata);
  return message;
}

/**
 * Converts core log level to a one that's known to the legacy platform.
 * @param level Log level from the core.
 */
function getLegacyLogLevel(level) {
  const logLevel = level.id.toLowerCase();

  if (logLevel === 'warn') {
    return 'warning';
  }

  if (logLevel === 'trace') {
    return 'debug';
  }

  return logLevel;
}
/**
 *  The "legacy" OpenSearch Dashboards uses Hapi server + even-better plugin to log, so we should
 *  use the same approach here to make log records generated by the core to look the
 *  same as the rest of the records generated by the "legacy" OpenSearch Dashboards. But to reduce
 *  overhead of having full blown Hapi server instance we create our own "light" version.
 *  @internal
 */


class LegacyLoggingServer {
  // Emulates Hapi's usage of the podium event bus.
  constructor(legacyLoggingConfig) {
    _defineProperty(this, "connections", []);

    _defineProperty(this, "events", new _podium.default(['log', 'request', 'response']));

    _defineProperty(this, "onPostStopCallback", void 0);

    // We set `ops.interval` to max allowed number and `ops` filter to value
    // that doesn't exist to avoid logging of ops at all, if turned on it will be
    // logged by the "legacy" OpenSearch Dashboards.
    const config = {
      logging: { ...legacyLoggingConfig,
        events: { ...legacyLoggingConfig.events,
          ops: '__no-ops__'
        }
      },
      ops: {
        interval: 2147483647
      }
    };
    (0, _logging.setupLogging)(this, _config.Config.withDefaultSchema(config));
  }

  register({
    plugin: {
      register
    },
    options
  }) {
    return register(this, options);
  }

  log({
    level,
    context,
    message,
    error,
    timestamp,
    meta = {}
  }) {
    const {
      tags = [],
      ...metadata
    } = meta;
    this.events.emit('log', {
      data: getDataToLog(error, metadata, message),
      tags: [getLegacyLogLevel(level), ...context.split('.'), ...tags],
      timestamp: timestamp.getTime()
    });
  }

  stop() {
    // Tell the plugin we're stopping.
    if (this.onPostStopCallback !== undefined) {
      this.onPostStopCallback();
    }
  }

  ext(eventName, callback) {
    // method is called by plugin that's being registered.
    if (eventName === 'onPostStop') {
      this.onPostStopCallback = callback;
    } // We don't care about any others the plugin registers

  }

  expose() {// method is called by plugin that's being registered.
  }

}

exports.LegacyLoggingServer = LegacyLoggingServer;