"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.GET_ALERTS_SORT_FILTERS = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

const GET_ALERTS_SORT_FILTERS = {
  MONITOR_NAME: 'monitor_name',
  TRIGGER_NAME: 'trigger_name',
  START_TIME: 'start_time',
  END_TIME: 'end_time',
  ACKNOWLEDGE_TIME: 'acknowledged_time'
};
exports.GET_ALERTS_SORT_FILTERS = GET_ALERTS_SORT_FILTERS;

class AlertService {
  constructor(esDriver) {
    _defineProperty(this, "getAlerts", async (context, req, res) => {
      const {
        from = 0,
        size = 20,
        search = '',
        sortDirection = 'desc',
        sortField = GET_ALERTS_SORT_FILTERS.START_TIME,
        severityLevel = 'ALL',
        alertState = 'ALL',
        monitorIds = [],
        monitorType = 'monitor'
      } = req.query;
      var params;

      switch (sortField) {
        case GET_ALERTS_SORT_FILTERS.MONITOR_NAME:
          params = {
            sortString: `${sortField}.keyword`,
            sortOrder: sortDirection
          };
          break;

        case GET_ALERTS_SORT_FILTERS.TRIGGER_NAME:
          params = {
            sortString: `${sortField}.keyword`,
            sortOrder: sortDirection
          };
          break;

        case GET_ALERTS_SORT_FILTERS.END_TIME:
          params = {
            sortString: sortField,
            sortOrder: sortDirection,
            missing: sortDirection === 'asc' ? '_last' : '_first'
          };
          break;

        case GET_ALERTS_SORT_FILTERS.ACKNOWLEDGE_TIME:
          params = {
            sortString: sortField,
            sortOrder: sortDirection,
            missing: '_last'
          };
          break;

        default:
          // If the sortField parsed from the URL isn't a valid option for this API, use a default option.
          params = {
            sortString: GET_ALERTS_SORT_FILTERS.START_TIME,
            sortOrder: sortDirection
          };
      }

      params.startIndex = from;
      params.size = size;
      params.severityLevel = severityLevel;
      params.alertState = alertState;
      params.searchString = search;
      if (search.trim()) params.searchString = `*${search.trim().split(' ').join('* *')}*`;

      if (monitorIds.length > 0) {
        const idField = monitorType === 'composite' ? 'workflowIds' : 'monitorId';
        params[idField] = !Array.isArray(monitorIds) ? monitorIds : monitorIds[0];
      }

      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alerting.getAlerts', params);
        const alerts = resp.alerts.map(hit => {
          const alert = hit;
          const id = hit.alert_id;
          const version = hit.alert_version;
          return {
            id,
            ...alert,
            version,
            alert_source: !!alert.workflow_id ? 'workflow' : 'monitor'
          };
        });
        const totalAlerts = resp.totalAlerts;
        return res.ok({
          body: {
            ok: true,
            alerts,
            totalAlerts
          }
        });
      } catch (err) {
        console.error(err.message);
        return res.ok({
          body: {
            ok: false,
            err: err.message
          }
        });
      }
    });

    _defineProperty(this, "getWorkflowAlerts", async (context, req, res) => {
      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alerting.getWorkflowAlerts', req.query);
        return res.ok({
          body: {
            ok: true,
            resp
          }
        });
      } catch (err) {
        console.error(err.message);
        return res.ok({
          body: {
            ok: false,
            err: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  }

}

exports.default = AlertService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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