/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.reports;

import jakarta.inject.Inject;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import org.jxls.common.Context;
import org.jxls.util.JxlsHelper;
import org.traccar.api.security.PermissionsService;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.helper.UnitsConverter;
import org.traccar.helper.model.DeviceUtil;
import org.traccar.helper.model.PositionUtil;
import org.traccar.helper.model.UserUtil;
import org.traccar.model.Device;
import org.traccar.model.Position;
import org.traccar.reports.common.ReportUtils;
import org.traccar.reports.model.SummaryReportItem;
import org.traccar.storage.Storage;
import org.traccar.storage.StorageException;
import org.traccar.storage.query.Columns;
import org.traccar.storage.query.Condition;
import org.traccar.storage.query.Order;
import org.traccar.storage.query.Request;

public class SummaryReportProvider {
    private final Config config;
    private final ReportUtils reportUtils;
    private final PermissionsService permissionsService;
    private final Storage storage;

    @Inject
    public SummaryReportProvider(Config config, ReportUtils reportUtils, PermissionsService permissionsService, Storage storage) {
        this.config = config;
        this.reportUtils = reportUtils;
        this.permissionsService = permissionsService;
        this.storage = storage;
    }

    private Position getEdgePosition(long deviceId, Date from, Date to, boolean end) throws StorageException {
        return this.storage.getObject(Position.class, new Request(new Columns.All(), new Condition.And(new Condition.Equals("deviceId", deviceId), new Condition.Between("fixTime", "from", from, "to", to)), new Order("fixTime", end, 1)));
    }

    private Collection<SummaryReportItem> calculateDeviceResult(Device device, Date from, Date to, boolean fast) throws StorageException {
        SummaryReportItem result = new SummaryReportItem();
        result.setDeviceId(device.getId());
        result.setDeviceName(device.getName());
        Position first = null;
        Position last = null;
        if (fast) {
            first = this.getEdgePosition(device.getId(), from, to, false);
            last = this.getEdgePosition(device.getId(), from, to, true);
        } else {
            List<Position> positions = PositionUtil.getPositions(this.storage, device.getId(), from, to);
            for (Position position : positions) {
                if (first == null) {
                    first = position;
                }
                if (position.getSpeed() > result.getMaxSpeed()) {
                    result.setMaxSpeed(position.getSpeed());
                }
                last = position;
            }
        }
        if (first != null && last != null) {
            long durationMilliseconds;
            boolean ignoreOdometer = this.config.getBoolean(Keys.REPORT_IGNORE_ODOMETER);
            result.setDistance(PositionUtil.calculateDistance(first, last, !ignoreOdometer));
            result.setSpentFuel(this.reportUtils.calculateFuel(first, last));
            if (first.hasAttribute("hours") && last.hasAttribute("hours")) {
                durationMilliseconds = last.getLong("hours") - first.getLong("hours");
                result.setEngineHours(durationMilliseconds);
            } else {
                durationMilliseconds = last.getFixTime().getTime() - first.getFixTime().getTime();
            }
            if (durationMilliseconds > 0L) {
                result.setAverageSpeed(UnitsConverter.knotsFromMps(result.getDistance() * 1000.0 / (double)durationMilliseconds));
            }
            if (!ignoreOdometer && first.getDouble("odometer") != 0.0 && last.getDouble("odometer") != 0.0) {
                result.setStartOdometer(first.getDouble("odometer"));
                result.setEndOdometer(last.getDouble("odometer"));
            } else {
                result.setStartOdometer(first.getDouble("totalDistance"));
                result.setEndOdometer(last.getDouble("totalDistance"));
            }
            result.setStartTime(first.getFixTime());
            result.setEndTime(last.getFixTime());
            return List.of(result);
        }
        return List.of();
    }

    private Collection<SummaryReportItem> calculateDeviceResults(Device device, ZonedDateTime from, ZonedDateTime to, boolean daily) throws StorageException {
        boolean fast = Duration.between(from, to).toSeconds() > this.config.getLong(Keys.REPORT_FAST_THRESHOLD);
        ArrayList<SummaryReportItem> results = new ArrayList<SummaryReportItem>();
        if (daily) {
            while (from.truncatedTo(ChronoUnit.DAYS).isBefore(to.truncatedTo(ChronoUnit.DAYS))) {
                ZonedDateTime fromDay = from.truncatedTo(ChronoUnit.DAYS);
                ZonedDateTime nextDay = fromDay.plus(1L, ChronoUnit.DAYS);
                results.addAll(this.calculateDeviceResult(device, Date.from(from.toInstant()), Date.from(nextDay.toInstant()), fast));
                from = nextDay;
            }
            results.addAll(this.calculateDeviceResult(device, Date.from(from.toInstant()), Date.from(to.toInstant()), fast));
        } else {
            results.addAll(this.calculateDeviceResult(device, Date.from(from.toInstant()), Date.from(to.toInstant()), fast));
        }
        return results;
    }

    public Collection<SummaryReportItem> getObjects(long userId, Collection<Long> deviceIds, Collection<Long> groupIds, Date from, Date to, boolean daily) throws StorageException {
        this.reportUtils.checkPeriodLimit(from, to);
        ZoneId tz = UserUtil.getTimezone(this.permissionsService.getServer(), this.permissionsService.getUser(userId)).toZoneId();
        ArrayList<SummaryReportItem> result = new ArrayList<SummaryReportItem>();
        for (Device device : DeviceUtil.getAccessibleDevices(this.storage, userId, deviceIds, groupIds)) {
            Collection<SummaryReportItem> deviceResults = this.calculateDeviceResults(device, from.toInstant().atZone(tz), to.toInstant().atZone(tz), daily);
            for (SummaryReportItem summaryReport : deviceResults) {
                if (summaryReport.getStartTime() == null || summaryReport.getEndTime() == null) continue;
                result.add(summaryReport);
            }
        }
        return result;
    }

    public void getExcel(OutputStream outputStream, long userId, Collection<Long> deviceIds, Collection<Long> groupIds, Date from, Date to, boolean daily) throws StorageException, IOException {
        Collection<SummaryReportItem> summaries = this.getObjects(userId, deviceIds, groupIds, from, to, daily);
        File file = Paths.get(this.config.getString(Keys.TEMPLATES_ROOT), "export", "summary.xlsx").toFile();
        try (FileInputStream inputStream = new FileInputStream(file);){
            Context context = this.reportUtils.initializeContext(userId);
            context.putVar("summaries", summaries);
            context.putVar("from", (Object)from);
            context.putVar("to", (Object)to);
            JxlsHelper.getInstance().setUseFastFormulaProcessor(false).processTemplate((InputStream)inputStream, outputStream, context);
        }
    }
}

